/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *   This file is based on the work by F. Zigterman, fzr@dds.nl
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <kapp.h>
#include <kpanner.h>
#include <qsplitter.h> 

#include "dialogmargin.h"
#include "pagedialog.h"


#define QT_14

CPageDialog::CPageDialog( int dialogType, int buttonMask, int defaultButton,
			  const char *title, QWidget *parent, 
			  const char *name, bool modal )
  : QDialog( parent, name, modal, WStyle_Customize|WStyle_DialogBorder )
{
  //
  // Set dialog caption
  //
  setCaption( title );

  //
  // Set some characteristics that are used to determine the layout
  //
  mDialogLayout = dialogType;
  mButtonMask   = buttonMask | CANCEL; // The cancel button must be present

  //
  // Create the list that holds the page widgets 
  //
  mPageList = new QList<QWidget>;
  mActivePageWidget = 0;
  mActivePageIndex  = -1;

  mTitleList.setAutoDelete( true );

  mTopLayout = new QVBoxLayout( this ); 
  if( mTopLayout == 0 ) { return; }

  if( mDialogLayout == TREE )
  {
    mPanner = new KPanner( this );    
    mPanner->setSeparator( 25 );
    mTopLayout->addWidget( mPanner, 10, AlignRight );

    //
    // Left panner
    //
    mTreeList = new KTreeList( mPanner->child0() );
    QHBoxLayout *hbox = new QHBoxLayout( mPanner->child0() );
    if( hbox == 0 ) { return; }
    hbox->addWidget( mTreeList );
    connect( mTreeList, SIGNAL(highlighted(int)), SLOT(showPage(int)) );

    //
    // Right panner
    //
    mPageLayout = new QVBoxLayout( mPanner->child1() );
    if( mPageLayout == 0 ) { return; }

    mTitleLabel = new QLabel( "Unnamed Dialog", mPanner->child1() );
    if( mTitleLabel == 0 ) { return; }
    mTitleLabel->setFrameStyle( QFrame::Panel|QFrame::Raised );
    mTitleLabel->setText( title );
    ((QFont*)&mTitleLabel->font())->setBold( true );
    mPageLayout->addWidget( mTitleLabel, 0, AlignCenter );

    mPageFrame = new QFrame(  mPanner->child1() );
    if( mPageFrame == 0 ) { return; }
    mPageFrame->setFrameStyle( QFrame::Panel|QFrame::Raised );
    mPageLayout->addWidget( mPageFrame );

    //
    // Make a separator above the action button area.
    //
    QFrame *hline = new QFrame( this );
    hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
    hline->setFixedHeight( hline->sizeHint().height() );
    mTopLayout->addWidget( hline );
  }
  else
  {
    mDialogLayout = TABS;
    mTitleLabel   = 0;
    mTabControl = new KTabCtl( this );
    if( mTabControl == 0 ) { return; }
    mTopLayout->addWidget( mTabControl, 10, AlignRight );
  }
  
  mTopLayout->addSpacing( INNER_SEP-OUTER_SEP+ACTION_SPACE);
  
  //
  // Make the action buttons
  //
  mButtonBox = new KButtonBox( this, KButtonBox::HORIZONTAL, 
			       OUTER_SEP, INNER_SEP );
  if( mButtonMask & HELP )
  {
    mHelpButton = mButtonBox->addButton( i18n("&Help"), false );
    if( defaultButton == HELP ) mHelpButton->setDefault( true );
  }
  if( mButtonMask & DEFAULT )
  {
    mDefaultButton = mButtonBox->addButton( i18n("&Default"), false );
    if( defaultButton == DEFAULT ) mDefaultButton->setDefault( true );
  }

  if( mButtonMask & (HELP|DEFAULT) )
    mButtonBox->addStretch();
 
  if( mButtonMask & PREVIEW )
  {
    mPreviewButton = mButtonBox->addButton( i18n("Pre&view"), false );
    if( defaultButton == PREVIEW ) mPreviewButton->setDefault( true );
  }
  if( mButtonMask & PRINT )
  {
    mPrintButton = mButtonBox->addButton( i18n("&Print"), false );
    if( defaultButton == PRINT ) mPrintButton->setDefault( true );
  }
  if( mButtonMask & OK )
  {
    mOKButton = mButtonBox->addButton( i18n("&OK"), false );
    if( defaultButton == OK ) mOKButton->setDefault( true );
  }
  if( mButtonMask & APPLY )
  {
    mApplyButton = mButtonBox->addButton( i18n("&Apply"), false );
    if( defaultButton == APPLY ) mApplyButton->setDefault( true );
  }
  if( mButtonMask & CANCEL )
  {
    mCancelButton = mButtonBox->addButton( i18n("&Cancel"), false );
    if( defaultButton == CANCEL ) mCancelButton->setDefault( true );
  }

  mButtonBox->layout();
  mButtonBox->setMinimumSize( mButtonBox->sizeHint() );
  mTopLayout->addWidget( mButtonBox );


  //
  // The title label gets the same height as the buttons. Looks nice!
  //
  if( mTitleLabel != 0 )
  {
    mTitleLabel->setFixedHeight( mCancelButton->sizeHint().height() );
  }

  connectButtons();
  connect( kapp, SIGNAL( kdisplayFontChanged() ), SLOT( fontChanged() ) );
}


CPageDialog::~CPageDialog()
{
}


void CPageDialog::fontChanged( void )
{
  mTitleLabel->setFont( kapp->generalFont );
  ((QFont*)&mTitleLabel->font())->setBold( true );
}

void CPageDialog::layout( void )
{
  mTopLayout->activate();
}






void CPageDialog::connectButtons( int buttons )
{
  if( mButtonMask & HELP )
    connect(mHelpButton, SIGNAL(clicked()), SLOT(slotHelp()));
  if( mButtonMask & DEFAULT )
    connect(mDefaultButton, SIGNAL(clicked()), SLOT(slotDefault()));
  if( mButtonMask & OK )
    connect(mOKButton, SIGNAL(clicked()), SLOT(slotOk()));
  if( mButtonMask & APPLY )
    connect(mApplyButton, SIGNAL(clicked()), SLOT(slotApply()));
  if( mButtonMask & CANCEL )
    connect(mCancelButton, SIGNAL(clicked()), SLOT(slotCancel()));
  if( mButtonMask & PRINT )
    connect(mPrintButton, SIGNAL(clicked()), SLOT(slotPrint()));
  if( mButtonMask & PREVIEW )
    connect(mPreviewButton, SIGNAL(clicked()), SLOT(slotPreview()));

}


QFrame *CPageDialog::addPage( int parentindex, const char *title, 
			      const char *longTitle, int position )
{
  QFrame *page = new QFrame( mDialogLayout == TABS ? mTabControl : mPageFrame,
			     title );
  page->setFrameStyle( QFrame::NoFrame );
  page->setEnabled( FALSE );
  page->hide();

  if( mDialogLayout == TREE )
  {
    mTreeList->insertItem( title, 0, parentindex );
    QVBoxLayout *vbox = new QVBoxLayout( mPageFrame, 2 );
    vbox->addWidget( page, 10 );
    vbox->activate();		
    page->lower();
  }
  else
  {
    mTabControl->addTab( page, title );
  }

  mPageList->append( page );
  QString *str = new QString( longTitle == 0 ? title : longTitle );
  if( str != 0 )
  {
    mTitleList.append( str );
  }

  return( page );
}


void CPageDialog::showPage( QWidget *w )
{	
  if( w == 0 )
  {
    return;
  }

  //
  // Bring the new page up front and hide the old
  //
  w->raise();
  if( mActivePageWidget != 0 )
  {
    mActivePageWidget->setEnabled( false );
    mActivePageWidget->hide();
  } 

  mActivePageWidget = w;
  mActivePageWidget->setEnabled( true );
  mActivePageWidget->show();
	
  //
  // Update treelist and local header text
  //
  int index = mPageList->findRef( w );
  if( mDialogLayout == TREE )
  {
    QString *str = mTitleList.at(index);
    mTitleLabel->setText( str == 0 ? w->name() : str->data() );
    //mTitleLabel->setMinimumWidth( mTitleLabel->sizeHint().width() );
    mTreeList->setCurrentItem( index );
  }
	
}

bool CPageDialog::showPage( int index )
{
  //
  // Locate page.
  //
  QWidget *page = mPageList->at( index );
  if( page == 0 ) 
  {	
    warning("Error: No page with index %d", index );
    return( false );
  }
  else
  {
    mActivePageIndex = index;
    showPage( page );
    return( true );
  }
}


int CPageDialog::activePage( void )
{
  return( mActivePageIndex );
}






bool CPageDialog::removePage( char *page, int index )
{
  warning("Removing pages is not yet implemented.");
  return FALSE;
}

void CPageDialog::setButtonsEnabled( int types )
{
  ActiveButtons = types;
  /*
  ApplyButton->setEnabled( types&APPLY != 0 );
  OKButton->setEnabled(    types&OK    != 0 );
  CancelButton->setEnabled(types&CANCEL!= 0 );
  */
}

int  CPageDialog::buttonsEnabled()
{
  return ActiveButtons;
}

void CPageDialog::setButton( int type, const char *text )
{
}


void CPageDialog::slotHelp( void ) 
{
  emit helpClicked();
}

void CPageDialog::slotDefault( void ) 
{
  emit defaultClicked();
}

void CPageDialog::slotOk( void ) 
{
  emit applyClicked();
  accept();
}

void CPageDialog::slotApply( void ) 
{
  emit applyClicked();
}

void CPageDialog::slotPrint( void ) 
{
  emit printClicked();
  accept();
}

void CPageDialog::slotPreview( void )
{
  emit previewClicked();
  accept();
}

void CPageDialog::slotCancel( void ) 
{
  reject();
}



int CPageDialog::pageListStatus( void )
{
  int status = 0;
  if( ! mPageList ) return NOLIST;
  if( ! mPageList->count() ) return NOPAGES;
  if( ! mPageList->current() ) return NOCURRENT;
  if( mPageList->current() == mPageList->getLast() ) status = LASTPAGE;
  if( mPageList->current() == mPageList->getFirst() ) status = FIRSTPAGE;
  return status;
}


bool CPageDialog::eventFilter( QObject *obj, QEvent *e )
{
  if( e->type() == Event_Resize )
  {
    adjustPageSize();
  }

  return( FALSE );
}


void CPageDialog::adjustPageSize( void )
{
  if( mDialogLayout != TREE )
  {
    return;
  }

  int currentPage = mPageList->at();

  //
  // Calculate the minimum size
  //
  int minHeight = 0;
  int minWidth  = 0;
  for( mPageList->first(); mPageList->current(); mPageList->next() )
  {
    int h = mPageList->current()->minimumSize().height();
    int w = mPageList->current()->minimumSize().width();

    if( h > minHeight ) { minHeight = h; }
    if( w > minWidth ) { minWidth = w; }
  }

  //
  // The tree covers 25% of the dialog width so the total width must be
  // W = 0.75x+0.25x  where y=0.75x, z=0.25x.
  //
  float y  = (float)minWidth;
  float x  = y / 0.75;
  float z  = 0.25 * x; 
  minWidth = (int) (y+z) + 8 + mPageFrame->frameWidth() * 2;

  minHeight += mTitleLabel->height() + mPageFrame->frameWidth() * 2;

  mPanner->setMinimumHeight( minHeight );
  mPanner->setMinimumWidth( minWidth );

  mPageList->at( currentPage );
}




