/**************************************************************************
                          cmapwidget.cpp
                      -------------------
    description          : Map Display Widget
    begin                : Fri Oct 22 1999
    copyright            : (C) 1999 by John-Paul Stanford
    email                : john-paul.stanford@virgin.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmapwidget.h"

#include <qdir.h>
#include <qfile.h>
#include <qkeycode.h>
#include <qstack.h>
#include <qimage.h>
#include <qmsgbox.h>
#include <qbitmap.h>

#include "dialogs/dlgmaptextproperties.h"
#include "dialogs/dlgmappathproperties.h"
#include "dialogs/dlgmaproomproperties.h"

#define MOVE_DELAY 2000

const char mapFileVerMajor=1;
const char mapFileVerMinor=3;

static unsigned char delete_cb_bits[] = {       // delete cursor bitmap
   0x0f, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00,
   0x89, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x02, 0x02, 0x00, 0x00,
   0x01, 0x04, 0x00, 0x00, 0x03, 0x08, 0x00, 0x00, 0x05, 0x10, 0x00, 0x00,
   0x09, 0x20, 0x00, 0x00, 0x11, 0x40, 0x00, 0x00, 0x22, 0x60, 0x00, 0x00,
   0x44, 0x50, 0x00, 0x00, 0x88, 0x48, 0x00, 0x00, 0x10, 0x45, 0x00, 0x00,
   0x20, 0x22, 0x00, 0x00, 0x40, 0x12, 0x00, 0x00, 0x80, 0x0a, 0x00, 0x00,
   0x00, 0x07, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};


static unsigned char delete_cm_bits[] = {       // delete cursor bitmap mask
   0x0f, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00,
   0xf9, 0x00, 0x00, 0x00, 0xfc, 0x01, 0x00, 0x00, 0xfe, 0x03, 0x00, 0x00,
   0xff, 0x07, 0x00, 0x00, 0xff, 0x0f, 0x00, 0x00, 0xff, 0x1f, 0x00, 0x00,
   0xff, 0x3f, 0x00, 0x00, 0xff, 0x7f, 0x00, 0x00, 0xfe, 0x7f, 0x00, 0x00,
   0xfc, 0x7f, 0x00, 0x00, 0xf8, 0x7f, 0x00, 0x00, 0xf0, 0x7f, 0x00, 0x00,
   0xe0, 0x3f, 0x00, 0x00, 0xc0, 0x1f, 0x00, 0x00, 0x80, 0x0f, 0x00, 0x00,
   0x00, 0x07, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static unsigned char path1_bits[] = {
   0x10, 0x00, 0x30, 0x00, 0x70, 0x00, 0xf0, 0x00, 0xf0, 0x01, 0xf0, 0x03,
   0xf0, 0x07, 0xf0, 0x0f, 0xf0, 0x01, 0xf0, 0x03, 0xb0, 0x07, 0x12, 0x0f,
   0x03, 0x06, 0x02, 0x02, 0x02, 0x00, 0x07, 0x00};

static unsigned char path2_bits[] = {
   0x10, 0x00, 0x30, 0x00, 0x70, 0x00, 0xf0, 0x00, 0xf0, 0x01, 0xf0, 0x03,
   0xf0, 0x07, 0xf0, 0x0f, 0xf0, 0x01, 0xf0, 0x03, 0xb0, 0x07, 0x12, 0x0f,
   0x05, 0x06, 0x04, 0x02, 0x02, 0x00, 0x07, 0x00};

// Sturcts used to save and load the maps
struct roomHeaderTyp
{
	int numRooms;
};

struct roomTyp
{
	int x;
	int y;
	int type;
	int red,green,blue;
	int useDefaultCol;

};

struct textHeaderTyp
{
	int numText;
};


struct pathHeaderTyp
{
	int numPaths;
};

struct pathTyp
{
	int srcX;
	int srcY;
	signed int srcLevel;
	int destX;
	int destY;
	signed destLevel;
	direction srcDir;
	direction destDir;
};

struct mainHeaderTyp
{	
	int numLevels;
};

CMapText::CMapText (QString str,int x,int y,QFont f)
{
	text = str;
	font = f;
 	col = black;
 	QFontMetrics fm(f);
	setCords(x,y,x+fm.width(str),y-fm.height());	
}

CMapText::~CMapText()
{

}

void CMapText::setColour(QColor colour)
{ col = colour; }

QColor CMapText::getColour(void)
{ return col; }

void CMapText::paint(QPainter *p)
{
	p->setPen(getColour());
	p->setBrush(getColour());
	QColor col = getColour();
	p->setFont(font);
	p->drawText(getLowX(),getLowY(),text,-1);

	if (getSelected())
	{
		p->setPen(getSelectColour());
		p->setBrush(getSelectColour());
		p->drawRect(getHiX()-3,getLowY()-3,6,6);
	}

}

void CMapText::dragPaint(QPainter *p,int offsetx,int offsety)
{
	p->setPen(black);
	p->setFont(font);
	p->drawText(getLowX()+offsetx,getLowY()+offsety,text,-1);
}


void CMapText::lowerPaint(QPainter *p)
{
}

void CMapText::higherPaint(QPainter *p)
{
}

CMapElement *CMapText::copy(void)
{
	CMapText *newText =  new CMapText (text,getLowX(),getLowY(),font);
	newText->setLevel(getLevel());
	newText->setColour(getColour());
	newText->setSelectColour(getSelectColour());
	return newText;
}


void CMapText::setFont(QFont f)
{ font = f; }

QFont CMapText::getFont(void)
{ return font; }


QString CMapText::getText(void)
{ return text; }


int CMapText::getLevel(void)
{ return lvl; }

void CMapText::setText(QString str)
{ text = str; }

void CMapText::setLevel(int level)
{ lvl = level; }


CMapElement *CMapPath::copy(void)
{
	CMapPath *path = new CMapPath (srcDir,destDir,destRoom,srcRoom);
	path->setLowHiX(getLowX(),getHiX());
	path->setLowHiY(getLowY(),getHiY());
	path->setSelected(getSelected());
	path->setLowerColour(getLowerColour());
	path->setDefaultColour(getDefaultColour());
	path->setHigherColour(getHigherColour());
	path->setSelectColour(getSelectColour());

	return path;
}

bool CMapPath::isTwoWay(void)
{
	CMapRoom *srcRoom = getSrcRoom();
	for (CMapPath *path = getDestRoom()->pathList.first() ; path!=0 ; path=getDestRoom()->pathList.next())
	{
		if (path->getDestRoom()==srcRoom)
			return true;
	}
	
	return false;
}

void CMapPath::lowerPaint(QPainter *p)
{
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;

	if (srcDir == UP || srcDir == DOWN || destDir == UP || destDir == DOWN) return;		

	x1 = getLowX()-5;
	y1 = getLowY()-5;
	x2 = getHiX()-5;
	y2 = getHiY()-5;

	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	

	switch(getDestDir())
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	
			
	p->setPen(getLowerColour());
	p->setBrush(getLowerColour());
			
	// Draw the path
	p->drawLine(x1,y1,x5,y5);
	p->drawLine(x5,y5,x6,y6);
	p->drawLine(x2,y2,x6,y6);

	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
}
      	
void CMapPath::higherPaint(QPainter *p)
{
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;

	if (srcDir == UP || srcDir == DOWN || destDir == UP || destDir == DOWN) return;	


	x1 = getLowX()+5;
	y1 = getLowY()+5;
	x2 = getHiX()+5;
	y2 = getHiY()+5;

	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	

	switch(getDestDir())
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	
			
	p->setPen(getHigherColour() );
	p->setBrush(getHigherColour() );
			
	// Draw the path
	p->drawLine(x1,y1,x5,y5);
	p->drawLine(x5,y5,x6,y6);
	p->drawLine(x2,y2,x6,y6);

	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
}


void CMapPath::paint(QPainter *p)
{
	if (srcDir == UP || srcDir == DOWN || destDir == UP || destDir == DOWN) return;

	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;
	int sel_x,sel_y;

	x1 = getLowX();
	y1 = getLowY();
	x2 = getHiX();
	y2 = getHiY();
		
	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	

	switch(getDestDir())
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	

			
	p->setPen(getDefaultColour());
	p->setBrush( getDefaultColour() );
			
	// Draw the path
	p->drawLine(x1,y1,x5,y5);
	p->drawLine(x5,y5,x6,y6);
	p->drawLine(x2,y2,x6,y6);

	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
			
	if (getSelected())
	{
		if (x1 > x2) sel_x=x1-3; else sel_x=x2-3;
		if (y1 > y2) sel_y=y1-3; else sel_y=y2-3;
	
		p->setPen(getSelectColour());
		p->setBrush(QColor(getSelectColour()));
		p->drawRect(sel_x,sel_y,6,6);
	}
}

void CMapPath::dragPaint(QPainter *p,int offsetx,int offsety)
{

	if (srcDir == UP || srcDir == DOWN) return;
	QPointArray a(3);
	signed int y1,x1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6;

	x1 = getLowX() + offsetx;
	y1 = getLowY()+ offsety;
	x2 = getHiX() + offsetx;
	y2 = getHiY()+ offsety;

	switch(getSrcDir())
	{
		case NORTH     : x5 = x1; y5=y1-5;  break;
		case SOUTH     : x5 = x1; y5=y1+5;  break;
		case WEST      : x5 = x1-5; y5=y1; break;
		case EAST      : x5 = x1+5;  y5=y1; break;
		case NORTHEAST : x5 = x1+5; y5=y1-5; break;
		case SOUTHEAST : x5 = x1+5; y5=y1+5; break;
		case SOUTHWEST : x5 = x1-5; y5=y1+5; break;
		case NORTHWEST : x5 = x1-5; y5=y1-5; break;
		case UP		   : break;
		case DOWN      : break;
	}	

	switch(getDestDir())
	{
		case NORTH     : y2--; x3=x2-3; y3=y2-3; x4=x2+3; y4=y2-3 ; x6 = x2  ; y6=y2-5;  break;
		case SOUTH     : y2++; x3=x2-3; y3=y2+3; x4=x2+3; y4=y2+3 ; x6 = x2  ; y6=y2+5;  break;
		case WEST      : x2--; x3=x2-3; y3=y2-3; x4=x2-3; y4=y2+3 ; x6 = x2-5; y6=y2; break;
		case EAST      : x2++; x3=x2+3; y3=y2-3; x4=x2+3; y4=y2+3 ; x6 = x2+5;  y6=y2; break;
		case NORTHEAST : x3=x2+5; y3=y2; x4=x2; y4=y2-5; x6 = x2+5; y6 = y2-5; break;
		case SOUTHEAST : x3=x2; y3=y2+5; x4=x2+5; y4=y2;x6 = x2+5 ; y6 = y2+5; break;
		case SOUTHWEST : x3=x2; y3=y2+5; x4=x2-5; y4=y2;x6 = x2-5 ; y6 = y2+5; break;
		case NORTHWEST : x3=x2-5; y3=y2; x4=x2; y4=y2-5;x6 = x2-5 ; y6 = y2-5; break;
		case UP		   : break;
		case DOWN      : break;
	}
		
	// Draw the path
	p->drawLine(x1,y1,x5,y5);
	p->drawLine(x5,y5,x6,y6);
	p->drawLine(x2,y2,x6,y6);

        // Draw the arrow head at the end of the path
	a.setPoint(0,x2,y2);
	a.setPoint(1,x3,y3);
	a.setPoint(2,x4,y4);
	p->drawPolygon(a,FALSE,0,3);
}

void CMapRoom::paint(QPainter *p)
{
	signed int y1,x1,x2,y2;
 	
	x1 = getLowX();
	y1 = getLowY();
	x2 = getHiX();
	y2 = getHiY();	

	switch (getType())
	{
		case 0 :
			p->setPen(white);
			p->drawLine(x1,y2,x1,y1);
			p->drawLine(x1,y1,x2-1,y1);
			p->setPen(black);
			p->drawLine(x1,y2,x2,y2);
			p->drawLine(x2,y2,x2,y1);
			if (login)
			{
				p->setBrush(loginCol);
				p->setPen(loginCol);
			}
			else
			{
				
				if (getUseDefaultCol())
				{
					p->setBrush(getDefaultColour());
					p->setPen(getDefaultColour());
				
				}
				else
				{
					p->setBrush(getColour());
					p->setPen(getColour());					
				
				}
			}			
			p->drawRect(x1+1,y1+1,ROOM_SIZE-3,ROOM_SIZE-3);
			break;
 		case 1 :
			p->setPen(getDefaultColour());
			p->setBrush(getDefaultColour());
			p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);
			break;
	}


	// If this is the current room the user is in
	// then draw the symbol to show that.
	if (getCurrentRoom())
	{
		p->setPen(red);
		p->setBrush( QColor(red) );
		p->drawEllipse(x1+4,y1+4,ROOM_SIZE - 9,ROOM_SIZE -9);
	}
	
	// If the room is selected then draw the selection indicator
	if (getSelected())
	{
		p->setPen(getSelectColour());
		p->setBrush(QColor(getSelectColour()));
		p->drawRect(x2-3,y2-3,6,6);
	}

	p->setPen(black);
	for (CMapPath *path=pathList.first();path!=0;path = pathList.next())
	{
		if (path->getSrcDir() == UP)
		{
			p->drawPoint(x1+4,y1+3);
			p->drawPoint(x1+3,y1+4);
			p->drawPoint(x1+4,y1+4);
			p->drawPoint(x1+5,y1+4);
		}

		if (path->getSrcDir() == DOWN)
		{
			p->drawPoint(x1+4,y1+ROOM_SIZE-5);
			p->drawPoint(x1+3,y1+ROOM_SIZE-6);
			p->drawPoint(x1+4,y1+ROOM_SIZE-6);
			p->drawPoint(x1+5,y1+ROOM_SIZE-6);
		}
	}

}

void CMapRoom::lowerPaint(QPainter *p)
{
	signed int y1,x1,x2,y2;

	
	x1 = getLowX()-5;
	y1 = getLowY()-5;
	x2 = getHiX()-5;
	y2 = getHiY()-5;	
	
	p->setPen(getLowerColour());
	QBrush brush(getLowerColour());
	brush.setStyle(Dense3Pattern);
	p->setBrush(brush);
	p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);
}

void CMapRoom::higherPaint(QPainter *p)
{
	signed int y1,x1,x2,y2;	
	
	x1 = getLowX()+5;
	y1 = getLowY()+5;
	x2 = getHiX()+5;
	y2 = getHiY()+5;	
	
	p->setPen(getHigherColour());
	QBrush brush(getHigherColour());
	brush.setStyle(Dense7Pattern);
	p->setBrush(brush);
	p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);
}


void CMapRoom::dragPaint(QPainter *p,int offsetx,int offsety)
{
	signed int y1,x1,x2,y2;
	
	x1 = getLowX() + offsetx;
	y1 = getLowY() + offsety;
	x2 = getHiX() + offsetx;
	y2 = getHiY() + offsety;
		
	p->setPen(black);
	p->setBrush(QColor(black));
	p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);	
	
	// If the room is selected then draw the selection indicator
	if (getSelected())
	{
		p->setPen(blue);
		p->setBrush(QColor(blue));
		p->drawRect(x2-3,y2-3,6,6);
	}

}


CMapRoom::CMapRoom(int type)
{
	roomType=type;
	current = false;
	setSelected(false);
	login = false;

	xpos = 0;
	ypos = 0;
	
	label = "";
	description = "";
	useDefaultCol = true;
}

CMapElement *CMapRoom::copy(void)
{
	CMapRoom *room = new CMapRoom(getType());
	room->setCurrentRoom(getCurrentRoom());
	room->setLevel(getLevel());
	room->setX(getX());
	room->setY(getY());
	room->setLowHiX(getLowX(),getHiX());
	room->setLowHiY(getLowY(),getHiY());
	room->setSelected(getSelected());
	room->setLowerColour(getLowerColour());
	room->setDefaultColour(getDefaultColour());
	room->setHigherColour(getHigherColour());
	room->setSelectColour(getSelectColour());
	room->setLoginColour(getLoginColour());

	return room;
}

void CMapRoom::setX(int x)
{
	xpos = x;	
	
	int newLoX = (x * ROOM_SIZE) - ROOM_SIZE + 1;
	int newHiX = (x * ROOM_SIZE) - 1;

	setLowHiX(newLoX,newHiX);
}

void CMapRoom::setY(int y)
{
	ypos = y;
	
	int newLoY = (y * ROOM_SIZE) - ROOM_SIZE + 1;
	int newHiY = y * ROOM_SIZE-1;
	
	setLowHiY(newLoY,newHiY);
}

void CMapRoom::addPath (CMapPath *path)
{
	pathList.append(path);		
}

CMapPath *CMapRoom::getPathDirection (direction dir)
{
	CMapPath *path;
	for (path=pathList.first(); path!=0; path=pathList.next())
	   if (path->getSrcDir()==dir)
			return path;

	return NULL;
}

void CMapRoom::setCurrentRoom(bool currentRoom)
{
   	current = currentRoom;
}

void CMapRoom::moveBy(int x, int y)	
{
	setX(getX()+(x / ROOM_SIZE));
	setY(getY()+(y / ROOM_SIZE));
}

CMapToolTip::CMapToolTip( QWidget * parent )
    : QToolTip( parent )
{
    // no explicit initialization needed
}


void CMapToolTip::maybeTip( const QPoint &pos )
{	
	if ( !parentWidget()->inherits( "CMapWidget" ) )
		return;
	
	QString s;
	CMapRoom *room;
	int level = ((CMapWidget*)parentWidget())->getCurrentLevel();
	int roomx = (pos.x() / ROOM_SIZE)+1;
	int roomy = (pos.y() / ROOM_SIZE)+1;	
	
	room = ((CMapWidget*)parentWidget())->findRoomAt(roomx,roomy,level);
	
	if (room)
	{		
		QString s = room->getLabel();
		if (!s.stripWhiteSpace().isEmpty())
		{
			QRect r(room->getLowX(),room->getLowY(),room->getHiX()-room->getLowX(),room->getHiY()-room->getLowY());
			tip( r, s );
		}
	}
}

CMapWidget::CMapWidget(KmudDoc *document,QString mudTitle,CMudProfile* mudPro,QScrollView *view,QWidget *parent ) : QWidget(parent,"mapwidget")
{
	setBackgroundMode(NoBackground);
	// buffer = NULL;
	xMax = 0;
	yMax = 0;
	mapView = view;
	pathToolMode = 0;
	pathStartRoom = NULL;
	
	doc = document;
	
	mudProfile = mudPro;

	dragPainter = new QPainter;

	// Setup default values
	XGridSpace = ROOM_SIZE;
	YGridSpace = ROOM_SIZE;	
	hasGrid = false;
	gridColour = lightGray;
	resize(ROOM_SIZE*3,ROOM_SIZE*3);
	mapName = mudTitle + ".map";
	selected.element = NULL;
	bDragging = false;

	QBitmap delete_cb( 32, 32, delete_cb_bits, TRUE );
	QBitmap delete_cm( 32, 32, delete_cm_bits, TRUE );
	deleteCursor = new QCursor( delete_cb, delete_cm, 1,1);
	
	QBitmap pathStart_cb(16,16, path1_bits,TRUE);
	pathStartCursor = new QCursor( pathStart_cb,pathStart_cb, 4,0);

	QBitmap pathEnd_cb(16,16, path2_bits,TRUE);
	pathEndCursor = new QCursor( pathEnd_cb, pathEnd_cb, 4,0);	
		
	// Create the rooms menu
	room_menu = new QPopupMenu();
	room_menu->insertItem(i18n("Set &Current Position"),ID_MAPPER_ROOM_MOVE_LOC);
	room_menu->insertItem(i18n("Set Room to &Login Point"),ID_MAPPER_ROOM_SET_LOGIN);
	room_menu->insertSeparator();
	room_menu->insertItem(i18n("&Speed walk to room"),ID_MAPPER_ROOM_MOVE_PLAYER);
	room_menu->insertSeparator();
	room_menu->insertItem(i18n("&Delete room"),ID_MAPPER_ROOM_DEL);
	room_menu->insertSeparator();
	room_menu->insertItem(i18n("&Properties..."),ID_MAPPER_ROOM_PROP);

	path_menu = new QPopupMenu();
	path_menu->setCheckable(true);
	path_menu->insertItem(i18n("&Delete Path"),ID_MAPPER_PATH_DEL);
	path_menu->insertSeparator();
	path_menu->insertItem(i18n("&One way"),ID_MAPPER_PATH_ONEWAY);
	path_menu->insertItem(i18n("&Two way"),ID_MAPPER_PATH_TWOWAY);
	path_menu->insertSeparator();
	path_menu->insertItem(i18n("&Properties"),ID_MAPPER_PATH_PROP);
	
	text_menu = new QPopupMenu();
	text_menu->insertItem(i18n("&Delete Text"),ID_MAPPER_TEXT_DEL);
	text_menu->insertSeparator();	
	text_menu->insertItem(i18n("&Properties"),ID_MAPPER_TEXT_PROP);	

	mouseDownTimer = new QTimer(this);
	connect (mouseDownTimer ,SIGNAL(timeout()),SLOT(slotStartDraging()));
	
	setFocusPolicy(QWidget::StrongFocus);
	setFocus();
	
	tip = new CMapToolTip(this);

	CONNECT_CMD(room_menu);
	CONNECT_CMD(path_menu);
	CONNECT_CMD(text_menu);		

	move_timer = new QTimer(this);
	connect(move_timer,SIGNAL(timeout()),SLOT(slotMoveTimerDone()));
	moveEnabled = true;
}

CMapWidget::~CMapWidget()
{
/*	if (!buffer)
		delete buffer;*/
		
	delete deleteCursor;
	delete pathStartCursor;
	delete pathEndCursor;
	delete mouseDownTimer;
	delete move_timer;
	delete room_menu;
	delete path_menu;
	delete text_menu;
	delete tip;
}

void CMapWidget::setSpeedWalkAbortActive(bool set)
{	speedWalkAbortActive = set; }

bool CMapWidget::getSpeedWalkAbortActive(void)
{	return speedWalkAbortActive; }

void CMapWidget::setSpeedWalkLimit(int limit)
{ speedWalkAbortLimit = limit; }

int CMapWidget::getSpeedWalkLimit(void)
{ return speedWalkAbortLimit; }

void CMapWidget::setViewOverview(bool visiable)
{
	bOverview = visiable;
}

/** Used to get the current level number */
int CMapWidget::getCurrentLevel(void)
{
	return currentLevelNum;
}

/** Used to set the default path type */
void CMapWidget::setDefaultPathTwoWay(bool set)
{ defaultTwoWayPath = set; }

/** Used to return the default path type */
bool CMapWidget::getDefaultPathTwoWay(void)
{ return defaultTwoWayPath; }

/** Get the totals for the map */
void CMapWidget::getTotals(int *lvl,int *room, int *path, int *text)
{
	*lvl = mapList.count();
	*room = 0;
	*path = 0;
	*text = 0;
	for (CMapList *map = mapList.first(); map!=0; map = mapList.next())
	{
		for(CMapRoom *room1 = map->getRoomList()->first(); room1!=0; room1=map->getRoomList()->next())
		{
			*path += room1->pathList.count();
		}
		*text+=map->getTextList()->count();
		*room+=map->getRoomList()->count();
	}
}

/** Get the login room */
CMapRoom *CMapWidget::getLoginRoom(void)
{
	return loginRoom;
}

/** Draw the grid if it's visable */
void CMapWidget::drawGrid(QPainter* p)
{
	int x=0,y=0;
	signed int x1,x2,y1,y2;
	int maxx = width();
	int maxy = height();

	// Is the grid visable
	if (hasGrid)
	{
		p->setPen(gridColour);
	 	// Draw the lines going across
		for (y=0;y<=maxy;y+=XGridSpace)
		{
			x1 = 5;
			y1 = y;
			x2 = maxx;
			y2 = y;

			p->drawLine(x1,y1,x2,y2);
		}

		// Draw the lines going down
		for (x=0;x<=maxx;x+=YGridSpace)
		{
			x1 = x;
			y1 = 5;
			x2 = x;
			y2 = maxy;

			p->drawLine(x1,y1,x2,y2);
		}
	}
}

/** Used to calulate and set the cords of a path */
void CMapWidget::setPathCords(CMapPath *path)
{	
	int x1,y1,x2,y2;
	
	int room_x = ((path->getSrcRoom()->getX() * ROOM_SIZE) - ROOM_SIZE) + HALF_ROOM_SIZE;
	int room_y = ((path->getSrcRoom()->getY() * ROOM_SIZE) - ROOM_SIZE) + HALF_ROOM_SIZE;

	// Get the start of the path
	directionToCord(path->getSrcDir(),HALF_ROOM_SIZE,&x1,&y1);

	x1+=room_x;
	y1+=room_y;

	// Get the end of the path
	directionToCord(path->getDestDir(),HALF_ROOM_SIZE,&x2,&y2);
	
	x2 += ((path->getDestRoom()->getX() * ROOM_SIZE) - ROOM_SIZE)  + HALF_ROOM_SIZE;
	y2 += ((path->getDestRoom()->getY() * ROOM_SIZE) - ROOM_SIZE)  + HALF_ROOM_SIZE;
	
	path->setCords(x1,y1,x2,y2);		
}


/** draw the map widget */
void CMapWidget::paintEvent(QPaintEvent *pe)
{
	QPixmap pm(size());
/*	if (!buffer)
	{
		delete buffer;
		buffer = NULL;
	}
	
	buffer = new QPixmap(size()); */
		
		
	QPainter p;		
	
	//p.begin(buffer);
	p.begin(&pm);
	
	// Check that there is a map in memory
	if (mapList.count() == 0 )
		return;

	// Clear the map
	p.fillRect(0,0,width(),height(),backColour);

	// Draw the grid
	drawGrid(&p);

	// Paint the map elements of the lower map
	if (bViewLowerMap)
		for ( CMapElement *element=elementListLower.first(); element != 0; element=elementListLower.next() )
		{
			element->lowerPaint(&p);	
		}	

	// Paint the map elements of the current map
	for ( CMapElement *element=elementList.first(); element != 0; element=elementList.next() )
	{
		element->paint(&p);	
	}	

	// Paint the map elements of the higher map
	if (bViewHigherMap)
		for ( CMapElement *element=elementListUpper.first(); element != 0; element=elementListUpper.next() )
		{
			if (element->getElementType()==ROOM || element->getElementType()==PATH)
				element->higherPaint(&p);				
		}	


	p.end();
	if (bOverview)
	{
		QPixmap pm2;
		// createZoomedPixmap(&pm2,*buffer,180,100);
		createZoomedPixmap(&pm2,pm,180,100);

		// p.begin(buffer);
		p.begin(&pm);
	
		QPainter p2;
		p2.begin(&pm2);
		p2.setPen(black);
		p2.drawRect(0,0,pm2.width(),pm2.height());
		p2.end();	

		int view_x,view_y;
		if (mapView->viewport()->width()>mapView->contentsWidth())
			view_x =mapView->contentsWidth();
		else
			view_x =mapView->viewport()->width();

		if (mapView->viewport()->height()>mapView->contentsHeight())
			view_y =mapView->contentsHeight();
		else
			view_y =mapView->viewport()->height();

		p.drawPixmap(mapView->contentsX()+view_x-180,mapView->contentsY()+view_y-100,pm2);
		p.end();
	}
	//bitBlt(this,0,0,buffer);
	bitBlt(this,0,0,&pm);
}

void CMapWidget::createZoomedPixmap(QPixmap *newPixmap, QPixmap pm,int w,int h)
{
	QPainter p;
	p.begin(&pm);
	p.setPen(black);
	p.drawRect(mapView->contentsX(),mapView->contentsY(),mapView->viewport()->width(),mapView->viewport()->height()	);
	p.end();
	QImage img = pm.convertToImage();
	
	newPixmap->convertFromImage(img.smoothScale(w,h),QPixmap::Auto);		
}

/** Turns the grid on or off */
void CMapWidget::setGridVisable(bool visable)
{
	hasGrid = visable;

	repaint(false);
}

/** Returns the visable state of the grid */
bool CMapWidget::GridVisable(void)
{ return hasGrid; }

void CMapWidget::setBackgroundColour(QColor colour)
{
	backColour = colour;
}

void CMapWidget::setGridColour(QColor colour)
{ gridColour = colour; }

void CMapWidget::setElementsColour(QColor loPath,QColor defPath,QColor hiPath,
                                   QColor loRoom,QColor defRoom,QColor hiRoom,
                                   QColor login,QColor sel,QColor defText )
{

	lowerRoomColour = loRoom;
	defaultRoomColour = defRoom;
	higherRoomColour = hiRoom;

	lowerPathColour = loPath;
	defaultPathColour = defPath;
	higherPathColour = hiPath;
	
	defaultTextColour = defText;
	selectedColour = sel;
	loginColour = login;

	for (CMapList *map=mapList.first(); map !=0;map = mapList.next())
	{
		for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
		{
			room->setLowerColour(loRoom);
			room->setDefaultColour(defRoom);
			room->setHigherColour(hiRoom);
			room->setSelectColour(sel);
			room->setLoginColour(login);			

			for (CMapPath *path=room->pathList.first();path!=0; path=room->pathList.next())
			{
				path->setLowerColour(loPath);
				path->setDefaultColour(defPath);
				path->setHigherColour(hiPath);
				path->setSelectColour(sel);

			}
		}
		
		for (CMapText *text=map->getTextList()->first();text !=0 ; text = map->getTextList()->next())
		{
			text->setSelectColour(sel);
		}
	}
		

}

QColor CMapWidget::getLoginColour(void)
{ return loginColour; }

QColor CMapWidget::getSelectedColour(void)
{ return selectedColour; }

QColor CMapWidget::getDefaultTextColour(void)
{ return defaultTextColour; }

QColor CMapWidget::getGridColour(void)
{ return gridColour; }

QColor CMapWidget::getBackgroundColour(void)
{	return backColour; }

QColor CMapWidget::getDefaultRoomColour(void)
{ return defaultRoomColour; }

QColor CMapWidget::getDefaultPathColour(void)
{ return defaultPathColour; }

QColor CMapWidget::getHigherRoomColour(void)
{ return higherRoomColour; }

QColor CMapWidget::getHigherPathColour(void)
{ return higherPathColour; }

QColor CMapWidget::getLowerRoomColour(void)
{ return lowerRoomColour; }

QColor CMapWidget::getLowerPathColour(void)
{ return lowerPathColour; }

CMapRoom *CMapWidget::getCurrentRoom(void)
{	return currentRoom; }

/** This method is used to delete a path on the map */
void CMapWidget::deletePath(CMapPath *path)
{
	CMapRoom *srcRoom = path->getSrcRoom();
	CMapRoom *destRoom = path->getDestRoom();

	if (srcRoom->getLevel()< currentLevelNum)
		elementListLower.remove(path);

	if (srcRoom->getLevel()> currentLevelNum)
		elementListUpper.remove(path);

	if (srcRoom->getLevel() == currentLevelNum)
		elementList.remove(path);

	destRoom->connectingPaths.remove(path);
	srcRoom->pathList.setAutoDelete(true);
	srcRoom->pathList.remove(path);
}

void CMapWidget::deleteText(CMapText *text)
{
	if (text->getLevel()< currentLevelNum)
		elementListLower.remove(text);

	if (text->getLevel()> currentLevelNum)
		elementListUpper.remove(text);

	if (text->getLevel() == currentLevelNum)
		elementList.remove(text);

	mapList.at(currentLevelNum)->getTextList()->setAutoDelete(true);
	mapList.at(currentLevelNum)->getTextList()->remove(text);

}

/** This checks the given size and resizes if needed */		
void CMapWidget::checkSize(int x,int y)
{
	int view_x,view_y;
	int newx,newy;	

	if (x > xMax) xMax = x;
	if (y > yMax) yMax = y;

	view_x =mapView->viewport()->width() + mapView->verticalScrollBar()->width();	
	view_y =mapView->viewport()->height() + mapView->horizontalScrollBar()->height();

	if (xMax > view_x)
	{
		newx = xMax;
	}
	else
	{
		newx = view_x;
	}

	if (yMax > view_y)
	{
		newy = yMax;
	}
	else
	{
		newy = view_y;
	}

	if (newy != height() || newx !=width())
	{
		resize(newx,newy);
	}
}

CMapText *CMapWidget::createText(QString str,int x,int y,QFont font,QColor col,int level)
{
	CMapText *text = new CMapText(str,x,y,font);
	text->setLevel(level);
	text->setSelected(false);
	text->setColour(col);
	text->setSelectColour(selectedColour);
	mapList.at(level)->getTextList()->append(text);
	elementList.append(text);

	checkSize(text->getHiX(),text->getHiY());

	return text;
}

/** This method is used to delete a room on the map */
void CMapWidget::deleteRoom(CMapRoom *room)
{	
	// Delete the paths for the room
	for (CMapPath *path=room->pathList.last(); path!=0; path=room->pathList.last())
		deletePath(path);

	if (room->getCurrentRoom())
		currentMap->first()->setCurrentRoom(true);

	// Delete any paths connecting with this room		
	for (CMapPath *path=room->connectingPaths.last(); path!=0; path = room->connectingPaths.last())
		deletePath(path);

	// delete the room
	elementList.remove(room);
	currentMap->remove(room);

	// if the map is now empty create a new room and make the current	
	if (currentMap->count()==0)
	{
		createRoom(3,3,currentLevelNum,0);
		currentMap->first()->setCurrentRoom(true);
	}
}

/** Used to create a new level of the map */
void CMapWidget::createLevel(direction dir)
{
	if (dir==UP)
		mapList.append( new CMapList() );	
	else
	{
		for (CMapList *map=mapList.first(); map !=0;map = mapList.next())
		{
			for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
			{
				room->setLevel(room->getLevel()+1);
			}

			for (CMapText *text=map->getTextList()->first();text!=0 ; text = map->getTextList()->next())
			{
				text->setLevel(text->getLevel()+1);
			}
		}
		mapList.insert( 0, new CMapList() );
	}
}

/** This method is used to create a new room on the map */
CMapRoom *CMapWidget::createRoom(int x,int y,signed int level,int type)
{
	CMapRoom *room;	

	if (findRoomAt(x,y,level)) return NULL;
	
	// Create the new room
	room = new CMapRoom(type);
	room->setX(x);
	room->setY(y);
	room->setLevel(level);
	room->setLowerColour(lowerRoomColour);
	room->setDefaultColour(defaultRoomColour);
	room->setHigherColour(higherRoomColour);	
	room->setSelectColour(selectedColour);
	room->setLoginColour(loginColour);

	// Add the room to the list
	if (level!=currentLevelNum)
	{
		if (level>=mapList.count() || level<0 )
		{
	 		if (level < 0)
			{				
				createLevel(DOWN);
				mapList.at(level+1)->getRoomList()->append( room );
				room->setLevel(level+1);
			}
			else
			{
				createLevel(UP);
				mapList.at(level)->getRoomList()->append( room );
			}
		}
		else
		{
			mapList.at(level)->getRoomList()->append( room );
		}

		
		
	}
	else
	{
		currentMap->append(room);
		elementList.append((CMapElement *)room);

		// Resize the map
		checkSize(room->getHiX(),room->getHiY());
	}

	return room;
}

/** This method is used to create a path between to rooms */
CMapPath *CMapWidget::createPath(int srcX,int srcY,signed int srcLevel,direction srcDir,int destX,int destY,signed int destLevel,direction destDir)
{
	CMapRoom *room=0;
	CMapRoom *srcRoom=0;
	CMapRoom *destRoom=0;
	CMapList *mapListElement;

	// Setup and find the src room for the new path
	mapListElement = mapList.at(srcLevel);
	for ( room=mapListElement->getRoomList()->first(); room != 0; room=mapListElement->getRoomList()->next() )
	{
		if (room->getX() == srcX && room->getY() == srcY)
		{
			srcRoom = room;
			break;
		}
	}

	// Setup and find the dest room for the path
	mapListElement = mapList.at(destLevel);
	for ( room=mapListElement->getRoomList()->first(); room != 0; room=mapListElement->getRoomList()->next() )
		if (room->getX() == destX && room->getY() == destY)
		{
			destRoom = room;
			break;
		}
	
	CMapPath *newPath = new CMapPath (srcDir,destDir,destRoom,srcRoom);

	newPath->setLowerColour(lowerPathColour);
	newPath->setDefaultColour(defaultPathColour);
	newPath->setHigherColour(higherPathColour);
	newPath->setSelectColour(selectedColour);
	
	setPathCords(newPath);
	srcRoom->addPath( newPath );
	destRoom->connectingPaths.append(newPath); 	
	elementList.append(newPath);
	
	return newPath;
}

/** This method is used to move the map by the given vector */
void CMapWidget::moveMap(signed int x,signed int y)
{	
	// Move the rooms
	for (CMapList *map=mapList.first(); map !=0;map = mapList.next())
	{
		for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
		{
			room->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));					

			for (CMapPath *path=room->pathList.first();path!=0; path=room->pathList.next())
				path->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));			
		}

		for (CMapText *text=map->getTextList()->first();text!=0; text = map->getTextList()->next())
		{
			text->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));						
		}
	}
	
	checkSize(xMax + (x * ROOM_SIZE),yMax + (y * ROOM_SIZE));
}

/** This method is used to save a map */
void CMapWidget::saveMap(void)
{
	QDir localKmudDir;

	// if it's not there... create it
 	localKmudDir.setCurrent(QDir::homeDirPath());
 	localKmudDir.mkdir(".kde",false);
 	localKmudDir.mkdir(".kde/share",false);
 	localKmudDir.mkdir(".kde/share/apps",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud/maps",false);

	// Setup maps path
 	localKmudDir.setPath(QDir::homeDirPath()+"/.kde/share/apps/kmud/maps/");

	// Check to see if we can write to the map direcrtory
 	if (localKmudDir.exists() && localKmudDir.isReadable())
		exportMap(localKmudDir.path() + QString::QString("/") + mapName);
		
}

void CMapWidget::exportMap(QString filename)
{
	struct roomHeaderTyp out_room_header;
	struct roomTyp       out_room;
	struct textHeaderTyp out_text_header;
	struct pathHeaderTyp out_path_header;
	struct pathTyp       out_path;
	struct mainHeaderTyp out_main_header;
	CMapRoom *room;
	CMapList *map;
	CMapPath *path;
	CMapText *text;
	int path_total,text_total;

	// Create and open for file to save the map to
	QFile f(filename);
	f.open(IO_WriteOnly);

	// Write out file format version information
	f.writeBlock((char*)&mapFileVerMinor,sizeof(char));
	f.writeBlock((char*)&mapFileVerMajor,sizeof(char));

	// Write out the main header for the map
	out_main_header.numLevels = mapList.count();
	f.writeBlock((char *)&out_main_header, sizeof(out_main_header));
	path_total = 0;
	text_total = 0;

	for ( map=mapList.first(); map != 0; map=mapList.next())
	{
		// Write out the room header
		out_room_header.numRooms = map->getRoomList()->count();
		f.writeBlock((char *)&out_room_header,sizeof(out_room_header));
		for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
		{
			// Write out room
			out_room.x = room->getX();
			out_room.y = room->getY();
			out_room.type = room->getType();
			out_room.useDefaultCol = room->getUseDefaultCol();
			out_room.red = room->getColour().red();
			out_room.green = room->getColour().green();
			out_room.blue = room->getColour().blue();
									
			f.writeBlock((char *)&out_room,sizeof(out_room));
			path_total +=room->pathList.count();
			
			writeStr(&f,room->getLabel());
			writeStr(&f,room->getDescription());
		}

		text_total += map->getTextList()->count();
	}

	// Write out the path header

	out_path_header.numPaths = path_total;
	f.writeBlock((char *)&out_path_header,sizeof(out_path_header));

	// Loop to add the paths to the bottom of the file for each room
	for ( map=mapList.first(); map != 0; map=mapList.next())
	{
		for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
		{
			for (path=room->pathList.first(); path != 0; path=room->pathList.next())
			{
				// Write out the detils for each path in the room
				out_path.srcX = room->getX();
				out_path.srcY = room->getY();
				out_path.srcLevel = room->getLevel();
				out_path.destX = path->getDestRoom()->getX();
				out_path.destY = path->getDestRoom()->getY();
				out_path.destLevel = path->getDestRoom()->getLevel();
				out_path.srcDir = path->getSrcDir();
				out_path.destDir = path->getDestDir();
				f.writeBlock((char *)&out_path,sizeof(out_path));
			}
		}
	}

	// Write out the text header
	out_text_header.numText = text_total;
	f.writeBlock((char *)&out_text_header,sizeof(out_text_header));

	// Save all the text elements
	for ( map=mapList.first(); map != 0; map=mapList.next())
	{
		for ( text=map->getTextList()->first(); text != 0; text=map->getTextList()->next() )
		{
			writeStr(&f,text->getText());

			writeInt(&f,text->getLevel());
			writeInt(&f,text->getLowX());
			writeInt(&f,text->getLowY());

			// write out the font details
			writeStr(&f,text->getFont().family());
			writeInt(&f,text->getFont().pointSize());
			writeInt(&f,text->getFont().weight());
			writeInt(&f,text->getFont().italic());
		
			// Write out the dummy colour value (yet to be implemented)
			writeInt(&f,text->getColour().red());
			writeInt(&f,text->getColour().green());
			writeInt(&f,text->getColour().blue());
		}
	}

	// Close the map, it has now been saved
	f.close();
}

int CMapWidget::importMap(QString filename)
{
	CMapList *mapListElement;
	struct roomHeaderTyp in_room_header;
	struct roomTyp       in_room;
	struct pathHeaderTyp in_path_header;
	struct pathTyp       in_path;
	struct mainHeaderTyp in_main_header;
	struct textHeaderTyp in_text_header;
	CMapRoom *room;
	int roomCount,pathCount;
	signed lvlCount;
	QString str;				
	QString family;
	int weight;
	bool italic;
	int level;
	int size;
	int red,green,blue;
	int x,y;
	char minor,major;
	int startx=-2;
	int starty=-2;
	int startlvl=-2;
	CMapRoom *firstRoom = NULL;
	
	// Create and open for file to save the map to
	QFile f(filename);

	if (!f.open(IO_ReadOnly))
		return -1;

	f.readBlock((char *)&minor,sizeof(char));
	f.readBlock((char *)&major,sizeof(char));	
	
	if (minor!=mapFileVerMinor || major!=mapFileVerMajor)
	{
			QMessageBox::information (this,"Kmud Mapper",
                                      "The file is the incorect version.\n"
                                      "or not a valid map file.");


		return -2;
	}

	eraseMap();

	f.readBlock((char *)&in_main_header, sizeof(in_main_header));
	for ( lvlCount=0 ; lvlCount<in_main_header.numLevels; lvlCount++)
	{
		// Add new level
		mapListElement = new CMapList();
		mapList.append( mapListElement );

		// Read in the room header
		f.readBlock((char *)&in_room_header,sizeof(in_room_header));
		for ( roomCount=0; roomCount < in_room_header.numRooms; roomCount++)
		{
			// Read the room
			f.readBlock((char *)&in_room,sizeof(in_room));

			// Create the new room
			room = new CMapRoom(in_room.type);
			room->setX(in_room.x);
			room->setY(in_room.y);				
			room->setLevel(lvlCount);
			room->setLowerColour(lowerRoomColour);
			room->setDefaultColour(defaultRoomColour);
			room->setHigherColour(higherRoomColour);			
			room->setUseDefaultCol(in_room.useDefaultCol);
			room->setLoginColour(loginColour);
			room->setSelectColour(selectedColour);

			room->setColour(QColor::QColor(in_room.red,in_room.green,in_room.blue));

			// Add the room to the list
			mapListElement->getRoomList()->append( room );
			elementList.append(room);
			
			room->setLabel(readStr(&f));
			room->setDescription(readStr(&f));
			
			if (!firstRoom) firstRoom = room;
		}
				
	}
	
	f.readBlock((char *)&in_path_header,sizeof(in_path_header));
	for (pathCount=0; pathCount<in_path_header.numPaths ; pathCount++)
	{
		f.readBlock((char *)&in_path,sizeof(in_path));
		createPath(in_path.srcX,in_path.srcY,in_path.srcLevel,in_path.srcDir,in_path.destX,in_path.destY,in_path.destLevel,in_path.destDir);
	}

	// Write out the text header
	f.readBlock((char *)&in_text_header,sizeof(in_text_header));

	// Save all the text elements
	for ( int textCount = 0; textCount < in_text_header.numText; textCount++)
	{
 		str = readStr(&f);

		level=readInt(&f);
		x=readInt(&f);
		y=readInt(&f);
				
		// read font details
		family = readStr(&f);
		size=readInt(&f);
		weight =readInt(&f);
		italic =readInt(&f);
		QFont font(family,size,weight,italic);				
		
		// Read the dummy colour value (yet to be implemented)
		red = readInt(&f);
		green = readInt(&f);
		blue = readInt(&f);
		
		QColor color(red,green,blue);

		if (str)
			CMapText *text = createText(str,x,y,font,color,level);
	}

	// Close the map, it has now been saved
	f.close();
	
	// Setup the characters starting positions

	CCharacterProfile* prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
	if (prof)
	{
	 	startx = prof->getMapStartX();
	 	starty =  prof->getMapStartY();
	 	startlvl = prof->getMapStartLevel();
	 	
	 	if (!findRoomAt(startx,starty,startlvl))
	 	{
	 		startx = -2;
	 		starty = -2;
	 		startlvl = firstRoom->getLevel();;
	 	}
	}
	else
	{
		startx = -2;
		starty = -2;
		startlvl = firstRoom->getLevel();
	}

	loginRoom = NULL;

	showLevel(startx,starty,startlvl);
	
	return 0;
}

QString CMapWidget::readStr(QFile *f)
{
	int len;
	QString str;
	char c;
	
	len = readInt(f);

	if (len == 0)
	{
		str = "";
		return str;
	}
		
	for (int i=0;i<len;i++)
	{
		f->readBlock((char *)&c,sizeof(char));

		str = str + c;
	}
	return str;
}

int CMapWidget::readInt(QFile *f)
{
	int i;
	
	f->readBlock((char *)&i,sizeof(int));
	
	return i;
}

void CMapWidget::writeStr(QFile *f,QString str)
{
	//int len =str.length();
	//f->writeBlock((char *)&len,sizeof(int));
	writeInt(f,str.length());
	f->writeBlock((const char*)str,strlen(str));
}

void CMapWidget::writeInt(QFile *f,int i)
{
	f->writeBlock((char *)&i,sizeof(int));
}

/** Turn on and off the display of the higher map */
void CMapWidget::setViewHigherMap(bool visiable)
{
	bViewHigherMap = visiable;

	repaint(false);
}

/** Turn on and off the display of the lower map */
void CMapWidget::setViewLowerMap(bool visiable)
{
	bViewLowerMap = visiable;

	repaint(false);
}


/** This method is used to load a map */
void CMapWidget::loadMap(void)
{
	QDir localKmudDir;

	// if it's not there... create it
 	localKmudDir.setCurrent(QDir::homeDirPath());
 	localKmudDir.mkdir(".kde",false);
 	localKmudDir.mkdir(".kde/share",false);
 	localKmudDir.mkdir(".kde/share/apps",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud/maps",false);

	// Setup maps path
 	localKmudDir.setPath(QDir::homeDirPath()+"/.kde/share/apps/kmud/maps/");

	// Check to see if we can write to the map direcrtory
 	if (localKmudDir.exists() && localKmudDir.isReadable())
 	{
		if (importMap(localKmudDir.path() + QString::QString("/") + mapName)!=0)
			createNewMap();
	}
}

void CMapWidget::createNewMap(void)
{
	CMapList *mapListElement;

	elementList.clear();
	elementListUpper.clear();
	elementListLower.clear();

	// Create empty rooms and paths elememts
	mapListElement = new CMapList();
	mapList.append( mapListElement );
	currentMap = mapListElement->getRoomList();
	currentLevelNum = mapList.count() - 1;
	
	resize(ROOM_SIZE*6,ROOM_SIZE*6);

	xMax = 0;
	yMax = 0;

	createRoom(3,3,currentLevelNum,0);

	currentMap->first()->setCurrentRoom(true);
	currentRoom = currentMap->first();	
	currentRoom->setLoginRoom(true);
	loginRoom = currentRoom;

	checkSize(xMax,yMax);	
}

void CMapWidget::createDummyMap(void)
{
	// Setup Example Map. This is only test code and will be remove
	// when loading and saveing of the map is implemented.
	createNewMap();

	// Create Paths
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTHEAST,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(SOUTHWEST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(UP,TRUE);
	movePlayerBy(UP,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(SOUTH,TRUE);
	movePlayerBy(DOWN,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(DOWN,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(UP,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(DOWN,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(UP,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(NORTH,TRUE);
	movePlayerBy(WEST,TRUE);
	movePlayerBy(EAST,TRUE);
	movePlayerBy(SOUTH ,TRUE);

	QFont font;
	createText("JP's Test Map",60,40,font,black,currentLevelNum);
	
	emit levelChange(0);
}


/** This method is used to erase a map from memory */
void CMapWidget::eraseMap(void)
{
	CMapRoom *room;
	CMapList *map;

	// Loop to add the paths to the bottom of the file for each room
	for ( map=mapList.first(); map != 0; map=mapList.next())
	{
		for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
		{
			room->pathList.setAutoDelete(true);
			room->pathList.clear();
			room->connectingPaths.clear();
		}
		map->getRoomList()->setAutoDelete(true);
		map->getRoomList()->clear();

		map->getTextList()->setAutoDelete(true);
		map->getTextList()->clear();
	}
	mapList.setAutoDelete(true);
	mapList.clear();
	
	elementList.clear();
	elementListUpper.clear();
	elementListLower.clear();
	

	xMax = 0;
	yMax = 0;

	currentMap = NULL;
	currentLevelNum = 0;
	currentRoom = NULL;
}

/** the showLevel method is used to display a level on the map */
void CMapWidget::showLevel(int startx,int starty,signed int level)
{
	CMapRoom *room;
	bool bFound = false;
	CMapList *map;


	
	currentLevelNum = level;

	map = mapList.at(level);
	currentMap = map->getRoomList();
	elementList.clear();
		
	xMax = 0;
	yMax = 0;

	// Display the level given as a paramerter to this function
	for ( room=currentMap->first(); room != 0; room=currentMap->next() )
	{
		// Set the start room
		if (room->getX() == startx && room->getY() == starty)
		{
			currentRoom = room;
			currentRoom->setCurrentRoom(true);
			if (loginRoom == NULL)
			{
				loginRoom = currentRoom;
				currentRoom->setLoginRoom(true);			
			}
			bFound = true;			
		}

		elementList.append(room);

		for (CMapPath *path=room->pathList.first(); path!=0;path = room->pathList.next())
			elementList.append(path);

		
		if (room->getHiX()>xMax) xMax = room->getHiX();		
		if (room->getHiY()>yMax) yMax = room->getHiY();
		
	}
	
	if ((startx == -2 || starty == -2) && startx != -1 && startx != -1)
	{
		if (!bFound)
		{
			currentRoom = currentMap->first();
			currentRoom->setCurrentRoom(true);
		}
		loginRoom = currentRoom;
		currentRoom->setLoginRoom(true);
	}
	
	if (currentRoom == NULL)
	{
		currentRoom = currentMap->first();
		currentRoom->setCurrentRoom(true);		
	}
	
	if (loginRoom == NULL)
	{
		loginRoom = currentRoom;
		currentRoom->setLoginRoom(true);
	}	
	
	for ( CMapText *text=map->getTextList()->first(); text!=0; text=map->getTextList()->next())
	{
		elementList.append(text);

		if (text->getHiX()>xMax) xMax = text->getHiX();		
		if (text->getHiY()>yMax) yMax = text->getHiY();
	}

	elementListUpper.clear();
	if (mapList.count()-1 >= level+1)
	{
		for ( room=mapList.at(level+1)->getRoomList()->first(); room != 0;
					room=mapList.at(level+1)->getRoomList()->next() )
		{
	
			elementListUpper.append(room);
	
			for (CMapPath *path=room->pathList.first(); path!=0;path = room->pathList.next())
				elementListUpper.append(path);
	
			if (room->getHiX()>xMax) xMax = room->getHiX();
			if (room->getHiY()>yMax) yMax = room->getHiY();
		}
	}
	
	elementListLower.clear();
	if (level-1>=0)
	{
		for ( room=mapList.at(level-1)->getRoomList()->first(); room != 0;
					room=mapList.at(level-1)->getRoomList()->next() )
		{
	
			elementListLower.append(room);
	
			for (CMapPath *path=room->pathList.first(); path!=0;path = room->pathList.next())
				elementListLower.append(path);
	
			if (room->getHiX()>xMax) xMax = room->getHiX();
			if (room->getHiY()>yMax) yMax = room->getHiY();
		}
	}

	emit levelChange(level);
	
	// Resize the contents of the scroll view
	checkSize(xMax,yMax);	
	
	repaint(false);
}

/** This method is used to convert a direction into a offset */
void CMapWidget::directionToCord(direction dir, int distance,signed int *x,signed int *y)
{
	switch (dir)
	{
		case NORTH     : *x = 0;
	                         *y = -distance;
				 break;
		case EAST      : *x = distance;
        	                 *y = 0;
				 break;
		case SOUTH     : *x = 0;
				 *y = distance;
				 break;
		case WEST      : *x = -distance;
				 *y = 0;
				 break;
		case NORTHEAST : *x = distance;
				 *y = -distance;
				 break;
		case NORTHWEST : *x = -distance;
				 *y = -distance;
				 break;
		case SOUTHEAST : *x = distance;
				 *y = distance;
				 break;
		case SOUTHWEST : *x = -distance;
				 *y = distance;
				 break;
	}
}

void CMapWidget::directionToCordInverted(direction dir, int distance,signed int *x,signed int *y)
{
	switch (dir)
	{
		case NORTH     : *x = 0;
                         *y = distance;
						 break;
		case EAST      : *x = -distance;
                         *y = 0;
						 break;
		case SOUTH     : *x = 0;
						 *y = -distance;
						 break;
		case WEST      : *x = distance;
						 *y = 0;
						 break;
		case NORTHEAST : *x = -distance;
						 *y = distance;
						 break;
		case NORTHWEST : *x = distance;
						 *y = distance;
						 break;
		case SOUTHEAST : *x = -distance;
						 *y = -distance;
						 break;
		case SOUTHWEST : *x = distance;
						 *y = -distance;
						 break;
	}
}

/** move the player relative to the current position of the player */
void CMapWidget::movePlayerBy(direction dir,bool create)
{
	signed int x,y,incx,incy;
	signed int x1,y1,x2,y2;
	CMapRoom *srcRoom;
	direction destDir;
	int movex,movey;
	signed int newLevel;
	
	CMapRoom *tmpRoom = currentRoom;


	if (currentLevelNum!=currentRoom->getLevel())
	{
		showLevel(-1,-1,currentRoom->getLevel());
	}

	currentRoom = tmpRoom;
	
	// Make the old room as not the current room
	currentRoom->setCurrentRoom(false);

	if (dir != UP && dir != DOWN)
	{
		// Find the destination of the path that was traveled and if no path
		// is exsits for the given direction create the room and path if necsarry
		CMapPath *path = currentRoom->getPathDirection(dir);

		if (path)
		{
			currentRoom=path->getDestRoom();
		}
		else
		{		

			bool bFound = false;
			srcRoom = currentRoom;
			
			// Check to see if there is a path in the opsite direction that we should
			// be using				
			for (CMapPath *path2=srcRoom->connectingPaths.first();path2!=0;path2=srcRoom->connectingPaths.next())
			{

				if (path2->getDestDir() == dir)
				{
					bFound = true;
					x = path2->getSrcRoom()->getX();
					y = path2->getSrcRoom()->getY();				
				}
			}

			if (!bFound)
			{
				directionToCord(dir,2,&incx,&incy);
				x = currentRoom->getX()+incx;
				y = currentRoom->getY()+incy;

				// Check to see if the map needs to be moved
				if (x<1 || y<1)
				{
					if (x<1)
					{
						movex = x * -1 +1;
						x=1;
					}
					else
						movex = 0;
					
					if (y<1)
					{
						movey = y * -1 +1;
						y=1;
					}
					else
						movey = 0;
		
					moveMap (movex,movey);
				}
			}

			// Check to see if the room already exsits
			currentRoom=findRoomAt(x,y,currentLevelNum);

			// Create the room if it does not exsit
			if (!currentRoom)
				if (create)
				{
					currentRoom=createRoom(x,y,currentLevelNum,0);
				}
				else
				{
					currentRoom=srcRoom;
					return;
				}				
			
			switch (dir)
			{
				case NORTH : destDir = SOUTH; break;
				case SOUTH : destDir = NORTH; break;
				case EAST : destDir = WEST; break;
				case WEST : destDir = EAST; break;
				case NORTHWEST : destDir = SOUTHEAST; break;
				case NORTHEAST : destDir = SOUTHWEST; break;
				case SOUTHWEST : destDir = NORTHEAST; break;
				case SOUTHEAST : destDir = NORTHWEST; break;
			}
		
			// Create the new path to the room
			CMapPath *newPath = new CMapPath(dir,destDir,currentRoom,srcRoom);
			newPath->setLowerColour(lowerPathColour);
			newPath->setDefaultColour(defaultPathColour);
			newPath->setHigherColour(higherPathColour);
			newPath->setSelectColour(selectedColour);

			setPathCords(newPath);
			srcRoom->addPath (newPath );
			currentRoom->connectingPaths.append(newPath);
			elementList.append(newPath);
			
			// Make the path two way if the default path type is two way
			if (defaultTwoWayPath && bFound == false)
			{
				CMapPath *newPath1 = new CMapPath(destDir,dir,srcRoom,currentRoom);
				newPath1->setLowerColour(lowerPathColour);
				newPath1->setDefaultColour(defaultPathColour);
				newPath1->setHigherColour(higherPathColour);
				newPath1->setSelectColour(selectedColour);
				
				setPathCords(newPath1);
				currentRoom->addPath(newPath1);
				srcRoom->connectingPaths.append(newPath1);
				elementList.append(newPath1);
			}
		}
		
		// Make the new room the current room
		currentRoom->setCurrentRoom(true);

		repaint(false);

	}
	else
	{
		// Find the destination of the path that was traveled and if no path
		// is exsits for the given direction create the room and path if necsarry
		CMapPath *path = currentRoom->getPathDirection(dir);

		if (path)
		{
			currentRoom=path->getDestRoom();
			showLevel(currentRoom->getX(),currentRoom->getY(),currentRoom->getLevel());
		}
		else
		{		
			srcRoom = currentRoom;
			x = currentRoom->getX();
			y = currentRoom->getY();

			if (dir == UP)
			{
				destDir = DOWN;
				newLevel = currentLevelNum + 1;
			}
			else
			{
				destDir = UP;
				newLevel = currentLevelNum - 1;
			}
			currentRoom=findRoomAt(x,y,newLevel);

			// Create the room if it does not exsit
			if (!currentRoom)
			{
				if (create)
				{
					currentRoom=createRoom(x,y,newLevel,0);
					if (newLevel < 0) newLevel = 0;					
				}
				else
				{
					currentRoom=srcRoom;
					return;
				}				
			}

			// Create the new path to the room
			CMapPath *newPath = new CMapPath(dir,destDir,currentRoom,srcRoom);
			newPath->setLowerColour(lowerPathColour);
			newPath->setDefaultColour(defaultPathColour);
			newPath->setHigherColour(higherPathColour);
			newPath->setSelectColour(selectedColour);

			newPath->setCords(0,0,0,0);
			srcRoom->addPath (newPath);

			currentRoom->connectingPaths.append(newPath);
			showLevel(currentRoom->getX(),currentRoom->getY(),newLevel);
		}
		
		
	}
}

/** This method is used to find a room at a given location in the current level of the map.
    Null is retured if the room can't be found otherwise a pointer is returned to the room. */
CMapRoom *CMapWidget::findRoomAt(int x, int y,signed int level)
{
	CMapRoom *room;
	
	if (level>mapList.count() -1) return NULL;

	QList<CMapRoom> *map = mapList.at(level)->getRoomList();

	for ( room=map->first(); room != 0; room=map->next() )
	{
		if (x==room->getX() && y==room->getY())
		{			
			return room;
		}
	}

	return NULL;
}

/** Used to unslected all the elements on the map */
bool CMapWidget::unselectAll(void)
{
	bool result = false;
		
	for ( CMapElement *element=elementList.first(); element != 0; element=elementList.next() )
	{
		if (element->getSelected()) result = true;
	        element->setSelected(false);	
	}
	
	return result;
}

/** Set the current tool */
void CMapWidget::setTool(int tool)
{
	
	
	if (currentTool == tool) return ;

	if (currentTool == ID_MAPPER_TOOL_SELECT)
	{
		if (unselectAll())
		{
			repaint(false);
		}		
	}

	if (currentTool == ID_MAPPER_TOOL_CREATE_ROOM)
	{
		repaint(false);
	}
	

	if (tool == ID_MAPPER_TOOL_CREATE_ROOM)
	{
		setMouseTracking(true);
		lastDragX=-100;
		lastDragY=-100;
	}
	else	
		setMouseTracking(false);

	if (tool == ID_MAPPER_TOOL_SELECT || currentTool == ID_MAPPER_TOOL_SELECT)
	{
		pathStartRoom = NULL;
		pathToolMode = 0;
	}
		
	currentTool = tool;

	// Set the cursor
	switch (tool)
	{
		case ID_MAPPER_TOOL_SELECT      : setCursor(arrowCursor); break;
		case ID_MAPPER_TOOL_CREATE_TEXT : setCursor(ibeamCursor); break;
		case ID_MAPPER_TOOL_CREATE_ROOM : setCursor(arrowCursor); break;
		case ID_MAPPER_TOOL_CREATE_PATH : setCursor(*pathStartCursor); break;
		case ID_MAPPER_TOOL_DELETE      : setCursor(*deleteCursor); break;		
	}

	
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION

void CMapWidget::slotSelectMouseMove(QMouseEvent *e)
{
	int x,y,offsetx,offsety;

	if (!bDragging) return;

	x=e->x();
	y=e->y();

	if (lastDragX != x || lastDragY !=y)
	{
		dragPainter->begin(this);
	
		dragPainter->setPen(black);

		dragPainter->setRasterOp(NotROP);

		if (moveDrag)
		{
			dragPainter->setRasterOp(NotROP);
			offsetx = (((int)(x / ROOM_SIZE)) * ROOM_SIZE) - (((int)((mouseDragX-6)/ROOM_SIZE))*ROOM_SIZE)-ROOM_SIZE;
			offsety = (((int)(y / ROOM_SIZE)) * ROOM_SIZE) - (((int)((mouseDragY-6)/ROOM_SIZE))*ROOM_SIZE)-ROOM_SIZE;

			for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
			{
				if (element->getSelected())
				{
					element->dragPaint(dragPainter,offsetx,offsety);
				}
			}

			offsetx = (((int)(lastDragX / ROOM_SIZE)) * ROOM_SIZE) - (((int)((mouseDragX-6)/ROOM_SIZE))*ROOM_SIZE)-ROOM_SIZE;
			offsety = (((int)(lastDragY / ROOM_SIZE)) * ROOM_SIZE) - (((int)((mouseDragY-6)/ROOM_SIZE))*ROOM_SIZE)-ROOM_SIZE;		
						
			for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
			{
				if (element->getSelected())
				{
					element->dragPaint(dragPainter,offsetx,offsety);
				}
			}

		}
		else
		{
			// Erase old rectangle
			dragPainter->drawRect(mouseDragX,mouseDragY,lastDragX-mouseDragX,lastDragY-mouseDragY);

			// Draw new rectangle
			dragPainter->drawRect(mouseDragX,mouseDragY,x-mouseDragX,y-mouseDragY);
			
		}

		lastDragX=x;
		lastDragY=y;

		dragPainter->end();
	}
}

void CMapWidget::mouseMoveEvent(QMouseEvent *e)
{
	if (currentTool == ID_MAPPER_TOOL_SELECT)
		slotSelectMouseMove(e);

	if (currentTool == ID_MAPPER_TOOL_CREATE_ROOM)
	{
		int posx = (((int)(e->x() / ROOM_SIZE)) * ROOM_SIZE);
		int posy = (((int)(e->y() / ROOM_SIZE)) * ROOM_SIZE);

		dragPainter->begin(this);
	
		dragPainter->setPen(black);
		dragPainter->setBrush(QColor(black));

		dragPainter->setRasterOp(NotROP);

		if (posx>=0 && posy>=0)
		{
			// Erase old one
			dragPainter->drawRect(lastDragX+1,lastDragY+1,ROOM_SIZE-2,ROOM_SIZE-2);	

			// Draw new one
			dragPainter->drawRect(posx+1,posy+1,ROOM_SIZE-2,ROOM_SIZE-2);	
   	}
		dragPainter->end();

		lastDragX = posx;
		lastDragY = posy;
	}
	
}

void CMapWidget::slotSelectMousePress(QMouseEvent *e)
{
	if (e->button()==LeftButton)
	{
		moveDrag = false;
		for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
		{
			if (element->mouseInResize(e->x(),e->y()))
			{			
				emit statusMsg(i18n("Drag the element a new position"));
				moveDrag = true;			
			}		
		}
		
		if (moveDrag)
		{
			mouseDragX =  e->x();
			mouseDragY =  e->y();
			lastDragX = -100;
			lastDragY = -100;
		}
		else
		{
			mouseDragX = lastDragX = e->x();
			mouseDragY = lastDragY = e->y();
		}

		mouseDownTimer->start(150,false);
	}	
}

void CMapWidget::mousePressEvent(QMouseEvent *e)
{
	if (bOverview)
	{
		int view_x,view_y;
		if (mapView->viewport()->width()>mapView->contentsWidth())
			view_x =mapView->contentsWidth();
		else
			view_x =mapView->viewport()->width();

		if (mapView->viewport()->height()>mapView->contentsHeight())
			view_y =mapView->contentsHeight();
		else
			view_y =mapView->viewport()->height();

		if (e->x()>=mapView->contentsX()+view_x-180 &&
			e->x()<=mapView->contentsX()+view_x &&
            e->y()>=mapView->contentsY()+view_y-100 &&
            e->y()<=mapView->contentsY()+view_y)
		{
			return;			
		}
	}

	if (e->button() ==RightButton) slotContexMenu(e);

	if (currentTool == ID_MAPPER_TOOL_SELECT)
		slotSelectMousePress(e);

	/*
	if (currentTool == ID_MAPPER_TOOL_CREATE_PATH)
		slotCreatePathMousePress(e);
	*/
}

void CMapWidget::slotMoveElement(QMouseEvent *e)
{
	CMapRoom *room;

	moveDrag = false;

	int offsetx = (((int)(e->x() / ROOM_SIZE)) * ROOM_SIZE) - (((int)((mouseDragX-6)/ROOM_SIZE))*ROOM_SIZE)-ROOM_SIZE;
	int offsety = (((int)(e->y() / ROOM_SIZE)) * ROOM_SIZE) - (((int)((mouseDragY-6)/ROOM_SIZE))*ROOM_SIZE)-ROOM_SIZE;

	for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
	{
		if (element->getSelected())
		{
			if (element->getElementType()==ROOM)
			{
				CMapRoom *findRoom = findRoomAt(((CMapRoom *)element)->getX() + (offsetx / ROOM_SIZE),
				                                ((CMapRoom *)element)->getY() + (offsety / ROOM_SIZE),currentLevelNum);
				
				if (findRoom)
					if (findRoom->getSelected()) findRoom = NULL;
					
				if (!findRoom)
				{
					room=(CMapRoom *)element;
					room->moveBy(offsetx,offsety);					
					
					for (CMapPath *path=room->connectingPaths.last(); path!=0; path = room->connectingPaths.prev())
					{		
						path->setLowHiX(path->getLowX(),path->getHiX()+offsetx);
						path->setLowHiY(path->getLowY(),path->getHiY()+offsety);
					}
					
					for (CMapPath *path=room->pathList.last(); path!=0; path = room->pathList.prev())
					{					
						path->setLowHiX(path->getLowX()+offsetx,path->getHiX());
						path->setLowHiY(path->getLowY()+offsety,path->getHiY());
					}
				}
			}	

			if (element->getElementType()!=PATH && element->getElementType()!=ROOM)
			{
				element->moveBy(offsetx,offsety);
			}
		}
	}	
		
	repaint(false);

}

void CMapWidget::slotContexMenu(QMouseEvent *e)
{
	bool found = false;
	
	for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
	{
		if (element->mouseInElement(e->x(),e->y()))
		{
			if (element->getElementType()==ROOM)
			{					
				room_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));
			}
					
			if (element->getElementType()==PATH)
			{
				path_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));
				bool twoWay = ((CMapPath *)element)->isTwoWay();
				path_menu->setItemChecked(ID_MAPPER_PATH_TWOWAY,twoWay);
				path_menu->setItemChecked(ID_MAPPER_PATH_ONEWAY,!twoWay);				
			}
			
			if (element->getElementType()==TEXT)
			{
				text_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));			
			}

			if (element->getLowX() < element->getHiX())
				selected.x = element->getLowX();	
			else
				selected.x = element->getHiX();	
				
			if (element->getLowY() < element->getHiY())
				selected.y = element->getLowY();	
			else
				selected.y = element->getHiY();	

			selected.element = element;
			found = true;
			break;
			
		}
 	}			
}

/** Called when the mouse is double clicked */
void CMapWidget::mouseDoubleClickEvent ( QMouseEvent * e )
{
}

void CMapWidget::mouseReleaseEvent(QMouseEvent *e)
{
	if (bOverview && !(bDragging && e->button() == LeftButton))
	{
		int view_x,view_y;
		if (mapView->viewport()->width()>mapView->contentsWidth())
			view_x =mapView->contentsWidth();
		else
			view_x =mapView->viewport()->width();

		if (mapView->viewport()->height()>mapView->contentsHeight())
			view_y =mapView->contentsHeight();
		else
			view_y =mapView->viewport()->height();

		if (e->x()>=mapView->contentsX()+view_x-180 &&
			e->x()<=mapView->contentsX()+view_x &&
            		e->y()>=mapView->contentsY()+view_y-100 &&
	                e->y()<=mapView->contentsY()+view_y)
		{
			return;			
		}
	}

	
	if (currentTool == ID_MAPPER_TOOL_CREATE_TEXT)
	{
		if (e->button() == LeftButton)
		{
			int x = ((int)(e->x() / ROOM_SIZE)) * ROOM_SIZE;
			int y = ((int)(e->y() / ROOM_SIZE)) * ROOM_SIZE;

			QFont font;
			QString str;
			QColor col;

			
			setUpdatesEnabled( FALSE );
			DlgMapTextProperties d(this);
			
			d.setColour(defaultTextColour);
			
			if (d.exec())
			{
				font = d.getFont();
				col = d.getColour();
				str = d.getText();
				setUpdatesEnabled( TRUE );
				createText(str,x,y+32,font,col,currentLevelNum);
				repaint(false);
			}									
			else
				setUpdatesEnabled( TRUE );
		}
	}

	if (currentTool == ID_MAPPER_TOOL_SELECT)
	{
		slotSelectMouseRelease(e);
	}

	if (currentTool == ID_MAPPER_TOOL_DELETE)
	{
		for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
		{
			if (element->mouseInElement(e->x(),e->y()))
			{				
				selected.element = element;
				slotDelElement();	
			}
		}
	}

	if (currentTool == ID_MAPPER_TOOL_CREATE_ROOM)
	{
		int x = (int)(e->x() / ROOM_SIZE) + 1;
		int y = (int)(e->y() / ROOM_SIZE) + 1;

		CMapRoom *room =createRoom(x,y,currentLevelNum,0);
		if (room)
		{
			QPainter p;
			p.begin(this);
			room->paint(&p);
			p.end();
		}
		
		lastDragX=-100;
		lastDragY=-100;
	}

	if (currentTool == ID_MAPPER_TOOL_CREATE_PATH)
	{
		if (pathToolMode==1)
		{
			int x = e->x();
			int y = e->y();

			CMapRoom *destRoom = NULL;

			for (CMapRoom *room=currentMap->first(); room!=0; room = currentMap->next())
			{
				if (room->mouseInElement(x,y))
				{
					destRoom = room;
					break;
				}
			}
				
			if ((destRoom && pathStartRoom) && (pathStartRoom!=destRoom))
			{					
				DlgMapPathProperties d(this, "Properties");

				if (defaultTwoWayPath)
					d.setTwoWay(true);
				
				if (d.exec())
				{	
					CMapPath *newPath = new CMapPath (d.getSrcDirection(),d.getDestDirection(),destRoom,pathStartRoom);
					newPath->setLowerColour(lowerPathColour);
					newPath->setDefaultColour(defaultPathColour);
					newPath->setHigherColour(higherPathColour);
					newPath->setSelectColour(selectedColour);
					setPathCords(newPath);							
										
					if (!pathStartRoom->getPathDirection(d.getSrcDirection()))
					{
						pathStartRoom->addPath( newPath );
						destRoom->connectingPaths.append(newPath); 	
	                        							
	     					elementList.append(newPath);												
					}

					if (d.getTwoWay())
					{
						CMapPath *newPath1 = new CMapPath (d.getDestDirection(),d.getSrcDirection(),pathStartRoom,destRoom);
						newPath1->setLowerColour(lowerPathColour);
						newPath1->setDefaultColour(defaultPathColour);
						newPath1->setHigherColour(higherPathColour);
						newPath1->setSelectColour(selectedColour);
						setPathCords(newPath1);
		
		                        	if (!destRoom->getPathDirection(d.getDestDirection()))
						{
							destRoom->addPath( newPath1 );
							pathStartRoom->connectingPaths.append(newPath1); 	

							elementList.append(newPath1);		
						}
					}
						
					repaint(false);	
				}				
			}
			
			pathToolMode = 0;
			pathStartRoom = NULL;
			setCursor(*pathStartCursor);			
			return;								
		}
		else
		{
			if (e->button()==LeftButton)
			{
				for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
				{
					if (element->mouseInElement(e->x(),e->y()) && element->getElementType() == ROOM)
					{		
						pathStartRoom = (CMapRoom *)element;
						emit statusMsg(i18n("Selected the destination of the path"));
						pathToolMode = 1;
						setCursor(*pathEndCursor);
						return;
					}			
				}		
			}						
		}
	
	}
	emit statusMsg("Ready.");
}

void CMapWidget::slotSelectMouseRelease(QMouseEvent *e)
{
	CMapRoom *room;
	CMapPath *path;
	bool found = false;
	CMapElement *element;

	mouseDownTimer->stop();

	if (bDragging && e->button() == LeftButton)
	{
		if (moveDrag)
			slotMoveElement(e);
		else
		{		
			found = false;

			if (!bCtrlPressed) found = unselectAll();

			for (element=elementList.first(); element !=0; element =elementList.next())
			{			
				if (element->elementIn(mouseDragX,mouseDragY,lastDragX,lastDragY))
				{			
					element->setSelected(true);		
					found = true;
				}
			}

			// Erase old rectangle		
			dragPainter->begin(this);
			dragPainter->setPen(black);
			dragPainter->setRasterOp(NotROP);
			dragPainter->drawRect(mouseDragX,mouseDragY,lastDragX-mouseDragX,lastDragY-mouseDragY);
			dragPainter->end();

			repaint(false);
		}
			
		bDragging = false;
		return;
	}

	// Check to see if the context menu needs displaying
	switch (e->button())
	{
		case RightButton:
			// slotContexMenu(e);
			break;
						
		case LeftButton:
			for (element=elementList.first(); element !=0; element =elementList.next())
			{
				if (element->mouseInElement(e->x(),e->y()))
			{				
					if (element->getSelected())
					{					
						if (!bCtrlPressed) unselectAll();
								
						element->setSelected(false);
					
					}
					else	
					{	
						if (!bCtrlPressed) unselectAll();			
	
						element->setSelected(true);
					}

			
					found = true;					
				}
			}

			if (!found)
			{
				if (unselectAll())
				{
					repaint(false);
				}
			}
			else
			{
				repaint(false);
			}
	}

}

/** Called when a key is pressed */
void CMapWidget::keyPressEvent(QKeyEvent *e)
{	
	if (e->key() == Key_Control) bCtrlPressed = true;
}

/** Called when a key is released */
void CMapWidget::keyReleaseEvent(QKeyEvent *e)
{	
	if (e->key() == Key_Control) bCtrlPressed = false;
}

void CMapWidget::slotSetCurrentPos(void)
{
	if (selected.element->getElementType() == ROOM)
	{
		currentRoom->setCurrentRoom(false);
		currentRoom=(CMapRoom *)selected.element;
		currentRoom->setCurrentRoom(true);
		selected.element = NULL;
		
		repaint(false);
	}
}

void CMapWidget::slotSetLogin(void)
{
	if (selected.element->getElementType() == ROOM)
	{
		CMapRoom *room =(CMapRoom *)selected.element;

		loginRoom->setLoginRoom(false);
		loginRoom = room;
		loginRoom->setLoginRoom(true);

		CCharacterProfile* prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
		if (prof)
		{
	 		prof->setMapStartX(loginRoom->getX());
	 		prof->setMapStartY(loginRoom->getY());
	 		prof->setMapStartLevel(loginRoom->getLevel());
	 		prof->writeData();
		}
				
		selected.element = NULL;

		repaint(false);
	}
}

void CMapWidget::slotMovePlayerTo(void)
{
	QStack<direction> pathToWalk;
	QStack<CMapPath> junctionList;
	QList<CMapRoom> visitedRoomList;
	CMapRoom *destRoom;
	CMapRoom *srcRoom;
	CMapPath *path;
	CMapRoom *foundRoom;
	signed int time = 0;
	bool bFound = false;
	direction dir;
	CMapPath *foundPath;	
	int speedWalkAbortCount = 0;
	

	// Reset the seach count for all the rooms
	for (CMapList *map=mapList.first(); map !=0;map = mapList.next())
	{
		for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
		{
			room->setMoveTime(-1);
		}
	}
	
	// Check selected element is a room as we can only move to rooms
	if (selected.element->getElementType() == ROOM)
	{
		// Init things for the start room
		srcRoom = currentRoom;		
		destRoom = (CMapRoom *)selected.element;
		srcRoom->setMoveTime(time++);

		// Push start rooms exits onto the stack
		for (path = srcRoom->pathList.first(); path !=0 ; path = srcRoom->pathList.next())
			junctionList.push(path);											
				
		// Find a path to the dest room from the src room and setup
		// time info for the rooms visted
		while (junctionList.count()>0)
		{
			// Get the next path to process
			path = junctionList.pop();			
			foundRoom = path->getDestRoom();
			
			// Check to see if this room has already been visited
			if (visitedRoomList.findRef(foundRoom)==-1)
			{
				foundRoom->setMoveTime(time++);
				visitedRoomList.append(foundRoom);
				if (foundRoom == destRoom)
				{
					bFound = true;
					break;
				}
				else
				{
					if (foundRoom->pathList.count()>0)
						// Push the paths for the found room onto the stack.
						for (path = foundRoom->pathList.first(); path !=0 ; path = foundRoom->pathList.next())
						{
							junctionList.push(path);
													
						}
				
				}
			}
			
			// Check to make sure that tings are not stuck in a loop and abort
			// if the move limit is reached
			speedWalkAbortCount++;
			if (speedWalkAbortCount == speedWalkAbortLimit)
			{
				QMessageBox::information (this,"Kmud Mapper",
                                      "Speedwalk abort because move limit was reached");
				
				return;
			}
	
		}

		// Check to see if we were unable to find any paths
		if (!bFound)
		{
			junctionList.clear();
			visitedRoomList.clear();
			pathToWalk.setAutoDelete(true);
			pathToWalk.clear();
			return;
		}
		
		speedWalkAbortCount++;		
		// Trace steps that need to be taken backwards from the dest to the src room
		while(destRoom != srcRoom)
		{
           		time = destRoom->connectingPaths.first()->getSrcRoom()->getMoveTime();
			foundRoom = destRoom->connectingPaths.first()->getSrcRoom();
			
			// Find the room with the shortes time as this is the room we
			// should be moving to.
			for (path=destRoom->connectingPaths.first();path!=0; path=destRoom->connectingPaths.next())
			{
				if ((path->getSrcRoom()->getMoveTime()<=time && path->getSrcRoom()->getMoveTime()!=-1) || time == -1)
				{
					time = path->getSrcRoom()->getMoveTime();
					foundRoom = path->getSrcRoom();
					foundPath = path;
				}
			}

			// Push the direction to walk onto the stack for later
			pathToWalk.push(new direction(foundPath->getSrcDir()));

			destRoom=foundRoom;			

			// Check to make sure that tings are not stuck in a loop and abort
			// if the move limit is reached					
			speedWalkAbortCount++;
			if (speedWalkAbortCount == speedWalkAbortLimit)
			{
				QMessageBox::information (this,"Kmud Mapper",
                                      "Speedwalk abort because move limit was reached");
				
				return;
			}	
		}

		// Walk the path
		while(!pathToWalk.isEmpty())
		{	
			direction *dir = pathToWalk.pop();
			movePlayerBy(*dir,FALSE);
			emit movePlayer(*dir);
		}
	
		// Tidy up
		junctionList.clear();
		visitedRoomList.clear();
		pathToWalk.setAutoDelete(true);
		pathToWalk.clear();
	}
}

void CMapWidget::slotDelElement(void)
{
	if (selected.element->getElementType() == ROOM)
		deleteRoom((CMapRoom *)selected.element);

	if (selected.element->getElementType() == PATH)
	{
		CMapPath *path = (CMapPath *)selected.element;
		for (CMapPath *path2 = path->getDestRoom()->pathList.first();path2!=0;path2=path->getDestRoom()->pathList.next())
		{			
			if (path2->getDestRoom()==path->getSrcRoom())
			{		
				deletePath(path2);
				break;
			}
		}
		
		deletePath(path);
	}

	if (selected.element->getElementType() == TEXT)
		deleteText((CMapText *)selected.element);

	selected.element = NULL;

	repaint(false);
}

void CMapWidget::slotCreateRoom(void)
{
	createRoom((selected.x/ROOM_SIZE)+1,(selected.y/ROOM_SIZE)+1,currentLevelNum,0);
	selected.element = NULL;

	repaint(false);
}

void CMapWidget::slotCopy(void)
{
	elementClipboard.clear();

	for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
		if (element->getSelected())
			if (element->getElementType()==PATH)
			{
				CMapPath *path =(CMapPath *)element;
				if (path->getSrcRoom()->getSelected() && path->getDestRoom()->getSelected())
					elementClipboard.append(element->copy());
			}
			else
				elementClipboard.append(element->copy());
}

void CMapWidget::slotCut(void)
{
	slotCopy();
	slotDelete();
}

void CMapWidget::slotPaste(void)
{

	unselectAll();

	for (CMapElement *element=elementClipboard.first(); element !=0; element =elementClipboard.next())
	{
		if (element->getElementType()==ROOM)
		{
			CMapRoom *room= (CMapRoom *)element;
			createRoom(room->getX()+1,room->getY()+1,currentLevelNum,room->getType());
		}

		if (element->getElementType()==PATH)
		{
			CMapPath *path= (CMapPath *)element;
			createPath(path->getSrcRoom()->getX()+1,path->getSrcRoom()->getY()+1,
					   path->getSrcRoom()->getLevel(),path->getSrcDir(),
                       path->getDestRoom()->getX()+1,path->getDestRoom()->getY()+1,
                       path->getDestRoom()->getLevel(),path->getDestDir());
		}

		if (element->getElementType()==TEXT)
		{
			CMapText *text = (CMapText *)element;
			createText(text->getText(),text->getLowX(),text->getHiY() + ROOM_SIZE,text->getFont(),text->getColour(),currentLevelNum);
		}
	}

	repaint(false);
}

void CMapWidget::slotDelete(void)
{
	bool found = false;

	CMapElement *element = elementList.last();

	// Delete All rooms that are selected
	while(element!=0)
	{
		if (element->getElementType()==ROOM && element->getSelected())
		{
			deleteRoom((CMapRoom*)element);
			element=elementList.last();
		}
		else
		{			
			element=elementList.prev();
		}
	}

	element = elementList.last();

	// Delete all elements that are selected but are not rooms
	while(element!=0)
	{
		if (element->getElementType()!=ROOM && element->getElementType()!=OTHER && element->getSelected())
		{
			switch (element->getElementType())
			{
				case ROOM :  break;
				case PATH :  deletePath((CMapPath*)element); break;
				case TEXT :  deleteText((CMapText*)element); break;
				case OTHER : break;			
			}

			elementList.last();
		}
		else
			element=elementList.prev();		
	}

	repaint(false);
}

void CMapWidget::slotSelectAll(void)
{
	for (CMapElement *element=elementList.last(); element!=0; element=elementList.prev())
		element->setSelected(true);

	repaint(false);
}

void CMapWidget::slotDeselectAll(void)
{
	unselectAll();

	repaint(false);
}

void CMapWidget::slotInvertSelection(void)
{
	for (CMapElement *element=elementList.last(); element!=0; element=elementList.prev())
		element->setSelected(!element->getSelected());
	
}

void CMapWidget::slotStartDraging(void)
{
	bDragging = true;
	mouseDownTimer->stop();
}

void CMapWidget::slotPathSetOneWay(void)
{
	CMapPath *path = (CMapPath *)selected.element;
	if (path->isTwoWay())
		for (CMapPath *path2 = path->getDestRoom()->pathList.first();path2!=0;path2=path->getDestRoom()->pathList.next())
		{			
			if (path2->getDestRoom()==path->getSrcRoom())
			{			
				deletePath(path2);
				break;
			}
		}			
}

void CMapWidget::slotPathSetTwoWay(void)
{
	CMapPath *path = (CMapPath *)selected.element;
	if (!path->isTwoWay())
	{
		CMapPath *newPath = new CMapPath (path->getDestDir(),path->getSrcDir(),path->getSrcRoom(),path->getDestRoom());
		newPath->setLowerColour(lowerPathColour);
		newPath->setDefaultColour(defaultPathColour);
		newPath->setHigherColour(higherPathColour);
		newPath->setSelectColour(selectedColour);
		setPathCords(newPath);
				
		path->getSrcRoom()->addPath( newPath );
		path->getDestRoom()->connectingPaths.append(newPath); 	
    elementList.append(newPath);				
	}
}

void CMapWidget::slotTextProperties(void)
{
	setUpdatesEnabled( FALSE );

	DlgMapTextProperties d(this);
	
	CMapText *text = (CMapText *)selected.element;

	if (!text) return;

	d.setFont(text->getFont());
	d.setColour(text->getColour());
	d.setText(text->getText());


	if (d.exec())
	{
		setUpdatesEnabled( TRUE );
		text->setFont(d.getFont());
		text->setColour(d.getColour());
		text->setText(d.getText());
	 	QFontMetrics fm(d.getFont());
		int x = text->getLowX();
		int y = text->getLowY();
		text->setCords(x,y,x+fm.width(d.getText()),y-fm.height());	
		repaint(false);
	}									
	else
		setUpdatesEnabled( TRUE );
}

void CMapWidget::slotPathProperties(void)
{
	setUpdatesEnabled( FALSE );
	DlgMapPathProperties d(this, "Properties");

	CMapPath *path = (CMapPath *)selected.element;
		
	d.setDirections(path->getSrcDir(),path->getDestDir());
	d.setTwoWay(path->isTwoWay());

	if (d.exec())
	{		
		setUpdatesEnabled( TRUE );
  		path->setSrcDir(d.getSrcDirection());
		path->setDestDir(d.getDestDirection());
		setPathCords(path);
		
		if (path->isTwoWay())
		{
			for (CMapPath *path2 = path->getDestRoom()->pathList.first();path2!=0;path2=path->getDestRoom()->pathList.next())
			{			
				if (path2->getDestRoom()==path->getSrcRoom())
				{		
					if (!d.getTwoWay())
					{
						deletePath(path2);
						break;
					}
					else
					{
						path2->setSrcDir(d.getDestDirection());
						path2->setDestDir(d.getSrcDirection());
						setPathCords(path2);
					}
				}							
			}
		}
		else
		{
			if (d.getTwoWay())
			{
				CMapPath *newPath = new CMapPath (path->getDestDir(),path->getSrcDir(),path->getSrcRoom(),path->getDestRoom());
				newPath->setLowerColour(lowerPathColour);
				newPath->setDefaultColour(defaultPathColour);
				newPath->setHigherColour(higherPathColour);
				newPath->setSelectColour(selectedColour);
				setPathCords(newPath);
				
				path->getSrcRoom()->addPath( newPath );
				path->getDestRoom()->connectingPaths.append(newPath); 	
			    elementList.append(newPath);				
			}
		}
		
		repaint(false);
	}
	else
		setUpdatesEnabled( TRUE );

	repaint(false);
}



void CMapWidget::commandCallback(int id_)
{
	switch (id_)
	{
		ON_CMD(ID_MAPPER_ROOM_MOVE_LOC,slotSetCurrentPos())
		ON_CMD(ID_MAPPER_ROOM_MOVE_PLAYER,slotMovePlayerTo())
		ON_CMD(ID_MAPPER_ROOM_DEL,slotDelElement())
		ON_CMD(ID_MAPPER_ROOM_SET_LOGIN,slotSetLogin())
		ON_CMD(ID_MAPPER_ROOM_PROP,slotRoomProperties())
		
		ON_CMD(ID_MAPPER_PATH_DEL,slotDelElement())
		ON_CMD(ID_MAPPER_PATH_ONEWAY,slotPathSetOneWay())
		ON_CMD(ID_MAPPER_PATH_TWOWAY,slotPathSetTwoWay())
		ON_CMD(ID_MAPPER_PATH_PROP,slotPathProperties())
		
		ON_CMD(ID_MAPPER_TEXT_DEL,slotDelElement())		
		ON_CMD(ID_MAPPER_TEXT_PROP,slotTextProperties())
	}		
}

void CMapWidget::statusCallback(int id_)
{
	switch (id_)
	{
        	ON_STATUS_MSG(ID_MAPPER_ROOM_MOVE_LOC,	 i18n("Set the current location to this room"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_MOVE_PLAYER,i18n("Speed walk the player to this room"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_DEL,	 i18n("Delete the room"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_SET_LOGIN,	 i18n("Set the players login position"))		
		ON_STATUS_MSG(ID_MAPPER_ROOM_PROP,	 i18n("Edit the room properties"))
		
		ON_STATUS_MSG(ID_MAPPER_PATH_DEL,i18n("Delete the path"))
		ON_STATUS_MSG(ID_MAPPER_PATH_ONEWAY,i18n("Make the path one way"))
		ON_STATUS_MSG(ID_MAPPER_PATH_TWOWAY,i18n("Make the path two way"))
		ON_STATUS_MSG(ID_MAPPER_PATH_PROP,i18n("Edit the paths properties"))
		
		ON_STATUS_MSG(ID_MAPPER_TEXT_DEL,i18n("Delete the text element"))
		ON_STATUS_MSG(ID_MAPPER_TEXT_PROP,i18n("Edit the text properties"))		
 	}

}

void CMapWidget::slotStatusHelpMsg(const char *text)
{
	emit statusHelpMsg(text);
}

void CMapWidget::slotLevelUp(void)
{
	if (currentLevelNum+1<mapList.count())
		showLevel(-1,-1,currentLevelNum+1);
}

void CMapWidget::slotLevelDown(void)
{
	if (currentLevelNum-1>=0)
		showLevel(-1,-1,currentLevelNum-1);
}

void CMapWidget::slotRoomProperties(void)
{
	setUpdatesEnabled( FALSE );

	DlgMapRoomProperties d(this,"RoomProperties");

	CMapRoom *room = (CMapRoom *)selected.element;

        for (CMapPath *path = room->pathList.first(); path !=0 ; path = room->pathList.next())
        {
        	switch (path->getSrcDir())
        	{
        		case NORTH     : d.addExit(mudProfile->getDirections()->north); break;
        		case NORTHEAST : d.addExit(mudProfile->getDirections()->northeast); break;
        		case EAST      : d.addExit(mudProfile->getDirections()->east); break;        		
        		case SOUTHEAST : d.addExit(mudProfile->getDirections()->southeast); break;
        		case SOUTH     : d.addExit(mudProfile->getDirections()->south); break;
        		case SOUTHWEST : d.addExit(mudProfile->getDirections()->southwest); break;
        		case WEST      : d.addExit(mudProfile->getDirections()->west); break;        		
        		case NORTHWEST : d.addExit(mudProfile->getDirections()->northwest); break;        		
        		case UP        : d.addExit(mudProfile->getDirections()->up); break;        		
        		case DOWN      : d.addExit(mudProfile->getDirections()->down); break;        		
        	}
        }

        d.setDescription(room->getDescription());
        d.setLabel(room->getLabel());

       	d.setUseDefaultCol(room->getUseDefaultCol());
        if (room->getUseDefaultCol())
        	d.setColour(room->getDefaultColour());
        else
        	d.setColour(room->getColour());

        if (d.exec())
        {
		setUpdatesEnabled( TRUE );
     	       QString name;
    	
		room->setLabel(d.getLabel());
		room->setDescription(d.getDescription());
		room->setColour(d.getColour());
	       	room->setUseDefaultCol(d.getUseDefaultCol());
	       	
	       	for (CMapPath *path = room->pathList.first(); path !=0 ; path = room->pathList.next())
	        {

			switch (path->getSrcDir())
	        	{
        			case NORTH     : name = mudProfile->getDirections()->north; break;
        			case NORTHEAST : name = mudProfile->getDirections()->northeast; break;
        			case EAST      : name = mudProfile->getDirections()->east; break;        		
        			case SOUTHEAST : name = mudProfile->getDirections()->southeast; break;
	        		case SOUTH     : name = mudProfile->getDirections()->south; break;
        			case SOUTHWEST : name = mudProfile->getDirections()->southwest; break;
	        		case WEST      : name = mudProfile->getDirections()->west; break;        		
        			case NORTHWEST : name = mudProfile->getDirections()->northwest; break;        		
        			case UP        : name = mudProfile->getDirections()->up; break;        		
        			case DOWN      : name = mudProfile->getDirections()->down; break;        		
	        	}
	        	
	        	
			bool FoundIt = false;	        	
	        	for (int i = d.getNumExits(); i>0;i--)
	       		{	       	
		       		if ((d.getExit(i-1))==name)
		        	{
       			       		FoundIt = true;
			       		break;
			       	}
	       		
		       	}
	       	
		       	if (!FoundIt)
		       		deletePath(path);

	        }	       		       		
	       	
        }
		else
			setUpdatesEnabled( TRUE );
}  

void CMapWidget::slotMoveTimerDone(void)
{
	moveEnabled = true;
}
