#include <qcolor.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qrect.h>
#include <qstring.h>

#include <kapp.h>
#include <qmsgbox.h>

#include "painter.h"
#include "board.h"

Painter::Painter( Board *b, QWidget *parent, Bitfont *font, QString pixDir)
{
    w = parent;
    board = b;

    pointPix = NULL;
    wallPix = NULL;
    prisonPix = NULL;
    switchPix = NULL;
    fruitPix = NULL;
    pacmanPix = NULL;
    dyingPix = NULL;
    eyesPix = NULL;
    monsterPix = NULL;
 
    setFont(font);
    setPixmaps(pixDir);
}

void Painter::setFont(Bitfont *font)
{
    bitfont = font;
}

void Painter::setPixmaps(QString pixDir)
{
    initPixmaps(pixDir);
    initbackPixmaps();
    initRoomPixmap();
}

QList<QPixmap> *Painter::loadPixmap(QWidget *parent, QString pixmapName, QList<QPixmap> *pixmaps)
{
    if (pixmaps == NULL) {
	pixmaps = new QList<QPixmap>;
	pixmaps->setAutoDelete(TRUE);
    }

    if (!pixmaps->isEmpty())
	pixmaps->clear();

    QPixmap PIXMAP(pixmapName);
    if (PIXMAP.isNull()) {
 	QString msg = i18n("The pixmap could not be contructed.\n\n"
			   "The file '@PIXMAPNAME@' does not exist,\n"
			   "or is of an unknown format.");
	msg.replace(QRegExp("@PIXMAPNAME@"), pixmapName);
	QMessageBox::information(parent, i18n("Initialization Error"),
			 	 (const char *) msg);
	return 0;
    }

    int height = PIXMAP.height();
    int	width = (height == 0) ? 0 : PIXMAP.width()/(PIXMAP.width()/height);
    
    QBitmap BITMAP;
    QBitmap MASK;

    BITMAP = *PIXMAP.mask();
    MASK.resize(width, height);

    int x;
    for (x = 0; x < PIXMAP.width()/width; x++) {
	QPixmap *pixmap = new QPixmap(width, height);
	pixmaps->append(pixmap);
	bitBlt(pixmap, 0, 0, &PIXMAP, x*width, 0, width, height, CopyROP, TRUE);
	bitBlt(&MASK, 0, 0, &BITMAP, x*width, 0, width, height, CopyROP, TRUE);
	pixmap->setMask(MASK);
    }

    return pixmaps;
}

/* Return the point of the upperleft pixel of the block representing that position on the
 * board.
 */
QPoint Painter::point(int pos)
{
    return QPoint((board->x(pos)-1)*BlockWidth, (board->y(pos)-1)*BlockHeight);
}


QRect Painter::rect(int pos, PixMap pix)
{
    if (pos == OUT)
	return QRect();

    QPixmap *PIXMAP = NULL;
    switch (pix) {
	case PacmanPix  : PIXMAP = pacmanPix->at(0);
		 	  break;
	case DyingPix   : PIXMAP = dyingPix->at(0);
		 	  break;
	case MonsterPix : PIXMAP = monsterPix->at(0);
			  break;
	case EyesPix    : PIXMAP = eyesPix->at(0);
			  break;
	case FruitPix   : PIXMAP = fruitPix->at(0);
			  break;
	case PointPix   : PIXMAP = pointPix->at(0);
			  break;
	case SwitchPix  : PIXMAP = switchPix->at(0);
			  break;
	default         : PIXMAP = wallPix->at(0);
    }
    if (PIXMAP == NULL)
	return QRect();

    QRect rect = PIXMAP->rect();
    QPoint point = this->point(pos);
    rect.moveCenter(QPoint(point.x()-1, point.y()-1));
  
    return rect;
}

QRect Painter::rect(int pos, QString str, int align)
{
    if (pos == OUT)			// return an empty rect if the position
	return QRect();			// is invalid
    QPoint point = this->point(pos);
    QRect rect = bitfont->rect(str);

    rect.moveCenter(QPoint(point.x()-1, point.y()-1));

    int dx = 0;
    int dy = 0;

    if (align & AlignLeft || align & AlignRight) {
        dx = (str.length()-1) * (bitfont->width()/2);
        if (align & AlignRight)
            dx *= -1;
    }

    if (align & AlignTop || align & AlignBottom) {
        dy = bitfont->height()/2;
        if (align & AlignBottom)
            dy *= -1;
    }

    if (dx != 0 || dy != 0)
        rect.moveBy(dx, dy);

    return rect;
}

QRect Painter::rect(QRect r1, QRect r2)
{
    QRect rect;
    rect.setLeft(r1.left() < r2.left() ? r1.left() : r2.left());
    rect.setTop(r1.top() < r2.top() ? r1.top() : r2.top());
    rect.setRight(r1.right() > r2.right() ? r1.right() : r2.right());
    rect.setBottom(r1.bottom() > r2.bottom() ? r1.bottom() : r2.bottom());

    return rect;
}

void Painter::erase(int pos, PixMap pix)
{
    if (pos == OUT)
	return;
    QRect rect = this->rect(pos, pix);
    bitBlt(&roomPix, rect.x(), rect.y(), &backPix,
	   rect.x(), rect.y(), rect.width(), rect.height());
}

int Painter::maxPixmaps(PixMap pix)
{
    switch (pix) {
	case WallPix    : return (int) wallPix->count();
	case PrisonPix  : return (int) prisonPix->count();
	case PointPix   : return (int) pointPix->count();
	case SwitchPix  : return (int) switchPix->count();
	case FruitPix   : return (int) fruitPix->count();
	case PacmanPix  : return (int) pacmanPix->count();
	case DyingPix   : return (int) dyingPix->count();
	case EyesPix    : return (int) eyesPix->count();
	case MonsterPix : return (int) monsterPix->count();
	default         : return 0;
    }
}

void Painter::draw(QPoint point, DrawWidget where, QPixmap pix)
{
    switch (where) {
	case Widget   : bitBlt(w, point.x(), point.y(), &pix);
		        break;
	case RoomPix  : bitBlt(&roomPix, point.x(), point.y(), &pix);
		        break;
	case BackPix  : bitBlt(&backPix, point.x(), point.y(), &pix);
		      	break;
    }	
}

void Painter::draw(QRect rect, DrawWidget where, QPixmap pix)
{
    draw(QPoint(rect.x(), rect.y()), where, pix);
}

void Painter::draw(int pos, DrawWidget where, PixMap pix, int i)
{
    QPixmap *PIXMAP = NULL;
    switch (pix) {
	case PacmanPix  : PIXMAP = pacmanPix->at(i);
			  break;
	case DyingPix   : PIXMAP = dyingPix->at(i);
			  break;
	case MonsterPix : PIXMAP = monsterPix->at(i);
			  break;
	case EyesPix    : PIXMAP = eyesPix->at(i);
			  break;
	case FruitPix   : PIXMAP = fruitPix->at(i);
			  break;
	default         : ;
    }

    if (PIXMAP == NULL)
	return;

    QRect rect = this->rect(pos, pix);

    switch (where) {
	case Widget   : bitBlt(w, rect.x(), rect.y(), PIXMAP); 
		        break;
	case RoomPix  : bitBlt(&roomPix, rect.x(), rect.y(), PIXMAP);
		        break;
	case BackPix  : bitBlt(&backPix, rect.x(), rect.y(), PIXMAP);
		      	break;
    }	
}

QPixmap Painter::draw(int pos, DrawWidget where, QString str, QColor fg, QColor bg, int align)
{
    QPixmap TEXT = bitfont->text(str, fg, bg);

    QRect rect = this->rect(pos, str, align);
    QPixmap SAVE = QPixmap(rect.width(), rect.height());

    switch (where) {
	case Widget  : bitBlt(&SAVE, 0, 0, w, rect.x(), rect.y());
		       bitBlt(w, rect.x(), rect.y(), &TEXT);
		       break;
	case RoomPix : bitBlt(&SAVE, 0, 0, &roomPix, rect.x(), rect.y());
		       bitBlt(&roomPix, rect.x(), rect.y(), &TEXT);
		       break;
	case BackPix : bitBlt(&SAVE, 0, 0, &backPix, rect.x(), rect.y());
		       bitBlt(&backPix, rect.x(), rect.y(), &TEXT);
		       break;
    }

    return SAVE;
}

QRect Painter::draw(int col, int row, DrawWidget where, QString str, QColor fg, QColor bg, int align)
{
    QPixmap TEXT = bitfont->text(str, fg, bg);

    QRect rect = this->rect(row*BoardWidth+col, str, align);
    draw(rect, where, TEXT);
     
    return rect;
}


void Painter::initPixmaps(QString pixDir)
{
    pointPix   = loadPixmap(w, pixDir+"point.xpm", pointPix);
    wallPix    = loadPixmap(w, pixDir+"wall.xpm", wallPix);
    prisonPix  = loadPixmap(w, pixDir+"prison.xpm", prisonPix);
    switchPix  = loadPixmap(w, pixDir+"switch.xpm", switchPix);
    fruitPix   = loadPixmap(w, pixDir+"fruit.xpm", fruitPix);
    pacmanPix  = loadPixmap(w, pixDir+"pacman.xpm", pacmanPix);
    dyingPix   = loadPixmap(w, pixDir+"dying.xpm", dyingPix);
    eyesPix    = loadPixmap(w, pixDir+"eyes.xpm", eyesPix);
    monsterPix = loadPixmap(w, pixDir+"monster.xpm", monsterPix);

    if (wallPix->isEmpty()) {
	BlockWidth = 0;
	BlockHeight = 0;
    } else {
	BlockWidth = wallPix->at(0)->width();
	BlockHeight = wallPix->at(0)->height();
    }
}

void Painter::initbackPixmaps()
{
    backgroundColor = BLACK;

    backPix.resize((BoardWidth-3)*BlockWidth, (BoardHeight-3)*BlockHeight );
    backPix.fill(backgroundColor);
}

void Painter::initRoomPixmap()
{
    roomPix.resize((BoardWidth-3)*BlockWidth, (BoardHeight-3)*BlockHeight );
    bitBlt(&roomPix,0,0, &backPix);

    for (unsigned int x = 0; x < board->size(); x++) {
	if (board->isBrick(x))
	    drawBrick(x);
        if (board->isPrison(x) || board->isGate(x))
	    drawPrison(x);
	if (board->isPoint(x))
	    drawPoint(x);
	if (board->isSwitch(x))
	    drawSwitch(x);
    }
}

void Painter::drawBrick(int pos)
{
    int border = 0;
    if (board->isBrick(board->move(pos, N ))) border |= (1 << 0);
    if (board->isBrick(board->move(pos, NE))) border |= (1 << 1);
    if (board->isBrick(board->move(pos, E ))) border |= (1 << 2);
    if (board->isBrick(board->move(pos, SE))) border |= (1 << 3);
    if (board->isBrick(board->move(pos, S ))) border |= (1 << 4);
    if (board->isBrick(board->move(pos, SW))) border |= (1 << 5);
    if (board->isBrick(board->move(pos, W ))) border |= (1 << 6);
    if (board->isBrick(board->move(pos, NW))) border |= (1 << 7);

    if (board->isOut(board->move(pos, N ))) border |= (1 << 8);
    if (board->isOut(board->move(pos, NE))) border |= (1 << 9);
    if (board->isOut(board->move(pos, E ))) border |= (1 << 10);
    if (board->isOut(board->move(pos, SE))) border |= (1 << 11);
    if (board->isOut(board->move(pos, S ))) border |= (1 << 12);
    if (board->isOut(board->move(pos, SW))) border |= (1 << 13);
    if (board->isOut(board->move(pos, W ))) border |= (1 << 14);
    if (board->isOut(board->move(pos, NW))) border |= (1 << 15);

    switch (border & 0xFF)  {
	case  31 : border =  0; break;
	case 159 : border =  0; break;
	case  63 : border =  0; break;
	case 241 : border =  1; break;
	case 249 : border =  1; break;
	case 243 : border =  1; break;
	case 124 : border =  2; break;
	case 252 : border =  2; break;
	case 126 : border =  2; break;
	case 199 : border =  3; break;
	case 231 : border =  3; break;
	case 207 : border =  3; break;
	case  28 : if ((border >> 8) > 0)
		      border = 24;
		   else
		      border =  4;
                   break;
	case 112 : if ((border >> 8) > 0)
		      border = 27;
		   else
		      border =  5;
                   break;
	case   7 : if ((border >> 8) > 0)
		      border = 25;
		   else
		      border =  6;
                   break;
	case 193 : if ((border >> 8) > 0)
		      border = 26;
		   else
		      border =  7;
                   break;
	case 247 : if ((border & (1 << 11)) > 0)
		      border = 23;
		   else
		      border =  8;
                   break;
	case 119 : if ((border & (1 << 15)) > 0)
		      border =  8;
		   if ((border & (1 << 11)) > 0)
		      border = 11;
	           break;
	case 223 : if ((border & (1 << 13)) > 0)
		      border = 20;
		   else
		      border =  9;
		   break;
	case 221 : if ((border & (1 << 13)) > 0)
		      border = 10;
		   if ((border & (1 << 9)) > 0)
		      border = 9;
	           break;
	case 253 : if ((border & (1 << 9)) > 0)
		      border = 21;
		   else
		      border = 10;
		   break;
	case 127 : if ((border & (1 << 15)) > 0)
		      border = 22;
		   else
		      border = 11;
                   break;
	case  30 : border = 12; break;
	case 240 : border = 13; break;
	case  15 : border = 14; break;
	case 225 : border = 15; break;
	case 135 : border = 16; break;
	case 195 : border = 17; break;
	case  60 : border = 18; break;
	case 120 : border = 19; break;
	case 255 : border = 28; break;
        default  : border = -1;
    }
    if (border != -1 &&  border < (int) wallPix->count()) {
	QRect rect = this->rect(pos, WallPix);
	bitBlt(&roomPix, rect.x(), rect.y(), wallPix->at((uint) border));
    }
}

void Painter::drawPrison(int pos)
{
    int border = 0;
    if (board->isPrison(board->move(pos, N ))) border |= (1 << 0);
    if (board->isPrison(board->move(pos, NE))) border |= (1 << 1);
    if (board->isPrison(board->move(pos, E ))) border |= (1 << 2);
    if (board->isPrison(board->move(pos, SE))) border |= (1 << 3);
    if (board->isPrison(board->move(pos, S ))) border |= (1 << 4);
    if (board->isPrison(board->move(pos, SW))) border |= (1 << 5);
    if (board->isPrison(board->move(pos, W ))) border |= (1 << 6);
    if (board->isPrison(board->move(pos, NW))) border |= (1 << 7);

    if (board->isGate(board->move(pos, N ))) border |= (1 << 8);
    if (board->isGate(board->move(pos, NE))) border |= (1 << 9);
    if (board->isGate(board->move(pos, E ))) border |= (1 << 10);
    if (board->isGate(board->move(pos, SE))) border |= (1 << 11);
    if (board->isGate(board->move(pos, S ))) border |= (1 << 12);
    if (board->isGate(board->move(pos, SW))) border |= (1 << 13);
    if (board->isGate(board->move(pos, W ))) border |= (1 << 14);
    if (board->isGate(board->move(pos, NW))) border |= (1 << 15);

    switch (border & 0xFF)  {
	case  31 : border =  0; break;
	case 159 : border =  0; break;
	case  63 : border =  0; break;
	case 241 : border =  1; break;
	case 249 : border =  1; break;
	case 243 : border =  1; break;
	case 124 : border =  2; break;
	case 252 : border =  2; break;
	case 126 : border =  2; break;
	case 199 : border =  3; break;
	case 231 : border =  3; break;
	case 207 : border =  3; break;
	case  28 : if ((border >> 8) != 0)
		      border = 12;
		   else
		      border =  4;
                   break;
	case 112 : if ((border >> 8) != 0)
		      border = 13;
		   else
		      border =  5;
                   break;
	case   7 : if ((border >> 8) != 0)
		      border = 14;
		   else
		      border =  6;
                   break;
	case 193 : if ((border >> 8) != 0)
		      border = 15;
		   else
		      border =  7;
                   break;
	case 247 : border =  8; break;
	case 223 : border =  9; break;
	case 253 : border = 10; break;
	case 127 : border = 11; break;
        default  : border = -1;
    }
    if (board->isGate(pos)) {
	if (board->isGate(board->move(pos, N)))
	    border = 17;
	else
	    border = 16;
    }

    if (border != -1 &&  border < (int) prisonPix->count()) {
	QRect rect = this->rect(pos, PrisonPix);
	bitBlt(&roomPix, rect.x(), rect.y(), prisonPix->at((uint) border));
    }
}

void Painter::drawPoint(int pos)
{
    if (!pointPix->isEmpty()) {
	QRect rect = this->rect(pos, PointPix);
	bitBlt(&roomPix, rect.x(), rect.y(), pointPix->at(0));
    }
}

void Painter::drawSwitch(int pos)
{
    if (!switchPix->isEmpty()) {
	QRect rect = this->rect(pos, SwitchPix);
	bitBlt(&roomPix, rect.x(), rect.y(), switchPix->at(0));
    }
}

