#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "avivideoclip.h"
#include "colorspace.h"

// ---- bttv stuff -----
#include <asm/types.h>        /* some warnings with glibc */ 

#ifdef HAVE_CONFIG_H
#include <config.h>  // config.h by configure
#endif

#ifdef USE_KERNEL_VIDEODEV
#include <linux/videodev.h>
#else
#include <videodev.h>
#endif


typedef unsigned long  uint32;
typedef unsigned short uint16;

static char graberror[256];
#define GRAB_PERROR(action,file) \
  {sprintf(graberror,"%s: %s: %s\n",action,file,strerror(errno));fprintf(stderr,graberror);}

#define TRAP(txt) fprintf(stderr,"%s:%d:%s\n",__FILE__,__LINE__,txt);exit(1);

/* ----------------------------------------------------------------------- */

/*
 * M$ vidcap avi video+audio layout
 *
 * riff avi
 *   list hdrl       header
 *     avih          avi header
 *     list strl     video stream header
 *       strh         
 *       strf        
 *     list strl     audio stream header
 *       strh        
 *       strf        
 *     istf          ??? software
 *     idit          ??? timestamp
 *   yunk            ??? 4k page pad
 *   list movi       data
 *     00db          video data
 *     yunk          ??? 4k page pad
 *     [ ... ]
 *     01wb          audio data
 *     [ ... ]
 *   idx1            video frame index
 *
 */

#define size_strl_vids (sizeof(struct RIFF_strh) + \
			sizeof(struct RIFF_strf_vids) + \
			4*5)
#define size_strl_auds (sizeof(struct RIFF_strh) + \
			sizeof(struct RIFF_strf_auds) + \
			4*5)


struct RIFF_avih {
  uint32 us_frame;          /* microsec per frame */
  uint32 bps;               /* byte/s overall */
  uint32 unknown1;          /* pad_gran (???) */
  uint32 flags;
  uint32 frames;            /* # of frames (all) */
  uint32 init_frames;       /* initial frames (???) */
  uint32 streams;
  uint32 bufsize;           /* suggested buffer size */
  uint32 width;
  uint32 height;
  uint32 scale;
  uint32 rate;
  uint32 start;
  uint32 length;
};

struct RIFF_strh {
  char   type[4];           /* stream type */
  uint32 unknown1;          /* fcc_handler */
  uint32 flags;
  uint32 priority;
  uint32 init_frames;       /* initial frames (???) */
  uint32 scale;
  uint32 rate;
  uint32 start;
  uint32 length;
  uint32 bufsize;           /* suggested buffer size */
  uint32 quality;
  uint32 samplesize;
  /* XXX 16 bytes ? */
};

struct RIFF_strf_vids {       /* == BitMapInfoHeader */
  uint32 size;
  uint32 width;
  uint32 height;
  uint16 planes;
  uint16 bit_cnt;
  uint32 compression;
  uint32 image_size;
  uint32 xpels_meter;
  uint32 ypels_meter;
  uint32 num_colors;        /* used colors */
  uint32 imp_colors;        /* important colors */
  /* may be more for some codecs */
};

struct RIFF_strf_auds {       /* == WaveHeader (?) */
  uint16 format;
  uint16 channels;
  uint32 rate;
  uint32 av_bps;
  uint16 blockalign;
  uint16 size;
};

struct AVI_HDR {
  char                     riff_id[4];
  uint32                   riff_size;
  char                     riff_type[4];

  char                       hdrl_list_id[4];
  uint32                     hdrl_size;
  char                       hdrl_type[4];

  char                         avih_id[4];
  uint32                       avih_size;
  struct RIFF_avih             avih;
} avi_hdr = {
  {'R','I','F','F'}, 0,                             {'A','V','I',' '},
  {'L','I','S','T'}, 0,                             {'h','d','r','l'},
  {'a','v','i','h'}, sizeof(struct RIFF_avih),      {}
};

struct AVI_HDR_VIDEO {
  char                         strl_list_id[4];
  uint32                       strl_size;
  char                         strl_type[4];

  char                           strh_id[4];
  uint32                         strh_size;
  struct RIFF_strh               strh;

  char                           strf_id[4];
  uint32                         strf_size;
  struct RIFF_strf_vids          strf;
} avi_hdr_video = {
  {'L','I','S','T'}, size_strl_vids,                {'s','t','r','l'},
  {'s','t','r','h'}, sizeof(struct RIFF_strh),      {{'v','i','d','s'}},
  {'s','t','r','f'}, sizeof(struct RIFF_strf_vids),
  {sizeof(struct RIFF_strf_vids)}
};

struct AVI_HDR_AUDIO {
  char                         strl_list_id[4];
  uint32                       strl_size;
  char                         strl_type[4];

  char                           strh_id[4];
  uint32                         strh_size;
  struct RIFF_strh               strh;

  char                           strf_id[4];
  uint32                         strf_size;
  struct RIFF_strf_auds          strf;
} avi_hdr_audio = {
  {'L','I','S','T'}, size_strl_auds,                {'s','t','r','l'},
  {'s','t','r','h'}, sizeof(struct RIFF_strh),      {{'a','u','d','s'}},
  {'s','t','r','f'}, sizeof(struct RIFF_strf_auds), {}
};

struct AVI_DATA {
  char                       data_list_id[4];
  uint32                     data_size;
  char                       data_type[4];

  /* audio+video data follows */
    
} avi_data = {
  {'L','I','S','T'}, 0,                   {'m','o','v','i'},
};

struct FRAME_HDR {
  char                       id[4];
  uint32                     size;
} frame_hdr = {
  {'0','0','d','b'}, 0
};

struct SOUND_HDR {
  char                       id[4];
  uint32                     size;
} sound_hdr = {
  {'0','1','w','b'}, 0
};

/* ----------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */

aviVideoClip::aviVideoClip():videoClip()
{
  
  strcpy(name,"AVI video+audio");
  video_formats=VIDEO_PALETTE_RGB555 | VIDEO_PALETTE_RGB24;
  audio_formats=AUDIO_PCM;
}

aviVideoClip::~aviVideoClip()
{
}

int aviVideoClip::open( char * filename, struct MOVIE_PARAMS * par )
{
  have_audio = (par->channels > 0);

#ifdef myDEBUG
  printf("avi: have_audio= %s\n",have_audio?"true":"false");
#endif

  if ( !have_audio ) { // AVI silent movie
    strcpy(name,"AVI silent movie");
    audio_formats=AUDIO_NONE;
  }

  /* reset */
  hdr_size = total_size = frames = audio_size = 0;
  
  
  strcpy(ext,avi_formats[par->video_format].ext);
  strcpy(file,filename);
  if (strstr(file,ext) == NULL) strcat(file,ext);
  strcpy(par->extension,ext);

  memcpy(&params,par,sizeof(params));
  params.video_format=avi_formats[par->video_format].bt848;

  frame_bytes  = params.width * params.height * 
    avi_formats[par->video_format].mul / avi_formats[par->video_format].div;

  screen_bytes = avi_formats[par->video_format].mul;
   
  framebuf = (unsigned char *)malloc(frame_bytes);

  if ( -1 == (fd = ::open(file,O_CREAT | O_RDWR | O_TRUNC, 0600)) )
    GRAB_PERROR("open",file);

  /* general */
  avi_hdr.avih.us_frame    = 1000000/params.fps;
  avi_hdr.avih.bps         = params.width * params.height * frame_bytes * params.fps +
    params.channels * (params.bits/8) * params.rate;
  avi_hdr.avih.streams     = have_audio ? 2 : 1;
  avi_hdr.avih.width       = params.width;
  avi_hdr.avih.height      = params.height;

  hdr_size += write(fd,&avi_hdr,sizeof(avi_hdr));

  /* video */
  frame_hdr.size                = frame_bytes;
  avi_hdr_video.strh.scale      = 1000000/params.fps;
  avi_hdr_video.strh.rate       = 1000000;
    
  avi_hdr_video.strf.width      = params.width;
  avi_hdr_video.strf.height     = params.height;
  avi_hdr_video.strf.planes     = 1;
  avi_hdr_video.strf.bit_cnt    = frame_bytes/params.width/params.height*8;
  avi_hdr_video.strf.image_size = frame_bytes;

  hdr_size += write(fd,&avi_hdr_video,sizeof(avi_hdr_video));

  /* audio */
  if (have_audio) {
    avi_hdr_audio.strh.scale      = params.channels * (params.bits/8);
    avi_hdr_audio.strh.rate       = params.channels * (params.bits/8) * params.rate;
    avi_hdr_audio.strh.samplesize = params.channels * (params.bits/8);

    avi_hdr_audio.strf.format     = WAVE_FORMAT_PCM;
    avi_hdr_audio.strf.channels   = params.channels;
    avi_hdr_audio.strf.rate       = params.rate;
    avi_hdr_audio.strf.av_bps     = params.channels * (params.bits/8) * params.rate;
    avi_hdr_audio.strf.blockalign = params.channels * (params.bits/8);
    avi_hdr_audio.strf.size       = params.bits;

    hdr_size += write(fd,&avi_hdr_audio,sizeof(avi_hdr_audio));

    // init dsp
    thedsp= new dsp();
    thedsp->open(&params);
    audiodatasize= thedsp->getBufSize();
    sound_hdr.size= thedsp->getBufSize();
  }

  /* data */
  if ( -1 == write(fd,&avi_data,sizeof(avi_data)) )
    GRAB_PERROR("write avi header",file);

  return 0;
}

int aviVideoClip::writeframe( char * data )
{
  // first, write dsp data
  if ( have_audio ) {
    sound_hdr.size = audiodatasize;
    if (-1 == write(fd,&sound_hdr,4*2))
      GRAB_PERROR("write audio header",file);
    if (-1 == write(fd,thedsp->readBuf(),audiodatasize))
      GRAB_PERROR("write audio",file);
    total_size += audiodatasize + 4*2;
    audio_size += audiodatasize;
  }

  // now, write frame data
  if ( -1 == write(fd,&frame_hdr,4*2) )
    GRAB_PERROR("write frame header",file);
  
  switch (params.video_format) {
  case VIDEO_PALETTE_RGB555:
  case VIDEO_PALETTE_RGB24:
    for (y = params.height-1; y >= 0; y--) {
      d = ((unsigned char*)data) + y*params.width * screen_bytes;
      rc = write(fd,d,screen_bytes*params.width);
    }
    break;
  case VIDEO_PALETTE_RGB32:
    for (y = params.height-1; y >= 0; y--) {
      d = ((unsigned char*)data) + y*params.width * screen_bytes;
      rgb32_to_rgb24(framebuf,d,params.width,0);
      rc = write(fd,framebuf,screen_bytes*params.width);
    }
    break;
  default:
    TRAP("unsupported video format");
  }
  if ( -1 == rc ) GRAB_PERROR("write frame",file);

  total_size += frame_bytes + 4*2;
  frames     += 1;
  return 0;
}

int aviVideoClip::close()
{
  /* fill in some statistic values ... */
  avi_hdr.riff_size         = total_size + hdr_size - 4*2;
  avi_hdr.hdrl_size         = hdr_size - 4*5;
  avi_hdr.avih.frames       = frames;
  avi_hdr_video.strh.length = frames;
  if (have_audio)
    avi_hdr_audio.strh.length = audio_size;
  avi_data.data_size        = total_size;

#ifdef myDEBUG
  printf("avi: close, write header");
#endif
  /* ... and write header again */
  lseek(fd,0,SEEK_SET);
  write(fd,&avi_hdr,sizeof(avi_hdr));
  write(fd,&avi_hdr_video,sizeof(avi_hdr_video));
  if (have_audio) write(fd,&avi_hdr_audio,sizeof(avi_hdr_audio));
  write(fd,&avi_data,sizeof(avi_data));

  ::close(fd);

  if ( have_audio ) {
    thedsp->close();
    delete thedsp;
  }

  free(framebuf);
  return 0;
}






