/* 
    xWinTV using Bt848 frame grabber driver

    Copyright (C) 1998 Moritz Wenk (wenk@mathematik.uni-kl.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// Hey, better do not change the order of the includes !!!
// REMEMBER: Qt-stuff BEFORE X11-stuff !!!

#include <wintv.h>

#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <linux/errno.h> // for RedHat

#include <wintvscreen.h>

#include <kiconloader.h>
#include <qpixmap.h>
//#include <qcolor.h>
#include <qpainter.h> 

// video clip stuff
#include <avivideoclip.h>
#include <rawvideoclip.h>
#include <ppmvideoclip.h>

// includes needed for X stuff 
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/IntrinsicP.h>
#include <X11/ObjectP.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xmd.h>

#include <qt_visual.h>

#ifdef HAVE_LIBXDPMS
# include <X11/extensions/dpms.h>
/* this library doesn't even have its own prototypes ... */
extern "C" { 
BOOL DPMSQueryExtension(Display*, int*, int*);
BOOL DPMSCapable(Display*);
void DPMSInfo(Display*, CARD16*, BOOL*);
void DPMSEnable(Display*);
void DPMSDisable(Display*);
}
BOOL dpms_on;
#endif

//#undef HAVE_LIBJPEG
#ifdef HAVE_LIBJPEG
#ifdef __cplusplus
extern "C" {          // jpeglib is pure C 
#endif
#include <jpeglib.h> // use KDE jpeglib.h
#ifdef __cplusplus
}
#endif
#endif

// debug switches
#ifdef myDEBUG
#define tvDEBUG
//#define X11DEBUG
//#define CLIPDEBUG
#include <xeventtype.h>
char * visibility[] = {"Un","Partially","Fully" , NULL };
#endif

#define HEX(N) hex << setw(N) << setfill(int('0')) 
#define DEC(N) dec << setw(N) << setfill(int('0')) 

extern int errno;  // for RedHat
XErrorHandler oldXErrorHandler;

#define USE_NEW_X11

//==============================================================================
//==============================================================================
// einige X errors muessen (sollten) ignoriert werden,
// z.b. BadWindow und so.
// Qt wuerde sonst mit fatal(...) aussteigen...
//
XErrorHandler xerror(Display *, XErrorEvent *event)
{
#ifdef tvDEBUG
  switch (event->error_code){
  case BadWindow:
    warning("screen: X error BadWindow ingnored");
    break;
  case BadMatch:
    warning("screen: X error BadMatch ingnored");
    break;
  case BadColor:
    warning("screen: X error BadColor ingnored");
    break;
  default:
    warning("screen: unknown X error %d ingnored",event->error_code);
  }
#endif
  return 0;
}

//============================= winTVScreenWidget =================================================
//============================= winTVScreenWidget =================================================

// local variables to avoid inclusion of X11 header stuff in wintvscreen.h
XColor xcolors[256];
Colormap xcolormap;       // colormap, needed when having only 256 colors -> pseudocolor

winTVScreenWidget::winTVScreenWidget( QWidget *_parent, const char *name,
				      int _capture, const char *_bttvdev, 
				      int _freq, int _input )
  : QWidget( _parent, name ), parent(_parent), devname(_bttvdev), captureIsOn(_capture),
    defaultFreq(_freq), defaultInput(_input)
{ 
#ifdef tvDEBUG
  debug("screen: begin, captureIsOn %d",captureIsOn);
#endif

    // do some widget stuff
    //setCaption("xWinTV - TV Screen");
    //    setBackgroundcolor
    setMinimumSize( 10, 10 );
    setMaximumSize(32000,32000);
    //resize(win_default_width(),win_default_height());

    QColor bkgc(0,0,0);
    setBackgroundColor(bkgc); 
    
    KIconLoader *loader = kapp->getIconLoader();
    title_pm = loader->loadIcon("kwintv_title.xpm");
    
    /////////////////////////////////////////////////////////////////////////
    // channel info
    chInfoText = new QLabel( NULL, "label", WStyle_Customize | WStyle_NoBorder | WStyle_Tool );
    chInfoText->setFrameStyle( QFrame::NoFrame );
    //chInfoText->setAutoResize(true);
    chInfoText->hide();

    chInfoBar = new QProgressBar ( 100 , NULL, "bar", 
				   WStyle_Customize | WStyle_NoBorder | WStyle_Tool );
    chInfoBar->setFrameStyle( QFrame::NoFrame );
    chInfoBar->reset();
    chInfoBar->hide();

    {
      QFont f("ledfixed");
      chInfoText->setFont( f );
      
      QColorGroup normal( QColor( QRgb(16777215) ), QColor( QRgb(0) ), 
			  QColor( QRgb(16777215) ), QColor( QRgb(16777215) ), 
			  QColor( QRgb(16777215) ), QColor( QRgb(16777215) ), 
			  QColor( QRgb(16777215) ) );
 
      QPalette palette( normal, normal, normal );
      chInfoBar->setPalette( palette );
      chInfoText->setPalette( palette );
    }

    chInfo=OSD_LEVEL_OFF;
    //chInfo=OSD_LEVEL_1; 
    infoTimer = new QTimer(this);
    connect(infoTimer,SIGNAL(timeout()),this,SLOT(unShowOSDSLOT()));
    // timer for view refresh actions
    viewRefreshTimer= new QTimer(this);
    connect(viewRefreshTimer,SIGNAL(timeout()),this,SLOT(refreshTVActionSLOT()));
    // timer for muting while frequency changes
    muteTimer = new QTimer(this);
    connect(muteTimer,SIGNAL(timeout()),this,SLOT(setMuteStateAfterFChangeSLOT()));
    // timer for audio mode detection
    audioDetectTimer = new QTimer(this);
    connect(audioDetectTimer,SIGNAL(timeout()),this,SLOT(autoDetectAudioModeSLOT()));

    // init everything (bttv)
#ifdef tvDEBUG
    debug("screen: initialize");
#endif 

    v4lx= new v4l1if( devname );
    //delete v4lx;

    v4lx->setCapture( off );

    // use my XErrorHandler 
    oldXErrorHandler= XSetErrorHandler((XErrorHandler) xerror);

    // init the rest
    initializeTV(parent->winId());

    // install filter for special X events
    installEventFilter( this );

    // we are running
    widget_is_active=1;
    captureWasOnQt=captureIsOn;
    captureWasOnX11=captureIsOn;

#ifdef tvDEBUG
  debug("screen: end");
#endif
}

//==============================================================================

void winTVScreenWidget::postInit()
{
  if ( captureIsOn ) {
    newTVGeometry( false );
    captureState=1;
    captureWasOnQt=captureIsOn;
    captureWasOnX11=captureIsOn;
  }
  v4lx->setCapture( captureIsOn );
  didViewRefresh=false;
}

//==============================================================================

winTVScreenWidget::~winTVScreenWidget()
{
  if (widget_is_active) {
    widget_is_active=0;
    destroyTV();
  } 
#ifdef tvDEBUG
  else 
    debug("screen: winTVScreenWidget already destroyed");
#endif
}

//==============================================================================
// X11 clipping stuff 

#ifdef USE_NEW_X11

void winTVScreenWidget::configOverlay()
{
  if ( mapWindow && (newVisibility != VisibilityFullyObscured) ) {

      if ( newVisibility == VisibilityPartiallyObscured ) {
	noldclips= nnewclips;
	nnewclips= getTVClipTab(false);
	if ((nnewclips != noldclips) || v4lx->changedCapAClip() ) doNewConfig= true;
	//doNewConfig= true;
      } else {
	nnewclips= 0;
      }
	
      
#ifdef CLIPDEBUG
      debug("CLIP: configOverlay, new %d, old %d, %s",nnewclips,noldclips,doNewConfig?"new":"ignore");
#endif
      if ( doNewConfig ) {
	doNewConfig= false;
	switch_capture(0);
	setTVClipTab( nnewclips );
	switch_capture(1);
	if ( allowViewRefresh && !didViewRefresh ) {
	  refreshTV();
	}
      }

  } else {
#ifdef CLIPDEBUG
    debug("CLIP: configOverlay, off");
#endif
    switch_capture(0);
    setTVClipTab( 0 );
    doNewConfig= false;
    if ( allowViewRefresh && !didViewRefresh ) {
      refreshTV();
    }
  }
}

// new
// X11 events des shell windows

void winTVScreenWidget::Shellx11EventSLOT( XEvent *event )
{

  if ( captureIsOn ) {
#ifdef CLIPDEBUG
    debug("CLIP: x11Event: (TV shell) <%s>",events[event->type]);
#endif

    switch ( event->type ) {
    case MapNotify:
      doNewConfig= true;
      mapWindow= true;
      configOverlay();
      break;
    case UnmapNotify:
      doNewConfig= true;
      mapWindow= false;
      configOverlay();
      break;
    } // switch
  }
}

// new
// X11 events des TV windows

bool winTVScreenWidget::x11Event( XEvent *event )
{
  if ( captureIsOn && isVisible() && (event->xvisibility.window = (Window) winId()) ) {

    switch ( event->type ) {
    case Expose:
      if ( !event->xexpose.count ) {
	if (newVisibility==VisibilityUnobscured) {
	  doNewConfig= true;
      	  configOverlay();
	}
	/*
	if ( didViewRefresh ) {
	  didViewRefresh= false;
#ifdef CLIPDEBUG
	  debug("CLIP: x11Event: (TV win) <%s> (refresh last expose)",events[event->type]);
#endif
	} else {
#ifdef CLIPDEBUG
	  debug("CLIP: x11Event: (TV win) <%s>",events[event->type]);
#endif
      	  doNewConfig= true;
      	  configOverlay();
	}
	*/
      }
      break;
    case VisibilityNotify:
      
      newVisibility= event->xvisibility.state;
#ifdef CLIPDEBUG
      debug("CLIP: x11Event: (TV win) <%s> -- <%s>, %s",
	    events[event->type],visibility[event->xvisibility.state],didViewRefresh?"y":"n");
#endif
      switch (event->xvisibility.state) {
      case VisibilityFullyObscured:
	// switch capture off
	if ( !didViewRefresh ) configOverlay();
	//else didViewRefresh= false;
	break;
      case VisibilityPartiallyObscured:
      case VisibilityUnobscured: 
	doNewConfig= true;
	mapWindow= true;
	if ( didViewRefresh ) {
	  didViewRefresh= false;
	} else {
	  configOverlay();
	}
	break;
      }

#ifdef CLIPDEBUG
      debug("CLIP: x11Event: (TV win) %s",didViewRefresh?"y":"n");
#endif
      //if ( mapWindow != captureIsOn ) debug("CLIP: x11Event: (TV win) call configOverlay()");
      

    } // switch (event->type)
  }
  return (false);
}

// new
// X11 events des root windows (desktop)

bool winTVScreenWidget::eventFilter( QObject *, QEvent *e )
{

  X11RWEvent *re = (X11RWEvent*)e;

  if ( captureIsOn && isVisible() && 
       (e->type() == XRWEvent) && 
       (re->XEventPtr()->xvisibility.window != (Window) winId()) ) {

#ifdef CLIPDEBUG
    if ( re->XEventTyp()==MapNotify || re->XEventTyp()==UnmapNotify || 
	 re->XEventTyp()==ConfigureNotify )
      debug("CLIP: eventFilter: (root) <%s>, %s",events[re->XEventTyp()],didViewRefresh?"y":"n");
#endif

    switch (re->XEventTyp()) {
    case MapNotify:
    case UnmapNotify:
    case ConfigureNotify:

      if ( !didViewRefresh ) 
	configOverlay();
      
      break;
    } // switch

    return true;                        // eat event
  }
#ifdef X11DEBUG
    else 
    debug("CLIP: eventFilter: (root) no root window event");
#endif

  return false;                         // standard event processing
}

#else // use the old stuff

// old
// X11 events des shell windows

void winTVScreenWidget::Shellx11EventSLOT( XEvent *event )
{
  switch_capture(0);

#ifdef CLIPDEBUG
  debug("CLIP: x11Event: (TV shell) <%s>",events[event->type]);
#endif

  switch ( event->type ) {
  case MapNotify:
    captureIsOn=captureWasOnX11;
    setTVClipTab(getTVClipTab(false));
    break;
  case UnmapNotify:
    captureWasOnX11=captureIsOn;
    captureIsOn=false;
    if ( allowViewRefresh ) {
      refreshTV();
    }
    break;
  } // switch
  
  switch_capture(1);
}

// old
// X11 events des TV windows

bool winTVScreenWidget::x11Event( XEvent *event )
{
  Window wwin;

  if ((!isVisible())||((wwin=(Window) winId()) == None) || (event->xvisibility.window != wwin)) 
    return(false);

  if (event->type == VisibilityNotify) {
#ifdef CLIPDEBUG
    debug("CLIP: x11Event: (TV win) <%s>",events[event->type]);
#endif
    switch_capture(0);

    switch (event->xvisibility.state) {
    case VisibilityFullyObscured:
      // switch capture off
      captureWasOnQt= captureIsOn;
      captureIsOn=false;
      break;
    case VisibilityPartiallyObscured:
#ifdef CLIPDEBUG
      debug("CLIP: x11Event: (TV win) subevent <VisibilityPartiallyObscured>");
#endif
      captureIsOn= captureWasOnQt;
      if ( captureIsOn ) {
	setTVClipTab(getTVClipTab(false));
	if ( allowViewRefresh && !didViewRefresh && allowViewRefreshVisibility ) {
	  refreshTV();
	} else didViewRefresh=false;
      } 
      break;
    case VisibilityUnobscured: 
#ifdef CLIPDEBUG
      debug("CLIP: x11Event: (TV win) subevent <VisibilityUnobscured>");
#endif
      captureIsOn= captureWasOnQt;
      didViewRefresh=false;
      if ( captureIsOn ) {
	setTVClipTab(getTVClipTab(false));
      }
      break;
    } // switch (event->xvisibility.state)
    switch_capture(1);     
  } // if (event->type)

  return (false);
}

// old
// X11 events des root windows (desktop)

bool winTVScreenWidget::eventFilter( QObject *, QEvent *e )
{
  Window wwin;

  if ((!isVisible()) || (wwin=(Window) winId()) == None) return(false);

  if ( e->type() == XRWEvent ) {
    
    X11RWEvent *re = (X11RWEvent*)e;

    switch (re->XEventTyp()) {
    case MapNotify:
    case UnmapNotify:
    case ConfigureNotify:
#ifdef CLIPDEBUG
    debug("CLIP: eventFilter: (root) <%s>",events[re->XEventTyp()]);
#endif
      if ( captureIsOn ) {
	switch_capture(0);
	setTVClipTab(getTVClipTab(false));
	switch_capture(1);
	captureWasOnQt= captureIsOn;
      } 
      break;
    } // switch

    return true;                        // eat event
  }

  return false;                         // standard event processing
}

#endif

//==============================================================================

void winTVScreenWidget::moveEvent ( QMoveEvent * theEvent )
{
  doNewConfig= true;
  switch_capture(0);
  unShowOSDSLOT();
  newTVGeometry( true );
  switch_capture(1);
#ifdef tvDEBUG
  debug("screen: move: %d, %d",theEvent->pos().x(),theEvent->pos().y());
#endif
}

//==============================================================================

void winTVScreenWidget::resizeEvent( QResizeEvent * theEvent )
{
  /*
  doNewConfig= true;
  switch_capture(0);
  unShowOSDSLOT();
  newTVGeometry( true );
  switch_capture(1);
#ifdef tvDEBUG
  debug("screen: resize: %d, %d",theEvent->size().width(),theEvent->size().height());
#endif
  */
}

//==============================================================================

void winTVScreenWidget::paintEvent( QPaintEvent * event)
{
  if ( !captureIsOn ) {
    if ( !title_shown ) {
      QPainter pa(this);
      pa.setClipRect(event->rect());
      pa.drawPixmap((width()/2)-(title_pm.width()/2),(height()/2)-(title_pm.height()/2),title_pm);
    } else {
      //title_shown=1;
    }
  }
}

//==============================================================================

void winTVScreenWidget::keyPressEvent ( QKeyEvent *keyE )
{
  emit keyPressedSIGNAL(keyE->key());
  keyE->ignore();
}

//==============================================================================
/*
void winTVScreenWidget::closeEvent (  QCloseEvent * e )
{
#ifdef tvDEBUG
  debug("screen: closeEvent");
#endif
  destroyTV();
  e->accept();
}
*/
//==============================================================================
/*
void winTVScreenWidget::mouseMoveEvent(QMouseEvent *me)
{
#ifdef tvDEBUG
    debug("screen: mouseMoveEvent");
#endif

  if (cursorState) {
    if (mtimer->isActive()) mtimer->stop();
    mtimer->start(MOUSE_DELAY,true);
#ifdef tvDEBUG
    debug("screen: timer started");
#endif
  } else {
    didViewChangesCursorStateSLOT();
  }
		     
}
*/
// =============================================================================== 

void winTVScreenWidget::mouseReleaseEvent ( QMouseEvent * me)
{
#ifdef tvDEBUG
  debug("screen: mouseReleaseEvent");
#endif
  if ((me->button() == RightButton) && (tvPopup != NULL)) {
    tvPopup->popup(mapToGlobal(me->pos()));
  }
  if (me->button() == MidButton) {
    emit toggleTVonlySIGNAL();
  }
}


//==============================================================================

void winTVScreenWidget::switch_capture(int on)
{
#ifdef X11DEBUG
  //debug("screen: switch_capture: captureIsOn = %s, on = %s",captureIsOn?"on":"off",on?"on":"off");
#endif
  if (on && captureIsOn) {
    if ( isVisible() ) 
      v4lx->setCapture( on );
  } else {
    v4lx->setCapture( off );
  }
}

//==============================================================================

void winTVScreenWidget::capture(int on)
{
#ifdef X11DEBUG
  //debug("screen: capture=%s",on?"on":"off");
#endif
  if (isVisible()) {
    if (on) {
      captureIsOn=true; 
      //      captureWasOnQt=captureIsOn;
      //      captureWasOnX11=captureIsOn;
      newTVGeometry( false );
    } else {
      captureIsOn=false;
      //      captureWasOnQt=captureIsOn;
      //      captureWasOnX11=captureIsOn;
    }
    v4lx->setCapture( captureIsOn );
  }
}
//==============================================================================

//==============================================================================

int winTVScreenWidget::initializeTV(WId parentWid)
{
  Window rootw;
  Display *disp;
  ushort i;
  int visual_class;  
  Visual *myvisual;
  char visname[255];

  // get the display
  disp=qt_xdisplay();

  // get the visual and check for doublebuffering
  find_visual(disp,NULL,&myvisual,&visual_class);  
  
  colorMap8=false;
  // we have to use PseudoColor, i.e. we have only 256 colors
  if (visual_class == PseudoColor) {
    Colormap xcmap;
    int r, rr;
#ifdef tvDEBUG
    debug("screen: using color class PseudoColor");
#endif
    colorMap8=true;
    for (i=0;i<256;i++) {
      xcolors[i].pixel=i;
      xcolors[i].flags = DoRed | DoGreen | DoBlue;
    }
    //XQueryColors(disp, xcolormap, xcolors, 256);
    xcmap=XCreateColormap(disp, DefaultRootWindow(disp),myvisual, AllocAll);
    for (i=0;i<225;i++){ // Set Colors 16 - 240 
      r=i%25;
      xcolors[i+16].green =(65535*(i/25)/8);  
      xcolors[i+16].red =(65535*(r/5)/4);  
      rr=r%5;
      xcolors[i+16].blue =(65535*rr/4);        
    }
    XStoreColors(disp,xcmap,xcolors,256);
    //XSetWindowColormap(disp, (Window)WId(), xcmap);
    XSetWindowColormap(disp, (Window)parentWid, xcmap);
#ifdef tvDEBUG
    debug("screen: set up colormap for 8 bpp");
#endif
  }

  // set up some X stuff
  rootw=DefaultRootWindow(disp);

  // set XEvents we want to receive
  const uint stdWidgetEventMask =                 // X event mask
        KeyPressMask | KeyReleaseMask |
        ButtonPressMask | ButtonReleaseMask |
        KeymapStateMask |
        ButtonMotionMask |
        EnterWindowMask | LeaveWindowMask |
        FocusChangeMask |
        ExposureMask | VisibilityChangeMask |
        StructureNotifyMask | SubstructureNotifyMask;

  // tv window -- need visibility, structure
  XSelectInput(disp,(Window) winId(), stdWidgetEventMask );
  //       VisibilityChangeMask | StructureNotifyMask  );

  // the shell window
  //XSelectInput(disp,parentWid, stdWidgetEventMask );

  // root window -- need visibility, sub-, structure
  XSelectInput(disp,rootw, 
	       VisibilityChangeMask | StructureNotifyMask | SubstructureNotifyMask );


  audioVolumeCan= v4lx->capAudioVolume();

  // init some variables
  doNewConfig=true;
  mapWindow=true;
  allowViewRefresh=true;
  didViewRefresh=true;
  allowViewRefreshVisibility=true;
  didViewMove=false;
  viewRefreshDelay=DEFAULT_DELAY;
  captureWasOnQt=false;
  captureWasOnX11=false;
  infodelay=INFO_DELAY;
  
  muteDelay= 100;

  saverState=1;
  cursorState=1;
  muteState=0;
  captureState=0;

  shotnumber=0;
  video_aktive=0;
  video_capturing=0;
  title_shown=0;

  oldxadj=oldyadj=0;
  nnewclips=noldclips=0;

  setTVMaxSize();

  return(0);
}

//==============================================================================

void winTVScreenWidget::setTVMaxSize()
{
  unsigned int  minw,minh;
  v4lx->capCapSize( &minw, &minh, &max_width, & max_height );
#ifdef tvDEBUG
  debug ("screen: screensize max_x: %d, max_y: %d",max_width,max_height);
#endif
}

//==============================================================================

void winTVScreenWidget::destroyTV()
{
#ifdef tvDEBUG
  debug("screen: destroyTV");
#endif

  capture(0);

  // reactivate screen saver
  if (!saverState) toggleScreenSaverSLOT();
    
  // reset X stuff
  XSetErrorHandler(oldXErrorHandler);
  if (colorMap8) XFreeColormap(qt_xdisplay(),xcolormap);
  XSelectInput(qt_xdisplay(),DefaultRootWindow(qt_xdisplay()),0);

  // kill the interface 
  delete v4lx;
  
#ifdef tvDEBUG
  debug("screen: destroyTV done");
#endif
}

//==============================================================================

void winTVScreenWidget::newTVGeometry( bool doViewRefresh )
{
  bool checkClipping;
  unsigned int view_width,view_height;
  int xp,yp,oldx,oldy,dx=0,dy=0;
  unsigned int neww,newh,oldw,oldh;

  if (!isVisible()) return;

  v4lx->getCapAClip( &oldx, &oldy, &oldw, &oldh );
#ifdef CLIPDEBUG
  debug("CLIP: newTVGeometry values getCapAClip %d %d %d %d",oldx,oldy,oldw,oldh);
#endif

  checkClipping=false;
  didViewMove=false;
  //didViewMove=true;

  // get new geometry values, need the global position 
  QPoint p=mapToGlobal(QPoint(0,0));
  view_height=geometry().height();
  view_width=geometry().width();

#if 0

  xp=p.x();
  yp=p.y();

  // checkClipping if size changed
  // set a new width
  if (view_width > max_width) {
    dx=(view_width - max_width)/2;
    neww= max_width;
  } else {
    neww= view_width;
  }
  if (oldw != neww) checkClipping=true;
  if (oldw > neww ) didViewMove=true;

  // set a new height
  if (view_height > max_height) {
    dy=(view_height - max_height)/2;
    newh= max_height;
  } else {
    newh= view_height;
  }
  if (oldh != newh) checkClipping=true;
  if (oldh > newh ) didViewMove=true;

#else

  xp=p.x() & ~3;
  yp=p.y() & ~3;

  // checkClipping if size changed
  // set a new width
  if (view_width > max_width) {
    dx=(view_width - max_width)/2;
    neww= max_width & ~3;
  } else {
    neww= view_width & ~3;
  }
  if (oldw != neww) checkClipping=true;
  if (oldw > neww ) didViewMove=true;

  // set a new height
  if (view_height > max_height) {
    dy=(view_height - max_height)/2;
    newh= max_height & ~3;
  } else {
    newh= view_height & ~3;
  }
  if (oldh != newh) checkClipping=true;
  if (oldh > newh ) didViewMove=true;

  if ( (p.x() & ~3) < p.x() ) {
    xp+=4;
    neww-=4;
#ifdef CLIPDEBUG
    debug("CLIP: xp, neww");
#endif
  }
  if ( (p.y() & ~3) < p.y() ) {
    yp+=4;
    newh-=4;
#ifdef CLIPDEBUG
    debug("CLIP: yp, newh");
#endif
  }

#endif

  // checkClipping if position changed
  if (oldx!=xp+dx) {
    didViewMove=true;
    checkClipping=true;
  } else 
  if (oldy!=yp+dy) {
    didViewMove=true;
    checkClipping=true;
  } 
#ifdef CLIPDEBUG
    debug("CLIP: checkClipping %s, didviewMove %s",checkClipping?"yes":"no",didViewMove?"yes":"no");
#endif
  if (checkClipping && captureIsOn) {
    setTVClipTab(getTVClipTab(true,xp+dx,yp+dy,neww,newh)); //,(xp+dx)-oldx,(yp+dy)-oldy)
    if ( allowViewRefresh && doViewRefresh ) refreshTV();
  }
}

//==============================================================================

void winTVScreenWidget::autoDetectAudioModeSLOT()
{
  v4lx->setAudioMode( 0 );
  if (audioMode != v4lx->getAudioMode()) {
    emit autoDetectAudioModeSIGNAL(v4lx->getAudioMode() | AUDIOMODE_AUTODETECT);
    if ( chInfo > OSD_LEVEL_1 ) 
      showOSDSLOT(NULL,CHINFO_NODISPVAL,0,0,audiomode[audioMode].str);
  }

#ifdef tvDEBUG
  //debug("screen: autoDetectAudioModeSLOT mode: %s",audiomode[vaudio[0].mode].str);
#endif
}

//==============================================================================

void winTVScreenWidget::refreshTVActionSLOT()
{
  int scr;
  XSetWindowAttributes xswa;
  unsigned long mask;
  Window rwin,rootW;
  Display *disp;

#ifdef X11DEBUG
  debug("X11: refreshTVAction queried");
#endif

  viewRefreshTimer->stop();

  if ( !didViewRefresh && ((newVisibility != VisibilityUnobscured) || didViewMove) ) {
#ifdef CLIPDEBUG
    debug("CLIP: do refreshTVAction %s",didViewMove?"(whole screen)":"");
#endif

    disp=qt_xdisplay();
    scr=DefaultScreen(disp);
    rootW= RootWindow(disp, scr);
    
    xswa.override_redirect = true;
    xswa.backing_store = NotUseful;
    xswa.save_under = false;
    mask = ( CWSaveUnder | CWBackingStore | CWOverrideRedirect );

    if (didViewMove) {
      rwin = XCreateWindow(disp, rootW, 0, 0, 
			   kapp->desktop()->width(),kapp->desktop()->height(),
			   /*vbuf.width, vbuf.height,*/
			   0, CopyFromParent, InputOutput, CopyFromParent,
			   mask, &xswa);
      didViewMove=false;
    } else {
#if 0
      int oldx,oldy;
      unsigned int oldw,oldh;
      v4lx->getCapAClip( &oldx, &oldy, &oldw, &oldh );
      rwin = XCreateWindow(disp,rootW,
			   oldx-40, oldy-40, 
			   oldw+80,oldh+80,
			   0,CopyFromParent, InputOutput, CopyFromParent,
			   mask, &xswa);
#ifdef CLIPDEBUG
      debug("CLIP: refreshTVAction area %d %d, %dx%d",oldx-40, oldy-40, 
	    oldw+80,oldh+80);
#endif
#else
      QPoint p= mapToGlobal(QPoint(0,0));
      rwin = XCreateWindow(disp,rootW,
			   p.x()-40, p.y()-40, 
			   geometry().width()+80,geometry().height()+80,
			   0,CopyFromParent, InputOutput, CopyFromParent,
			   mask, &xswa);
#ifdef CLIPDEBUG
      debug("CLIP: refreshTVAction area %d %d, %dx%d",p.x()-40, p.y()-40, 
	    geometry().width()+80,geometry().height()+80);
#endif
#endif

    }



    XMapWindow(disp, rwin);
    XUnmapWindow(disp, rwin);
    XDestroyWindow(disp, rwin);

    didViewRefresh=true;
  }
}

//==============================================================================

void  winTVScreenWidget::refreshTV()
{
#ifdef X11DEBUG
  debug("X11: query refreshTV");
#endif
  didViewRefresh=false;
  if ( viewRefreshTimer->isActive() ) viewRefreshTimer->stop();
  viewRefreshTimer->start(viewRefreshDelay);
}

//==============================================================================

int winTVScreenWidget::getTVClipTab( bool set, int wx, int wy, unsigned int ww, unsigned int wh, 
				     int xadj, int yadj )
{
  int nclips;
  int x,y,x2,y2;
  Display *disp;
  XWindowAttributes wts;
  Window parent,win,root,rroot,*children;
  uint nchildren, i;

  if (!isVisible()) return(0);

  v4lx->resetCapAClip();
  if ( !set ) {
    v4lx->getCapAClip( &wx, &wy, &ww, &wh );
    //xadj= oldxadj;
    //yadj= oldyadj;
  } else {
    oldxadj= xadj;
    oldyadj= yadj;
  }

//  #ifdef tvDEBUG
//    debug("screen: getTVClip old vals: %d, %d, %d, %d, set=%s",wx,wy,ww,wh,set?"true":"false");
//  #endif

  if (wx<0)
    v4lx->addCapAClip(0, 0, (uint)(-wx), wh);
  if (wy<0)
     v4lx->addCapAClip(0, 0, ww, (uint)(-wy));
  if ((wx+ww) > v4lx->getFBwidth())
    v4lx->addCapAClip(v4lx->getFBwidth()-wx, 0, ww, wh);
  if ((wy+wh) > v4lx->getFBheight())
    v4lx->addCapAClip(0, v4lx->getFBheight()-wy, ww, wh);

  disp=qt_xdisplay();
  win=(Window)winId();
  root=DefaultRootWindow(disp);
  
  for (;;) {
    XQueryTree(disp, win, &rroot, &parent, &children, &nchildren);
    XFree((char *) children);
    //debug("screen: getTVClip: win=0x%x, parent=0x%x\n",me,parent);
    if (root == parent) break;
    win = parent;
  }
  XQueryTree(disp, root, &rroot, &parent, &children, &nchildren);

  for  (i = 0; i < nchildren; i++ )
    if ( children[i]==win )
      break;

//  #ifdef tvDEBUG
//    debug("screen: getTVClip: nchildren %d, %d",nchildren,i);
//  #endif

  for ( i++; i < nchildren; i++  ) {
    XGetWindowAttributes(disp, children[i], &wts);
    if ( !(wts.map_state & IsViewable) ) continue;
    
    x=wts.x-wx; 
    y=wts.y-wy;
    x2=x+wts.width+2*wts.border_width;
    y2=y+wts.height+2*wts.border_width;

    if ( (x2 < 0) || (x >= (int)ww) || (y2 < 0) || (y >= (int)wh) ) continue;

    if ( x < 0 ) x=0;
    if ( y < 0 ) y=0;
    if ( x2 >= (int)ww ) x2=ww;
    if ( y2 >= (int)wh ) y2=wh;

    nclips=v4lx->addCapAClip(x, y, x2, y2, xadj, yadj );

  }
  XFree((char *) children);

  return v4lx->setCapAClip( wx, wy, ww, wh );
}

//==============================================================================

int winTVScreenWidget::setTVClipTab( int num )
{
  return v4lx->applyCapAClip( num );
}

//==============================================================================

void flip(char *mem, int len)
{
  int i;
  char tmp;

  for (i=0;i<len;i++) {
    tmp=mem[2];
    mem[2]=mem[0];
    mem[0]=tmp;
    mem+=3;
  }
}

//==============================================================================
//---------------------------------- SLOTS -------------------------------------
//==============================================================================

void winTVScreenWidget::snapShotSLOT(const char *prefix, int fixed, int width, int height, 
				     int format, int quality)
{
  QString name,ext;
  char *tm;
  int bpp= 3;

  if (captureIsOn && !video_aktive) {

    // set image params and capture one 
    v4lx->grabSetParams( fixed, &width, &height, VIDEO_PALETTE_RGB24 );
    tm= v4lx->grabCapture( true ); 

    switch (format) {
    case SNAPSHOT_JPEG:
      ext=".jpg";
      break;
    case SNAPSHOT_GIF:
      ext=".gif";
    case SNAPSHOT_TIFF:
      ext=".tif";
    case SNAPSHOT_PNM:
    default:
      ext=".pnm";
    }
    // dont overwrite existing files...
    bool file_exists=true;
    while (file_exists) {
      QTextStream oname( name, IO_WriteOnly );
      oname << prefix << HEX(3) << shotnumber << ext << char(0);
      ifstream os_t(name);
      if (! os_t) {
	file_exists=false;
	os_t.close();
      } else {
	shotnumber++;
	os_t.close();
      }
    }

    QTextStream oname( name, IO_WriteOnly );
    oname << prefix << HEX(3) << shotnumber << ext << char(0);

    switch (format) {
#ifdef HAVE_LIBJPEG
    case SNAPSHOT_JPEG:
      {
	int rs,k;
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	FILE *outfile; 

	if (!(outfile=fopen((const char*)name, "wb"))){
	  warning("screen: Can't open file %s: %s",(const char*)name,strerror(errno));
	  return;
	}

	flip(tm, width*height);

	cinfo.err=jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
	jpeg_stdio_dest(&cinfo,outfile);
	cinfo.image_width=width;
	cinfo.image_height=height;
	cinfo.input_components=bpp;
	cinfo.in_color_space=JCS_RGB;
      
	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo,quality,true);
    
	jpeg_start_compress(&cinfo,true);
      
	rs=width*3;
	JSAMPROW rp[height];
	for (k=0; k<height; k++)
	  rp[k]=(unsigned char *)&tm[rs*k];
	jpeg_write_scanlines(&cinfo,rp,height);
      
	/* while(cinfo.next_scanline < cinfo.image_height){
	   JSAMPROW rp[1];
	   rp[0]=& mem[rs*cinfo.next_scanline];
	   int sl=jpeg_write_scanlines(&cinfo,rp,1);
	   }
	   */
	jpeg_finish_compress(&cinfo);
	jpeg_destroy_compress(&cinfo);

	fclose(outfile);
      }
    break;
#endif 
    case SNAPSHOT_GIF:
    case SNAPSHOT_TIFF:
    case SNAPSHOT_PNM:
    default:
      {
	ofstream os((const char*)name);
	if (!os) {
	  warning("screen: Can't open file %s: %s",(const char*)name,strerror(errno));
	  return;
	}
	os << "P6\n" << dec << width << endl << dec << height << "\n255\n";
      
	// FIX ME do we need this??
	flip(tm, width*height);
	os.write(tm,width*height*3);
	os.close();
	chown(name,getuid(),getgid());
      }
    }

#ifdef tvDEBUG
    debug("screen: snap shot %s",(const char*)name);
#endif

    if ( chInfo > OSD_LEVEL_1 ) 
      showOSDSLOT(NULL,CHINFO_NODISPVAL,0,0,(const char*)(QString("wrote ")+name));

  } // captureIsOn
}

//==============================================================================

void winTVScreenWidget::startVideoClipSLOT(struct MOVIE_PARAMS *movieparams)
{
  if ( !video_aktive && captureIsOn ) {
    int palette;
    video_aktive=1;

    switch (movieparams->video_clip) {
    case VIDEOCLIP_AVI: // avi
      video= new aviVideoClip();
      palette= avi_formats[movieparams->video_format].bt848;
      break;
    case VIDEOCLIP_PPM: // ppm
      video= new ppmVideoClip();
      palette= file_formats[movieparams->video_format].bt848;
      break;
    case VIDEOCLIP_RAW: // raw
      video= new rawVideoClip();
      palette= raw_formats[movieparams->video_format].bt848;
      break;
    default:
      fatal("screen: unsupported video format %d",movieparams->video_clip);
    }

#ifdef tvDEBUG
    debug("screen: opening video-file %s",movieparams->filename);
    debug("screen: audio: ch %d, b %d, r %d",movieparams->channels,movieparams->bits,
	  movieparams->rate);
#endif
    QString fname(movieparams->filename);
    fname.append(VIDNAME);

    video->open( (char *)(const char *)fname, movieparams );
    v4lx->grabSetParams( true, &(movieparams->width), &(movieparams->height), palette );

    writeVideoClipFrameSLOT();
  }
}

//==============================================================================

void winTVScreenWidget::writeVideoClipFrameSLOT()
{
  long lostframes=0;
  long frames=0;
  int fps=video->parameter()->fps;
  int lfps=fps-1;

  emit lostVideoClipFrameSIGNAL(lostframes);
  qApp->processEvents( 1000 ); 

  videoTime.start();

  while (video_aktive) {
    
    qApp->processEvents( 20 ); 

    elapsed=videoTime.elapsed();
    if ( (((frames * 1000.0) / (double)elapsed) <= fps) ) {

      video->writeframe( v4lx->grabCapture( false ) );

      if (!(frames % 5)) emit wroteVideoClipFrameSIGNAL(frames);
      frames++;

      elapsed=videoTime.elapsed();
      while ( ((frames * 1000.0) /(double)elapsed) < lfps ) {
	// lost a frame!
	lostframes++;
	frames++;
	elapsed=videoTime.elapsed();
      }

      if ((lostframes % 5)) emit lostVideoClipFrameSIGNAL(lostframes);
    } 
  }

  elapsed=videoTime.elapsed();

  video->close();

  frames=video->nframes();
  emit wroteVideoClipFrameSIGNAL(frames);
  emit statVideoClipFrameSIGNAL((frames*1000.0)/(double)elapsed);
  emit lostVideoClipFrameSIGNAL(lostframes);

  delete video;

#ifdef tvDEBUG
  debug("screen: wrote %d frames, %d msec. , %3.1f fps",frames,elapsed,(double)frames/elapsed*1000);
#endif
}

//==============================================================================

void winTVScreenWidget::endVideoClipSLOT()
{
  if (video_aktive) {
    video_aktive=0;
#ifdef tvDEBUG
    debug("screen: video clip: stop");
#endif
  }
}

//==============================================================================

void winTVScreenWidget::changeFrequencySLOT(int value)
{
  if ( defaultFreq != value ) {
    defaultFreq=value;
    v4lx->setFreq( value );
  }
  
#ifdef tvDEBUG
  debug("screen: changeFrequencySLOT %d",value);
#endif
}

//==============================================================================

void winTVScreenWidget::changeCaptureSLOT(int value)
{
  captureState=!captureState;
  capture(value);
}

//==============================================================================

void winTVScreenWidget::changeBrightnessSLOT(int bri)
{
#ifdef tvDEBUG
  debug("screen: changeBrightnessSLOT %d",bri);
#endif
  if (Bright != bri)
    {
      Bright=bri;
      v4lx->setPicBrightness( bri );
      if ( chInfo > OSD_LEVEL_2 ) 
	showOSDSLOT(NULL,bri,-127,127,"Brightness");
    }
}

//==============================================================================

void winTVScreenWidget::changeContrastSLOT(int contr)
{
#ifdef tvDEBUG
  debug("screen: changeContrastSLOT %d",contr);
#endif
  if (Contrast != contr)
    {
      Contrast=contr;
      v4lx->setPicConstrast( contr );
      if ( chInfo > OSD_LEVEL_2 ) 
	showOSDSLOT(NULL,contr,0,511,"Contrast");
    }
}

//==============================================================================

void winTVScreenWidget::changeColorSLOT(int color)
{
#ifdef tvDEBUG
  debug("screen: changeColorSLOT %d",color);
#endif
  if (Color != color)
    {
      Color=color;
      v4lx->setPicColor( color );
      if ( chInfo > OSD_LEVEL_2 ) 
	showOSDSLOT(NULL,color,0,511,"Color");
    }
}

//==============================================================================

void winTVScreenWidget::changeHueSLOT(int hue)
{
#ifdef tvDEBUG
  debug("screen: changeHueSLOT %d",hue);
#endif
  if (Hue != hue)
    {
      Hue=hue;
      v4lx->setPicHue( hue );
      if ( chInfo > OSD_LEVEL_2 ) 
	showOSDSLOT(NULL,hue,-127,127,"Hue");
    }
}

//==============================================================================

void winTVScreenWidget::changeSourceSLOT(int s)
{
#ifdef tvDEBUG
  debug("screen: changeSourceSLOT %d",s);
#endif
  if (defaultInput != s){
    defaultInput=s;
    v4lx->setChannel( s );
  }
}

//==============================================================================

void winTVScreenWidget::changeNormSLOT(int n)
{
#ifdef tvDEBUG
  debug("screen: changeNormSLOT %d",n);
#endif
  if (n != defaultNorm) {
    //v4lx->setTunerMode( n );
    v4lx->setChannelNorm( n );
    defaultNorm=n;
    setTVMaxSize();
    newTVGeometry( true );
  }
}

//==============================================================================

void winTVScreenWidget::changeChannelValuesSLOT(channel *c, int id)
{
  if (!c) return;

#ifdef tvDEBUG
  debug("screen: change channel values: %s, %d",c->name,c->frequency);
#endif

  switch_capture(0);

  if ( !muteState && v4lx->capChannelTuner(c->input) ) { // not muted -> mute for some milisecs...
    v4lx->setAudioMute(true);
    if ( muteTimer->isActive() ) muteTimer->stop();
    muteTimer->start(muteDelay);
  }

  changeSourceSLOT(c->input);

  if ( v4lx->capChannelTuner(c->input) ) {
    changeFrequencySLOT(c->frequency);
    changeNormSLOT(c->norm);
  }

  changeBrightnessSLOT(c->brightness);
  changeColorSLOT(c->color);
  changeHueSLOT(c->hue);
  changeContrastSLOT(c->contrast);

  switch_capture(1);

  c->id=id;
  if ( chInfo > OSD_LEVEL_OFF ) 
    showOSDSLOT(c,CHINFO_NODISPVAL,0,0,NULL);

#ifdef tvDEBUG
  debug("screen: changed values");
#endif
}

//==============================================================================

void winTVScreenWidget::setMuteStateAfterFChangeSLOT()
{
#ifdef tvDEBUG
  debug("screen: setMuteStateAfterFChangeSLOT: unmuted=%s",muteState?"no":"yes");
#endif
  muteTimer->stop();
  if ( !muteState ) { // not muted -> unmute again
    v4lx->setAudioMute(false);
  }
}

//==============================================================================

void winTVScreenWidget::toggleAudioMuteSLOT()
{
  muteState=!muteState;

#ifdef tvDEBUG
  debug("screen: changeMuteSLOT, muted=%s",muteState?"yes":"no");
#endif

  v4lx->setAudioMute(muteState);

  if (muteState) {
    if ( chInfo > OSD_LEVEL_1 ) 
      showOSDSLOT(NULL,CHINFO_NODISPVAL,0,0,"Sound muted");
  } else {
    if ( chInfo > OSD_LEVEL_1 ) 
      showOSDSLOT(NULL,CHINFO_NODISPVAL,0,0,"Sound unmuted");
  }
}

//==============================================================================

void winTVScreenWidget::changeAudioVolumeSLOT(int vl, int vr)
{
#if 0
#ifdef tvDEBUG
    debug("screen: changeAudioVolumeSLOT set audio volume to %d, %d",vl,vr);
#endif
  if (!audioVolumeCan) return;

  v4lx->setAudioVolume( vr );
  v4lx->setAudioBalance( vl );
#endif
}

//==============================================================================

void winTVScreenWidget::changeAudioModeSLOT(int mode)
{
  if ((mode > 0) && (mode < 5)) {
#ifdef tvDEBUG
    debug("screen: changeAudioModeSLOT set audio mode to %s",audiomode[mode].str);
#endif
    if (audioDetectTimer->isActive()) audioDetectTimer->stop();
    audioMode=audiomode[mode].nr;
    // try to set mode...
    v4lx->setAudioMode( audioMode );
    audioMode=v4lx->getAudioMode();
    emit autoDetectAudioModeSIGNAL(audioMode);
    if ( chInfo > OSD_LEVEL_1 ) 
      showOSDSLOT(NULL,CHINFO_NODISPVAL,0,0,audiomode[audioMode].str);
  } else {
    // autodetection, mode==0
    // set up timer
    audioMode=audiomode[mode].nr;
    if (audioDetectTimer->isActive()) audioDetectTimer->stop();
    audioDetectTimer->start(AUDIOMODE_DELAY);
  }

}

//==============================================================================

void winTVScreenWidget::changeTVMultiDisplaySLOT(int x, int y, int w, int h)
{
#ifdef tvDEBUG
  debug("screen: changeTVMultiDisplaySLOT %d, %d, %d, %d",x,y,w,h);
#endif
}

//==============================================================================

void winTVScreenWidget::toggleScreenSaverSLOT()
{
#ifdef HAVE_LIBXDPMS
  CARD16 dpms_state;
  int dpms_dummy;
#endif
  int t,i,p,a;
  Display *displ=qt_xdisplay();
  

  saverState=!saverState;
  if (saverState) { // activate screensaver
    XGetScreenSaver(displ,&t,&i,&p,&a);
    if (t == 0 && i == ss_interval && p == ss_prefer_blanking && a == ss_allow_exposures)
      XSetScreenSaver(displ,ss_timeout,ss_interval,ss_prefer_blanking,ss_allow_exposures);
#ifdef HAVE_LIBXDPMS
    if ( (DPMSQueryExtension(displ, &dpms_dummy, &dpms_dummy) ) && 
	 (DPMSCapable(displ)) && (dpms_on) ) {
      DPMSEnable(displ);
    }
#endif
  } else { // deactivate screensaver
#ifdef HAVE_LIBXDPMS
    if ( (DPMSQueryExtension(displ, &dpms_dummy, &dpms_dummy)) && (DPMSCapable(displ)) ) {
      DPMSInfo(displ, &dpms_state, &dpms_on);
      DPMSDisable(displ);
    }
#endif
    XGetScreenSaver(displ,&ss_timeout,&ss_interval,&ss_prefer_blanking,&ss_allow_exposures);
    XSetScreenSaver(displ,0,ss_interval,ss_prefer_blanking,ss_allow_exposures);
  }
#ifdef tvDEBUG
  debug("screen: screensaver %s",saverState?"activated":"deactivated");
#endif
}

//==============================================================================

void winTVScreenWidget::toggleMouseCursorSLOT()
{
  cursorState=!cursorState;

#ifdef tvDEBUG
  debug("screen: cursorState = %s",cursorState?"off":"on");
#endif

  if (!cursorState) {
    orgCursor=cursor();
    setCursor(blankCursor);
  } else {
    setCursor(orgCursor);
  }
}


//==============================================================================

void winTVScreenWidget::showOSDEnableSLOT( int level )
{
  chInfo= level;
}

//==============================================================================

void winTVScreenWidget::showOSDSLOT(channel *ch,
					    int val, int min_val, int max_val, 
					    const char *text)
{
  int wx,wy;
  unsigned int wwidth,wheight;
  
  if ( !isVisible() || (chInfo == OSD_LEVEL_OFF) ) return;

  v4lx->getCapAClip( &wx, &wy, &wwidth, &wheight );

  chInfoText->setGeometry(wx+(wwidth/10),wy+wheight-(wheight/10)-10,
			  (wwidth/10 * 8),(wheight/10));
  
  chInfoBar->setGeometry(wx+(wwidth/10),wy+wheight-(wheight/10)*2,
			 8*(wwidth/10),wheight/20);
  
  if (ch) {
    chInfoBar->hide();

    if (!chInfoText->isVisible()) chInfoText->show();
    chInfoText->raise();

    QString tout;
    tout.setNum(ch->id);
    tout.insert(0," ");
    tout+=" -- ";
    tout+=ch->name;
    tout=tout.leftJustify(25,' ');
    tout+=QTime::currentTime().toString();
    chInfoText->setText(tout);
  }

  if ( text ) {
    chInfoBar->hide();

    if (!chInfoText->isVisible()) chInfoText->show();
    chInfoText->raise();
    //    chInfoText->repaint();
    QString tout(" ");
    tout+=text;
    chInfoText->setText(tout);
  }

  if ( val != CHINFO_NODISPVAL ) {
    if (! text ) chInfoText->hide();
    if (!chInfoBar->isVisible()) chInfoBar->show();
    //chInfoBar->setTotalSteps(max_val);
    chInfoBar->reset();
    chInfoBar->setProgress( (100/max_val)*val );
    chInfoBar->raise();
    //chInfoBar->repaint();
  }

  if (infoTimer->isActive()) infoTimer->stop();
  infoTimer->start(infodelay,true);
#ifdef tvDEBUG
  debug("screen: showOSDSLOT");
#endif
}

//==============================================================================

void winTVScreenWidget::unShowOSDSLOT()
{
  if (chInfoText->isVisible()) {
    chInfoText->hide();
  }
  if (chInfoBar->isVisible()) 
    chInfoBar->hide();

  infoTimer->stop();
}

//==============================================================================
//==============================================================================
// never change the next line!
#include "wintvscreen.moc"

// end of file









