#include <iostream.h>
#include "StringToken.h"
#include "fileListBox.h"
#include "fileListBox.moc"

void FileFilterBox::init(const char* Filter)
{
  setInsertionPolicy(QComboBox::AtTop);
  StringToken st(Filter," :");
  QStrList list;
  list.setAutoDelete(true);
  QString s;
  while(st.hasMoreTokens()){
    s = st.nextToken();
    list.append(s);
  }
  insertStrList(&list);
  setCurrentItem(0);
}


FileFilterBox::FileFilterBox(QWidget* parent, const char* Name, const char* Filter) :
  QComboBox(parent, Name)
{
  init(Filter);
}

FileFilterBox::FileFilterBox(bool flag, QWidget* parent, const char* Name, const char* Filter) :
  QComboBox(flag, parent, Name)
{
  init(Filter);
}

void FileFilterBox::setFilter(const char* Filter)
{
  if (!Filter && count()>0){
     if (currentItem()<0)
        setCurrentItem(0);
     emit activated(currentText());
     return;
  }
  QString s=Filter;
  for (int i=0;i<count();i++){
     if (!stricmp(s, text(i))){
        setCurrentItem(i);
	emit activated(s);
	return;
     }
  }
  insertItem(s, 0);
  emit activated(s);
}

void FileFilterBox::setFilterList(QStrList& list)
{
  clear();
  insertStrList(&list);
  if (list.count())
     emit activated(text(0));
}


// FileDirBox

FileDirBox::FileDirBox(QWidget* parent, const char* name, const char* Dir):
  QComboBox(parent, name)
{
  dir.setFilter(QDir::Dirs|QDir::Hidden);
  connect(this,
	  SIGNAL(activated    (int)),
	  SLOT  (slotActivated(int)));
  if (Dir)
     setPath(Dir);
}

FileDirBox::~FileDirBox()
{
}

void FileDirBox::setPath(const char* path)
{
#ifdef EDEBUG
  cout << "PATH:"<<path<<endl;
#endif
  if (!stricmp(path, ".."))
     dir.cdUp();
  else
     dir.setPath(path);

  dir.convertToAbs();
  curDir=dir.absPath();

  clear();
  QStrList list;
  list.setAutoDelete(true);
  list.append(curDir);

  const QFileInfoList *il = dir.entryInfoList();
  if (il){
     QFileInfoListIterator it(*il); 
     QFileInfo* fi; 
  
     while((fi=it.current())!=0L){
       list.append(fi->fileName());
       ++it; 
     }
  }
  insertStrList(&list);
  emit signActivated(curDir);
}
  
void FileDirBox::slotActivated(int idx)
{
  QString s=text(idx);
  if (!stricmp(s, curDir))
     setPath(curDir);
  else
     setPath(curDir+"/"+s);
}


// FileListBox
FileListBox::FileListBox(QWidget* parent, const char* name, int column) :
  KTabListBox(parent, name, column)
{
  curFilter = "*";
  curDir    = "" ;

  setColumn(0, "Filename");
  setColumn(1, "Size");
  setColumnWidth(0, 150);
  setColumnWidth(1, 100);
  setSeparator('\t');
}


void FileListBox::slotFilterActivated(const char* filter)
{
#ifdef EDEBUG
  cout << "FileListBox::slotFilterActivated("<<filter<<")"<<endl;
#endif
  if (curDir.isNull() || curDir.isEmpty())
     return;
  curFilter = filter;
  slotDirActivated(curDir);
}

// dir is absolute path
void FileListBox::slotDirActivated(const char* Dir)
{
#ifdef EDEBUG
  cout << "FileListBox::slotDirActivated("<<Dir<<")"<<endl;
#endif
  curDir = Dir;

  QDir dir(curDir);
  dir.setNameFilter(curFilter);
  dir.setFilter(QDir::Files|QDir::Hidden|QDir::NoSymLinks);

  clear();
  QStrList list;
  list.setAutoDelete(true);

  const QFileInfoList *il = dir.entryInfoList();
  if (il){
     QFileInfoListIterator it(*il); 
     QFileInfo* fi; 
     QString s;
     while((fi=it.current())!=0L){
       QString ss;
       ss.setNum(fi->size());
       s = fi->fileName()+separator()+ss;
       list.append(s);
       ++it; 
     }
  }
  appendStrList(&list);
}


// FileBox
FileBox::FileBox(QWidget* parent, const char* name, const char* filter, const char* dir) :
  QWidget(parent, name)
{
  topLayout = new QVBoxLayout(this, 0);

  filterBox = new FileFilterBox(TRUE, this,"filterBox", filter);
  topLayout->addWidget(filterBox);
  filterBox->setFixedHeight(25);

  dirBox = new FileDirBox(this,"dirBox", dir);
  topLayout->addWidget(dirBox);
  dirBox->setFixedHeight(25);
  topLayout->addSpacing(10);

  frame = new QFrame(this);
  frame->setLineWidth(2);
  frame->setFrameStyle(QFrame::Panel|QFrame::Sunken);
  topLayout->addWidget(frame, 10);
  listBox = new FileListBox(frame);

  topLayout->activate();

  connect(filterBox, SIGNAL(activated(const char*)),
	  listBox,   SLOT  (slotFilterActivated(const char*)));

  connect(dirBox,    SIGNAL(signActivated(const char*)),
	  listBox,   SLOT  (slotDirActivated(const char*)));

}

void FileBox::resizeEvent(QResizeEvent*)
{
  listBox->setGeometry(3,
		       3,
		       frame->width()-6,
		       frame->height()-6);
		       
}

QString FileBox::selectedFile()
{
  int idx = listBox->currentItem();
  return QString(dirBox->currentText())+"/"+listBox->text(idx, 0);
}

QString FileBox::fileSize()
{
  int idx = listBox->currentItem();
  return listBox->text(idx, 1);
}

void FileBox::refresh()
{
  filterBox->setFilter(NULL);
}





