#include <time.h>
#include <iostream.h>
#include <qregexp.h> 
#include "ircUser.h"
#include "StringToken.h"
#include "ircClient.h"

QString toUsermask(const char* ident)
{
  // Format: nick!user@host
  QString nick="*";
  QString user="*";
  QString host="*";
  
  StringToken st(ident);
  if (st.hasMoreTokens(" !")){
     nick=st.nextToken();
     if (st.hasMoreTokens("!@")){
        user=st.nextToken();
	if (st.hasMoreTokens(" @"))
	   host=st.nextToken();
     }
  }
  
  return  nick+"!"+user+"@"+host;
}

bool isChannel(const char* c)
{
  if (c && (*c=='&' || *c=='#'))
     return true;
  return false;
}


IrcUser::IrcUser(const char* nick, const char* user, const char* host):
  IrcCore(0, nick)
{
  lastActive  = time(0L);
  activeCount = 0;
  if (!nick || !user || !host)
     return;

  mNick = nick;
  mUser = user;
  mHost = host;
  mInfo = "";
  mServer = "";
}

IrcUser::IrcUser(const char* name)
{
  lastActive  = time(0L);
  activeCount = 0;
  StringToken st(name);
  QString nick, user, host;
  if (!(nick=st.nextToken(" !@")))
     return;
  setName(nick);
  if (!(user=st.nextToken()))
     user="";
  if (!(host=st.nextToken()))
     host="";
  mNick = nick;
  mUser = user;
  mHost = host;
  mInfo = "";
  mServer = "";
}

QString IrcUser::toName()
{
  return mNick+"!"+mUser+"@"+mHost;
}

bool IrcUser::checkFlood(int inSecond, int maxLine)
{
#ifdef EDEBUG
  cout << "IrcUser::checkFlood:"<<inSecond<<":"<<maxLine<<endl;
#endif
  long curTime = time(0L);
  activeCount++;
  if ( (curTime-lastActive)<inSecond){
     if (activeCount>maxLine){
        lastActive  = curTime;
	activeCount = 0;
	return true;
     }
  }
  else {
     lastActive  = curTime;
     activeCount = 0;
  }
  return false;
}

FriendUser::FriendUser(const char* ident, int level, const char* c)
{
  mIdent = toUsermask(ident);
  mLevel = level;
  QString channel=c;
  if (channel.isEmpty())
     mChannel = "*";
  else if (c && (*c!='&' || *c!='#') && *c!='*')
     mChannel = "*";
  else
     mChannel = c;
}

FriendUser::~FriendUser()
{
}

void FriendUser::addChannel(const char* channel)
{
  if (!channel)
     return;
  if (*channel=='#' ||*channel=='&'){
     if (matchChannel(channel))
        return;
     mChannel += ":"+QString(channel);
  }
}

bool FriendUser::matchChannel(const char* channel)
{
  StringToken st(mChannel, ": ");
  while (st.hasMoreTokens()){
     QRegExp re(st.nextToken(), false, true);
     int pos, len;
     pos = re.match(channel, 0, &len);
      if (len==(int)strlen(channel))
	 return true;
  }
  return false;
}

FriendList::FriendList(IrcClient* client)
{
  this->client = client;
  mList.setAutoDelete(true);
  mUpdate = false;
}

FriendList::~FriendList()
{
}

// Internal, wird von abgeleiteten Klasse aufgerufen
void FriendList::delUserIn(const char* list, const char* ident, bool wildcard)
{
  FriendUser* fu;
  QString s = toUsermask(ident);
  for (fu=mList.first();fu!=0L;){
      if (wildcard){
	 QRegExp re(fu->mIdent, false, wildcard);
	 int pos, len;
	 pos = re.match(s, 0, &len);
	 if (len!=(int)s.length()){
	    fu = mList.next();
	    continue;
	 }
      }
      else{
	 if (stricmp(fu->mIdent, s)){
	    fu = mList.next();
	    continue;
	 }
      }
      mList.remove();
      fu=mList.current();
      mUpdate=true;
      QString s(1024);
      s.sprintf("Remove %s : %s", list, ident);
      client->slotWriteMsg(TYPE_INFO|TYPE_IMG, s);
  }
}

FriendUser* FriendList::getUser(const char* ident, bool wildcard)
{
  FriendUser* fu;
  QString s = toUsermask(ident);
  for (fu=mList.first();fu!=0L;fu=mList.next()){
      if (wildcard){
	 QRegExp re(fu->mIdent, false, wildcard);
	 int pos, len;
	 pos = re.match(s, 0, &len);
	 if (len==(int)s.length())
	    return fu;
      }
      else{
	 if (!stricmp(fu->mIdent, s))
	    return fu;
      }
  }
  return 0L;
}

int FriendList::matchUser(const char* ident, int level, bool wildcard, const char* channel, int* ret)
{
  FriendUser* fu;
  QString s = toUsermask(ident);
  for (fu=mList.first();fu!=0L;fu=mList.next()){
      QRegExp re(fu->mIdent, false, wildcard);
      int pos, len;
      pos = re.match(s, 0, &len);
      if (len==(int)s.length()){
	 if (channel && !fu->matchChannel(channel))
	    continue;
	 if (ret != 0L)
	    *ret = fu->mLevel;
	 if (level < fu->mLevel)
	    return 0;
	 else
	    return 1;
      }
  }
  return -1;
}

FriendUser* FriendList::addUser(const char* ident, int level, const char* channel)
{
   FriendUser* fu = getUser(ident);
   if (fu==0L){
      mList.append(fu=new FriendUser(ident, level, channel));
      mUpdate=true;
      QString s(1024);
      s.sprintf("Add Friend : %s level %d on channel %s", ident, level, channel);
      client->slotWriteMsg(TYPE_INFO|TYPE_IMG, s);
      return fu;
   }
   return 0L;
}

void FriendList::delUser(const char* ident, bool wildcard)
{
  delUserIn("Friend", ident, wildcard);
}







