#include "kldap.h"

Kldapw::Kldapw(QWidget* parent) : KTreeList(parent)
{
  refill = true;
  bound = false;
  loader = kapp->getIconLoader();

  setBackgroundColor(*(new QColor(255, 255, 255)));
  setAutoUpdate(true);
  setTreeDrawing(true);

  QObject::connect(this, SIGNAL(expanded(int)), this, SLOT(expandLDAPTree(int)));
  QObject::connect(this, SIGNAL(selected(int)), this, SLOT(selectedSlot(int)));
}

void Kldapw::expandLDAPTree(int itemIndex)
{
  if(isRefill())
  {
    LDAPItem* item;
    char* dn;

    item = (LDAPItem*)itemAt(itemIndex);
    dn = strdup(item->getDN());
    buildTree(dn, itemIndex);
  }
}

void Kldapw::selectedSlot(int index)
{
  /* Expanded oder collapsed Tree wenn Childs vorhanden sind */
  if(itemAt(index)->hasChild())
  {
    expandOrCollapseItem(index);
  }
  /* Sind keine Childs vorhanden wird mit expandTree eine Abfrage gestartet*/
  else
  {
    expandLDAPTree(index);
    if(itemAt(index)->hasChild())
    {
	    setRefill(false); // Verhindert das erneute Auffllen durch expandCollapseItem()
	    expandOrCollapseItem(index);
	    setRefill(true);
	  }
   }
}

void Kldapw::buildTree(char* dn, int index, int scope)
{
  LDAPMessage* result;
  LDAPMessage* ep = NULL;
  char** objectclass;
  QPixmap pix;
  int childIndex = 0;
  int error;
  LDAPItem* newItem;
  char* displayName;
  int i = 0;

  /*Zuerst die bestehenden Elemente lschen weil zwichenzeitlich ev. nderungen aufgetreten sind*/
  if(visibleCount() > 0) // Schaut, ob ueberhaupt Elemente zu loeschen sind
  {
    for(childIndex = itemIndex(itemAt(index)->getChild()); childIndex != -1; childIndex = itemIndex(itemAt(index)->getChild()))
    {
	    removeItem(childIndex);
	  }
  }
  
  /* Ldap Anbfrage starten */
  if(dn == NULL || strcmp(dn, "") == 0) dn = NULL;
  error = ldap_search_s(ldap, dn, scope, "objectClass=*", NULL, 0, &result);
  
  if(error == 0)
  {
    error = ldap_sort_entries(ldap, &result, NULL, strcasecmp);
    cout << ldap_err2string(error);

    /* Fr jeden gefundenen Eintrag ein Element mit entsprechendem Bild hinzufgen */
    for(ep = ldap_first_entry(ldap, result); ep != NULL; ep = ldap_next_entry(ldap, ep))
	  {
	    objectclass = (char**)ldap_get_values(ldap, ep, "objectclass");
	    if(strcmp(objectclass[i], "top") == 0) // Achtung. Ohne == 0 gibt es ein segfault
	    {
        i++;
      }

	    displayName = (ldap_explode_dn(ldap_get_dn(ldap, ep), 1))[0];

	    /*
	       Z.B. beim Novell LDAP-Server (gesehen bei Version1.03) ist es moeglich, das er ein "Unknown" Objekt zurueckgibt.
	       Das hat keine DN. Weil displayName dann keine Wert hat, weist folgende Zeile ihm einen Wert zu ("Unknown")
	    */
	    if(strcmp(displayName, "")==0) displayName = "Unknown";

	    newItem = new LDAPItem(displayName, &(pix = getPixmap(objectclass[i])), ldap_get_dn(ldap, ep));
	  
	    if(visibleCount() > 0)
      {
       addChildItem(newItem, index);
      }
	    else
      {
        insertItem(newItem);
      }
    }
  }
  else
  {
    if(error == LDAP_NO_SUCH_OBJECT)
      QMessageBox::warning(this, "kldap", "This server needs a DN to connect." "Please enter one. (Error: NO SUCH OBJECT)", "OK", 0);
    else
      QMessageBox::warning(this, "kldap operations", ldap_err2string(error), "OK", 0);
  }

  ldap_msgfree(result);
  ldap_msgfree(ep);
}

bool Kldapw::connect(char* host, int port, char* rootDN)
{
  QPixmap pix;
  clear();
  bool tryAgain = true;
  bool success = false;

  if(port == 0) port = LDAP_PORT;
  this->rootDN = rootDN;
  
  while(tryAgain)
  {
    ldap = ldap_open(host, port);
    if(ldap == NULL)
	  {
      switch(QMessageBox::warning( this, "kldap", "Could not connect to the server.\n", "Try again", "Cancel", 0, 1))
	    {
	    case 0: 
	      tryAgain = true;
	      break;
	    case 1: 
	      tryAgain = false;
	      return (success = false);
	      break;
	    }
    }
    else
    {
	    bind("", "", LDAP_AUTH_SIMPLE);
	    tryAgain = false;
   
	  /*if(rootDN == NULL || (strcmp(rootDN, "")==0))
	    {
	      cout << "rootDN = NULL !!!!!!!!!!\n";
	      buildTree(rootDN, 0, LDAP_SCOPE_BASE);
	      //insertItem(new LDAPItem("[ROOT]", &(pix = getPixmap("root")), ""));
	    }
	  else 
	    {
	      buildTree(rootDN, 0, LDAP_SCOPE_BASE);
	      //insertItem(new LDAPItem(rootDN, &(pix = getPixmap("world")), rootDN));
	      }*/

	    buildTree(rootDN, 0, LDAP_SCOPE_BASE);

	    return (success = true);
    }
  }

  return false; // etwas unelegant. Aber sonst gibt es 'ne Warnung
}

void Kldapw::bind(char* who, char* cred, int method)
{
  int error;
  const char* errmsg;

  error = ldap_bind_s(ldap, who, cred, method);

  if(error == 0)
  {
    bound = true;
  }
  else
  {
    errmsg = (new QString("Could not bind: "))->append(ldap_err2string(error));
    //bound  = false;
    QMessageBox::warning(this, "kldap", errmsg, "OK", 0);
  }
}

void Kldapw::unbind()
{
  int error;
  const char* errmsg = NULL;

  if(bound)
  {
    error = ldap_unbind_s(ldap);

    if(error == LDAP_SUCCESS)
	  {
	    bound = false;
	  }
    else
	  {
	    errmsg = (new QString("Could not unbind: "))->append(ldap_err2string(error));
	    QMessageBox::warning(this, "kldap", errmsg, "OK", 0);
	  }
  }
  else
  {
    QMessageBox::warning(this, "kldap", errmsg, "You are not bound to the Server");
  }
}

int Kldapw::modify(char* dn, int mod_op, char* mod_type, char* value)
{
  LDAPMod ldapmod;
  LDAPMod* ldapmods[2] = { &ldapmod };
  int error = 0;
  char* values[2];

  if(isBound() == false)
  {
    QMessageBox::warning(this, "kldap", "You are not bound to the server", "OK", 0);
    return 1; // Ev. noch andere Fehlernummer
  }

  if(value == NULL)
  {
      //cout << "value = " << value << "\n";
  }
  else
  {
    values[0] = strdup(value);
  }
  values[1] = NULL;

  ldapmod.mod_op = mod_op;
  ldapmod.mod_type = (char*)mod_type;
  if(value == NULL)
  {
    //cout << "value = NULL\n";
    ldapmod.mod_values = NULL;
  }
  else
  {
    //cout << "values != NULL\n";
    ldapmod.mod_values = values;
  }
  
  error = ldap_modify_s(ldap, dn, ldapmods);

  if(error == LDAP_SUCCESS || error == LDAP_NO_SUCH_ATTRIBUTE) // NO_SUCH_ATTRIBUTE ist ein Workaround weil dieser Fehler immer angezeigt wird und ich nicht weiss warum.
  {
    return 0;
  }
  else
  {
    QMessageBox::warning(this, "kldap", ldap_err2string(error), "OK", 0);
    return error;
  }
}

void Kldapw::deleteEntry()
{
  int error;
  char* dn;
  LDAPMessage* result;
  LDAPMessage* ep;

  if(isBound() == false)
  {
    QMessageBox::warning(this, "kldap", "You are not bound to the server", "OK", 0);
    return;
  }

  dn = getSelectedDN();

  if(strcmp(dn, "") == 0)
  {
    QMessageBox::warning(this, "kldap", "No Item selected.\n", "OK", 0);
    return;
  }

  /* 
     Beim Versuch ein Objekt zu lschen das noch Childs hat, mach der OpenLdap-Server die Schraube 
     Darum berprfe ich mal zuerst ob er noch Childs hat
  */
  error = ldap_search_s(ldap, dn, LDAP_SCOPE_ONELEVEL, "objectClass=*", NULL, 0, &result);
  if(error == LDAP_SUCCESS)
  {
    ep = ldap_first_entry(ldap, result);
    if(ep != NULL) // Childs vorhanden
	  {
	    QMessageBox::warning(this, "kldap", "Container not emty\n", "OK", 0);
	    return;
	  }
  }
  else
  {
    QMessageBox::warning(this, "kldap", ldap_err2string(error), "OK", 0);
  }
 
  error = ldap_delete_s(ldap, dn);
 
  if(error == 0)
  {
    removeItem(currentItem());
  }
  else
  {
    char* errmsg = ldap_err2string(error);
    QMessageBox::warning(this, "kldap", errmsg, "OK", 0);
  }
}

QPixmap Kldapw::getPixmap(char* objectclass)
{
  QPixmap pix;
  const char* filename;
  
  QString s(objectclass);
  s = s.lower();
  s.append(".xpm");
  
  filename = s;

  pix = loader->loadIcon(filename);
  if(pix.isNull()) pix = loader->loadIcon("unknown.xpm");

  return pix;
}

void Kldapw::setRootDN(char* rootDN)
{
  QPixmap pix;
  clear();
  insertItem(new LDAPItem(rootDN, &(pix = getPixmap("world")), rootDN));
  this->rootDN = rootDN;
}

LDAP* Kldapw::getLDAP()
{
  return ldap;
}

char* Kldapw::getSelectedDN()
{
  int itemIndex = currentItem();
  if(itemIndex == -1) return "";

  return ((LDAPItem*)itemAt(itemIndex))->getDN();
}

void Kldapw::setldap(LDAP* ldap)
{
  this->ldap = ldap;
}

LDAP* Kldapw::getldap()
{
  return ldap;
}
  
char* Kldapw::getRootDN()
{
  return rootDN;
}

void Kldapw::setRefill(bool r)
{
  refill = r;
}

bool Kldapw::isRefill()
{
  return refill;
}

bool Kldapw::isBound()
{
  return bound;
}

/************************************************************/

LDAPItem::LDAPItem(char* text, QPixmap* pixmap, char* dn) : KTreeListItem(text, pixmap)
{
  this->dn = dn;
}

void LDAPItem::setDN(char* dn)
{
  this->dn = dn;
}

char* LDAPItem::getDN()
{
  return dn;
}



  

