/* Copyright (C) 1999 Claus Wilke <claus.wilke@gmx.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qlayout.h>
#include <qpushbutton.h>
#include <qstring.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qmultilineedit.h>
#include <qgroupbox.h>
#include <qdom.h>
#include <stdio.h>
#include <klocale.h>

#include "biblined.h" 
#include "bibentryfieldviews.h"
#include "kbiblioshell.h"
#include "entrymanipulator.h"


//--------------------------------------------------
//  BibEntryFieldView

BibEntryFieldView::BibEntryFieldView( QDomElement entry, const QString &fieldName, KBiblioShell *shell, QWidget *parent, const char *name )
    : QObject( parent, name )
{
    mEntry = entry;
    mFieldName = fieldName;
    mShell = shell;

    mManipulator = new EntryManipulator( mEntry, mShell );
}

BibEntryFieldView::~BibEntryFieldView()
{
    delete mManipulator;
}

void
BibEntryFieldView::storeData()
{
}

void
BibEntryFieldView::setFocus()
{
}


//--------------------------------------------------
//  StringFieldView


StringFieldView::StringFieldView( QDomElement entry, const QString &fieldName, KBiblioShell *shell, QGridLayout* gridLayout, int row, QWidget *parent, const char *name )
    : BibEntryFieldView( entry, fieldName, shell, parent, name )
{
    QLabel *label;

    label = new QLabel( mShell->fieldDescription( mFieldName ) + ":", parent );
    gridLayout->addWidget( label, row, 0 );
    mLineEdit = new BibLined( mShell->document()->lookupCompletionsList( mFieldName), parent );
    mLineEdit->setText( mManipulator->fieldFlatText( mFieldName ) );
    mLineEdit->setMinimumWidth( 40*mLineEdit->fontMetrics().width( "x" ) );
    gridLayout->addWidget( mLineEdit, row, 1 );

    connect( mLineEdit, SIGNAL( returnPressed() ), SIGNAL( returnPressed() ) );
    connect( mLineEdit, SIGNAL( textChanged ( const QString & ) ), SIGNAL( hasChanged() ) );

}

StringFieldView::~StringFieldView()
{
}


void
StringFieldView::storeData()
{
    mManipulator->setValue( mFieldName, mLineEdit->text() );
}



void
StringFieldView::setFocus()
{
    mLineEdit->setFocus();
}


//--------------------------------------------------
//  TextFieldView

TextDialogMainWidget::TextDialogMainWidget( QString *text, const QString &fieldDescr, QWidget *parent, const char *name )
    : QWidget( parent, name ) // Don't change to not-blocking dialog
    // as long as you don't know exactly what you are doing. The dialog doesn't
    // know how to handle a change in the list that comes from somewhere else!

{
    QBoxLayout *topLayout = new QVBoxLayout( this, 0 );

    QLabel *label = new QLabel( fieldDescr, this );
    topLayout->addWidget( label );


    meText = new QMultiLineEdit( this );
    meText->setWordWrap( QMultiLineEdit::DynamicWrap );
    meText->setText( *text );
    topLayout->addWidget( meText, 10 );

    topLayout->activate();
}

TextDialogMainWidget::~TextDialogMainWidget()
{
}


QString
TextDialogMainWidget::text( void ) const
{
    return meText->text();
}


TextDialog::TextDialog( QString *text, const QString &dialogTitle, const QString &fieldDescr, QWidget *parent, const char *name )
    : KDialogBase ( parent, name, true, dialogTitle, KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok, false )
    // Don't change to non-blocking dialog
    // as long as you don't know exactly what you are doing. The dialog doesn't
    // know how to handle a change in the list that comes from somewhere else!
{
    mText = text;
    mTextWidget =  new TextDialogMainWidget ( text, fieldDescr, this );

    setMainWidget ( mTextWidget );

    //  set to the size of 50 x'es and 15 lines
    //
    resize ( QSize( 60*fontMetrics().width( "x" ), 20*fontMetrics().lineSpacing() ) );
}

TextDialog::~TextDialog()
{
}

void
TextDialog::slotOk()
{
    *mText = mTextWidget->text();
    accept();
}


TextFieldView::TextFieldView( QDomElement entry, const QString &fieldName, KBiblioShell *shell, QGridLayout* gridLayout, int row, QWidget *parent, const char *name )
    : BibEntryFieldView( entry, fieldName, shell, parent, name )
{
    QBoxLayout *hbox;
    QLabel *label;

    label = new QLabel( mShell->fieldDescription( mFieldName ) + ":", parent );
    gridLayout->addWidget( label, row, 0 );

    hbox = new QHBoxLayout();

    mTextLine = new QLabel( parent );
    mTextLine->setFrameStyle( QFrame::Panel | QFrame::Sunken );
    mTextLine->setMinimumWidth( 40*mTextLine->fontMetrics().width( "x" ) );
    hbox->addWidget( mTextLine, 10 );

    mEditPush = new QPushButton( i18n( "Edit" ), parent );
    mEditPush->setMaximumSize( mEditPush->sizeHint() );
    hbox->addWidget( mEditPush );
    gridLayout->addLayout( hbox, row, 1 );

    // connect the button
    connect( mEditPush, SIGNAL( clicked() ), SLOT( editText() ) );

    connect( this, SIGNAL( hasChanged() ), SLOT( displayData() ) );

    displayData();
}

TextFieldView::~TextFieldView()
{
    // nothing needs to be done
}

void
TextFieldView::editText()
{
    QString tmpText = mManipulator->fieldFlatText( mFieldName );
    QString descr =  mShell->fieldDescription( mFieldName );
    TextDialog *ed = new TextDialog ( &tmpText, descr + " " + i18n( "Dialog" ), descr + ":", 0 );
    ed->exec();

    if ( ed->result() == QDialog::Accepted ){
	mManipulator->setValue( mFieldName, tmpText );
	emit hasChanged();
    }

    delete ed;
}

void
TextFieldView::displayData()
{
    QString line = mManipulator->fieldFlatText( mFieldName, 45 );
    if ( line.isEmpty() ) return;
    
    line += "...";
		   
    mTextLine->setText( line );
}


#include "bibentryfieldviews.moc"








