/*  pcl727.c
 *  (c) 1997,98 Samuel Kvasnica
 * 
 *  This is the Linux PCL-727 driver.
 * 
 * 
 *  Code of MLP-16 driver written by Sebastian Kuzminsky used
 */


#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/timer.h>

#include <linux/fcntl.h>

#include <asm/segment.h>
#include <asm/io.h>


#include "pcl727-driver.h"

#include "pcl727.h"

#define PCL727_DEBUG

#define BASE_ADDRESS 0x2c0

unsigned short base=BASE_ADDRESS;

#ifdef PCL727_DEBUG
   #define DPRINTK1(a)          printk(PCL727_ID a)
   #define DPRINTK2(a, b)       printk(PCL727_ID a, b)
   #define DPRINTK3(a, b, c)    printk(PCL727_ID a, b, c)
   #define DPRINTK4(a, b, c, d) printk(PCL727_ID a, b, c, d)
   #define DPRINTK5(a, b, c, d, e) printk(PCL727_ID a, b, c, d, e)
#else
   #define DPRINTK1(a)
   #define DPRINTK2(a, b)
   #define DPRINTK3(a, b, c)
   #define DPRINTK4(a, b, c, d)
   #define DPRINTK5(a, b, c, d, e)
#endif

/*  Global structures for card info.                                         */
static struct pcl727_data pcl727_card;


static int pcl727_read(
   struct inode *inode,
   struct file *file,
   char *buffer,
   int count
) {
//   int i;
//   unsigned char c;
   unsigned int minor;
   minor = MINOR(inode->i_rdev);

   if(minor == PCL727_MINOR_DIO) {
/*      for(i = 0; i < count; i ++) {
	 c = PCL727_DIGITAL_GET(cardnum);
	 put_fs_byte(c, buffer);

	 DPRINTK3("card %c, digital lines, read 0x%02x\n", 'A' + cardnum, c);

	 buffer ++;
      }

      return(count);*/
   }

   printk(PCL727_ID " bungled minor number %d\n", minor );
   return(0);
}

static int pcl727_write(
   struct inode *inode,
   struct file *file,
   const char *buffer,
   int count
) {
   int i;

   unsigned int minor = MINOR(inode->i_rdev);

   if(minor < PCL727_MINOR_DIO) {
      for(i = 0; i+1 < count; i+=2) {
	 pcl727_card.dac[minor]=get_fs_word(buffer);
	 PCL727_DAC_SET(get_fs_word(buffer), minor);
 	 DPRINTK3("dac %d, wrote 0x%04x\n", minor, (unsigned short)get_fs_word(buffer));
	 buffer+=2;
      }

      return(count);
   }

/*  To make the compiler happy:                                              */
   return(0);
}

static int pcl727_ioctl(
   struct inode *inode,
   struct file *file,
   unsigned int command,
   unsigned long argument
) {
/*
   
   int minor = MINOR(inode->i_rdev);

   switch(command) {

    case PCL727_SETBIT_DO:
      if(minor == PCL727_MINOR_DIGITAL) {
	 int bitnum = (argument & 0xf0) >> 4;
	 unsigned char val = (argument & 0x01);
	 
	 DPRINTK4("card %c, setting bit %d of digital lines to %d\n", 'A' + cardnum, bitnum, val);
	 
	 cli();
	 
	 if(val == 0x01) {
	    pcl727_card[cardnum].diglines |= (0x01 << bitnum);
	 }
	 else {
	    pcl727_card[cardnum].diglines &= 0xff - (0x01 << bitnum);
	 }
	 
	 PCL727_DIGITAL_SET(pcl727_card[cardnum].diglines, cardnum);
	 
	 sti();
	 
	 break;
      }
      else {
	 printk(PCL727_ID "bungled minor number %d in ioctl SETBIT\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
      }
    case PCL727_SET_DO:
      if(minor == PCL727_MINOR_DIGITAL) {
	 unsigned int val = argument;
	 
	 DPRINTK3("card %c, setting digital lines to %d\n", 'A' + cardnum, val);
	 
	 cli();
	 
	 pcl727_card[cardnum].diglines = val;
	 
	 PCL727_DIGITAL_SET(pcl727_card[cardnum].diglines, cardnum);
	 
	 sti();
	 
	 break;
      }
      else {
	 printk(PCL727_ID "bungled minor number %d in ioctl SETBIT\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
      }
    case PCL727_SET_DAC:
      if((minor == PCL727_MINOR_DAC0) || (minor == PCL727_MINOR_DAC1)) {
	 unsigned int val = argument;

	 minor-=PCL727_MINOR_DAC0;

	 pcl727_card[cardnum].dac[minor] = val;
	 
	 DPRINTK3("card %c, setting DAC0 to %d\n", 'A' + cardnum, val);
	 
	 cli();

	 
	 PCL727_DAC_SET(val,minor,cardnum);
	 
	 sti();
	 
	 break;
      }
      else {
	 printk(PCL727_ID "bungled minor number %d in ioctl SETBIT\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
      }
      
      default:
         return -EINVAL;
   }
*/
   return 0;
}




static int pcl727_open(struct inode *inode, struct file *file) {
   unsigned int minor = MINOR(inode->i_rdev);

   if (minor > PCL727_MINOR_LARGEST) {
      printk(PCL727_ID "trying to open bad minor number %d\n", minor);
      return -ENODEV;
   }


   switch(file->f_flags & O_ACCMODE) {
    case O_RDONLY:

      if (minor != PCL727_MINOR_DIO) {
	 printk(PCL727_ID "trying to open DAC minor number %d for reading\n", minor);
	 return -ENODEV;
      }
      
      if((pcl727_card.read_in_use & (1L << minor)) != 0) {
#ifdef PCL727_DEBUG
	    printk("digital lines already opened for reading\n");
#endif
	    return -EBUSY;
	 }

	 pcl727_card.read_in_use |= (1L << minor);

#ifdef PCL727_DEBUG
	 printk("digital lines opened for reading\n");
#endif
         break;

      case O_WRONLY:
	 if((pcl727_card.write_in_use & (1L << minor)) != 0) {
#ifdef PCL727_DEBUG
	    if(minor == PCL727_MINOR_DIO) {
	       printk("digital lines ");
	    }
	    else if(minor < PCL727_MINOR_DIO) {
	       printk("analog output line %d ", minor);
	    }
	    printk("already opened for writing\n");
#endif
	    return -EBUSY;
	 }

	 pcl727_card.write_in_use |= (1L << minor);

#ifdef PCL727_DEBUG
	 if(minor == PCL727_MINOR_DIO) {
	    printk("digital lines ");
	 }
	 else if(minor < PCL727_MINOR_DIO) {
	    printk("analog output line %d ", minor);
	 }
	 printk("opened for writing\n");
#endif
	 break;


      case O_RDWR:
         if (minor != PCL727_MINOR_DIO) {

	    DPRINTK2("trying to open DAC minor number %d for reading and writing\n", minor);

            return -EPERM;
         }

	 if (((pcl727_card.write_in_use & (1L << minor)) != 0) || \
	    ((pcl727_card.read_in_use & (1L << minor)) != 0)) {
#ifdef PCL727_DEBUG
	    printk("digital lines already opened for reading and/or writing\n");
#endif
	    return -EBUSY;
	 }

	 pcl727_card.read_in_use |= (1L << minor);
	 pcl727_card.write_in_use |= (1L << minor);

#ifdef PCL727_DEBUG
	 printk("digital lines opened for reading and writing\n");
#endif

	 break;
   }
	 
    return 0;
}

static void pcl727_close(struct inode *inode, struct file *file) {

   unsigned int minor = MINOR(inode->i_rdev);

   switch(file->f_flags & O_ACCMODE) {
      case O_RDONLY:
	 pcl727_card.read_in_use &= ~(1L << minor);
         break;

      case O_WRONLY:
	 pcl727_card.write_in_use &= ~(1L << minor);
	 break;

      case O_RDWR:
	 pcl727_card.read_in_use &= ~(1L << minor);
	 pcl727_card.write_in_use &= ~(1L << minor);
	 break;
   }

#ifdef PCL727_DEBUG
   if(minor == PCL727_MINOR_DIO) {
      printk("digital lines ");
   }
   else if(minor < PCL727_MINOR_DIO) {
      printk("analog output line %d ", minor);
   }
   printk("closed\n");
#endif

}

static struct file_operations pcl727_fops =
{
   NULL,                   /*  pcl727_lseek               */
   pcl727_read,             /*  pcl727_read                */
   pcl727_write,            /*  pcl727_write               */
   NULL,                   /*  pcl727_readaddr            */
   NULL,                   /*  pcl727_select              */
   pcl727_ioctl,            /*  pcl727_ioctl               */
   NULL,                   /*  pcl727_mmap                */
   pcl727_open,             /*  pcl727_open                */
   pcl727_close,            /*  pcl727_close               */
   NULL,                   /*  pcl727_fsync               */
   NULL,                   /*  pcl727_fasync              */
   NULL,                   /*  pcl727_check_media_change  */
   NULL                    /*  pcl727_revalidate          */
};

int init_module(void) {
   
   unsigned short probe_address;

   probe_address=base;
   
   printk(PCL727_ID PCL727_VERSION ", loading...\n");

   /*  Register the device with the kernel.                                     */
   DPRINTK2("registering driver, major number %d\n", PCL727_MAJOR);
   if(register_chrdev(PCL727_MAJOR, "pcl727", &pcl727_fops)) {
      printk(PCL727_ID "unable to get major number %d\n", PCL727_MAJOR);
      return -EBUSY;
   }

   DPRINTK2("checking 0x20 I/O ports starting at 0x%03x\n", probe_address);
   if(check_region(probe_address, 32)) {
      DPRINTK1("I/O ports not available\n");

      DPRINTK2("unregistering device %d\n", PCL727_MAJOR);
      if(unregister_chrdev(PCL727_MAJOR, "pcl727")) {
	 printk(PCL727_ID "device unregistering failed\n");
      }
      
      printk(PCL727_ID "no cards found, driver not loaded\n");
      
      return -EBUSY;
   }
   DPRINTK1("I/O ports are available\n");
   
   pcl727_card.baseaddress=probe_address;
   
   /*  Claim the I/O ports.                                                     */
   DPRINTK2("requesting I/O port 0x%03x\n", pcl727_card.baseaddress);
   request_region(pcl727_card.baseaddress, 32, "pcl727");

   printk(PCL727_ID "pcl727 card initialized (addr=0x%03x)\n", pcl727_card.baseaddress);
   
   return 0;
}


void cleanup_module(void) {
   
   DPRINTK2("releasing io port 0x%03x\n", pcl727_card.baseaddress);
   release_region(pcl727_card.baseaddress, 32);

   DPRINTK1("unregistering device\n");
   if(unregister_chrdev(PCL727_MAJOR, "pcl727")) {
      printk(PCL727_ID "device unregistering failed\n");
   } else {
      printk(PCL727_ID "driver unloaded\n");
   }
}
