/**************************************************************************
 * $Id: SamplinCurve.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3 beta $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "SamplinCurve.h"
#include "qwt_math.h"
#include <qpntarry.h>
#include <qpainter.h>
#include <limits.h>
#include <qbitmap.h>

#include <stdio.h>

static char *cstyleNames[]={"None","Line","Sticks","Steps","Dots","Spline","Fit",NULL
};
static char *lstyleNames[]={"None","Solid","Dash","Dot","Dash-dot","Dash-dot-dot",NULL
};
static char *sstyleNames[]={"None","Circle","Rectangle","Diamond","Triangle",
   "DTriangle","UTriangle","LTriangle","RTriangle","Cross","XCross",NULL
};


#define isInsideR(x,y) ( rx1 <= x && rx2 >= x && ry1 <= y && ry2 >= y )
#define isInside(x1,x2,x) ( qwtMin(x1,x2) <= x && qwtMax(x1,x2) >=x )

//#define isInsideR(x,y) ( rx1 < x && rx2 > x && ry1 < y && ry2 > y )
//#define isInside(x1,x2,x) ( qwtMin(x1,x2) < x && qwtMax(x1,x2) >x )

int SamplinCurveData::pointDist(QPoint &pt, const QwtDiMap &mx, const QwtDiMap& my)
{
   int dist=INT_MAX,new_dist;
   int u,v,x,y;
   uint i,len=size();
//   int size=qwtMin(d_x.size(),d_y.size());
   
   x=pt.x();
   y=pt.y();
   
   //   printf("mx:%i,my:%i\n",mx.contains(x), my.contains(y));


   
   setMap(mx,my);
   
   //   if(mx.contains(x) && my.contains(y))
   for(i=0;i<len;++i){
      u = d_xMap.transform(d_x[i]);
      v = d_yMap.transform(d_y[i]);
      new_dist=(x-u)*(x-u)+(y-v)*(y-v);
      if(new_dist<dist)dist=new_dist;
   }

//   if(d_xoff)mx.setOffset(0);
//   if(d_yoff)mx.setOffset(0);
   
   return dist;
}


uint SamplinCurveData::size() const
{

   return qwtMin(d_x.size(),d_y.size());

}

bool SamplinCurveData::isEmpty() const
{

   return !(qwtMin(d_x.size(),d_y.size()) || (d_style==Fit && d_parser.isParsed()));

}


void SamplinCurveData::init(const char *name)
{
   d_visible=TRUE;
   d_xAxis=xBottom;
   d_yAxis=yLeft;
   
   d_pen = QPen(red, 0);
    
    d_splineSize = 250;
    d_splineType = Auto;
    d_xraw = false;
    d_yraw = false;
    
    if (name)
       d_name = name;
    else
       d_name = "";
	
   d_style = Lines;
   
   d_dataStyle = XandY;
   d_begin=0;
   d_step=1;
   d_xoff=0;
   d_yoff=0;
   
//   d_ix=-1;d_fman=0;
//   d_range1=0;d_range2=0;
   d_parser.setData(&d_x,&d_y);
   //updateFunctionRange();
   
   
   bcolor=white;
   updatePixmap();
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::~SamplinCurveData
//	Destructor
//
//.s	Syntax
//.f	 SamplinCurveData::~SamplinCurveData()
//
//------------------------------------------------------------
SamplinCurveData::~SamplinCurveData()
{
    if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
    if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());

}

void SamplinCurveData::setStyle(CurveStyle cs)
{
	d_style = cs;
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::setSymbol
//	Assign a symbol
//
//.s	Parameters:
//.p	const QwtSymbol &s	--	 symbol
//.s	See also
//	\l{qwt_symbol.html}{QwtSymbol}
//------------------------------------------------------------
void SamplinCurveData::setSymbol(const QwtSymbol &s )
{
   d_sym = s;
   updatePixmap();
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::setPen
//	Assign a pen
//
//.s	Parameters
//.p	const QPen &p	-- New pen
//
//.s	Syntax
//.f	void SamplinCurveData::setPen(const QwtPen &p)
//
//------------------------------------------------------------
void SamplinCurveData::setPen(const QPen &p)
{
   d_pen = p;
   updatePixmap();
}


//------------------------------------------------------------
//
//.S	SamplinCurveData::setData
//	Copy x-y data from specified arrays
//
//.s	Parameters
//.p	double *x, double *y, int size
//
//.s	Syntax
//.f	void SamplinCurveData::setData(double *x, double *y, int size)
//
//------------------------------------------------------------
void SamplinCurveData::setData(double *x, double *y, int size)
{
    if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
    if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());
    d_x.duplicate(x, size);
    d_y.duplicate(y, size);
    d_xraw = false;
    d_yraw = false;

   d_dataStyle = XandY;
   
   update();
}

void SamplinCurveData::putPoint(int index, double x, double y)
{
   int i, xsize, ysize;
   
   xsize=d_x.size();
   ysize=d_y.size();
   index=qwtMin(index,xsize);
   index=qwtMin(index,ysize);
   if(index==-1)index=xsize;
   index=qwtMax(index,0);
   
   
   if(d_xraw==false){
      d_x.resize(xsize+1);
      for(i=index;i<xsize;++i)
	d_x[i+1]=d_x[i];
      d_x[xsize]=x;
   }
   if(d_yraw==false){
      d_y.resize(ysize+1);
      for(i=index;i<ysize;++i)
	d_y[i+1]=d_y[i];
      d_y[ysize]=y;
   }
   
   
   update();
}

void SamplinCurveData::removePoint(int index)
{   int i, xsize, ysize;
   
   xsize=d_x.size();
   ysize=d_y.size();
   
   if(index<0 || index > xsize-1 || index > ysize-1)return;
   
   if(index<xsize && d_xraw==false){
      for(i=index;i<xsize-1;++i)
	d_x[i]=d_x[i+1];
      d_x.resize(xsize-1);
   }

   if(index<ysize && d_yraw==false){
      for(i=index;i<ysize-1;++i)
	d_y[i]=d_y[i+1];
      d_y.resize(ysize-1);
   }
     
   update();
}

void SamplinCurveData::setData(double beg, double stp, double *y, int size)
{
   int i;
   
   if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
   if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());

   d_x.resize(size);
   for(i=0;i<size;++i)
     d_x[i]=beg+i*stp;
   d_y.duplicate(y, size);

   d_xraw = false;
   d_yraw = false;

   d_step=stp;
   d_begin=beg;
   
   d_dataStyle = Yonly;
   
   update();
}

void SamplinCurveData::setData(double *x, double beg, double stp, int size)
{
   int i;
   
   if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
   if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());

   d_x.duplicate(x, size);

   d_y.resize(size);
   for(i=0;i<size;++i)
     d_y[i]=beg+i*stp;

   d_xraw = false;
   d_yraw = false;

   d_step=stp;
   d_begin=beg;
   
   d_dataStyle = Xonly;
   
   update();
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::setRawData
//	Attach raw data
//
//.s	Parameters
//.p	double *x, double *y, int size
//
//.s	Description
//	setRawData is provided for efficiency. In contrast to setData,
//	it does not copy the data, so it is important to keep the pointers
//	valid while they are attached.  
//	The SamplinCurveData destructor does not delete the attached data.
//	
//.s	Syntax
//.f	void SamplinCurveData::setRawData(double *x, double *y, int size)
//
//------------------------------------------------------------
void SamplinCurveData::setRawData(double *x, double *y, int size)
{
   if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
   if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());
   d_x.setRawData(x, size);
   d_y.setRawData(y, size);
   d_xraw = true;
   d_yraw = true;

   d_dataStyle = XandY;
   
   update();
}

void SamplinCurveData::setRawData(double beg, double stp, double *y, int size)
{
   int i;
   
   if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
    if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());

   d_x.resize(size);
   for(i=0;i<size;++i)
     d_x[i]=beg+i*stp;
   
   d_y.setRawData(y, size);
   
   d_xraw = false;
   d_yraw = true;

   d_step=stp;
   d_begin=beg;
   
   d_dataStyle = Yonly;
   
   update();
}

void SamplinCurveData::setRawData(double *x, double beg, double stp, int size)
{
   int i;
   
   if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
   if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());
   
   d_y.resize(size);
   for(i=0;i<size;++i)
     d_y[i]=beg+i*stp;
   
   d_x.setRawData(x, size);
   
   d_xraw = true;
   d_yraw = false;

   d_step=stp;
   d_begin=beg;
   
   
   d_dataStyle = Xonly;
   
   update();
}

//------------------------------------------------------------
//
// .S SamplinCurveData::setName
//	Assign a name to a curve
//
//.s  Parameters
//	const char *name
//.s Syntax
//.f	void SamplinCurveData::setName(const char *name)
//------------------------------------------------------------
void SamplinCurveData::setName(const char *name)
{
	d_name = name;
}

//------------------------------------------------------------
//
//.S SamplinCurveData::minXValue
//	find the smallest x value
//
//.s Return Type
//	double
//------------------------------------------------------------
double SamplinCurveData::minXValue() const
{
   double ret;
   
   if(d_style==Fit && d_parser.rangeMan())
     ret=qwtMin( qwtGetMin(d_x.data(),d_x.size()), d_parser.xMin());
   else
     ret= qwtGetMin(d_x.data(),d_x.size());
   
   
//   debug("xmin=%lf",ret);
   return ret;
}

//------------------------------------------------------------
//
//.S SamplinCurveData::minYValue
//	find the smallest y value
//
//.s Return Type
//	double
//
//------------------------------------------------------------
double SamplinCurveData::minYValue() const
{
   if(d_style==Fit && d_parser.rangeMan())
     return qwtMin( qwtGetMin(d_y.data(), d_y.size()), d_parser.yMin()) ;
   else
     return qwtGetMin(d_y.data(), d_y.size());
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::maxXValue
//	Find tha largest x value
//
//.s Return Type
//	double
//
//------------------------------------------------------------
double SamplinCurveData::maxXValue() const
{
   double ret;
   
   if(d_style==Fit && d_parser.rangeMan())
     ret= qwtMax( qwtGetMax(d_x.data(), d_x.size()), d_parser.xMax());
   else
     ret= qwtGetMax(d_x.data(), d_x.size());
   
   //debug("xMax=%lf",ret);
   
   return ret;
}

//------------------------------------------------------------
//
//.S SamplinCurveData::maxYValue
//	Find the largest y value
//
//.s Return Type
//	double
//
//------------------------------------------------------------
double SamplinCurveData::maxYValue() const
{
   if(d_style==Fit && d_parser.rangeMan())
     return qwtMax( qwtGetMax(d_y.data(), d_y.size()), d_parser.yMax());
   else
     return qwtGetMax(d_y.data(), d_y.size());
}


//------------------------------------------------------------
//
//.S SamplinCurveData::SamplinCurveData
//	Constructor
//.s Parameters
//.p	const char *name  -- name of the curve	
//------------------------------------------------------------
SamplinCurveData::SamplinCurveData(const char *name)
{
	init (name);
}


//------------------------------------------------------------
//
//.S SamplinCurveData::SamplinCurveData
//	Constructor
//
//.s Parameters
//.p double *xval, double *yval, int nValues, const char *name
// Description
//	This constructor  builds a new curve and copies the data.
//------------------------------------------------------------
SamplinCurveData::SamplinCurveData(double *xval, double *yval, int nValues, const char *name)
{
    init(name);
    d_x.duplicate(xval, nValues);
    d_y.duplicate(yval, nValues);
}



//------------------------------------------------------------
//
//.S SamplinCurve::draw
//	Draw the curve
//
//.s Parameters
//.p QPainter *p, const QwtDiMap &xMap, const QwtDiMap &yMap
//
//------------------------------------------------------------
void SamplinCurve::draw(QPainter *p, const QwtDiMap &xMap, const QwtDiMap &yMap)
{
//   printf("Xrange: %f,%f\n",xMap.d1(),xMap.d2());
//  printf("Yrange: %f,%f\n",yMap.d1(),yMap.d2());   
   setMap(xMap,yMap);

   draw(p);
}

//------------------------------------------------------------
//
//.S	SamplinCurve::draw
//	Draw the curve into a rectangle
//
//.s	Parameters
//.p	QPainter *p, const QRect &r
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s    Note
//	Before calling this function, you have to specify the ranges
//	for the x and y axes using \R{SamplinCurve::setRange}!
//
//.s	Syntax
//.f	void SamplinCurve::draw(QPainter *p, const QRect &r)
//
//------------------------------------------------------------
void SamplinCurve::draw(QPainter *p, const QRect &r)
{
//    setRect(r);
    draw(p);
}

//------------------------------------------------------------
//.-
//.S	SamplinCurve::draw
//	Draw the curve
//
//.s	Parameters
//.p	QPainter *p
//
//.s	Syntax
//.f	void SamplinCurve::draw(QPainter *p)
//
//------------------------------------------------------------
void SamplinCurve::draw(QPainter *p)
{
//   p->setRasterOp(OrROP);

   rx1 = qwtMin(d_xMap.c1(),d_xMap.c2());
   rx2 = qwtMax(d_xMap.c1(),d_xMap.c2());
   ry1 = qwtMin(d_yMap.c1(),d_yMap.c2());
   ry2 = qwtMax(d_yMap.c1(),d_yMap.c2());
   
   p->setBrush(NoBrush);
/*   p->drawLine(d_xMap.c1(),d_yMap.c1(),d_xMap.c1(),d_yMap.c2()); 
   p->drawLine(d_xMap.c2(),d_yMap.c1(),d_xMap.c2(),d_yMap.c2()); 
   p->drawLine(d_xMap.c1(),d_yMap.c1(),d_xMap.c2(),d_yMap.c1()); 
   p->drawLine(d_xMap.c1(),d_yMap.c2(),d_xMap.c2(),d_yMap.c2()); */

//   debug("double margins: %f,%f,%f,%f",rx1,rx2,ry1,ry2);
//   debug("int margins: %i,%i,%i,%i",d_xMap.i1(),d_xMap.i2(),d_yMap.i1(),d_yMap.i2());
   
//   debug("curve style %i",d_style);
   
   if(isEmpty())return;

   switch (d_style)
     {
      case NoCurve:
	break;
      case Lines:
	drawLines(*p);
	break;
      case Sticks:
	drawSticks(*p);
	break;
      case Steps:
	drawSteps(*p);
	break;
      case Dots:
	drawDots(*p);
	break;
      case Spline:
	drawSpline(*p);
	break;
      case Fit:
	drawFit(*p);
	break;
     default:
	drawDots(*p);
    }
    if (d_sym.style() != QwtSymbol::None)
    {
       drawSymbols(*p);
    }
    
    
}

//------------------------------------------------------------
//.-
// Name: SamplinCurve::drawLines
//
// Purpose: 
//
// Parameters:
//	
//
// Return Type: void
//
// Return Value:
//
// Description:
//
//------------------------------------------------------------
void SamplinCurve::drawLines(QPainter &p)
{
   int i;
   double u1,v1,u2,v2;
   
   int size = qwtMin(d_x.size(), d_y.size());
	
   p.setPen(d_pen);
   
   for (i=1;i<size;i++)
	{
	   transform(d_x[i], d_y[i], u2,v2);
	   transform(d_x[i-1],d_y[i-1],u1,v1);
	   drawClipLine(p, u1, v1, u2, v2);
	}
}


//------------------------------------------------------------
//.-
//.S	SamplinCurve::drawSpline
//
//
//.s	Parameters
//.p	QPainter *p
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurve::drawSpline(QPainte *p)
//
//------------------------------------------------------------
void SamplinCurve::drawSpline(QPainter &p)
{
 
   double xmin, xmax, ymin, ymax, delta;
   double u1,v1,u2,v2;
   double *param;
   double dtmp;
   int i;
   int stype;
   int rc;
   int size = qwtMin(d_x.size(), d_y.size());
   double *txval = new double[size];
   double *tyval = new double[size];
	

   p.setPen(d_pen);
   
    if ( (!txval) | (!tyval)/* | (!d_pa.resize(d_splineSize))*/ )
    {
	if (txval) delete[] txval;
	if (tyval) delete[] tyval;
	return;
    }

    //
    // Transform x and y values to window coordinates
    // to avoid a distinction between linear and
    // logarithmic scales.
    //
    for (i=0;i<size;i++)
    {
	txval[i] = d_xMap.xTransform(d_x[i]);
	tyval[i] = d_yMap.xTransform(d_y[i]);
    }
    
    if (d_splineType == Auto)
    {
	
	if (qwtChkMono(txval, size))
	{
	    stype = Yfx;
	}
	else
	{
	    
	    if(qwtChkMono(tyval, size))
	    {
		stype = Xfy;
	    }
	    else
	    {
		stype = Parametric;
		if ( (d_x[0] == d_x[size-1])
		    && (d_y[0] == d_y[size-1]))
		{
		    stype = Parametric | Periodic;
		}
		
	    }
	}
    }
    else
    {
	stype = d_splineType;
    }
    
    if (stype & Parametric)
    {
	param = new double[size];
	if (param)
	{
	    //
	    // setup parameter vector
	    //
	    param[0] = 0.0;
	    for (i=1;i<size; i++)
	    {
		delta = sqrt( qwtSqr(txval[i] - txval[i-1]) + qwtSqr( tyval[i] - tyval[i-1]));
		param[i] = param[i-1] + qwtMax(delta, 1.0);
	    }
	    
	    //
	    // setup splines
	    rc = d_spx.recalc(param, txval, size, stype & Periodic);
	    if (!rc) rc = d_spy.recalc(param, tyval, size, stype & Periodic);
	    
	    if (rc)
	    {
		drawLines(p);
	    }
	    else
	    {
		
		// fill point array
		delta = param[size - 1] / double(d_splineSize-1);
		for (i=0;i<d_splineSize;i++)
		{
		   dtmp = delta * double(i);
		   
		   u1=u2;
		   v1=v2;
		   u2=int(rint(d_spx.value(dtmp)));
		   v2=int(rint(d_spy.value(dtmp)));
		   if(i>0)drawClipLine(p,u1,v1,u2,v2);
		   //d_pa.setPoint(i, int(rint(d_spx.value(dtmp))),
//				  int(rint(d_spy.value(dtmp))));
		}
	    }
	    
	    
	    delete[] param;
	}
    }
    else if (stype & Xfy)
    {
	if (tyval[size-1] < tyval[0])
	{
	    qwtTwistArray(txval, size);
	    qwtTwistArray(tyval, size);
	}
	
	// 1. Calculate spline coefficients
	rc = d_spx.recalc(tyval, txval, size, stype & Periodic);
	
	if (rc)				// an error occurred
	{
	    drawLines(p);
	}
	else				// Spline OK
	{	
	    ymin = qwtGetMin(tyval, size);
	    ymax = qwtGetMax(tyval, size);
	    delta = (ymax - ymin) / double(d_splineSize - 1);
	    
	    for (i=0;i<d_splineSize;i++)
	    {
	       dtmp = ymin + delta * double(i);

	       u1=u2;
	       v1=v2;
	       u2=int(rint(d_spx.value(dtmp)));
	       v2=int(rint(dtmp));
	       if(i>0)drawClipLine(p,u1,v1,u2,v2);
	       
	       //		d_pa.setPoint(i, int(rint(d_spx.value(dtmp))),
//			      int(rint(dtmp)));
	    }
	}
    }		
    else
    {
	if (txval[size-1] < txval[0])
	{
	    qwtTwistArray(tyval, size);
	    qwtTwistArray(txval, size);
	}
	
	
	// 1. Calculate spline coefficients
	rc = d_spy.recalc(txval, tyval, size, stype & Periodic);
	
	if (rc)				// error
	{
	    drawLines(p);
	}
	else				// Spline OK
	{
	    xmin = qwtGetMin(txval, size);
	    xmax = qwtGetMax(txval, size);
	    delta = (xmax - xmin) / double(d_splineSize - 1);
	    
	    for (i=0;i<d_splineSize;i++)
	    {
	       dtmp = xmin + delta * double(i);

	       u1=u2;
	       v1=v2;
	       u2=int(rint(dtmp));
	       v2=int(rint(d_spy.value(dtmp)));
	       if(i>0)drawClipLine(p,u1,v1,u2,v2);
	       
//		d_pa.setPoint(i, int(rint(dtmp)),
//			      int(rint(d_spy.value(dtmp))));
	    }
	}
    }
    

//    p.drawPolyline(d_pa);
    
    delete[] txval;
    delete[] tyval;
    
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::setSplineType
//	Change the spline type		
//
//.s	Parameters
//.p	int t	--	new type
//
//.s	Syntax
//.f	void SamplinCurveData::setSplineType(int t)
//
//------------------------------------------------------------
void SamplinCurveData::setSplineType(int t)
{
    d_splineType = t;
}

//------------------------------------------------------------
//
//.S	SamplinCurveData::setSplineSize
//	Change the number of interpolated points
//
//.s	Parameters
//.p	int s	--	new size
//
//.s	Note
//		The default is 250 points.
//.s	Syntax
//.f	void SamplinCurveData::setSplineSize(int s)
//
//------------------------------------------------------------
void SamplinCurveData::setSplineSize(int s)
{
    d_splineSize = qwtMax(s, 10);
}


//------------------------------------------------------------
//.-
//.S	SamplinCurve::drawSticks
//
//
//.s	Parameters
//.p	QPainter &p
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurve::drawSticks(QPainter &p)
//
//------------------------------------------------------------
void SamplinCurve::drawSticks(QPainter &p)
{
    int i;
    double u,v, v0;
    int size = qwtMin(d_x.size(), d_y.size());
    
    if (qwtMax(d_yMap.d1(), d_yMap.d2()) < 0.0)
       v0 = qwtMin(d_yMap.i1(), d_yMap.i2());
    else if (qwtMin(d_yMap.d1(), d_yMap.d2()) > 0.0)
       v0 = qwtMax(d_yMap.i1(), d_yMap.i2());
    else
       v0 = d_yMap.xTransform(0.0);
    
    p.setPen(d_pen);
    
    for (i=0;i<size;i++)
     {
	transform(d_x[i], d_y[i], u,v);
	drawClipLine(p, u, v0, u, v);
     }
    
}

//------------------------------------------------------------
//.-
//.S	SamplinCurve::drawDots
//
//
//.s	Parameters
//.p	QPainter &p
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurve::drawDots(QPainter &p)
//
//------------------------------------------------------------
void SamplinCurve::drawDots(QPainter &p)
{
    int i;
    double u,v;
    int size = qwtMin(d_x.size(), d_y.size());
    
    p.setPen(d_pen);
    
    for (i=0;i<size;i++)
    {
       transform(d_x[i], d_y[i], u,v);
       if(isInsideR(u,v))
	 p.drawPoint(u,v);
    }
    
}


//------------------------------------------------------------
//.-
//.S	SamplinCurve::drawSteps
//
//
//.s	Parameters
//.p	QPainter &p
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurve::drawSteps(QPainter &p)
//
//------------------------------------------------------------
void SamplinCurve::drawSteps(QPainter &p)
{
    int i;
    double u1,v1,u2,v2;
    int w1,w2;
    int size = qwtMin(d_x.size(), d_y.size());
    
   /*
    if (d_pa.size() != uint (2*size - 1))
    {
	if (!d_pa.resize(uint(2*size - 1)))
	   return;
    }*/
    
    p.setPen(d_pen);
    
//    w1 = (d_pen.width() +1) /2;
//    w2 = d_pen.width() / 2;
    
//    transform(d_x[0], d_y[0], u1, v1);
//    d_pa.setPoint(0, u1,v1);   
    
    for (i=1;i<size;i++)
    {
       transform(d_x[i-1], d_y[i-1], u1,v1);
       transform(d_x[i], d_y[i], u2,v2);
       drawClipLine(p,u1,v1,u2,v1);
       drawClipLine(p,u2,v1,u2,v2);
       
//       d_pa.setPoint(2*i - 1, u2,v1);   
//	d_pa.setPoint(2*i, u2,v2);
//	u1 = u2;
//	v1 = v2;
    }
//    p.drawPolyline(d_pa);
    
    
}

//------------------------------------------------------------
//.-
//.S	SamplinCurveData::transform
//
//
//.s	Parameters
//.p	double x, double y, int &u, int &v
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurveData::transform(double x, double y, int &u, int &v)
//
//------------------------------------------------------------
void SamplinCurveData::transform(double x, double y, double &u, double &v)
{
    u = d_xMap.xTransform(x);
    v = d_yMap.xTransform(y);
}


//------------------------------------------------------------
//.-
//.S	SamplinCurve::drawSymbols
//
//
//.s	Parameters
//.p	QPainter &p
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurve::drawSymbols(QPainter &p)
//
//------------------------------------------------------------
void SamplinCurve::drawSymbols(QPainter &p)
{
   int i;
   double u, v;
   int size = qwtMin(d_x.size(), d_y.size());
    
   int w2 = d_sym.size().width() / 2;
   int h2 = d_sym.size().height() / 2;
   
    for (i=0;i<size;i++)
     {
	transform(d_x[i], d_y[i], u,v);
	if(isInsideR(u,v))
	  d_sym.draw(&p, u - w2, v - h2);
     }
}


//------------------------------------------------------------
//
//.S	SamplinCurveData::setRange
//	Specify x and y ranges
//
//.s	Parameters
//.p	double x1	--	left boundary of the x axis
//	double x2	--	right boundary of the y axis
//	int xlog	--	logarithmic x division if nonzero
//	double y1	--	lower boundary of the y axis
//	double y2	--	upper boundary of the y axis
//	int ylog	--	logarithmic y division if nonzero
//
//.s	Syntax
//.f	void SamplinCurveData::setRange(double x1, double x2, double y1, double y2)
//
//------------------------------------------------------------
void SamplinCurveData::setRange(double x1, double x2, int xlog, double y1, double y2, int ylog)
{
    d_xMap.setDblRange(x1, x2, xlog);
    d_yMap.setDblRange(y1, y2, ylog);
    
}

//------------------------------------------------------------
//.-
//.S	SamplinCurveData::setRect
//	Specify a bounding rectangle
//
//.s	Parameters
//.p	const QRect &r
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurveData::setRect(const QRect &r)
//
//------------------------------------------------------------
void SamplinCurveData::setRect(const QRect &r)
{
    d_xMap.setIntRange(r.left(), r.right());
    d_yMap.setIntRange(r.top(), r.bottom());
}

//------------------------------------------------------------
//.-
//.S	SamplinCurveData::setMap
//	Assign the maps for the x and y axes
//
//.s	Parameters
//.p	const QwtDiMap &xmap, const QwtDiMap &yMap
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void SamplinCurveData::setMap(const QwtDiMap &xmap, const QwtDiMap &yMap)
//
//------------------------------------------------------------
void SamplinCurveData::setMap(const QwtDiMap &xmap, const QwtDiMap &ymap)
{
   d_xMap = xmap;
   d_yMap = ymap;
   
   d_xMap.setOffset(d_xoff);
   d_yMap.setOffset(-d_yoff);

   /*
   clipRect.setLeft(d_xMap.i1());
   clipRect.setRight(d_xMap.i2());
   clipRect.setTop(d_yMap.i2());
   clipRect.setBottom(d_yMap.i1());
    */
}

 
void SamplinCurveData::setVisible(bool vis){
   d_visible=vis;
   updatePixmap();
}
 
 

void SamplinCurve::updatePixmap(void){

   QBitmap bit(16,16);
   QPainter p;
   pix.resize(16,16);
   int xoff, yoff;
   
   xoff=d_sym.size().width()/2;
   yoff=d_sym.size().height()/2;
   
   p.begin(&pix);
   p.eraseRect(0,0,16,16);
   p.fillRect(2,3,13,13,QBrush(bcolor));
   p.end();
   
   if(d_visible){
      p.begin(&pix);
      p.setPen(d_pen);
      if(d_style!=NoCurve)
	p.drawLine(2,9,15,9);
      d_sym.draw(&p,8-xoff,9-yoff);
      p.setPen(black);
      p.setBrush(NoBrush);
      p.end();
   }

   p.begin(&bit);
   p.eraseRect(0,0,16,16);
   p.fillRect(2,3,13,13,QBrush(white));
   p.end();
   pix.setMask(bit);
   
}

QPixmap &SamplinCurve::pixmap(void){

   return pix;
}

void SamplinCurve::drawIcon(QPainter &p, const QRect &r)
{
   int xoff, yoff;
   
   xoff=d_sym.size().width()/2;
   yoff=d_sym.size().height()/2;
   
   p.fillRect(r.x(),r.y(),r.width(),r.height(),QBrush(bcolor));
   
   if(d_visible){
      if(d_style!=NoCurve){
	 p.setPen(d_pen);
	 p.drawLine(r.x(),(r.top()+r.bottom())/2,r.right(),(r.top()+r.bottom())/2);
//	 debug("icon line %i - %i",r.x(),r.right());
      }
      d_sym.draw(&p,(r.right()+r.left())/2-xoff,(r.top()+r.bottom())/2-yoff);
      p.setPen(black);
      p.setBrush(NoBrush);
   }
}
 
 
void SamplinCurveData::setBackgroundColor(const QColor &c){

   if(bcolor!=c){
      bcolor=c;
      updatePixmap();
   }
}


QDataStream &operator<<( QDataStream &s, SamplinCurveData &series ){
   uint len;
   
   len = series.size();
   
   s << series.name();
//   debug("%i,%s",strlen(series.name().data()),series.name().data());
   s << series.legend();
//   debug("%i,%s",strlen(series.legend().data()),series.legend().data());
   
   s << series.xAxis();
//   debug("%i",series.xAxis());
   s << series.yAxis();
//   debug("%i",series.yAxis());
   
   s << series.style();
   s << series.pen();
   s << series.symbol();
   
   //   s << series.symbol().size();
   //   s << series.symbol().pen();
   //   s << series.symbol().style();
   //   s << series.symbol().brush();
     
   s << series.splineSize();
   s << series.splineType();
     
   s << (int)(!series.visible()); //is visible ?
   s << (int)1; //d_parser follows
   s << series.d_parser; //save parser data
   s << (int)0; //reserved for extension
   
   s << len;
   s << series.dataStyle();
   
   
   if(series.dataStyle()==SamplinCurveData::XandY){
      while(len>0){
	 --len;
	 s << series.x(len);
	 s << series.y(len);
      }
   }
   else{
      s << series.begin();
      s << series.step();
      while(len>0){
	 --len;
	 if(series.dataStyle()==SamplinCurveData::Xonly)
	   s << series.x(len);
	   else s << series.y(len);
      }
   }

   return(s);
}

QDataStream &operator>>( QDataStream &s, SamplinCurveData &series ){

   QString str;
   //   QColor color;
   QPen p;
   QwtSymbol sym;
   int i,style;
   double begin,step;
   uint count,len;
   QArray<double> xData,yData;
   //   QwtCurve::CurveStyle cstyle;
   
   s >> str;
   series.setName(str);
   s >> str;
   series.setLegend(str);
   s >> i;
   series.setxAxis(i);
   s >> i;
   series.setyAxis(i);
   
//   debug("%i,%s",strlen(series.name().data()),series.name().data());
//   debug("%i,%s",strlen(series.legend().data()),series.legend().data());
   
//   debug("%i",series.xAxis());
//   debug("%i",series.yAxis());
   
   
   s >> i;
   series.setStyle((SamplinCurveData::CurveStyle)i);
   s >> p;
   series.setPen(p);
   s >> sym;
   series.setSymbol(sym);
   //   s << series.symbol().size();
   //   s << series.symbol().pen();
   //   s << series.symbol().style();
   //   s << series.symbol().brush();
   s >> i;
   series.setSplineSize(i);
   s >> i;
   series.setSplineType(i);
   
   s >> i; //reserved for extension
   series.setVisible(!i);

   s >> i; //was parser saved (ver>=1.5.0)
   if(i==1){ 
      s >> series.d_parser;
      s >> i; //reserved for extension
   }
   
   s >> count;
   len=count;
   s >> style;
   
   if(style==SamplinCurve::XandY){
      xData.resize(len);
      yData.resize(len);
      while(count>0){
	 --count;
	 s >> xData[count];
	 s >> yData[count];
      }
      series.setData( xData.data(), yData.data(), len );
   }
   else{
      s >> begin;
      s >> step;
      xData.resize(len);
      while(count>0){
	 --count;
	 s >> xData[count];
	 //      s >> d;
	 //       if(style==SamplinCurve::Xonly)
	 //         s >> xData[count];
	 
	 //       else s >> yData[count];
      }
      if(style==SamplinCurve::Yonly)
	series.setData( begin, step, xData.data(), len );
      else
	series.setData( xData.data(), begin, step, len );
   }
   xData.resize(0);
   yData.resize(0);
   
   return(s);
}

int SamplinCurveData::curveStyle(const char *str)
  {
     char **s=cstyleNames;
     int i=0;
     
     while(*s!=0){
	if(!strcasecmp(*s,str)&&strlen(*s)==strlen(str))return i;
	++s;++i;
     }
     
     return -1;
  }

int SamplinCurveData::lineStyle(const char *str)
  {
     char **s=lstyleNames;
     int i=0;
     
     while(*s!=0){
	if(!strcasecmp(*s,str)&&strlen(*s)==strlen(str))return i;
	++s;++i;
     }
     
     return -1;
  }

int SamplinCurveData::symbolStyle(const char *str)
  {
     char **s=sstyleNames;
     int i=0;
     
     while(*s!=0){
	if(!strcasecmp(*s,str)&&strlen(*s)==strlen(str))return i;
	++s;++i;
     }
     
     return -1;
  }

void SamplinCurveData::setOffset(int x, int y)
{
   d_xoff=x;
   d_yoff=y;
}

int SamplinCurveData::xOffset(void)
{
   return d_xoff;
}

int SamplinCurveData::yOffset(void)
{
   return d_yoff;
}


void SamplinCurve::drawClipLine(QPainter &p, double x1, double y1, double x2, double y2)
{
//   double x1, x2, y1, y2;
   double dx,dy;
   double cx[2], cy[2], tmp;
   int done=0;
   int u1, v1, u2, v2;

//   x1=d_xMap.xTransform(_x1);
//   x2=d_xMap.xTransform(_x2);
//   y1=d_yMap.xTransform(_y1);
//   y2=d_yMap.xTransform(_y2);

//   if(!finite(x1) || !finite(x2) || !finite(y1) || !finite(y2))return;
   
   dx=x2-x1;
   dy=y2-y1;
   
//   debug("drawline %lf,%f,%f,%f",x1,y1,x2,y2);
   
   if(isInside(rx1,rx2,x1) && isInside(ry1,ry2,y1)){
      cx[done]=x1;
      cy[done]=y1;
      ++done;
//      printf("*L1");
   }
   
   if(isInside(rx1,rx2,x2) && isInside(ry1,ry2,y2)){
      cx[done]=x2;
      cy[done]=y2;
      ++done;
//      printf("*L2");
   }

   if(done==2){
      p.drawLine(cx[0],cy[0],cx[1],cy[1]);
      return;
   }
   
   if(x1<rx1 && x2<rx1)return;
   if(x1>rx2 && x2>rx2)return;
   if(y1<ry1 && y2<ry1)return;
   if(y1>ry2 && y2>ry2)return;
   
   //test top edge
   if(done<2)
     if(dy!=0 && isInside(y1,y2,ry1)){ 
	tmp = dx/dy*(ry1-y1)+x1;
	if(finite(tmp) && isInside(rx1,rx2,tmp)){
	   cx[done]=tmp;
	   cy[done]=ry1;
	   ++done;
//	   printf("*L3");
	}
     }
   //test bottom edge
   
   if(done<2)
     if(dy!=0 && isInside(y1,y2,ry2)){ 
	tmp = dx/dy*(ry2-y1)+x1;
	if(finite(tmp) && isInside(rx1,rx2,tmp)){
	   cx[done]=tmp;
	   cy[done]=ry2;
	   ++done;
//	   printf("*L4");
	}
     }
   
      //test left edge
   if(done<2)
     if(dx!=0 && isInside(x1,x2,rx1)){ 
	tmp = dy/dx*(rx1-x1)+y1;
	if(finite(tmp) && isInside(ry1,ry2,tmp)){
	   cx[done]=rx1;
	   cy[done]=tmp;
	   ++done;
//	   printf("*L5");
	}
     }
   
   //test right edge
   if(done<2)
     if(dx!=0 && isInside(x1,x2,rx2)){ 
	tmp = dy/dx*(rx2-x1)+y1;
	if(finite(tmp) && isInside(ry1,ry2,tmp)){
	   cx[done]=rx2;
	   cy[done]=tmp;
	   ++done;
//	   printf("*L6");
	}
     }
   
   if(done==2){
      p.drawLine(cx[0],cy[0],cx[1],cy[1]);
//    printf("\ndrawline %f,%f,%f,%f -> %f,%f,%f,%f\n",
//	     x1,y1,x2,y2,cx[0],cy[0],cx[1],cy[1]);
   }
//   else printf("sorry\n");
}

/*
int SamplinCurveData::setFunction(const char *s)
{
   return d_parser.setFunction(s);
}


void SamplinCurveData::updateFunctionRange(void)
{
   double step,var,tmp;
   int i;
   
   d_fmin=DBL_MAX;d_fmax=DBL_MIN;
   d_vmin=DBL_MAX;d_vmax=DBL_MIN;   

   if(d_fman && d_range1!=d_range2 && d_parser.isParsed()){
      d_vmin=qwtMin(d_range1,d_range2);
      d_vmax=qwtMax(d_range1,d_range2);
      
      step=(d_vmax-d_vmin)/2048;
      var=d_vmin;
      for(i=0;i<=2048;++i){
	 tmp=d_parser.function(var);
//	 if(isinf(tmp)==-1)tmp=-DBL_MAX/10;
//	 if(isinf(tmp)==1)tmp=DBL_MAX/10;
	 if(!isnan(tmp) && !isinf(tmp)){
	    if(tmp>d_fmax)d_fmax=tmp;
	    if(tmp<d_fmin)d_fmin=tmp;
	 }
	 var+=step;
      }
//   debug("vmin=%f, vmax=%f, fmin=%f, fmax=%f",d_vmin,d_vmax,d_fmin,d_fmax);
   }


   
}
*/

void SamplinCurve::drawFit(QPainter &p)
{
   int i;
   double u1,v1,u2,v2;

   double vrange,delta,var,res,oldvar,oldres;
   double b1,b2;
   int npoints;
   bool isfirst=TRUE;
   
   //if(d_ix==-1)return;
   if(!d_parser.isParsed())return;
   

   if(d_parser.rangeMan()){
      b1=d_parser.rangeLow();
      b2=d_parser.rangeHigh();
   }
   else if(d_x.size() && d_y.size()){
      b1=minXValue();
      b2=maxXValue();
   }
   else {
      b1=d_xMap.invTransform(d_xMap.c1());
      b2=d_xMap.invTransform(d_xMap.c2());
//      debug("draw: empty, b1=%lf,b2=%lf, c1=%i,c2=%i, d1=%lf,d2=%lf",b1,b2,d_xMap.c1(),d_xMap.c2(),d_xMap.d1(),d_xMap.d2());
   }

   vrange=fabs(b2-b1);
   npoints=2*(int)fabs(d_xMap.transform(b1)-d_xMap.transform(b2));
   delta=(b2-b1)/(float)npoints;

   p.setPen(d_pen);
   
   var=b1;
//   oldvar=b1;
//   oldres=d_parser.deffkt(b1);
   
   for(i=0;i<npoints+1;++i){
      res=d_parser.function(var);
      if(!isnan(res) && !isinf(res)){
	 transform(var, res, u2,v2);
	 if(!isfirst){
	    transform(oldvar,oldres,u1,v1);
	    drawClipLine(p, u1, v1, u2, v2);
	 }
	 isfirst=FALSE;
      }else isfirst=TRUE;

      oldvar=var;
      oldres=res;
      var+=delta;
   }

}

void SamplinCurve::update(void)
{
   if(d_style==Fit && d_parser.isParsed() && d_parser.autoFit()){
      d_parser.regress();
   }
   
}
