/**************************************************************************
 * $Id: SamplinPlot.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3 beta $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include <samplin.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <sys/param.h>
#include <sys/types.h>
#include <math.h>
#include <float.h>

//#include <stdio.h>
//#include <limits.h>
#include "SamplinPlot.h"
#include <qwt_math.h>
#include <qintdict.h>
#include <qpdevmet.h>
#include <qprinter.h>
#include <qfiledlg.h>
#include <qfontmetrics.h>
#include <qdrawutil.h>
#include <qprintdialog.h>
#include <qpicture.h>
#include <qimage.h>
//#include <qwidget.h>
//#include <qapplication.h>
//#include <qlabel.h>

#define FRAMEWIDTH 2
#define MINIMUM_WIDTH 260
#define MINIMUM_HEIGHT 200

#include "SamplinPlot.moc"


/*
// internal class

class PlotLabel : public QLabel
{
   Q_OBJECT
 public:
   PlotLabel(QString const & text) : QLabel( 0, "PlotLabel",
			WStyle_Customize + WStyle_NoBorder + WStyle_Tool )
     {
	setMargin(1);
//	setIndent(0);
//	setAutoMask( FALSE );

	//	if ( use_style_override ) {
	//	   setFont( QToolTip::font() );
	//	   setPalette( QToolTip::palette(), TRUE );
	//	}

	QColorGroup cg(black, QColor(255,255,220),
		       QColor(96,96,96), black, black,
		       black, QColor(255,255,220) );
	QPalette pal( cg, cg, cg );
	setPalette( pal );
	
	setFrameStyle( QFrame::Plain | QFrame::Box );
	setLineWidth( 1 );
	setAlignment( AlignLeft | AlignTop );
	setAutoResize(TRUE);
	//	polish();
	setText(text);
	adjustSize();
     }
};
*/

SamplinPlot::SamplinPlot(QWidget *p, const char *name)
: QFrame(p,name)
{
   int i;

   setMouseTracking(TRUE);
   setFrameStyle(QFrame::NoFrame);
//   setLineWidth(0);   
   d_waitflag=FALSE;
   
   d_titleFont=QFont("Helvetica", 14, QFont::Bold);
   d_panelColor=QColor(200,230,220);
   d_plotColor=QColor("grey");  
   d_label=name;
   d_marginX=d_marginY=d_margin = 6;
   d_autoReplot = false;
   d_zooming=FALSE;
   d_pointing=FALSE;
   d_zoom=FALSE;
   d_axisEnabled[yLeft] = TRUE;
   d_axisEnabled[yRight] = FALSE;
   d_axisEnabled[xBottom] = TRUE;
   d_axisEnabled[xTop] = FALSE;

   d_nvisible=0;
   d_x3d=0; d_y3d=0;
   d_stacked=FALSE;

   d_currentKey=0;
   d_currentObject=OBJ_GRAPH;
   
   for (i=0;i<axisCnt;i++)
    {
       d_scale[i]->adjust(0.0,1000.0,TRUE);
       d_map[i].setDblRange(d_scale[i]->scaleDiv().lBound(), d_scale[i]->scaleDiv().hBound(), d_scale[i]->scaleDiv().logScale());
    }
   d_scale[0]->setLabelFormat(SamplinScaleBar::Scientific,2);    

   d_grid.setXDiv(d_scale[xBottom]->scaleDiv());
   d_grid.setYDiv(d_scale[yLeft]->scaleDiv());
  
   d_printfit=TRUE;
   d_printwidth=0;
   d_printheight=0;
   d_printinfo=TRUE;
   
   setSaved();
   
   d_coordTip=NULL;
   connect(&d_timer,SIGNAL(timeout()),SLOT(showCoord()));
}

SamplinPlot::~SamplinPlot()
{
}

void SamplinPlot::setPlotColor(const QColor &c)
{
   SamplinCurve *crv;
   
   if(c!=d_plotColor){
      d_plotColor=c;
      for(crv=d_curveList.first();crv!=0;crv=d_curveList.next())
	crv->setBackgroundColor(c);
      autoRefresh();
   }
}

int SamplinPlot::whichObject(QPoint &pt, uint *key=0)
{
   uint ckey;
   
   if(d_axisEnabled[yLeft] && rc[yLeft].contains(pt))
     return OBJ_YLEFT;
   if(d_axisEnabled[yRight] && rc[yRight].contains(pt))
     return OBJ_YRIGHT;
   if(d_axisEnabled[xTop] && rc[xTop].contains(pt))
     return OBJ_XTOP;
   if(d_axisEnabled[xBottom] && rc[xBottom].contains(pt))
     return OBJ_XBOTTOM;
   if(rPlot.contains(pt)){
      ckey=nearestCurve(pt);
      if(key!=0)*key=ckey;
      if(ckey>0)
	return OBJ_SERIES;
      else return OBJ_PLOT;
   }
   return OBJ_GRAPH;
   
}

uint SamplinPlot::nearestCurve(QPoint &pt)
{
   SamplinCurve *c;
   uint key=0;
   int dist=INT_MAX,new_dist;
   
   c=d_curveList.first();
   while(c){
      if(c->visible()){
	 new_dist=c->pointDist(pt, d_map[c->xSAxis()], d_map[c->ySAxis()]);
	 if(new_dist<dist){
	    key=c->key();
	    dist=new_dist;
	 }
      }
      c=d_curveList.next();
   }
   return key;
}

void SamplinPlot::paintEvent(QPaintEvent *e)
{
   QRect r = contentsRect();
   QRect ri;
   QPainter p;

   d_tmppix=d_pix;
   
   if(d_pointing || d_zooming){
      QPainter pp(&d_tmppix);      
      if(d_pointing)drawCross(pp);
      if(d_zooming)drawZoomer(pp);
   }
   
   p.begin(this);
   
   //if necessary, redraw frame
   if (!r.contains(e->rect(), TRUE))
     drawFrame(&p);
   
   // redraw pixmap portion

   if (r.intersects(e->rect())){
      ri = r.intersect(e->rect());
      p.drawPixmap(ri.x(), ri.y(), d_tmppix,
		   ri.x() - r.x(), ri.y() - r.y(),
		   qwtMin(ri.width(), d_pix.width()),
		   qwtMin(ri.height(), d_pix.height()));
   }
   
   p.end();

}

void SamplinPlot::drawContents(QPainter *p)
{
}

bool SamplinPlot::exportPic(const char *fn)
{
   bool ret;
   
   QPicture pic;
   QPainter p;

   if(QFile::exists(fn))
     rename(fn,QString(fn)+"~");
   
   p.begin( &pic );
   draw(p);
   p.end();
   
   ret=pic.save(fn);

   return ret;

}

bool SamplinPlot::exportBMP(const char *fn)
{
   bool ret;
   QImageIO iio;
   QImage im;
   
   im=d_pix;
   iio.setImage(im.convertDepth(8));
   iio.setFileName(fn);
   iio.setFormat("BMP");
   
   if(QFile::exists(fn))
     rename(fn,QString(fn)+"~");

   ret=iio.write();

   return ret;
}


void SamplinPlot::drawZoomer(QPainter &p)
{
//   QPainter p(this);
   QColor zcolor;
   int w,h;
   
   
   zcolor.setRgb(255-d_plotColor.red(),255-d_plotColor.green(),255-d_plotColor.blue());
   p.setPen(zcolor);
   p.setRasterOp(CopyROP);

   if(d_p2.x()<rPix.x())d_p2.setX(rPix.x());
   if(d_p2.x()>rPix.right()-d_x3d)d_p2.setX(rPix.right()-d_x3d);   
   if(d_p2.y()<rPix.y()+d_y3d)d_p2.setY(rPix.y()+d_y3d);
   if(d_p2.y()>rPix.bottom())d_p2.setY(rPix.bottom());   
   
   if(d_p1.x()<rPix.x())d_p1.setX(rPix.x());
   if(d_p1.x()>rPix.right()-d_x3d)d_p1.setX(rPix.right()-d_x3d);   
   if(d_p1.y()<rPix.y()+d_y3d)d_p1.setY(rPix.y()+d_y3d);
   if(d_p1.y()>rPix.bottom())d_p1.setY(rPix.bottom());   
   
   w=d_p2.x()-d_p1.x();
   h=d_p2.y()-d_p1.y();
   if(w>0)++w;
   if(w<0)--w;
   if(h>0)++h;
   if(h<0)--h;
   if(w==0)++w;
   if(h==0)++h;
   
   p.drawRect(d_p1.x(),d_p1.y(),w,h);

//   p.end();
   
}

void SamplinPlot::drawCross(QPainter &p)
{
//   QPainter p(this);
   QColor zcolor;
   int w,h;
   int xoff=0,yoff=0;
   SamplinCurve *c;
   QRect rBound=rPix;
   
   zcolor.setRgb(255-d_plotColor.red(),255-d_plotColor.green(),255-d_plotColor.blue());
   p.setPen(zcolor);
   p.setRasterOp(CopyROP);

   c=findCurve(d_currentKey);
   
   if(d_stacked && c!=NULL){
      xoff=c->xOffset();
      yoff=c->yOffset();
      rBound=QRect(rPix.x()+xoff,rPix.y()+d_y3d-yoff,rPix.right()-d_x3d-rPix.x(),rPix.bottom()-rPix.y()-d_y3d);
   }
   
   if(d_p2.x()<rBound.x())d_p2.setX(rBound.x());
   if(d_p2.x()>rBound.right())d_p2.setX(rBound.right());   
   if(d_p2.y()<rBound.y())d_p2.setY(rBound.y());
   if(d_p2.y()>rBound.bottom())d_p2.setY(rBound.bottom());   

   p.drawLine(d_p1.x()-5,d_p1.y(),d_p1.x()+5,d_p1.y());
   p.drawLine(d_p1.x(),d_p1.y()-5,d_p1.x(),d_p1.y()+5);
   p.drawLine(d_p1.x(),d_p1.y(),d_p2.x(),d_p2.y());
   
   if(!d_stacked){
      p.drawLine(rBound.x(),d_p2.y(),rBound.right(),d_p2.y());
      p.drawLine(d_p2.x(),rBound.y(),d_p2.x(),rBound.bottom());
   }
   else{
      p.drawLine(rBound.x(),d_p2.y(),rBound.right(),d_p2.y());
      p.drawLine(rPix.x(),d_p2.y()+yoff,rBound.x(),d_p2.y());
      
      p.drawLine(d_p2.x(),rBound.y(),d_p2.x(),rBound.bottom());
      p.drawLine(d_p2.x()-xoff,rPix.bottom(),d_p2.x(),rBound.bottom());      
   }

//   p.end();
   
}



void SamplinPlot::updateCursor(int x, int y)
{
   if(d_waitflag)setCursor(waitCursor);
   else{
      if(rPix.contains(QPoint(x,y),FALSE))
	setCursor(crossCursor);
      else setCursor(arrowCursor);   
   }
}

void SamplinPlot::mouseMoveEvent(QMouseEvent *m)
{
   SamplinCurve *c;
   uint key;
   int object;
   QPoint pt(m->x(),m->y());
   double x=0,y=0;
   
   updateCursor(m->x(),m->y());

   if(d_currentObject==OBJ_SERIES || d_currentObject==OBJ_PLOT){
      c=findCurve(d_currentKey);
      if(c!=NULL){
	 x=d_map[c->xSAxis()].limInvTransform(m->pos().x()-c->xOffset());
	 y=d_map[c->ySAxis()].limInvTransform(m->pos().y()+c->yOffset());
      }
      else{
	 x=d_map[xBottom].limInvTransform(m->pos().x());
	 y=d_map[yLeft].limInvTransform(m->pos().y());	 
      }
   }
   
   
   if(d_zooming){
      d_p2 = m->pos();
      repaint(FALSE);

      emit objClick(d_currentObject/*object*/,d_currentKey/*key*/, m->button(),x,y);
   }
   if(d_pointing){
      d_p2 = m->pos();
      repaint(FALSE);
      if(d_coordTip){
	 delete d_coordTip;
	 d_coordTip=NULL;
      }
      d_timer.start(750,TRUE);
      
      emit objClick(d_currentObject/*object*/,d_currentKey/*key*/, m->button(),x,y);
   }
   
   emit mouseMoved(*m);
}

void SamplinPlot::mousePressEvent(QMouseEvent *m)
{
   SamplinCurve *c;
   uint key;
   int object;
   bool flag=FALSE;
   QRect hrc;
   QPoint pt(m->x(),m->y());
   double x=m->x(),y=m->y();
   
   
   updateCursor(m->x(),m->y());   

   object=whichObject(pt,&key);
  
   if(rPix.contains(pt)){
      if(m->button()==MidButton){
	 if (!d_zooming)
	   {
	      // store position
	      d_p1 = m->pos();
	      d_p2 = m->pos();
	      //enable zooming
	      d_zooming = TRUE;
	      repaint(FALSE);
	   }
      }
      if(m->button()==LeftButton){
	 if (!d_pointing)
	   {
	      // store position
	      d_p1 = m->pos();
	      d_p2 = m->pos();
	      //enable pointing
	      d_pointing = TRUE;
	      
	      d_currentObject=object;
	      
	      if(object==OBJ_SERIES){
		 d_currentKey=key;
	      }
	      if(d_coordTip){
		 delete d_coordTip;
		 d_coordTip=NULL;
	      }
	      repaint(FALSE);	      
	      d_timer.start(750,TRUE);
	   }
      }
   }
   
   if((object==OBJ_SERIES || object==OBJ_PLOT) && m->button()!=RightButton){
      c=findCurve(key);
      if(c!=NULL){
	 x=d_map[c->xSAxis()].limInvTransform(m->pos().x()-c->xOffset());
	 y=d_map[c->ySAxis()].limInvTransform(m->pos().y()+c->yOffset());
	 
      }
      else{
	 x=d_map[xBottom].limInvTransform(m->pos().x());
	 y=d_map[yLeft].limInvTransform(m->pos().y());	 
      }
   }
   
   emit objClick(/*d_currentObject*/object,/*d_currentKey*/key, m->button(),x,y);
}

void SamplinPlot::blinkObject(int obj, uint key)
{
   QRect hrc;
   int t;
   
   switch(obj){
    case OBJ_YLEFT:
      hrc=rc[yLeft];
      break;
    case OBJ_YRIGHT:
      hrc=rc[yRight];
      break;
    case OBJ_XTOP:
      hrc=rc[xTop];
      break;
    case OBJ_XBOTTOM:
      hrc=rc[xBottom];
      break;
    case OBJ_GRAPH:
    case OBJ_PLOT:
      hrc=contentsRect();
      break;
    case OBJ_SERIES:
      break;
    default:
      return;
   }
   
   if(obj!=OBJ_SERIES){
      QPainter p(this);
      p.setPen(d_panelColor);
      p.setRasterOp(NotROP);
      p.drawRect(hrc);
      p.flush();
      usleep(30000);
      p.drawRect(hrc);
      p.flush();
      usleep(30000);
      p.drawRect(hrc);
      p.flush();
      usleep(30000);
      p.drawRect(hrc);      
      p.end();
   }
   else if(key!=0){
      SamplinCurve *c=findCurve(key);
      if(c->visible()){
	 
	 QPainter p(this);
	 
	 for(t=0;t<2;++t){
	    p.setRasterOp(NotCopyROP);
	    c->draw(&p, d_map[c->xSAxis()], d_map[c->ySAxis()]);
	    p.flush();
	    usleep(20000);
	    p.setRasterOp(CopyROP);
	    c->draw(&p, d_map[c->xSAxis()], d_map[c->ySAxis()]);
	    p.flush();
	    usleep(20000);
	 }
	 p.end();
	 replot();
      }

   }
}

void SamplinPlot::mouseReleaseEvent(QMouseEvent *m)
{
   int x1, x2, y1, y2,x,y;
   int lim;
   
   switch(m->button()){
    case LeftButton:
      if(d_pointing){
	 d_pointing=FALSE;
	 repaint(FALSE);
	 d_timer.stop();
      }
      break;
    case MidButton:
      
      if (d_zooming){
	 x=m->pos().x();
	 y=m->pos().y();

	 x=qwtMax(rPix.x(),x);
	 x=qwtMin(x,rPix.right());
	 y=qwtMax(rPix.y(),y);
	 y=qwtMin(y,rPix.bottom());
	 
	 // Don't invert any scales which aren't inverted
	 
	 x1 = qwtMin(d_p1.x(), x);
	 x2 = qwtMax(d_p1.x(), x);
	 y1 = qwtMin(d_p1.y(), y);
	 y2 = qwtMax(d_p1.y(), y);

	 d_zooming = FALSE;
	 
	 if(x1!=x2 && y1!=y2){
	    // limit selected area to a minimum of 11x11 points
	    lim = 5 - (y2 - y1) / 2;
	    if (lim > 0){
	       y1 -= lim;
	       y2 += lim;
	    }
	    lim = 5 - (x2 - x1 + 1) / 2;
	    if (lim > 0){
	       x1 -= lim;
	       x2 += lim;
	    }
	    
	    // Set fixed scales
//	    printf("zoomX: %f,%f\n",transform(x1, xBottom), transform(x2, xBottom));
//	    printf("zoomY: %f,%f\n",transform(y1, yLeft), transform(y2, yLeft));
	    if(axisValid(yLeft))
	      d_scale[yLeft]->zoom(transform(y2, yLeft), transform(y1, yLeft));
	    if(axisValid(yRight))
	      d_scale[yRight]->zoom(transform(y2, yRight), transform(y1, yRight));
	    if(axisValid(xBottom))
	      d_scale[xBottom]->zoom(transform(x1, xBottom), transform(x2, xBottom));	       
	    if(axisValid(xTop))
	      d_scale[xTop]->zoom(transform(x1, xTop), transform(x2, xTop));	 
	    replot();
	    d_zoom=TRUE;
	 }
	 else if(d_zoom){
	    //zoom out
	    d_zoom=FALSE;
	    if(axisValid(yLeft))
	      d_scale[yLeft]->unZoom();
	    if(axisValid(yRight))
	      d_scale[yRight]->unZoom();
	    if(axisValid(xBottom))
	      d_scale[xBottom]->unZoom();	       
	    if(axisValid(xTop))
	      d_scale[xTop]->unZoom();	 
    
	    replot();
	 }
//	 drawZoomer();
//	 replot();
	 

      }
      
   }
   emit mouseReleased(*m);   
}

void SamplinPlot::mouseDoubleClickEvent(QMouseEvent *m)
{
   uint key=0;
   int object;
   QPoint pt(m->x(),m->y());

   object=whichObject(pt,&key);
   emit objDoubleClick(m,object,key);
}

void SamplinPlot::autoRefresh(){
   if (d_autoReplot)
     replot();
}

void SamplinPlot::resizeEvent(QResizeEvent *e)
{
   d_pix.resize(contentsRect().size());
//   updatePixmap(); ??????????
   replot();
}

void SamplinPlot::resizeObjects(QPainter &p, const QRect &r)
{
   QRect mr;
   int htitle;
   int hDist, vDist;
   int dTitle;
   QSize csize,lsize;
   int bshrink,rshrink;
   SamplinCurve *c;
   int x3d,y3d;

   d_x3d=0;
   d_y3d=0;
   
   rPainter=r;
   rGraph=r;
   bshrink=0;rshrink=0;
   
   d_stacked=(d_stack && d_stackx && d_stacky);
   
   c=d_curveList.first();
   d_nvisible=0;
   while(c)
     {
	if(!c->isEmpty() && c->visible()){
	   ++d_nvisible;
	}
	c=d_curveList.next();
     }

   hDist = vDist = FRAMEWIDTH + 2;
   
   if (d_axisEnabled[xTop] && !d_stacked)
     hDist = qwtMax(hDist,d_scale[xTop]->minScaleOffset(p));
   if (d_axisEnabled[xBottom])
     hDist = qwtMax(hDist,d_scale[xBottom]->minScaleOffset(p));
   
   if (d_axisEnabled[yLeft])
     vDist = qwtMax(vDist, d_scale[yLeft]->minScaleOffset(p));
   if (d_axisEnabled[yRight] && !d_stacked)
     vDist = qwtMax(vDist, d_scale[yRight]->minScaleOffset(p));

//   vDist=0;
//   debug("hdist=%i, vdist=%i",hDist, vDist);
   if(d_stacked)vDist=FRAMEWIDTH;

   
   
   p.setFont(d_titleFont);
   QFontMetrics fm=p.fontMetrics();
   
   if (!d_title.isEmpty())
     {
	rTitle.setY(rGraph.y()+d_marginY);
	rTitle.setHeight(fm.lineSpacing());
     }
   else
     {
	rTitle.setY(rGraph.y());
	rTitle.setHeight(0);
    }

   rc[xTop].setY(rTitle.bottom() + 1);
   if (d_axisEnabled[xTop] && !d_stacked)
     rc[xTop].setHeight(d_scale[xTop]->sizeHint(p).height());
   else
     rc[xTop].setHeight(d_marginY);

   if (d_axisEnabled[yLeft]){
      rc[yLeft].setWidth(d_scale[yLeft]->sizeHint(p).width());
   }
   else 
     rc[yLeft].setWidth(qwtMax(d_marginX, hDist - vDist/*+d_margin*/));

   rc[yLeft].setX(rGraph.x());
   rc[yLeft].setY(rc[xTop].bottom()+1/* +y3d*/);
   
   if (d_axisEnabled[xBottom])
     rc[xBottom].setHeight(d_scale[xBottom]->sizeHint(p).height());
   else
     rc[xBottom].setHeight(d_marginY);

   if(!d_comment.isEmpty() && d_cpos!=LegendPosition(None)){
      p.setFont(d_cFont);
      QFontMetrics fmm=p.fontMetrics();
      csize=fmm.size(ShowPrefix/*|WordBreak*/,d_comment.data()); //!!!!!!!!!!!!!
   }
   else csize=QSize(0,0);
   
   d_litemwidth=legendItemSize(p);
   d_litemcount=curveCount();
   if(d_lpos==LegendPosition(None)){
      lsize=QSize(0,0);
      rshrink=(d_cpos==LegendPosition(Right))*(csize.width()+d_marginX) +
	(d_lpos==LegendPosition(Right))*(lsize.width()+d_marginX) +
	(d_axisEnabled[yRight]==TRUE && !d_stacked)*(d_lpos==LegendPosition(Right) || d_cpos==LegendPosition(Right))*d_marginX;      
      bshrink=(d_cpos==LegendPosition(Bottom))*(csize.height()+d_marginY) +
	(d_lpos==LegendPosition(Bottom))*(lsize.height()+d_marginY);
      rGraph.setWidth(MAX(250,rGraph.width()-rshrink));
      rGraph.setHeight(MAX(160,rGraph.height()-bshrink));
      if (d_axisEnabled[yRight] && !d_stacked)
	rc[yRight].setLeft( rGraph.right() - d_scale[yRight]->sizeHint(p).width() + 1);
      else
	rc[yRight].setLeft(rGraph.right() - qwtMax(d_marginX, hDist - vDist /*+d_margin*/) + 1);

      rc[xBottom].setTop(rGraph.bottom()  - rc[xBottom].height() + 1);
      rc[yLeft].setBottom(rc[xBottom].top() - 1);
   }
   if(d_lpos==LegendPosition(Right)){
      bshrink=(d_cpos==LegendPosition(Bottom))*(csize.height()+d_marginY) +
	(d_lpos==LegendPosition(Bottom))*(lsize.height()+d_marginY);
      rGraph.setHeight(MAX(160,rGraph.height()-bshrink));

      rc[xBottom].setTop(rGraph.bottom()  - rc[xBottom].height() + 1);
      rc[yLeft].setBottom(rc[xBottom].top() - 1);
      
      p.setFont(d_lFont);
      QFontMetrics fml=p.fontMetrics();
      d_litemheight=MAX(16,fml.height());      
      
      if(d_litemcount && d_litemwidth){
	 d_llineitems= rc[yLeft].height()/d_litemheight; // rPlot.height()
	 if(d_llineitems>0){
	    d_llines=d_litemcount/d_llineitems;
	    if(d_llines*d_llineitems<d_litemcount)++d_llines;

	    lsize.setWidth(d_litemwidth*d_llines);
	 }
      }
      
      rshrink=(d_cpos==LegendPosition(Right))*(csize.width()+d_marginX) +
	(d_lpos==LegendPosition(Right))*(lsize.width()/*+d_margin*/) +
	(d_axisEnabled[yRight]==TRUE && !d_stacked)*(d_lpos==LegendPosition(Right) || d_cpos==LegendPosition(Right))*d_marginX;            
      rGraph.setWidth(MAX(250,rGraph.width()-rshrink));

      if (d_axisEnabled[yRight] && !d_stacked)
	rc[yRight].setLeft( rGraph.right() - d_scale[yRight]->sizeHint(p).width() + 1);
      else
	rc[yRight].setLeft(rGraph.right() - qwtMax(d_marginX, hDist - vDist) + 1);

   }
   if(d_lpos==LegendPosition(Bottom)){
      rshrink=(d_cpos==LegendPosition(Right))*(csize.width()+d_marginX) +
	(d_lpos==LegendPosition(Right))*(lsize.width()+d_marginX) +
	(d_axisEnabled[yRight]==TRUE && !d_stacked)*(d_lpos==LegendPosition(Right) || d_cpos==LegendPosition(Right))*d_marginX;      
      rGraph.setWidth(MAX(250,rGraph.width()-rshrink));
      if (d_axisEnabled[yRight] && !d_stacked)
	rc[yRight].setLeft( rGraph.right() - d_scale[yRight]->sizeHint(p).width() + 1);
      else
	rc[yRight].setLeft(rGraph.right() - qwtMax(d_marginX, hDist - vDist) + 1);

      if(d_litemcount && d_litemwidth){
	 d_llineitems=(rc[yRight].left()-rc[yLeft].right()-2)/d_litemwidth; // rPlot.width()
	 if(d_llineitems>0){
	    d_llines=d_litemcount/d_llineitems;
	    if(d_llines*d_llineitems<d_litemcount)++d_llines;
	    
	    p.setFont(d_lFont);
	    QFontMetrics fml=p.fontMetrics();
	    d_litemheight=MAX(16,fml.height());
	    lsize.setHeight(d_litemheight*d_llines);
	 }
      }
      
      bshrink=(d_cpos==LegendPosition(Bottom))*(csize.height()+d_marginY) +
	(d_lpos==LegendPosition(Bottom))*(lsize.height()+d_marginY);
      rGraph.setHeight(MAX(160,rGraph.height()-bshrink));

      rc[xBottom].setTop(rGraph.bottom()  - rc[xBottom].height() + 1);
      rc[yLeft].setBottom(rc[xBottom].top() - 1);
   }

   rc[xBottom].setBottom(rGraph.bottom());   
   
   rc[yRight].setY(rc[yLeft].y());
   rc[yRight].setBottom(rc[xBottom].top() - 1);

   rc[yRight].setRight(rGraph.right());
   
   rPlot.setX(rc[yLeft].right() + 1);
   rPlot.setRight(rc[yRight].left() - 1 /*- x3d*/);
   rPlot.setY(rc[yLeft].y());
   rPlot.setBottom(rc[yLeft].bottom());

   rTitle.setX(rPlot.x()); //rGraph
   rTitle.setWidth(rPlot.width() /*+ x3d*/); //rGraph
   
   rLegend.setY(rGraph.bottom()+1);
   rLegend.setX(rGraph.right()+d_marginX*(d_axisEnabled[yRight]==TRUE && !d_stacked)+1);
   rLegend.setHeight((d_lpos==LegendPosition(Bottom))*lsize.height());
   rLegend.setWidth((d_lpos==LegendPosition(Right))*lsize.width());   
//   debug("legend x:%i,y:%i,w:%i,h:%i",rLegend.x(),rLegend.y(),rLegend.width(),rLegend.height());

   rComment.setY(rLegend.bottom()+1+(d_lpos==LegendPosition(Bottom))*d_marginY);
   rComment.setX(rLegend.right()+1+(d_lpos==LegendPosition(Right))*d_marginX);
   rComment.setHeight((d_cpos==LegendPosition(Bottom))*csize.height());   
   rComment.setWidth((d_cpos==LegendPosition(Right))*csize.width());
//   debug("comment x:%i,y:%i,w:%i,h:%i",rComment.x(),rComment.y(),rComment.width(),rComment.height());
   
   if(d_cpos==LegendPosition(Bottom)){
      rComment.setX(rPlot.x());
      rComment.setWidth(rPlot.width()/* + x3d*/);
   }
   if(d_cpos==LegendPosition(Right)){
      rComment.setY(rPlot.y());
      rComment.setHeight(rPlot.height() /*+ y3d*/);      
   }
   if(d_lpos==LegendPosition(Bottom)){
      rLegend.setX(rPlot.x());
      rLegend.setWidth(rPlot.width());
   }
   if(d_lpos==LegendPosition(Right)){
      rLegend.setY(rPlot.y());
      rLegend.setHeight(rPlot.height());      
   }

//-------------------------------------------
   rc[xTop].setLeft(rPlot.x() + vDist - hDist);
   rc[xTop].setRight(rPlot.right() + hDist - vDist);
   rc[xBottom].setLeft(rc[xTop].left());
   rc[xBottom].setRight(rc[xTop].right());
//--------------------------------------------   
    
//==========================================
   if (d_axisEnabled[yLeft])
     d_scale[yLeft]->setScaleOffset(vDist, vDist);

   if (d_axisEnabled[yRight] && !d_stacked)
     d_scale[yRight]->setScaleOffset(vDist, vDist);
   
   if (d_axisEnabled[xTop] && !d_stacked)
     d_scale[xTop]->setScaleOffset(hDist, hDist);
   
   if (d_axisEnabled[xBottom])
     d_scale[xBottom]->setScaleOffset(hDist, hDist);
   
//=============================================   
   mr.setRect(0,0,rPlot.width() - 2*FRAMEWIDTH,
	      rPlot.height() - 2*FRAMEWIDTH);
   vDist = qwtMax(0,vDist - FRAMEWIDTH);


   rPix.setRect(rPlot.x()+FRAMEWIDTH,rPlot.y()+FRAMEWIDTH,rPlot.width()-2*FRAMEWIDTH,rPlot.height()-2*FRAMEWIDTH);

   // Offset setting (vDist) !

   if(d_stack){
      d_x3d=(int)(rPix.width()/(fabs(d_stackx)+100.0)*fabs(d_stackx));
      d_y3d=(int)(rPix.height()/(fabs(d_stacky)+100.0)*fabs(d_stacky));
   }
   
   if(d_stacked && (d_stackx || d_stacky) /*&& d_nvisible*/){
      x3d=d_x3d;
      y3d=d_y3d;
      //      x3d=d_stackx*(d_nvisible-1)*rPix.width()/100.0;
      //      y3d=d_stacky*(d_nvisible-1)*rPix.height()/100.0;
      rc[yLeft].setY(rc[yLeft].y()+y3d);
//      rc[yLeft].setHeight(rc[yLeft].height()-y3d);
      rc[xBottom].setWidth(rc[xBottom].width()-x3d);

   }
   else {
      x3d=0;
      y3d=0;
   }
   
   d_map[xBottom].setIntRange(rPix.x()+vDist, rPix.x()+mr.right() - vDist - x3d);
   d_map[xTop].setIntRange(rPix.x()+vDist, rPix.x()+mr.right() - vDist -x3d);
   d_map[yLeft].setIntRange(rPix.y()+mr.bottom() - vDist , rPix.y()+vDist + y3d);
   d_map[yRight].setIntRange(rPix.y()+mr.bottom() - vDist , rPix.y()+vDist + y3d);
   
//   debug("dmap %i/%i",rPix.y()+mr.bottom() - vDist , rPix.y()+vDist + y3d);
   
   d_map[xBottom].setClipRange(rPix.x(), rPix.x()+mr.right() - x3d);
   d_map[xTop].setClipRange(rPix.x(), rPix.x()+mr.right() - x3d);
   d_map[yLeft].setClipRange(rPix.y()+mr.bottom(), rPix.y() + y3d);
   d_map[yRight].setClipRange(rPix.y()+mr.bottom(), rPix.y() + y3d);
   
//   printf("rTitle: %i,%i,%i,%i\n",rTitle.x(),rTitle.y(),rTitle.width(),rTitle.height());
/*   printf("rc[xTop]: %i,%i,%i,%i\n",rc[xTop].x(),rc[xTop].y(),rc[xTop].width(),rc[xTop].height());
   printf("rPlot: %i,%i,%i,%i\n",rPlot.x(),rPlot.y(),rPlot.width(),rPlot.height());
   printf("rPix: %i,%i,%i,%i\n",rPix.x(),rPix.y(),rPix.width(),rPix.height());   
   printf("rc[yLeft]: %i,%i,%i,%i\n",rc[yLeft].x(),rc[yLeft].y(),rc[yLeft].width(),rc[yLeft].height());
   printf("rc[xBottom]: %i,%i,%i,%i\n",rc[xBottom].x(),rc[xBottom].y(),rc[xBottom].width(),rc[xBottom].height());
 */ 
   
   updateStacked();

}

void SamplinPlot::setAxisAutoScale(int axis)
{
    if (axisValid(axis))
    {
       d_scale[axis]->setAutoScale(TRUE);
       autoRefresh();
    }
}

void SamplinPlot::setAxisScale(int axis, double min, double max, double step)
{
    if (axisValid(axis))
    {
       d_scale[axis]->setScale(min,max,step);
       autoRefresh();
    }
    
}

void SamplinPlot::replot()
{
   //updateStacked();
   updateAxes();
   updatePixmap();


   
   repaint(FALSE);
}

void SamplinPlot::updateAxes()
{
    int i;
    SamplinCurve *c;

//   if(d_curveList.isEmpty())return;
   //
    //  Adjust autoscalers
    //
   for (i=0; i<axisCnt; i++)
     if(d_scale[i]->autoScale())d_scale[i]->reset();

   c=d_curveList.first();

   while(c)
     {
	if(!c->isEmpty() && c->visible()){
	   if(!d_stack){
	     if(d_scale[c->xAxis()]->autoScale()){
		 d_scale[c->xAxis()]->adjust(c->minXValue()/*+c->xOffset()*/, c->maxXValue()/*+c->xOffset()*/);
//		 debug("adjust axis %f - %f",c->minXValue()+c->xOffset(),c->maxXValue()+c->xOffset());
	      }
	      if(d_scale[c->yAxis()]->autoScale()){
		 d_scale[c->yAxis()]->adjust(c->minYValue()/*+c->yOffset()*/, c->maxYValue()/*+c->yOffset()*/);	
	      }
	   }
	   else{
	     if(d_scale[xBottom]->autoScale()){
		 d_scale[xBottom]->adjust(c->minXValue()/*+c->xOffset()*/, c->maxXValue()/*+c->xOffset()*/);
//		 debug("adjust axis %f - %f",c->minXValue()+c->xOffset(),c->maxXValue()+c->xOffset());
	      }
	      if(d_scale[yLeft]->autoScale()){
		 d_scale[yLeft]->adjust(c->minYValue()/*+c->yOffset()*/, c->maxYValue()/*+c->yOffset()*/);	
	      }	      
	      
	   }
	}
	c=d_curveList.next();
     }

   if(d_stack && (d_stackx==0 || d_stacky==0)){
      double tmp1,tmp2;
      
      tmp1=d_scale[yLeft]->scaleMin();
      tmp2=d_scale[yLeft]->scaleMax();
      d_scale[yLeft]->adjust(tmp1,tmp1+(tmp2-tmp1)*(1.0+d_stacky/100.0));
      
      tmp1=d_scale[xBottom]->scaleMin();
      tmp2=d_scale[xBottom]->scaleMax();
      d_scale[xBottom]->adjust(tmp1,tmp1+(tmp2-tmp1)*(1.0+d_stackx/100.0));
   }
   
   
   for (i=0; i<axisCnt; i++){
      if(d_scale[i]->isReset()){
	 d_scale[i]->adjust(0.0,1000.0,TRUE);
      }
//      d_scale[i]->build();

//      if(stacked) ... add x%, y%
   }
    //
    // Adjust scales
    //
    for (i=0; i<axisCnt; i++)
    {
       d_scale[i]->build();
       d_map[i].setDblRange(d_scale[i]->scaleDiv().lBound(), d_scale[i]->scaleDiv().hBound(), d_scale[i]->scaleDiv().logScale());
       //debug("low: %lf, high: %lf",d_scale[i]->scaleDiv().lBound(), d_scale[i]->scaleDiv().hBound(), d_scale[i]->scaleDiv().logScale());
    }

   if(!d_stack){
      d_grid.setXDiv(d_scale[d_grid.xAxis()]->scaleDiv());
      d_grid.setYDiv(d_scale[d_grid.yAxis()]->scaleDiv());
   }
   else{
      d_grid.setXDiv(d_scale[xBottom]->scaleDiv());
      d_grid.setYDiv(d_scale[yLeft]->scaleDiv());      
   }
}

void SamplinPlot::setAutoReplot(bool tf)
{
    d_autoReplot = tf;
}

void SamplinPlot::setAxisFont(int axis, QFont &f)
{
    if (axisValid(axis))
	d_scale[axis]->setFont(f);
}

void SamplinPlot::setAxisOptions(int axis, int opt)
{
    if (axisValid(axis))
    {
	d_scale[axis]->setOptions(opt);
	autoRefresh();
    }
}

void SamplinPlot::changeAxisOptions(int axis, int opt, bool additive)
{
    if (axisValid(axis))
    {
	d_scale[axis]->changeOptions(opt, additive);
	autoRefresh();
    }
}

void SamplinPlot::setAxisTitleFont(int axis, const QFont &f)
{
    if (axisValid(axis))
	d_scale[axis]->setTitleFont(f);
}

void SamplinPlot::updatePixmap()
{
   int i;
   QPainter p;
   QRect rt,rr[axisCnt];
   
   p.begin(&d_pix);

   /*
   QPaintDeviceMetrics metrics(this);
   d_marginX=float(metrics.widthMM())/float(metrics.width())*3;
   d_marginY=float(metrics.heightMM())/float(metrics.height())*3;
    */
   
   resizeObjects(p,QRect(0,0,d_pix.width(),d_pix.height()));
   draw(p);

   
   
   p.end();   
}

void SamplinPlot::draw(QPainter &p, bool clip)
{
   int i;
   
   QPointArray pa;
   QRect rr[axisCnt];
   
   if(!clip){
      p.setClipping(TRUE);
      p.setClipRect(rPainter);   
   }
   
//   p.fillRect(rPainter,d_panelColor);

   if(!d_stacked){
      if(clip){
	 p.setPen(NoPen);
	 p.setBrush(d_panelColor);
	 p.drawRect(rPainter);
	 p.setBrush(d_plotColor);
	 p.drawRect(rPix);
	 qDrawShadePanel( &p, rPlot, colorGroup(), TRUE, FRAMEWIDTH );
      }
      else {
	 p.setBrush(d_panelColor);
	 p.setPen(black);
	 p.drawRect(rPainter);      
	 p.setPen(d_grid.majPen().color());
	 p.setBrush(d_plotColor);
	 p.drawRect(rPix);
      }
   }
   if(d_stacked){
      p.setPen(NoPen);
      p.setBrush(d_panelColor);
      p.drawRect(rPainter);      
      p.setBrush(d_plotColor);
      p.setPen(d_grid.majPen().color());
      
      pa.setPoints(6,rPix.left(),rPix.bottom(),rPix.left(),rPix.top()+d_y3d,
		   rPix.left()+d_x3d,rPix.top(),rPix.right(),rPix.top(),
		   rPix.right(),rPix.bottom()-d_y3d,rPix.right()-d_x3d,rPix.bottom());
      p.drawPolygon(pa);
      p.drawLine(rPix.left(),rPix.bottom(),rPix.left()+d_x3d,rPix.bottom()-d_y3d);
      p.drawLine(rPix.left()+d_x3d,rPix.bottom()-d_y3d,rPix.left()+d_x3d,rPix.top());      
      p.drawLine(rPix.left()+d_x3d,rPix.bottom()-d_y3d,rPix.right(),rPix.bottom()-d_y3d);
      
      p.setPen(d_grid.majPen());

//      if(d_nvisible>2)
      for(i=1;i<d_nvisible-1;++i){
	 p.drawLine(rPix.left()+(i*d_x3d)/(d_nvisible-1),rPix.bottom()-(d_y3d*i)/(d_nvisible-1),rPix.right()-d_x3d+(i*d_x3d)/(d_nvisible-1),rPix.bottom()-(d_y3d*i)/(d_nvisible-1));
	 p.drawLine(rPix.left()+(i*d_x3d)/(d_nvisible-1),rPix.bottom()-(d_y3d*i)/(d_nvisible-1),rPix.left()+(i*d_x3d)/(d_nvisible-1),rPix.top()+d_y3d-(d_y3d*i)/(d_nvisible-1));
      }
   }
   
   p.setFont(d_titleFont);
   p.setPen(d_titleColor);
//   printf("Drawing title:%s, color r%i,g%i,b%i\n",d_title.data(),d_titleColor.red(),d_titleColor.green(),d_titleColor.blue());
//   printf("title font:%s, size: %i\n",d_titleFont.family(),d_titleFont.pointSize());
//   printf("title rect:%i,%i,%i,%i\n",rTitle.x(),rTitle.y(),rTitle.height(),rTitle.width());
//   QFontMetrics fm=p.fontMetrics();
//   printf("metrics: h%i,a%i,l%i,line%i, w:%i\n",fm.height(),fm.ascent(),fm.leading(),fm.lineSpacing(),fm.width(d_title));
   
   p.drawText(rTitle,AlignHCenter|AlignBottom,d_title);
//   p.drawText(rTitle,AlignHCenter,d_title);
//   p.drawText(rTitle,AlignBottom,d_title);
//   p.drawText(rTitle.x(),rTitle.y(),d_title);

//   p.drawRect(rc[yLeft]);
//   p.drawRect(rc[xBottom]);
   
   if(d_cpos!=LegendPosition(None)){
      p.setFont(d_cFont);
      p.setPen(d_cColor);
//      p.setBrush(white);
//      p.drawRect(rComment);
      p.setBrush(d_cColor);
      p.drawText(rComment,ShowPrefix|WordBreak,d_comment);
   }

   if(d_lpos!=LegendPosition(None) && d_litemcount){
      p.setPen(d_lColor);
//      p.setBrush(NoBrush);
//      p.drawRect(rLegend);
      p.setBrush(d_lColor);
      p.setFont(d_lFont);
      QFontMetrics fml=p.fontMetrics();
      SamplinCurve *crv;
      QString tmp_name;
      int y=0,x=0;
      int x_off,y_off;
      
      for(crv=d_curveList.first();crv!=0;crv=d_curveList.next())
	if(crv->visible()){

	   if(crv->legend().isEmpty())
	     tmp_name=crv->name();
	   else tmp_name=crv->legend();
	   
	   if(d_lpos==LegendPosition(Bottom)){
	      x_off=x*d_litemwidth;
	      y_off=y*d_litemheight;
	   }
	   else {
	      x_off=y*d_litemwidth;
	      y_off=x*d_litemheight;
	   }
	   
//	   p.drawPixmap(rLegend.x()+x_off,fml.height()/2-7+rLegend.y()+y_off,crv->pixmap(),2,3,13,13);

	   crv->drawIcon(p,QRect(rLegend.x()+x_off,fml.height()/2-7+rLegend.y()+y_off,13,13));
	   p.drawText(rLegend.x()+d_marginX+16+x_off,rLegend.y()+y_off+fml.ascent()-1,tmp_name);
	   
	   ++x;
	   if(x==d_llineitems){
	      ++y;x=0;
	   }
	}
      
   }
   
   if(clip){
      p.setClipping(TRUE);
      p.setClipRect(rPix);   
   }
   
   d_grid.draw(&p, rPix, d_map[d_grid.xSAxis()], d_map[d_grid.ySAxis()]); 
   
   SamplinCurve *curCurve;
/*   int nvisible=0;
   curCurve=d_curveList.last();
   while(curCurve){
      if(curCurve->visible()){
	 ++nvisible;
      }
      curCurve=d_curveList.prev();
   }
 */
   
//   if(nvisible){
      curCurve=d_curveList.last();
      while(curCurve){
	 if(curCurve->visible()){
	    curCurve->draw(&p, d_map[curCurve->xSAxis()], d_map[curCurve->ySAxis()]);
	 }
	 curCurve=d_curveList.prev();
      }
//   }
   
   if(clip)p.setClipping(FALSE);


   if(d_stacked){
      p.setPen(d_grid.majPen().color());
      pa.setPoints(6,rPix.left(),rPix.bottom(),rPix.left(),rPix.top()+d_y3d,
		   rPix.left()+d_x3d,rPix.top(),rPix.right(),rPix.top(),
		   rPix.right(),rPix.bottom()-d_y3d,rPix.right()-d_x3d,rPix.bottom());
      p.drawPolyline(pa);
   }
   
//   p.setPen(black);

   for(i=yLeft;i<axisCnt;++i){
//      p.resetXForm();
      if(d_axisEnabled[i] &&(!d_stacked || i==yLeft || i==xBottom))
	d_scale[i]->draw(p,rc[i]);
   }
   
}

void SamplinPlot::print(void)
{
   int i;
   int pwidth,pheight;
   QRect rr[axisCnt],rt;
   QwtDiMap mx,my;
//   SamplinCurve *curCurve;
   QFont ifnt=QFont("Helvetica",8);
   
   QString info1,info2,info3;
   char tmp[64];
   
   QPrinter prt;
   prt.setOrientation(QPrinter::Landscape);
//   prt.setPrintProgram("print");
   prt.setDocName(fileName());
   prt.setCreator(QString("SampLin ")+SAMPLIN_VERSION);
//   QPrintDialog pdlg(&prt);
//   pdlg.setPrinter(&prt,TRUE);

   
   QPaintDeviceMetrics metrics(&prt);
/*   d_marginX=float(metrics.widthMM())/float(metrics.width())*3;
   d_marginY=float(metrics.heightMM())/float(metrics.height())*3;
    */
   
   if ( /*pdlg.getPrinterSetup(&prt)*/ prt.setup() ) {
      QPainter p;
      p.begin( &prt );


      
//      printf("Paper size: %imm x %imm, %ipt x %ipt\n",metrics.widthMM(),metrics.heightMM(),metrics.width(),metrics.height());
      
      if(d_printfit){
      pwidth=(int)(0.952*(float)metrics.width());
      pheight=(int)(0.965*(float)metrics.height()); 
      }
      else{
	 int mmw, mmh;
	 mmw=(int)MIN(MAX(d_printwidth,30),0.94*(float)metrics.widthMM());
	 mmh=(int)MIN(MAX(d_printheight,20),0.96*(float)metrics.heightMM());
	 pwidth=(int)((float)metrics.width()/(float)metrics.widthMM()*mmw);
	 pheight=(int)((float)metrics.height()/(float)metrics.heightMM()*mmh);
      }

      
      p.setFont(ifnt);
      
      resizeObjects(p,QRect(0,0,pwidth,pheight-d_printinfo*p.fontMetrics().height())); // preserve room for info !!!

      draw(p,FALSE);
      
      if(d_printinfo){
	 time_t tm;
	 struct passwd *pw;
	 
	 pw=getpwuid(getuid());
	 tm=time(NULL);
	 
	 gethostname(tmp,sizeof(tmp)-1);
	 info2=QString(pw->pw_name)+"@"+tmp;
	 info1=QString(fileName());
	 info3=QString(ctime(&tm));
	 info3.truncate(info3.length()-1);
	 
	 info1+=QString(", ")+info3+QString(", ")+info2;
	 
	 p.setPen(black);
	 p.setFont(ifnt);

	 p.setClipRect(0,0,pwidth,pheight);
	 
	 p.drawText(QRect(0,pheight-p.fontMetrics().height(),pwidth,p.fontMetrics().height()),AlignLeft,info1);
//	 p.drawText(QRect(0,pheight-p.fontMetrics().height(),pwidth,p.fontMetrics().height()),AlignRight,info1);	 
      }

      p.end();

      updatePixmap();
   }
   
}

void SamplinPlot::setAxisMaxMinor(int axis, int maxMinor)
{
    if (axisValid(axis))
    {
	d_scale[axis]->setMaxMinor(maxMinor);
	autoRefresh();
    }
}

void SamplinPlot::setAxisMaxMajor(int axis, int maxMajor)
{
    if (axisValid(axis))
    {
	d_scale[axis]->setMaxMajor(maxMajor);
	autoRefresh();
    }
}

void SamplinPlot::setAxisReference(int axis, double value)
{
    if (axisValid(axis))
    {
	d_scale[axis]->setReference(value);
	autoRefresh();
    }
}

void SamplinPlot::setAxisTitle(int axis, const char *t)
{
    if (axisValid(axis))
    {
       d_scale[axis]->setTitle(t);
       autoRefresh();
    }
    
}

double SamplinPlot::transform(int pos, int axis) const
{
    if (axisValid(axis))
       return(d_map[axis].invTransform(pos));
    else
       return 0.0;
}

int SamplinPlot::legendItemSize(QPainter &p) const
{
   int size=0;
   SamplinCurve *crv;

   p.setFont(d_lFont);
   QFontMetrics fm=p.fontMetrics();
   QString tmp_name;
   
   for(crv=d_curveList.first();crv!=0;crv=d_curveList.next())
     if(crv->visible())
     {
	if(crv->legend().isEmpty())
	  tmp_name=crv->name();
	  else tmp_name=crv->legend();
	
	//      size=MAX(size,fm.width(crv->name())+3*d_margin+16);
	if(d_lpos==LegendPosition(Bottom))
	  size=MAX(size,fm.width(tmp_name)+3*d_marginX+16);
	else size=MAX(size,fm.width(tmp_name)+2*d_marginX+16);
     }
   
   return size;
}

void SamplinPlot::setFileName(const char *str)
{
   filename=str;
   emit filenameChanged();
}

void SamplinPlot::updateStacked(void)
{
   SamplinCurve *c;
   int i;
   
   c=d_curveList.first();
   i=0;
   while(c)
     {
	if(!c->isEmpty() && c->visible()){
	   if(d_stack && d_nvisible>1)c->setOffset(d_x3d/(double)(d_nvisible-1)*i,d_y3d/(double)(d_nvisible-1)*i);
	   else c->setOffset(0.0,0.0);
	   ++i;
	}
	c=d_curveList.next();
     }
   
   if(d_stacked/* && d_nvisible*/)d_grid.setOffset(fabs(d_stackx),fabs(d_stacky));
   else d_grid.setOffset(0.0,0.0);
}

void SamplinPlot::showCoord()
{
   SamplinCurve *c;
   double x1,y1,x2,y2;
   int xoff=0,yoff=0;
   QString tmpstr;
   
   if(d_coordTip)delete d_coordTip;

   c=findCurve(d_currentKey);
   if(c!=NULL){
      xoff=c->xOffset();
      yoff=c->yOffset();
      x1=d_map[c->xSAxis()].limInvTransform(d_p1.x()-xoff);
      y1=d_map[c->ySAxis()].limInvTransform(d_p1.y()+yoff);
      x2=d_map[c->xSAxis()].limInvTransform(d_p2.x()-xoff);
      y2=d_map[c->ySAxis()].limInvTransform(d_p2.y()+yoff);
   }
   else{
      x1=d_map[xBottom].limInvTransform(d_p1.x());
      y1=d_map[yLeft].limInvTransform(d_p1.y());
      x2=d_map[xBottom].limInvTransform(d_p2.x());
      y2=d_map[yLeft].limInvTransform(d_p2.y());      
   }
   
   if(fabs(x2-x1)>DBL_MIN)
     tmpstr.sprintf("x1=%g, y1=%g\nx2=%g, y2=%g\ndx=%g, dy=%g\nds=%g\ndy/dx=%g",
      x1,y1,x2,y2,x2-x1,y2-y1,sqrt((y2-y1)*(y2-y1)+(x2-x1)*(x2-x1)),(y2-y1)/(x2-x1));
   else
     tmpstr.sprintf("x1=%g, y1=%g\nx2=%g, y2=%g\ndx=%g, dy=%g",x1,y1,x2,y2,x2-x1,y2-y1);

   d_coordTip=new PlotLabel(tmpstr);
   connect(d_coordTip,SIGNAL(clicked()),SLOT(tipClicked()));

   
   QPoint p = mapToGlobal( d_p2 ) + QPoint( 12, 12 );
   if ( p.x() + d_coordTip->width() > QApplication::desktop()->width() )
     p.setX( QApplication::desktop()->width() - d_coordTip->width() );
   if ( p.y() + d_coordTip->height() > QApplication::desktop()->height() )
             p.setY( p.y() - 20 - d_coordTip->height() );

   
   d_coordTip->move( p );
   d_coordTip->show();
   d_coordTip->raise();
}

void SamplinPlot::tipClicked()
{
   d_coordTip=NULL;
}

void SamplinPlot::setBusy(bool busy)
{
   d_waitflag=busy;
   updateCursor(0,0);
}
