// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "calculationParameter.h"

#include <kconfig.h>

#include <stdio.h> 



// static data

const char InertialConstants::name[ 2 ][ end ][ 8 ] = {
    { "Ag", "Bg", "Cg", "DXg", "DYg", "DZg", "DJg", "DJKg", "DKg", "dJg", "dKg" },
    { "Ae", "Be", "Ce", "DXe", "DYe", "DZe", "DJe", "DJKe", "DKe", "dJe", "dKe" }
};

const char InertialConstants::flagName[ 2 ][ end ][ 8 ] = {
    { "fitAg", "fitBg", "fitCg", "fitDXg", "fitDYg", "fitDZg",
      "fitDJg", "fitDJKg", "fitDKg", "fitdJg", "fitdKg" },
    { "fitAe", "fitBe", "fitCe", "fitDXe", "fitDYe", "fitDZe",
      "fitDJe", "fitDJKe", "fitDKe", "fitdJe", "fitdKe" }
};

const char CalculationParameter::floatName[ fEnd ][ 8 ] = {
    "T", "T2", "TWeight", "Phi", "Theta", "Chi", "Origin", "LPU", "HybridA", "HybridB", "HybridC"
};

const char CalculationParameter::floatFlagName[ fEnd ][ 12 ] = {
    "fitT", "fitT2", "fitTWeight", "fitPhi", "fitTheta", "fitChi",
    "fitOrigin", "fitLPU", "fitHybridA", "fitHybridB", "fitHybridC"
};

const char CalculationParameter::intName[ iEnd ][ 8 ] = {
    "Jmin", "Jmax", "DKmax", "NSSWee", "NSSWeo", "NSSWoe", "NSSWoo"
};

const char CalculationParameter::intFlagName[ iEnd ][ 12 ] = {
    "fitJmin", "fitJmax", "fitDKmax", "fitNSSWee", "fitNSSWeo", "fitNSSWoe", "fitNSSWoo"
};


// functions

void InertialConstants::readConfig( KConfig *config )
{
    KROT_LAUNCH( "Launching InertialConstants::readConfig" );
    config->setGroup( KROT_CONFIG_CALCULATION );
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=begin; i<end; incr( i ) ) {
	    constant( s, i ) = config->readDoubleNumEntry( name[ s ][ i ], 0.0 );
	    fit( s, i ) = config->readBoolEntry( flagName[ s ][ i ], 0.0 );
	}
    return;
}



void InertialConstants::writeConfig( KConfig *config )
{
    KROT_LAUNCH( "Launching InertialConstants::writeConfig" );
    config->setGroup( KROT_CONFIG_CALCULATION );
    for( State s=GroundState; s<=ExcitedState; incr( s ) )
	for( Constant i=begin; i<end; incr( i ) ) {
	    config->writeEntry( name[ s ][ i ], constant( s, i ) );
	    config->writeEntry( flagName[ s ][ i ], fit( s, i ) );
	}
    return;
}



void CalculationParameter::readConfig( KConfig *config )
{
    KROT_LAUNCH( "Launching CalculationParameter::readConfig" );
    config->setGroup( KROT_CONFIG_CALCULATION );
    InertialConstants::readConfig( config );
    for( ParameterFloat p=fBegin; p<fEnd; incr( p ) ) {
	constant( p ) = config->readDoubleNumEntry( floatName[ p ], 0.0 );
	fit( p ) = config->readBoolEntry( floatFlagName[ p ], 0.0 );
    }
    for( ParameterInt p=iBegin; p<iEnd; incr( p ) ) {
	constant( p ) = config->readNumEntry( intName[ p ], 0 );
	fit( p ) = config->readBoolEntry( intFlagName[ p ], 0 );
    }
    return;
}



void CalculationParameter::writeConfig( KConfig *config )
{
    KROT_LAUNCH( "Launching CalculationParameter::writeConfig" );
    config->setGroup( KROT_CONFIG_CALCULATION );
    InertialConstants::writeConfig( config );
    for( ParameterFloat p=fBegin; p<fEnd; incr( p ) ) {
	config->writeEntry( floatName[ p ], constant( p ) );
	config->writeEntry( floatFlagName[ p ], fit( p ) );
    }
    for( ParameterInt p=iBegin; p<iEnd; incr( p ) ) {
	config->writeEntry( intName[ p ], constant( p ) );
	config->writeEntry( intFlagName[ p ], fit( p ) );
    }
    return;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
