// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "calculation.h"
#include "krot.h"
#include "lined.h"

#include <iostream.h>

#include <qbuttongroup.h> 
#include <qcheckbox.h>
#include <qfont.h>
#include <qgroupbox.h> 
#include <qlayout.h>
#include <qradiobutton.h>
#include <qtabdialog.h>

#include <kapp.h>
#include <kbuttonbox.h>
#include <kconfig.h>
#include <klocale.h>



AxisTiltBox::AxisTiltBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Axis Tilting" ), parent, name )
{
    KROT_LAUNCH( "Launching AxisTiltBox constructor" );
    QGridLayout *layout = new QGridLayout( this, 2, 3, 20, 5 );
    flag = new QCheckBox( i18n( "Rotate excited state frame" ), this );
    flag->setMinimumWidth( flag->sizeHint().width() );
    flag->setFixedHeight( flag->sizeHint().height() );
    layout->addMultiCellWidget( flag, 0, 0, 0, 2 );
    angle[ 0 ] = new LblFloated( " j ", 1, par.constant( CalculationParameter::AxisTiltPhi ), this );
    angle[ 0 ]->setLabelFont( QFont( "symbol" ) );
    angle[ 0 ]->setMinimumWidth( angle[0]->sizeHint().width() );
    angle[ 0 ]->setFixedHeight( angle[0]->sizeHint().height() );
    layout->addWidget( angle[0], 1, 0 );
    angle[ 1 ] = new LblFloated( " q ", 1, par.constant( CalculationParameter::AxisTiltTheta ), this );
    angle[ 1 ]->setLabelFont( QFont( "symbol" ) );
    angle[ 1 ]->setMinimumWidth( angle[1]->sizeHint().width() );
    angle[ 1 ]->setFixedHeight( angle[1]->sizeHint().height() );
    layout->addWidget( angle[1], 1, 1 );
    angle[ 2 ] = new LblFloated( " c ", 1, par.constant( CalculationParameter::AxisTiltChi ), this );
    angle[ 2 ]->setLabelFont( QFont( "symbol" ) );
    angle[ 2 ]->setMinimumWidth( angle[2]->sizeHint().width() );
    angle[ 2 ]->setFixedHeight( angle[2]->sizeHint().height() );
    layout->addWidget( angle[2], 1, 2 );
    layout->activate();
    setEnabled( par.axisTilting() );
    connect( flag, SIGNAL( toggled( bool ) ),
  	     this, SLOT( enableAxisTilt( bool ) ) );
    return;
}



ControlParameterBox::ControlParameterBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Calculation Control Parameter" ), parent, name )
{
    KROT_LAUNCH( "Launching ControlParameterBox constructor" );
    QBoxLayout *layout = new QHBoxLayout( this, 20, 5 );
    jmax = new LblInted( "Jmax", 1, 10000, par.constant( CalculationParameter::Jmax ), this );
    jmax->setMinimumWidth( jmax->sizeHint().width() );
    jmax->setFixedHeight( jmax->sizeHint().height() );
    layout->addWidget( jmax );
    dkmax = new LblInted( "DKmax", 0, 10000, par.constant( CalculationParameter::DKmax ), this );
    dkmax->setMinimumWidth( dkmax->sizeHint().width() );
    dkmax->setFixedHeight( dkmax->sizeHint().height() );
    layout->addWidget( dkmax );
    layout->activate();
    return;
}



CentrifugalBox::CentrifugalBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Centrifugal Distortion Terms [MHz]" ), parent, name )
{
    KROT_LAUNCH( "Launching CentrifugalBox constructor" );
    QGridLayout *layout = new QGridLayout( this, 5, 2, 20, 5, "CentrifugalBox layout" );
    terms[0][0] = new ChkLblFloated( "DJ''", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::DJ ),
				  this );
    terms[0][0]->setMinimumWidth( terms[0][0]->sizeHint().width() );
    terms[0][0]->setFixedHeight( terms[0][0]->sizeHint().height() );
    layout->addWidget( terms[0][0], 0, 0 );
    terms[0][1] = new ChkLblFloated( "DJK''", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::DJK ),
				  this );
    terms[0][1]->setMinimumWidth( terms[0][1]->sizeHint().width() );
    terms[0][1]->setFixedHeight( terms[0][1]->sizeHint().height() );
    layout->addWidget( terms[0][1], 1, 0 );
    terms[0][2] = new ChkLblFloated( "DK''", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::DK ),
				  this );
    terms[0][2]->setMinimumWidth( terms[0][2]->sizeHint().width() );
    terms[0][2]->setFixedHeight( terms[0][2]->sizeHint().height() );
    layout->addWidget( terms[0][2], 2, 0 );
    terms[0][3] = new ChkLblFloated( "dJ''", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::dJ ),
				  this );
    terms[0][3]->setMinimumWidth( terms[0][3]->sizeHint().width() );
    terms[0][3]->setFixedHeight( terms[0][3]->sizeHint().height() );
    layout->addWidget( terms[0][3], 3, 0 );
    terms[0][4] = new ChkLblFloated( "dK''", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::dK ),
				  this );
    terms[0][4]->setMinimumWidth( terms[0][4]->sizeHint().width() );
    terms[0][4]->setFixedHeight( terms[0][4]->sizeHint().height() );
    layout->addWidget( terms[0][4], 4, 0 );
    terms[1][0] = new ChkLblFloated( "DJ'", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::DJ ),
				  this );
    terms[1][0]->setMinimumWidth( terms[1][0]->sizeHint().width() );
    terms[1][0]->setFixedHeight( terms[1][0]->sizeHint().height() );
    layout->addWidget( terms[1][0], 0, 1 );
    terms[1][1] = new ChkLblFloated( "DJK'", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::DJK ),
				  this );
    terms[1][1]->setMinimumWidth( terms[1][1]->sizeHint().width() );
    terms[1][1]->setFixedHeight( terms[1][1]->sizeHint().height() );
    layout->addWidget( terms[1][1], 1, 1 );
    terms[1][2] = new ChkLblFloated( "DK'", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::DK ),
				  this );
    terms[1][2]->setMinimumWidth( terms[1][2]->sizeHint().width() );
    terms[1][2]->setFixedHeight( terms[1][2]->sizeHint().height() );
    layout->addWidget( terms[1][2], 2, 1 );
    terms[1][3] = new ChkLblFloated( "dJ'", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::dJ ),
				  this );
    terms[1][3]->setMinimumWidth( terms[1][3]->sizeHint().width() );
    terms[1][3]->setFixedHeight( terms[1][3]->sizeHint().height() );
    layout->addWidget( terms[1][3], 3, 1 );
    terms[1][4] = new ChkLblFloated( "dK'", 1,
				  par.constant( CalculationParameter::GroundState, CalculationParameter::dK ),
				  this );
    terms[1][4]->setMinimumWidth( terms[1][4]->sizeHint().width() );
    terms[1][4]->setFixedHeight( terms[1][4]->sizeHint().height() );
    layout->addWidget( terms[1][4], 4, 1 );
    layout->activate();
    return;
}



HybridBox::HybridBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Band Type" ), parent, name )
{
    KROT_LAUNCH( "Launching HybridBox constructor" );
    QBoxLayout *layout = new QHBoxLayout( this, 20, 5 );
    hybrid[0] = new LblFloated( "%a", 1, par.constant( CalculationParameter::HybridA ), this );
    hybrid[0]->setMinimumWidth( hybrid[0]->sizeHint().width() );
    hybrid[0]->setFixedHeight( hybrid[0]->sizeHint().height() );
    layout->addWidget( hybrid[0] );
    hybrid[1] = new LblFloated( "%b", 1, par.constant( CalculationParameter::HybridB ), this );
    hybrid[1]->setMinimumWidth( hybrid[1]->sizeHint().width() );
    hybrid[1]->setFixedHeight( hybrid[1]->sizeHint().height() );
    layout->addWidget( hybrid[1] );
    hybrid[2] = new LblFloated( "%c", 1, par.constant( CalculationParameter::HybridC ), this );
    hybrid[2]->setMinimumWidth( hybrid[2]->sizeHint().width() );
    hybrid[2]->setFixedHeight( hybrid[2]->sizeHint().height() );
    layout->addWidget( hybrid[2] );
    layout->activate();
    return;
}



LinearBox::LinearBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Linear Distortion Terms [MHz]" ), parent, name )
{
    KROT_LAUNCH( "Launching LinearBox constructor" );
    QGridLayout *layout = new QGridLayout( this, 3, 2, 20, 5, "LinearBox layout" );
    terms[0][0] = new ChkLblFloated( "DX''", 1,
				     par.constant( CalculationParameter::GroundState, CalculationParameter::DX ),
				     this );
    terms[0][0]->setMinimumWidth( terms[0][0]->sizeHint().width() );
    terms[0][0]->setFixedHeight( terms[0][0]->sizeHint().height() );
    layout->addWidget( terms[0][0], 0, 0 );
    terms[0][1] = new ChkLblFloated( "DY''", 1,
				     par.constant( CalculationParameter::GroundState, CalculationParameter::DY ),
				     this );
    terms[0][1]->setMinimumWidth( terms[0][1]->sizeHint().width() );
    terms[0][1]->setFixedHeight( terms[0][1]->sizeHint().height() );
    layout->addWidget( terms[0][1], 1, 0 );
    terms[0][2] = new ChkLblFloated( "DZ''", 1,
				     par.constant( CalculationParameter::GroundState, CalculationParameter::DZ ),
				     this );
    terms[0][2]->setMinimumWidth( terms[0][2]->sizeHint().width() );
    terms[0][2]->setFixedHeight( terms[0][2]->sizeHint().height() );
    layout->addWidget( terms[0][2], 2, 0 );
    terms[1][0] = new ChkLblFloated( "DX'", 1,
				     par.constant( CalculationParameter::GroundState, CalculationParameter::DX ),
				     this );
    terms[1][0]->setMinimumWidth( terms[1][0]->sizeHint().width() );
    terms[1][0]->setFixedHeight( terms[1][0]->sizeHint().height() );
    layout->addWidget( terms[1][0], 0, 1 );
    terms[1][1] = new ChkLblFloated( "DY'", 1,
				     par.constant( CalculationParameter::GroundState, CalculationParameter::DY ),
				     this );
    terms[1][1]->setMinimumWidth( terms[1][1]->sizeHint().width() );
    terms[1][1]->setFixedHeight( terms[1][1]->sizeHint().height() );
    layout->addWidget( terms[1][1], 1, 1 );
    terms[1][2] = new ChkLblFloated( "DZ'", 1,
				     par.constant( CalculationParameter::GroundState, CalculationParameter::DZ ),
				     this );
    terms[1][2]->setMinimumWidth( terms[1][2]->sizeHint().width() );
    terms[1][2]->setFixedHeight( terms[1][2]->sizeHint().height() );
    layout->addWidget( terms[1][2], 2, 1 );
    layout->activate();
    return;
}



SpinStatisticBox::SpinStatisticBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Nuclear Spin Statistical Weights" ), parent, name )
{
    KROT_LAUNCH( "Launching SpinStatisticBox constructor" );
    QGridLayout *layout = new QGridLayout( this, 1, 4, 20, 5, "SpinStatisticBox layout" );
    nssw[0] = new LblInted( "ee", 0, INT_MAX, par.constant( CalculationParameter::NSSWee ), this );
    nssw[0]->setMinimumWidth( nssw[0]->sizeHint().width() );
    nssw[0]->setFixedHeight( nssw[0]->sizeHint().height() );
    layout->addWidget( nssw[0], 0, 0 );
    nssw[1] = new LblInted( "eo", 0, INT_MAX, par.constant( CalculationParameter::NSSWeo ), this );
    nssw[1]->setMinimumWidth( nssw[1]->sizeHint().width() );
    nssw[1]->setFixedHeight( nssw[1]->sizeHint().height() );
    layout->addWidget( nssw[1], 0, 1 );
    nssw[2] = new LblInted( "oe", 0, INT_MAX, par.constant( CalculationParameter::NSSWoe ), this );
    nssw[2]->setMinimumWidth( nssw[2]->sizeHint().width() );
    nssw[2]->setFixedHeight( nssw[2]->sizeHint().height() );
    layout->addWidget( nssw[2], 0, 2 );
    nssw[3] = new LblInted( "oo", 0, INT_MAX, par.constant( CalculationParameter::NSSWoo ), this );
    nssw[3]->setMinimumWidth( nssw[3]->sizeHint().width() );
    nssw[3]->setFixedHeight( nssw[3]->sizeHint().height() );
    layout->addWidget( nssw[3], 0, 3 );
    layout->activate();
    return;
}



OriginBox::OriginBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Vibronic Origin Frequency" ), parent, name ),
      config( kapp->getConfig() )
{
    KROT_LAUNCH( "Launching OriginBox constructor" );
    config->setGroup( KROT_CONFIG_CALCULATION );
    QGridLayout *layout = new QGridLayout( this, 2, 2, 20, 5, "OriginLayout" );
    origin = new ChkLblFloated( "n0", 6, 0, this );
    origin->setLabelFont( QFont( "symbol" ) );
    origin->setMinimumWidth( origin->sizeHint().width() );
    origin->setFixedHeight( origin->sizeHint().height() );
    layout->addMultiCellWidget( origin, 0, 0, 0, 1 );
    QButtonGroup *unitBG = new QButtonGroup( 0, "OriginUnitGroup" );
    cmRB = new QRadioButton( i18n( "cm^-1" ), this, "cmRB" );
    cmRB->setChecked( true );
    cmRB->setFixedHeight( cmRB->sizeHint().height() );
    unitBG->insert( cmRB, ID_CM );
    layout->addWidget( cmRB, 1, 0 );
    mhzRB = new QRadioButton( i18n( "MHz" ), this, "mhzRB" );
    mhzRB->setFixedHeight( mhzRB->sizeHint().height() );
    unitBG->insert( mhzRB, ID_MHZ );
    mhzRB->setChecked( config->readBoolEntry( config_OriginUnit, false ) );
    layout->addWidget( mhzRB, 1, 1 );
    layout->activate();
    if( mhzRB->isChecked() )
	origin->setValue( par.constant( CalculationParameter::Origin ) );
    else
	origin->setValue( par.constant( CalculationParameter::Origin ) / Constants::cm2MHz() );
    connect( unitBG, SIGNAL( clicked( int ) ),
	     this, SLOT( unitChanged( int ) ) );
    return;
}



void OriginBox::unitChanged( int )
{
    config->setGroup( KROT_CONFIG_CALCULATION );
    if( cmRB->isChecked() ) {
	origin->setValue( origin->value() / Constants::cm2MHz() );
	config->writeEntry( config_OriginUnit, false );
    } else {
	origin->setValue( origin->value() * Constants::cm2MHz() );
	config->writeEntry( config_OriginUnit, true );
    }
    return;
}



RotationalConstantsBox::RotationalConstantsBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Rotational Constants [MHz]" ), parent, name )
{
    KROT_LAUNCH( "Launching RotationalConstantsBox constructor" );
    QGridLayout *layout = new QGridLayout( this, 3, 2, 20, 5 );
    Ag = new ChkLblFloated( "A''", 6, 0, this );
    Ag->setValue( par.constant( CalculationParameter::GroundState, CalculationParameter::A ) );
    Ag->setMinimumWidth( Ag->sizeHint().width() );
    Ag->setFixedHeight( Ag->sizeHint().height() );
    layout->addWidget( Ag, 0, 0 );
    Bg = new ChkLblFloated( "B''", 6, 0, this );
    Bg->setValue( par.constant( CalculationParameter::GroundState, CalculationParameter::B ) );
    Bg->setMinimumWidth( Bg->sizeHint().width() );
    Bg->setFixedHeight( Bg->sizeHint().height() );
    layout->addWidget( Bg, 1, 0 );
    Cg = new ChkLblFloated( "C''", 6, 0, this );
    Cg->setValue( par.constant( CalculationParameter::GroundState, CalculationParameter::C ) );
    Cg->setMinimumWidth( Cg->sizeHint().width() );
    Cg->setFixedHeight( Cg->sizeHint().height() );
    layout->addWidget( Cg, 2, 0 );
    Ae = new ChkLblFloated( "A'", 6, 0, this );
    Ae->setValue( par.constant( CalculationParameter::ExcitedState, CalculationParameter::A ) );
    Ae->setMinimumWidth( Ae->sizeHint().width() );
    Ae->setFixedHeight( Ae->sizeHint().height() );
    layout->addWidget( Ae, 0, 1 );
    Be = new ChkLblFloated( "B'", 6, 0, this );
    Be->setValue( par.constant( CalculationParameter::ExcitedState, CalculationParameter::B ) );
    Be->setMinimumWidth( Be->sizeHint().width() );
    Be->setFixedHeight( Be->sizeHint().height() );
    layout->addWidget( Be, 1, 1 );
    Ce = new ChkLblFloated( "C'", 6, 0, this );
    Ce->setValue( par.constant( CalculationParameter::ExcitedState, CalculationParameter::C ) );
    Ce->setMinimumWidth( Ce->sizeHint().width() );
    Ce->setFixedHeight( Ce->sizeHint().height() );
    layout->addWidget( Ce, 2, 1 );
    layout->activate();
    return;
}



TemperatureBox::TemperatureBox( const CalculationParameter& par, QWidget *parent, const char *name )
    : ParameterBox( i18n( "Rotational Temperature [K]" ), parent, name )
{
    KROT_LAUNCH( "Launching TemperatureBox constructor" );
    QBoxLayout *layout = new QHBoxLayout( this, 20, 5, "TemperatureLayout" );
    temp = new LblFloated( i18n( "Temperature" ), 3, par.constant( CalculationParameter::Temperature ), this );
    temp->setMinimumWidth( temp->sizeHint().width() );
    temp->setFixedHeight( temp->sizeHint().height() );
    layout->addWidget( temp );
    layout->activate();
    return;
}



PageCentrifugal::PageCentrifugal( const CalculationParameter& par, QWidget *parent, const char *name )
    : TabPage( parent, name )
{
    KROT_LAUNCH("Launching PageCentrifugal constructor" );
    QBoxLayout *layout = new QVBoxLayout( this, 5, 5, "PageCentrifugalVLayout" );
    // temperature
    cent = new CentrifugalBox( par, this, "PageCentrifugal centrifugal box" );
    layout->addWidget( cent );
    layout->addStretch( 10 );
    layout->activate();
    return;
}



PageIntensity::PageIntensity( const CalculationParameter& par, QWidget *parent, const char *name )
    : TabPage( parent, name )
{
    KROT_LAUNCH("Launching PageIntensity constructor" );
    QBoxLayout *layout = new QVBoxLayout( this, 5, 5, "PageIntensityVLayout" );
    // temperature
    temp = new TemperatureBox( par, this, "PageIntensity temperature box" );
    layout->addWidget( temp );
    // hybrid character
    hybrid = new HybridBox( par, this, "PageIntensity hybrid box" );
    layout->addWidget( hybrid );
    // axis tilting
    axisTilt = new AxisTiltBox( par, this, "PageIntensity axis tilting box" );
    layout->addWidget( axisTilt );
    // nuclear spin statistic
    nssw = new SpinStatisticBox( par, this, "PageIntensity spin statistics box" );
    layout->addWidget( nssw );
    layout->addStretch( 10 );
    layout->activate();
    return;
}



PageLinear::PageLinear( const CalculationParameter& par, QWidget *parent, const char *name )
    : TabPage( parent, name )
{
    KROT_LAUNCH("Launching PageLinear constructor" );
    QBoxLayout *layout = new QVBoxLayout( this, 5, 5, "PageLinearVLayout" );
    // linear distortion terms
    lin = new LinearBox( par, this, "PageLinear LinearBox" );
    layout->addWidget( lin );
    layout->addStretch( 10 );
    layout->activate();
    return;
}



PageOneArnirot::PageOneArnirot( const CalculationParameter& par, QWidget *parent, const char *name )
    : TabPage( parent, name )
{
    KROT_LAUNCH( "Launching PageOneArnirot constructor" );
    layout = new QVBoxLayout( this, 5, 5, "PageOneVLayout" );
    // rotational constants
    rotConstants = new RotationalConstantsBox( par, this, "Rotational Constants GroupBox" );
    layout->addWidget( rotConstants );
    // origin
    origin = new OriginBox( par, this, "Origin" );
    layout->addWidget( origin );
    // control parameter
    control = new ControlParameterBox( par, this, "Control" );
    layout->addWidget( control );
    // flags to implement other calculations
    QGroupBox *flags = new QGroupBox( i18n( "Additional Calculation Parameter" ), this, "CalcParam" );
    QGridLayout *flagsLayout = new QGridLayout( flags, 2, 2, 15, 5 );
    linDist  = new QCheckBox( i18n( "Linear Distortion Terms" ), flags );
    linDist->setChecked( par.linearDistortion() );
    linDist->setFixedHeight( linDist->sizeHint().height() );
    linDist->setMinimumWidth( linDist->sizeHint().width() );
    flagsLayout->addWidget( linDist, 0, 0 );
    centDist = new QCheckBox( i18n( "Centrifugal Distortion" ), flags );
    centDist->setChecked( par.centrifugalDistortion() );
    centDist->setFixedHeight( centDist->sizeHint().height() );
    centDist->setMinimumWidth( centDist->sizeHint().width() );
    flagsLayout->addWidget( centDist, 1, 0 );
    flagsLayout->activate();
    layout->addWidget( flags );
    connect( centDist, SIGNAL( toggled(bool) ),
 	     this, SLOT( slotCentDist(bool) ) );
    connect( linDist, SIGNAL( toggled(bool) ),
	     this, SLOT( slotLinDist(bool) ) );
    layout->addStretch( 10 );
    layout->activate();
    resize( sizeHint() );
    return;
}



DialogCalculation::DialogCalculation( const class CalculationParameter& param, QWidget *parent, const char *name )
    : QTabDialog( parent, name, true ),
      par( param ),
      reallyDone( false )
{
    KROT_LAUNCH( "Launching DialogCalculation constructor." );
    setCaption( i18n( "KRot [Calculation]" ) );
    // be aware of the "misuse" of this buttons to make the help button the rightmost one !
    setOkButton( i18n( "OK" ) );
    setDefaultButton( i18n( "Cancel" ) );
    setCancelButton( i18n( "Help" ) );
    connect( this, SIGNAL( applyButtonPressed() ),
	     this, SLOT( accept() ) );
    connect( this, SIGNAL( defaultButtonPressed() ),
	     this, SLOT( reject() ) );
    connect( this, SIGNAL( cancelButtonPressed() ),
	     this, SLOT( help() ) );
    return;
}



DialogCalculationArnirot::DialogCalculationArnirot( const CalculationParameter& param, QWidget *parent, const char *name )
    : DialogCalculation( param, parent, name )
{
    KROT_LAUNCH( "Launching DialogCalculationArnirot constructor." );
    setCaption( i18n( "KRot [Arnirot Calculation]" ) );
    one = new PageOneArnirot( param, this, nameOne );
    intensity = new PageIntensity( param, this, nameIntensity );
    lin  = new PageLinear( param, this, nameLinearDistortion );
    cent = new PageCentrifugal( param, this, nameCentrifugalDistortion );
    addTab( one, i18n( "&General" ) );
    addTab( intensity, i18n( "&Intensity" ) );
    addTab( lin, i18n( "&Linear Distortion" ) );
    addTab( cent, i18n( "&Centrifugal Distortion" ) );
    setTabEnabled( nameOne, true );
    setTabEnabled( nameIntensity, true );
    setTabEnabled( nameLinearDistortion, par.linearDistortion() );
    setTabEnabled( nameCentrifugalDistortion, par.centrifugalDistortion() );
    resize( minimumSize() );
    connect( one, SIGNAL( centDistToggled(bool) ),
 	     this, SLOT( enableCentDist(bool) ) );
    connect( one, SIGNAL( linDistToggled(bool) ),
 	     this, SLOT( enableLinDist(bool) ) );
    // ensure we see page one when we first enter the dialog
    showPage( one );
    return;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
