// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "calculation.h"
#include "deleteSet.h"
#include "editAssignment.h"
#include "editSets.h"
#include "importRawData.h"
#include "krotData.h"
#include "mainwindow.h"
#include "selectSet.h"
#include "workarea.h"

#include <kapp.h>
#include <kfiledialog.h>
#include <kmsgbox.h> 
#include <ktmainwindow.h>

#include <qevent.h>
#include <qpopupmenu.h>
#include <qstring.h>
#include <qtimer.h>



QList<MainWindow> MainWindow::windowList;



MainWindow::MainWindow( const char* name )
    : KTMainWindow( name ),
      config( kapp->getConfig() ),
      dialogFoldSet( 0 ), dialogGeomTransform( 0 ), dialogImport( 0 ),
      // dialogPreferences( 0 ),
      msgCount( 0 ), msgTimer( new QTimer( this ) ),
      datFilter( QString( "*.dat *.dat.gz *.dat.Z|" ) + QString( i18n( "All Data files" ) )
		 + QString( "\n*.dat|" ) + QString( i18n( "Uncompressed Data files" ) )
		 + QString( "\n*.dat.gz *.dat.Z|" ) + QString( i18n( "Compressed Data files" ) )
		 + QString( "\n*|" ) + QString( i18n( "All Files" ) ) ),
      krtFilter( QString( "*.krt *.krt.gz *.krt.Z|" ) + QString( i18n( "All Data files" ) )
		 + QString( "\n*.krt|" ) + QString( i18n( "Uncompressed Data files" ) )
		 + QString( "\n*.krt.gz *.krt.Z|" ) + QString( i18n( "Compressed Data files" ) )
		 + QString( "\n*|" ) + QString( i18n( "All Files" ) ) )
{
    KROT_LAUNCH( "Launching MainWindow constructor." );
    windowList.setAutoDelete( false );
    windowList.append( this );
    setCaption( "KRot" );
    krotData = new KRotData;
    menu();
    status();
    tool();
    view = new WorkArea( krotData, this, "WorkArea" );
    setView( view );
    
    connect( krotData, SIGNAL( limits( const DataRanges& ) ),
             view, SLOT( limits( const DataRanges& ) ) );
    connect( krotData, SIGNAL( message( const QString& ) ),
	     this, SLOT( setMessage( const QString& ) ) );
    connect( krotData, SIGNAL( newSet() ),
             view, SLOT( update() ) );
    connect( krotData, SIGNAL( position( const int64_t ) ),
             view, SLOT( setPosition( const int64_t ) ) );
    connect( this, SIGNAL( exportAssignments( const QString& ) ),
	     krotData, SLOT( exportAssignments( const QString& ) ) );
    connect( this, SIGNAL( exportLines( const QString&, const QString& ) ),
	     krotData, SLOT( exportLines( const QString&, const QString& ) ) );
    connect( this, SIGNAL( exportXY( const QString&, const QString& ) ),
	     krotData,   SLOT( exportXY( const QString&, const QString& ) ) );
    connect( this, SIGNAL( importLines( const QString& ) ),
	     krotData,   SLOT( importLines( const QString& ) ) );
    connect( this, SIGNAL( importXY( const QString& ) ),
	     krotData,   SLOT( importXY( const QString& ) ) );
    connect( this, SIGNAL( newAveraging(const unsigned int ) ),
	     krotData,   SLOT( setAveraging(const unsigned int ) ) );
    connect( msgTimer, SIGNAL( timeout() ),
	     this, SLOT( clearMessage() ) );
    connect( this, SIGNAL( newAveraging( const unsigned int ) ),
	     view, SLOT( setAveraging( const unsigned int ) ) );
    connect( view, SIGNAL( assignment( const AssignedFrequency& ) ),
	     krotData, SLOT( add( const AssignedFrequency& ) ) );
    connect( view, SIGNAL( message( const QString& ) ),
	     this, SLOT( setMessage( const QString& ) ) );
    connect( view, SIGNAL( freq( const int64_t ) ),
	     this, SLOT( setFreq( const int64_t ) ) );

    // frequently used child dialogs
    dialogArnirot = new DialogCalculationArnirot( krotData->parameter(), 0, "DialogCalculationArnirot" );
    connect( dialogArnirot, SIGNAL( message( const QString& ) ),
	     this, SLOT( setMessage( const QString& ) ) );
    connect( dialogArnirot, SIGNAL( calculate( const CalculationParameter& ) ),
	     krotData, SLOT( arnirot( const CalculationParameter& ) ) );
    dialogGeomTransform = new DialogGeometryTransform( krotData );
    dialogSelectSet = new DialogSelectSet( krotData );
}



void MainWindow::autocorrelate()
{
    KROT_LAUNCH( "Launching MainWindow::autocorrelate" );
    if( ! krotData->empty() ) {
	if( QDialog::Accepted == dialogSelectSet->exec() ) {
	    DataSet *set = dialogSelectSet->getSelection();
	    if( 0 != set )
		krotData->add( set->autocorrelate() );
	    else
		setMessage( i18n( "No DataSet for autocorrelation selected." ) );
	}
    } else
	setMessage( i18n( "No DataSet loaded right now." ) );
    return;
}



void MainWindow::closeEvent( QCloseEvent *event )
{
    int button = KMsgBox::yesNo( this, i18n( "Quit" ),
     				 i18n( "Do you really want to quit this project ?" ),
				 KMsgBox::QUESTION | KMsgBox::DB_SECOND,
				 i18n( "Yes" ), i18n( "No" ) );
    if( button == 1 ) {
	windowList.remove( this );
	if( 0 == windowList.count() ) {
	    config->sync();
	    // deleteAll();
	    kapp->quit();
	} else
	    event->accept();
    } else
	event->ignore();
    return;
}



void MainWindow::crosscorrelate()
{
    KROT_LAUNCH( "Launching MainWindow::crosscorrelate" );
    if( ! krotData->empty() ) {
	if( QDialog::Accepted == dialogSelectSet->exec() ) {
	    DataSet *one = dialogSelectSet->getSelection();
	    if( 0 != one )
		if( QDialog::Accepted == dialogSelectSet->exec() ) {
		    DataSet *two = dialogSelectSet->getSelection();
		    if( 0 != two )
			krotData->add( one->crosscorrelate( *two ) );
		} else
		    setMessage( i18n( "No second DataSet for crosscorrelation selected." ) );
	}
	else
	    setMessage( i18n( "No DataSet for crosscorrelation selected." ) );
    } else
	setMessage( i18n( "No DataSet loaded right now." ) );
}



void MainWindow::deleteSet()
{
    KROT_LAUNCH( "Launching MainWindow::deleteSet" );
    if( ! krotData->empty() ) {
	DialogDeleteSet dialog( krotData );
	connect( &dialog, SIGNAL( deleted() ),
		 view, SLOT( update() ) );
	connect( &dialog, SIGNAL( message( const QString& ) ),
		 this, SLOT( setMessage( const QString& ) ) );
    } else
	setMessage( i18n( "No DataSet loaded right now." ) );
}



void MainWindow::difference()
{
    KROT_LAUNCH( "Launching MainWindow::difference" );
    if( ! krotData->empty() ) {
	if( QDialog::Accepted == dialogSelectSet->exec() ) {
	    DataSet *one = dialogSelectSet->getSelection();
	    if( 0 != one )
		if( QDialog::Accepted == dialogSelectSet->exec() ) {
		    DataSet *two = dialogSelectSet->getSelection();
		    if( 0 != two ) {
			try {
			    krotData->add( one->difference( *two ) );
			}
			catch ( NoDataSetOverlap ) {
			    KMsgBox::message( this, i18n( "Error" ),
					      i18n( "Selected DataSets do not overlap,\n"
						    "Cannot calculate difference spectrum" ),
					      KMsgBox::EXCLAMATION );
			}
		    }
		} else
		    setMessage( i18n( "No second DataSet for crosscorrelation selected." ) );
	} else
	    setMessage( i18n( "No DataSet for crosscorrelation selected." ) );
    } else
	setMessage( i18n( "No DataSet loaded right now." ) );
    return;
}



void MainWindow::editAssignments()
{
    KROT_LAUNCH( "Launching MainWindow::editAssignments" );
    if( ! krotData->empty( KRotData::ASSIGNMENTS ) ) {
	DialogEditAssignments dialog( krotData );
	dialog.show();
    } else
	setMessage( i18n( "No assignments made yet." ) );
    return;
}



void MainWindow::exportAssignments()
{
    KROT_LAUNCH( "Launching MainWindow::exportAssignments" );
    QString file = KFileDialog::getSaveFileName( 0, "*.asn", this );
    if ( ! file.isEmpty() ) {
	emit exportAssignments( file );
    }
    return;
}



void MainWindow::exportLines()
{
    KROT_LAUNCH( "Launching MainWindow::exportLines." );
    if( ! krotData->empty( KRotData::SIMULATIONS ) ) {
#warning Change !
	const QString& simName = dialogSelectSet->getSelectionString();
	if( ! simName.isEmpty() ) {
	    QString fileName = KFileDialog::getSaveFileName( config->readEntry( config_CurrentDir, 0 ),
							     "*.sim" ).data();
	    if( ! fileName.isNull() ) {
		emit exportLines( simName, fileName );
	    }
	}
    } else
	KMsgBox::message( this, i18n( "No Simulation" ),
			  i18n( "There is no Simulation you could export loaded right now." ),
			  KMsgBox::EXCLAMATION, i18n( "OK" ) );
    return;
}



void MainWindow::exportXY()
{
    KROT_LAUNCH( "Launching MainWindow::exportXY." );
    if( ! krotData->empty( KRotData::DATASETS ) ) {
	if( QDialog::Accepted == dialogSelectSet->exec() ) {
	    const QString setName = dialogSelectSet->getSelectionString();
	    if( ! setName.isEmpty() ) {
		QString fileName = KFileDialog::getSaveFileName( config->readEntry( config_CurrentDir, 0 ),
								 "*.dat" );
		if( ! fileName.isNull() ) {
		    emit exportXY( setName, fileName );
		}
	    }
	}
    } else
	KMsgBox::message( this, i18n( "No Data" ),
			  i18n( "There is no data you could export loaded right now." ),
			  KMsgBox::EXCLAMATION, i18n( "OK" ) );
    return;
}



void MainWindow::foldSimulations()
{
    KROT_LAUNCH( "Launching MainWindow::editSimulations" );
    if( ! krotData->empty( KRotData::SIMULATIONS ) ) {
	if( ! dialogFoldSet ) {
	    dialogFoldSet = new DialogFoldSet( krotData );
	    connect( dialogFoldSet, SIGNAL( changed() ),
		     view, SLOT( update() ) );
	}
	dialogFoldSet->exec();
    } else
	setMessage( i18n( "No Simulation loaded right now." ) );
    return;
}



void MainWindow::importLines()
{
    KROT_LAUNCH( "Launching MainWindow::importLines" );
    const QString name = KFileDialog::getOpenFileName( config->readEntry( config_CurrentDir, 0 ),
						       "*.dat");
    if( ! name.isNull() ) {
	emit importLines( name );
    }
    return;
}



void MainWindow::iar()
{
    KROT_LAUNCH( "Launching MainWindow::iar" );
//     DialogCalculation *dlg = new DialogCalculationIAR( krotData->parameter() );
//     connect( dlg, SIGNAL( calculate( CalculationParameter ) ),
// 	     krotData, SLOT( iar( CalculationParameter ) ) );
//     dlg->exec();
//     delete dlg;
}



void MainWindow::importJba()
{
    KROT_LAUNCH( "Launching MainWindow::importJba" );
    QString dir = config->readEntry( config_CurrentDir, 0 );
    QString name = KFileDialog::getOpenFileName( dir, "*.dat" );
    if( ! name.isEmpty() ) {
	config->setGroup( KROT_CONFIG_IMPORT );
	config->writeEntry( config_CurrentDir, name.left( name.findRev( '/' ) ) );
	QFile file ( name );
	if( file.exists() ) {
	    // see importKHiDaq
	} else
	    KMsgBox::message( 0, i18n( "File Error" ),
			      i18n( "Could not open file to import Jba data." ),
			      KMsgBox::EXCLAMATION, i18n( "OK" ) );
    }
    return;
}



void MainWindow::importKHiDaq()
{
    KROT_LAUNCH( "Launching MainWindow::importKHiDaq" );
    config->setGroup( KROT_CONFIG_IMPORT );
    QString dir = config->readEntry( config_CurrentDir, 0 );
    QString name = KFileDialog::getOpenFileName( dir, datFilter );
    if(! name.isEmpty() ) {
	config->setGroup( KROT_CONFIG_IMPORT );
	config->writeEntry( config_CurrentDir, name.mid( 0, name.findRev( '/' ) ) );
	QFile file ( name );
	if( file.exists() ) {
	    KHiDaqData *khidaq = new KHiDaqData( name );
	    QString msg( i18n( " KHiDaq data from " ) );
	    msg += file.name();
	    msg += i18n( " loaded" );
	    setMessage( msg );
	    dialogImport = new DialogImportKHiDaq( *khidaq );
	    dialogImport->show();
	    connect( dialogImport, SIGNAL( marker( const int ) ),
		     khidaq, SLOT( toggleMarker( const int ) ) );
	    connect( dialogImport, SIGNAL(import( const RawData& )),
		     krotData, SLOT(importKHiDaq( const RawData& )) );
	} else
	    KMsgBox::message( 0, i18n( "File Error" ),
			      i18n( "Could not open file to import KHiDaq data." ),
			      KMsgBox::EXCLAMATION, i18n( "OK" ) );
    }
    return;
}



void MainWindow::importXY()
{
    KROT_LAUNCH( "Launching MainWindow::importXY" );
    config->setGroup( KROT_CONFIG_IMPORT );
    QString dir = config->readEntry( config_CurrentDir, 0 );
    QString name = KFileDialog::getOpenFileName( dir, "*.dat *.xy" );
    if( ! name.isEmpty() ) {
	config->setGroup( KROT_CONFIG_IMPORT );
	config->writeEntry( config_CurrentDir, name.left( name.findRev( '/' ) ) );
	emit importXY( name );
    }
    return;
}



void MainWindow::jarot()
{
    KROT_LAUNCH( "Launching MainWindow::jarot" );
//     DialogCalculation *dlg = new DialogCalculationJarot( krotData->parameter() );
//     connect( dlg, SIGNAL( calculate( CalculationParameter ) ),
// 	     krotData, SLOT( jarot( CalculationParameter ) ) );
//     dlg->exec();
//     delete dlg;
}



void MainWindow::menu()
{
    KROT_LAUNCH( "Launching MainWindow::menu" );
    // ** file menu **
    QPopupMenu *file = new QPopupMenu;
    file->insertItem( i18n( "New Window" ), this, SLOT( newWindow() ) );
    file->insertSeparator();
    file->insertItem( i18n( "Open" ) );
    file->insertItem( i18n( "Save" ) );
    file->insertItem( i18n( "Save As" ) );
    file->insertSeparator();
    // import
    QPopupMenu *import = new QPopupMenu;
    import->insertItem( i18n( "KHiDaq file" ), this, SLOT( importKHiDaq() ) );
    import->insertItem( i18n( "jba file" ), this, SLOT( importJba() ) );
    import->insertItem( i18n( "XY data" ), this, SLOT( importXY() ) );
    import->insertItem( i18n( "lines file" ), this, SLOT( importLines() ) );
    file->insertItem( i18n( "Import" ), import);
    // export
    QPopupMenu *export = new QPopupMenu;
    export->insertItem( i18n( "Assignments" ), this, SLOT( exportAssignments() ) );
    export->insertItem( i18n( "lines file" ), this, SLOT( exportLines() ) );
    export->insertItem( i18n( "XY data" ), this, SLOT( exportXY() ) );
    file->insertItem( i18n( "Export" ), export);
    // quit
    file->insertSeparator();
    file->insertItem( i18n( "Close" ), this, SLOT( close() ) );
    file->insertItem( i18n( "Quit" ), this, SLOT( closeAll() ) );
    // ** edit menu **
    QPopupMenu *edit = new QPopupMenu;
    edit->insertItem( i18n( "Fold simulations" ), this, SLOT( foldSimulations() ) );
    edit->insertItem( i18n( "Edit spectra" ), this, SLOT( editSpectra() ) );
    edit->insertItem( i18n( "Geometric transformations" ), this, SLOT( geomTransform() ) );
    edit->insertItem( i18n( "Delete datasets" ), this, SLOT( deleteSet() ) );
    edit->insertSeparator();
    edit->insertItem( i18n( "Edit assignments" ), this, SLOT( editAssignments() ) );
    edit->insertSeparator();
    edit->insertItem( i18n( "Preferences" ), this, SLOT( preferences() ) );
    
    // ** calculation menu **
    QPopupMenu *calc = new QPopupMenu;
    calc->insertItem( i18n( "arnirot" ), this, SLOT( arnirot() ) );
    calc->setItemEnabled( calc->insertItem( i18n( "iar" ), this, SLOT( iar() ) ), false );
    calc->setItemEnabled( calc->insertItem( i18n( "jarot" ), this, SLOT( jarot() ) ), false );
    calc->insertSeparator();
    calc->insertItem( i18n( "auto-correlation" ), this, SLOT( autocorrelate() ) );
    calc->insertItem( i18n( "cross-correlation" ), this, SLOT( crosscorrelate() ) );
    calc->insertItem( i18n( "difference" ), this, SLOT( difference() ) );
    // ** help menu **
    QPopupMenu *help = kapp->getHelpMenu( true, aboutKRot );
    // ** menubar **
    menuBar()->insertItem( i18n( "File" ), file);
    menuBar()->insertItem( i18n( "Edit" ), edit);
    menuBar()->insertItem( i18n( "Calculation" ), calc);
    menuBar()->insertSeparator();
    menuBar()->insertItem( i18n( "Help" ), help);
    return;
}



void MainWindow::newWindow()
{
    MainWindow *win = new MainWindow( "krot" );
    win->show();
}



void MainWindow::preferences()
{
}



void MainWindow::status()
{
    KROT_LAUNCH( "Launching MainWindow::status" );
    statusBar()->setBorderWidth( 3 );
    statusBar()->setInsertOrder( KStatusBar::RightToLeft );
    statusBar()->insertItem( "00000000000000 MHz", ID_FREQUENCY );
    statusBar()->setAlignment( ID_FREQUENCY, AlignRight );
    statusBar()->insertItem( "", ID_MESSAGE );
    statusBar()->setAlignment( ID_FREQUENCY, AlignRight );
    setMessage( i18n( "Wellcome to krot." ) );
    return;
}



void MainWindow::tool()
{
    KROT_LAUNCH( "Launching MainWindow::tool" );
    toolBar()->setTitle( i18n("krot [ToolBar]") );
    toolBar()->insertCombo( "1", ID_AVERAGING, true,
			    SIGNAL(activated(const char *)), this, SLOT(averaging(const char *)),
			    true,
			    i18n( "Frequency interval [MHz] to average into one point." ),
			    80 );
    toolBar()->insertComboItem( 0, "2", -1 );
    toolBar()->insertComboItem( 0, "5", -1 );
    toolBar()->insertComboItem( 0, "10", -1 );
    toolBar()->insertComboItem( 0, "25", -1 );
    toolBar()->insertComboItem( 0, "50", -1 );
    toolBar()->insertComboItem( 0, "100", -1 );
    toolBar()->setCurrentComboItem( ID_AVERAGING, 0 );
    return;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
