// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "dataset.h"
#include "functor.h"
#include "krot.h"
#include "krotData.h"
#include "simulation.h"
#include "transition.h"

#include <kapp.h>

#include <algorithm>
#include <fstream.h>



void Simulation::fold()
{
    KROT_LAUNCH( "Launching Simulation::fold" );
    KROT_DEBUG2( "Simulation::fold: short name of simulation to fold: %s, consisting of %d lines",
		 shortName().data(), lns.size() );
    int64_t limit;
    if( 0 < lns.size() ) {
	if( ! profilevalid )
	    reProfile();
	switch( profile ) {
	case LINESHAPE_STICK:
	    // this is just to have some baseline before/after the first/last line.
	    limit = 10;
	    break;
	case LINESHAPE_GAUSS:
	case LINESHAPE_LORENTZ:
	case LINESHAPE_VOIGT:
	    limit = profilecenter;
	    break;
	}
 	int64_t start = static_cast< int64_t >( lns.front().frequency() ) - limit;
 	int64_t stop = static_cast< int64_t >( lns.back().frequency() ) + limit;
	int64_t size = stop - start + 1;
	if( stop - start < 2 * limit )
	    throw ShortDataSet( *this );
	double *newdata = new double[ size ];
	fill_n( newdata, size, 0.0 );
	switch( profile ) {
	case LINESHAPE_STICK:
	    for( vector<Transition>::iterator iter = lns.begin(); iter != lns.end(); iter++ )
		newdata[ static_cast< int64_t >( iter->frequency() ) - start ] += iter->intensity() * scaling;
	    break;
	case LINESHAPE_GAUSS:
	case LINESHAPE_LORENTZ:
	case LINESHAPE_VOIGT:
	    for( vector<Transition>::iterator iter=lns.begin(); iter != lns.end(); iter++ )
		for( int64_t i = static_cast< int64_t >( iter->frequency() ) - limit;
		     i<= static_cast< int64_t >( iter->frequency() ) + limit;
		     i++ )
		    newdata[ i - start ] +=
			profiledata[ static_cast< int64_t >( iter->frequency() ) + - i + profilecenter ]
			* iter->intensity() * scaling;
	    break;
	}
	setData( start, stop, newdata );
    } else {
	KROT_DEBUG( "No lines to fold !" );
    }
    return;
}



double Simulation::humlicek_v12( double x, double y )
{
    KROT_LAUNCH( "Launching Simulation::foldVoigt" );
    static const double T_v12[ 6 ] = {
	0.314240376254359,    0.947788391240164,    1.597682635152605,
	2.27950708050106,     3.020637025120890,    3.889724897869782
    };
    static const double alpha_v12[ 6 ] = {
       -1.393236997981977,   -0.231152406188676,    0.155351465642094,
       -6.21836623696556e-3, -9.190829861057113e-5, 6.275259577497896e-7
    };
    static const double beta_v12[ 6 ] = {
	1.011728045548831,   -0.751971469674635,    1.255772699323164e-2,
	1.0022008145159e-2,  -2.42068134815573e-4,  5.008480613664573e-7
    };
    static const double  y0_v12 = 1.50;
    double yp, xp, xm, sum, yp2, xp2, xm2;

    sum = 0;
    yp = y + y0_v12;
    yp2 = yp * yp;
    if( ( y > 0.85 ) || ( fabs( x ) < ( 18.1 * y + 1.65 ) ) ) {
	// Bereich I
	for( int k=0; k<6; k++ ) {
	    xp = x + T_v12[ k ];
	    xm = x - T_v12[ k ];
	    sum += ( ( alpha_v12[ k ] * xm + beta_v12[ k ] * yp ) / ( xm * xm + yp2 )
		     + ( beta_v12[ k ] * yp - alpha_v12[ k ] * xp ) / ( xp * xp + yp2 ) );
	}
    } else {
	// Bereich II
	for( int k=0; k<6; k++ ) {
	    xp = x + T_v12[k];
	    xp2 = xp * xp;
	    xm = x - T_v12[k];
	    xm2 = xm * xm;
	    sum += ( ( ( beta_v12[ k ] * ( xm2 - y0_v12 * yp ) - alpha_v12[ k ] * xm * ( yp + y0_v12 ) )
		       / ( ( xm2 + yp2 ) * ( xm2 + y0_v12 * y0_v12 ) ) )
		     + ( ( beta_v12[ k ] * ( xp2 - y0_v12 * yp ) + alpha_v12[ k ] * xp * (yp + y0_v12))
			 / ( ( xp2 + yp2 ) * ( xp2 + y0_v12 * y0_v12 ) ) ) );
	}
	if( fabs( x ) < 100 )
	    sum = y * sum + exp( -x * x );
	else
	    sum *= y;
    }
    return sum;
}



bool Simulation::loadLinesDat( const QString& name )
{
    KROT_LAUNCH( "Launching Simulation::loadLinesDat" );
    QFile file( name );
    if(! file.open( IO_ReadOnly ) )
	return false;
    QTextStream stream( &file );
    
    vector< QNum > qn;
    vector< double > fr;
    vector< double > in;

    while( ! stream.eof() ) {
	Transition line;
	stream >> line;
	stream.readLine();
	lns.push_back( line );
    }
    KROT_DEBUG_VERBOSE( "Last line discarded!" );
    lns.pop_back();

    // sort Assignments by frequency (using Transition::operator<).
    // we use stable_sort since it has a better worst case complexity then sort itself !
    // Moreover it doesn't reorder equal assignments /There shouldnt actually be any here :-)
    stable_sort( lns.begin(), lns.end() );
    KROT_DEBUG2( " lns in the frequency range %d  - %d read.", lns.front().frequency(), lns.back().frequency() );
    setName( name );
    fold();
    return true;
}



void Simulation::reProfile()
{
    KROT_LAUNCH( "Launching Simulation::reProfile" );
    if( ! profilevalid ) {
	switch( profile ) {
	case LINESHAPE_GAUSS:
	    profilecenter = static_cast< int >( gaussLimit() );
	    break;
	case LINESHAPE_LORENTZ:
	    profilecenter = static_cast< int >( lorentzLimit() );
	    break;
	case LINESHAPE_VOIGT:
	    profilecenter = static_cast< int >( voigtLimit() );
	    break;
	case LINESHAPE_STICK:
	    profilecenter = 0;
	break;
	}
	if( 0 != profiledata )
	    delete [] profiledata;
	profiledata = new double[ profilecenter * 2 + 1 ];
	switch( profile ) {
	case LINESHAPE_GAUSS:
	    for( int i = -profilecenter; i <= +profilecenter; i++ )
		profiledata[ i + profilecenter ] = gauss( i );
	    break;
	case LINESHAPE_LORENTZ:
	    for( int i = -profilecenter; i <= +profilecenter; i++ )
		profiledata[ i + profilecenter ] = lorentz( i );
	    break;
	case LINESHAPE_VOIGT:
	    for( int i = -profilecenter; i <= +profilecenter; i++ )
		profiledata[ i + profilecenter ] = voigt( i );
	    break;
	case LINESHAPE_STICK:
	    profiledata[ 0 ] = 1.0;
	break;
	}
	profilevalid = true;
    }
    return;
}



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
