/*============================================================================*/
/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> NRC_IAR.C <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/
/*============================================================================*/
/*>>>>>>>>>>>>>>>>>>>>> NUMERICAL RECIPES ROUTINES <<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include        <stdio.h>
#include 	<math.h>

/*
MACHINE PRECISION BASED
#define TOL	1.0e-5
*/
#define TOL	1.0e-20
#define TINY	1.0e-20

#define SIGN(a,b)	((b) < 0.0 ? -fabs(a) : fabs(a))
#define SWAP(a,b)	{float temp=(a);(a)=(b);(b)=temp;}
#define MAX(a,b)	(maxarg1 = (a), maxarg2 = (b), (maxarg1) > (maxarg2) ? (maxarg1) : (maxarg2))
#define SQR(a)		(sqrarg = (a), sqrarg * sqrarg)

#define PYTHAG(a,b)	((at = fabs(a)) > (bt = fabs(b)) ? (ct = bt / at, at * sqrt(1.0 + ct * ct)) : \
			(bt ? (ct = at / bt, bt * sqrt(1.0 + ct * ct)) : 0.0))

static double maxarg1, maxarg2 ;
static double sqrarg ;
static double at, bt, ct ;




short svdfit( double *x, double *y, long ndata, double *a, long ma, long *lista, long mfit, double **u, double **v, double *w, double *chisq, double *funcs, double *sig ) ;
short svdcmp( double **a, long m, long n, double *w, double **v ) ;
short svbksb( double **u, double *w, double **v, long m, long n, double *b, double *x, long *lista ) ;
short svdvar( double **v, long mfit, double *w, double **cvm ) ;
short lfit( double *x, double *y, long ndata, double *a, long ma, long *lista, long mfit, double **covar, double *chisq, double *funcs ) ;
short gaussj( double **a, long n, double **b, long m ) ;
short ludcmp( double **a, long n, long *indx, double *d ) ;
short lubksb( double **a, long n, long *indx, double *b ) ;
short tred2( double **a, long n, double *d, double *e ) ;
short tqli( double *d, double *e, long n, double **z ) ;
short LldSort123( long n, long *idx, long *is1, long *is2, double *ds3 ) ;
short Csi6sdrSort( long n, long *idx, char *cs, short *ss, long *is, short *s6s, double *ds ) ;
short LIndexx( long n, long *arrin, long *indx ) ;
short DIndexx( long n, double *arrin, long *indx ) ;

short CAlloc1Dim( char **ptpt, long sz ) ;
void CFree1Dim( char **ptpt ) ;
short SAlloc1Dim( short **ptpt, long sz ) ;
void SFree1Dim( short **ptpt ) ;
short LAlloc1Dim( long **ptpt, long sz ) ;
void LFree1Dim( long **ptpt ) ;
short DAlloc1Dim( double **ptpt, long sz ) ;
void DFree1Dim( double **ptpt ) ;
short DAlloc2Dim( double ***ptptpt, long sz1, long sz2 ) ;
void DFree2Dim( double ***ptptpt, long sz1 ) ;





/*LINEAR LEAST SQUARES FIT TO OBSERVED TRANSITION ENERGIES*/
short svdfit( x, y, ndata, a, ma, lista, mfit, u, v, w, chisq, funcs, sig )
   long ndata, ma, lista[], mfit ;
   double x[], y[], a[], **u, **v, w[], *chisq, *funcs, *sig ;
   {
   long j, i ;
   double ym, wmax, thresh, sum, *b, *afunc ;
   double tmp ;



/*ALLOCATE SPACE TO POINTER*/
   b = NULL ;
   if( DAlloc1Dim( &b, ndata + 1 ) == -1 )
      {
      printf( "\nSVDFIT (B) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }



/*ACCUMULATE COEFFICIENTS OF FITTING MATRIX*/
   for( i = 1; i <= ndata; i++ )
      {

   /*DERIVATIVES REPLACE FUNCTION CALL*/
      afunc = funcs + i * (ma + 1) ;

   /*ASSIGN TO TEMPORARY VALUE -> DO NOT CHANGE ARRAY VALUES*/
      ym = y[ i ] ;

   /*ADJUST OBSERVED DIFFERENCES FOR NON-FIT PARAMETERS*/
      if( mfit < ma )
         for( j = mfit + 1; j <= ma; j++ )
            ym -= a[ lista[ j ] ] * afunc[ lista[ j ] ] ;

      tmp = 1.0 / sig[ i ] ;

   /*LOOP THROUGH ALL FIT PARAMETERS -> DERIVATIVES REPLACE FUNCTION CALL*/
      for( j = 1; j <= mfit; j++ )
         u[ i ][ j ] = afunc[ lista[ j ] ] * tmp ;

      b[ i ] = ym * tmp ;
      }



/*SINGLE VALUE DECOMPOSITION*/
   if( svdcmp( u, ndata, mfit, w, v ) == -1 )
      return( -1 ) ;



/*EDIT THE SINGULAR VALUES, GIVEN TOL FROM THE DEFINE STATEMENT*/
   wmax = 0.0 ;
   for( j = 1; j <= mfit; j++ )
      if( w[ j ] > wmax )
         wmax = w[ j ] ;

   thresh = TOL * wmax ;

   for( j = 1; j <= mfit; j++ )
      if( w[ j ] < thresh )
         w[ j ] = 0.0 ;


   if( svbksb( u, w, v, ndata, mfit, b, a, lista ) == -1 )
      return( -1 ) ;



/*EVALUATE CHI-SQUARE OF THE FIT*/
   *chisq = 0.0 ;
   for( i = 1; i <= ndata; i++ )
      {
   
   /*DERIVATIVES REPLACE FUNCTION CALL*/
      afunc = funcs + i * (ma + 1) ;

      for( sum = 0.0, j = 1; j <= ma; j++ )
         sum += a[ j ] * afunc[ j ] ;
      *chisq += (tmp = (y[ i ] - sum) / sig[ i ], tmp * tmp) ;
      }



/*FREE SPACE*/
   DFree1Dim( &b ) ;


/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*SOLUTION TO LINEARLY INDEPENDENT SET OF EQUATIONS*/
short svdcmp( a, m, n, w, v )
   long m, n ;
   double **a, *w, **v ;
   {
   long flag, i, its, j, jj, k, l, nm ;
   double c, f, h, s, x, y, z ;
   double anorm = 0.0, g = 0.0, scale = 0.0 ;
   double *rv1 ;


   if( m < n )
      {
      printf( "\nSVDCMP -> %ld %ld YOU MUST AUGMENT A WITH EXTRA ZERO ROWS\n", m, n ) ;
      return( -1 ) ;
      }


/*ALLOCATE SPACE TO POINTER*/
   rv1 = NULL ;
   if( DAlloc1Dim( &rv1, n + 1 ) == -1 )
      {
      printf( "\nSVDCMP (RV1) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }


   for( i = 1; i <= n; i++ )
      {

      l = i + 1 ;
      rv1[ i ] = scale * g ;
      g = s = scale = 0.0 ;

      if( i <= m )
         {

         for( k = i; k <= m; k++ )
            scale += fabs( a[ k ][ i ] ) ;

         if( scale )
            {

            for( k = i; k <= m; k++ )
               {
               a[ k ][ i ] /= scale ;
               s += a[ k ][ i ] * a[ k ][ i ] ;
               }

            f = a[ i ][ i ] ;
            g = -SIGN( sqrt( s ), f ) ;
            h = f * g - s ;
            a[ i ][ i ] = f - g ;

            if( i != n )
               {

               for( j = l; j <= n; j++ )
                  {

                  for( s = 0.0, k = i; k <= m; k++ )
                     s += a[ k ][ i ] * a[ k ][ j ] ;

                  f = s / h ;

                  for( k = i; k <= m; k++ )
                     a[ k ][ j ] += f * a[ k ][ i ] ;
                  }
               }

            for( k = i; k <= m; k++ )
               a[ k ][ i ] *= scale ;
            }
         }


      w[ i ] = scale * g ;
      g = s = scale = 0.0 ;


      if( i <= m && i != n )
         {
         for( k = l; k <= n; k++ )
            scale += fabs( a[ i ][ k ] ) ;


         if( scale )
            {

            for( k = l; k <= n; k++ )
               {
               a[ i ][ k ] /= scale ;
               s += a[ i ][ k ] * a[ i ][ k ] ;
               }

            f = a[ i ][ l ] ;
            g = -SIGN( sqrt( s ), f ) ;
            h = f * g - s ;
            a[ i ][ l ] = f - g ;

            for( k = l; k <= n; k++ )
               rv1[ k ] = a[ i ][ k ] / h ;

            if( i != m )
               {

               for( j = l; j <= m; j++ )
                  {

                  for( s = 0.0, k = l; k <= n; k++ )
                     s += a[ j ][ k ] * a[ i ][ k ] ;

                  for( k = l; k <= n; k++ )
                     a[ j ][ k ] += s * rv1[ k ] ;
                  }
               }

            for( k = l; k <= n; k++ )
               a[ i ][ k ] *= scale ;
            }
         }


      anorm = MAX( anorm, (fabs( w[ i ] ) + fabs( rv1[ i ] )) ) ;
      }



   for( i = n; i >= 1; i-- )
      {

      if( i < n )
         {

         if( g )
            {

            for( j = l; j <= n; j++ )
               v[ j ][ i ] = (a[ i ][ j ] / a[ i ][ l ]) / g ;

            for( j = l; j <= n; j++ )
               {

               for( s = 0.0, k = l; k <= n; k++ )
                  s += a[ i ][ k ] * v[ k ][ j ] ;

               for( k = l; k <= n; k++ )
                  v[ k ][ j ] += s * v[ k ][ i ] ;
               }
            }

         for( j = l; j <= n; j++ )
            v[ i ][ j ] = v[ j ][ i ] = 0.0 ;
         }


      v[ i ][ i ] = 1.0 ;
      g = rv1[ i ] ;
      l = i ;
      }



   for( i = n; i >= 1; i-- )
      {
      l = i + 1 ;
      g = w[ i ] ;

      if( i < n )
         for( j = l; j <= n; j++ )
            a[ i ][ j ] = 0.0 ;
      
      if( g )
         {

         g = 1.0 / g ;

         if( i != n )
            {

            for( j = l; j <= n; j++ )
               {
               for( s = 0.0, k = l; k <= m; k++ )
                  s += a[ k ][ i ] * a[ k ][ j ] ;

               f = (s / a[ i ][ i ]) * g ;

               for( k = i; k <= m; k++ )
                  a[ k ][ j ] += f * a[ k ] [ i ] ;
               }
            }

         for( j = i; j <= m; j++ )
            a[ j ][ i ] *= g ;
         }


      else
         {
         for( j = i; j <= m; j++ )
            a[ j ][ i ] = 0.0 ;
         }

      ++a[ i ][ i ] ;
      }


   for( k = n; k >= 1; k-- )
      {

      for( its = 1; its <= 30; its++ )
         {
         flag = 1 ;

         for( l = k; l >= 1; l-- )
            {
            nm = l - 1 ;

            if( fabs( rv1[ l ]) + anorm == anorm )
               {
               flag = 0 ;
               break ;
               }

            if( fabs( w[ nm ] ) + anorm == anorm )
               break ;
            }


         if( flag )
            {

/*
            c = 0.0 ;
*/
            s = 1.0 ;

            for( i = l; i <= k; i++ )
               {

               f = s * rv1[ i ] ;

               if( fabs( f ) + anorm != anorm )
                  {

                  g = w[ i ] ;
                  h = PYTHAG( f, g ) ;
                  w[ i ] = h ;
                  h =  1.0 / h ;
                  c = g * h ;
                  s = (-f * h) ;

                  for( j = 1; j <= m; j++ )
                     {
                     y = a[ j ][ nm ] ;
                     z = a [ j ][ i ] ;
                     a[ j ][ nm ] = y * c + z * s ;
                     a[ j ][ i ] = z * c - y * s ;
                     }
                  }
               }
            }

         z = w[ k ] ;

         if( l == k )
            {

            if( z < 0.0 )
               {
               w[ k ] = -z ;

               for( j = 1; j <= n; j++ )
                  v[ j ][ k ] = (-v[ j ][ k ]) ;
               }
            break ;
            }


         if( its == 30 )
            {
            printf( "SVDCMP -> NO CONVERGENCE IN 30 SVDCMP ITERATIONS\n" ) ;
            return( -1 ) ;
            }


         x = w[ l ] ;
         nm = k - 1 ;
         y = w[ nm ] ;
         g = rv1[ nm ] ;
         h = rv1[ k ] ;

         f = ((y - z) * (y + z) + (g - h) * (g + h)) / (2.0 * h * y) ;
         g = PYTHAG( f, 1.0 ) ;

         f = ((x - z) * (x + z) + h * ((y / (f + SIGN( g, f ))) - h)) / x ;
         c = s = 1.0 ;

         for( j = l; j <= nm; j++ )
            {
            i = j + 1 ;
            g = rv1[ i ] ;
            y = w[ i ] ;
            h = s * g ;
            g = c * g ;
            z = PYTHAG( f, h ) ;
            rv1[ j ] = z ;
            c = f / z ;
            s = h / z ;
            f = x * c + g * s ;
            g = g * c - x * s ;
            h = y * s ;
            y = y * c ;

            for( jj = 1; jj <= n; jj++ )
               {
               x = v[ jj ][ j ] ;
               z = v[ jj ][ i ] ;
               v[ jj ][ j ] = x * c + z * s ;
               v[ jj ][ i ] = z * c - x * s ;
               }

            z = PYTHAG( f, h ) ;

            w[ j ] = z ;

            if( z )
               {
               z = 1.0 / z ;
               c = f * z ;
               s = h * z ;
               }

            f = (c * g) + (s * y) ;
            x = (c * y) - (s * g) ;

            for( jj = 1; jj <= m; jj++ )
               {
               y = a[ jj ][ j ] ;
               z = a[ jj ][ i ] ;
               a[ jj ][ j ] = y * c + z * s ;
               a[ jj ][ i ] = z * c - y * s ;
               }
            }

         rv1[ l ] = 0.0 ;
         rv1[ k ] = f ;
         w[ k ] = x ;
         }
      }


/*FREE SPACE*/
   DFree1Dim( &rv1 ) ;


/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*SOLUTION TO LINEARLY INDEPENDENT SET OF EQUATIONS*/
short svbksb( u, w, v, m, n, b, x, lista )
   double **u, w[], **v, b[], x[] ;
   long m, n, lista[] ;
   {
   long jj, j, i ;
   double s, *tmp ;


/*ALLOCATE SPACE TO POINTER*/
   tmp = NULL ;
   if( DAlloc1Dim( &tmp, n + 1 ) == -1 )
      {
      printf( "\nSVBKSB (TMP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }

   

   for( j = 1; j <= n; j++ )
      {
      s = 0.0 ;

      if( w[ j ] )
         {
         for( i = 1; i <= m; i++ )
            s += u[ i ][ j ] * b[ i ] ;
         s /= w[ j ] ;
         }

      tmp[ j ] = s ;
      }


   for( j = 1; j <= n; j++ )
      {
      s = 0.0 ;

      for( jj = 1; jj <= n; jj++ )
         s += v[ j ][ jj ] * tmp[ jj ] ;

      x[ lista[ j ] ] = s ;
      }



/*FREE SPACE*/
   DFree1Dim( &tmp ) ;

/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*CALCULATE COVARIANCES FOR FIT PARAMETERS*/
short svdvar( v, mfit, w, cvm )
   long mfit ;
   double **v, w[], **cvm ;
   {
   long k, j, i ;
   double sum, *wti ;


/*ALLOCATE SPACE TO POINTER*/
   wti = NULL ;
   if( DAlloc1Dim( &wti, mfit + 1 ) == -1 )
      {
      printf( "\nSVDVAR (WTI) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      return( -1 ) ;
      }


   for( i = 1; i <= mfit; i++ )
      {
      wti[ i ] = 0.0 ;
      if( w[ i ] )
         wti[ i ] = 1.0 / (w[ i ] * w[ i ]) ;
      }


/*SUM CONTRIBUTIONS TO COVARIANCE MATRIX*/
   for( i = 1; i <= mfit; i++ )
      {
      for( j = 1; j <= i; j++ )
         {
         for( sum = 0.0, k = 1; k <= mfit; k++ )
            sum += v[ i ][ k ] * v[ j ][ k ] * wti[ k ] ;

         cvm[ j ][ i ] = cvm[ i ][ j ] = sum ;
         }
      }


/*FREE SPACE*/
   DFree1Dim( &wti ) ;

/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*LINEAR LEAST SQUARES FIT TO OBSERVED TRANSITION ENERGIES*/
short lfit( x, y, ndata, a, ma, lista, mfit, covar, chisq, funcs )
   long ndata, ma, lista[], mfit ;
   double x[], y[], a[], **covar, *chisq, *funcs ;
   {
   long k, kk, j, ihit, i ;
   double ym, wt, sum, **beta ;
   double sig = 1.0, sig2i = 1.0 ;
   double *afunc ;




/*ALLOCATE SPACE TO POINTERS*/
   beta = NULL ;
   if( DAlloc2Dim( &beta, ma + 1, 2 ) == -1 )
      {
      printf( "\nLFIT (BETA) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }




/*CHECK TO SEE THAT LISTA CONTAINS A PROPER PERMUTATION OF THE COEFF AND FILL IN ANY MISSING MEMBERS*/
   kk = mfit + 1 ;

   for( j = 1; j <= ma; j++ )
      {
      ihit = 0 ;

      for( k = 1; k <= mfit; k++ )
         if( lista[ k ] == j )
            ihit++ ;

      if( ihit == 0 )
         lista[ kk++ ] = j ;

      else if( ihit > 1 )
         {
         printf( "\nLFIT -> BAD LISTA PERMUTATION - 1 !\n" ) ;
         return( -1 ) ;
         }
      }


   if( kk != ma + 1 )
      {
      printf( "\nLFIT -> BAD LISTA PERMUTATION - 2 !\n" ) ;
      return( -1 ) ;
      }





/*INIALIZE ARRAY AND MATRIX*/
   for( j = 1; j <= mfit; j++ )
      {
      for( k = 1; k <= mfit; k++ )
         covar[ j ][ k ] = 0.0 ;
      beta[ j ][ 1 ] = 0.0 ;
      }





/*ACCUMULATE COEFFICIENTS OF NORMAL EQUATIONS*/
   for( i = 1; i <= ndata; i++ )
      {
      afunc = funcs + i * (ma + 1) ;

   /*ASSIGN TO TEMPORARY VALUE -> DO NOT CHANGE ARRAY VALUES*/
      ym = y[ i ] ;

   /*ADJUST OBSERVED DIFFERENCES FOR NON-FIT PARAMETERS*/
      if( mfit < ma )
         for( j = mfit + 1; j <= ma; j++ )
            ym -= a[ lista[ j ] ] * afunc[ lista[ j ] ] ;

/*
      sig2i = 1.0 / SQR( sig[ i ] ) ;
*/

   /*LOOP THROUGH OBSERVED LINES*/
      for( j = 1; j <= mfit; j++ )
         {

         wt = afunc[ lista[ j ] ] * sig2i ;

      /*GENERATE COVAR MATRIX*/
         for( k = 1; k <= j; k++ )
            covar[ j ][ k ] += wt * afunc[ lista[ k ] ] ;

         beta[ j ][ 1 ] += ym * wt ;
         }
      }


/*FILL IN ABOVE DIAGONAL FROM SYMMETRY*/
   if( mfit > 1 )
      for( j = 2; j <= mfit; j++ )
         for( k = 1; k <= j - 1; k++ )
            covar[ k ][ j ] = covar[ j ][ k ] ;




/*SOLVE LINEAR SYSTEM OF EQUATIONS*/
   if( gaussj( covar, mfit, beta, 1 ) == -1 )
      return( -1 ) ;





/*PARTITION SOLUTION TO APPROPIATE COEFFICIENTS A*/
   for( j = 1; j <= mfit; j++ )
      a[ lista[ j ] ] = beta[ j ][ 1 ] ;


/*INITIALIZE CHISQ*/
   *chisq = 0.0 ;

/*EVALUATE CHI-SQUARE OF THE FIT*/
   for( i = 1; i <= ndata; i++ )
      {
      afunc = funcs + i * (ma + 1) ;

      for( sum = 0.0, j = 1; j <= ma; j++ )
         sum += a[ j ] * afunc[ j ] ;
      *chisq += SQR( (y[ i ] - sum) / sig ) ;
      }


/*SORT COARIANCE MATRIX TO TRUE ORDER OF FITTING COEFFICIENTS*/
/*
   covsrt( covar, ma, lista, mfit ) ;
*/



/*FREE SPACE*/
   DFree2Dim( &beta, ma + 1 ) ;


/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*GAUSS-JORDAN ELIMINATION -> SOLUTION TO LINEAR INDEPENDENT SET OF EQUATIONS*/
short gaussj( a, n, b, m )
   double **a, **b ;
   long n, m ;
   {
   long *indxc, *indxr, *ipiv ;
   long i, icol, irow, j, k, l, ll ;
   float big, dum, pivinv ;




/*ALLOCATE SPACE*/
   indxc = NULL ;
   if( LAlloc1Dim( &indxc, n + 1 ) == -1 )
      {
      printf( "\nGAUSSJ (INDXC) -> CAN'T ALLOCATE ENOUGH DOUBLE SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }

   indxr = NULL ;
   if( LAlloc1Dim( &indxr, n + 1 ) == -1 )
      {
      printf( "\nGAUSSJ (INDXR) -> CAN'T ALLOCATE ENOUGH DOUBLE SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }

   ipiv = NULL ;
   if( LAlloc1Dim( &ipiv, n + 1 ) == -1 )
      {
      printf( "\nGAUSSJ (IPIV) -> CAN'T ALLOCATE ENOUGH DOUBLE SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }
      



   for( j = 1; j <= n; j++ )
      ipiv[j] = 0 ;

   for( i = 1; i <= n; i++ )
      {
      big = 0.0 ;

      for( j = 1; j <= n; j++ )
         if( ipiv[j] != 1 )
            for( k = 1; k <= n; k++ )
               {

               if( ipiv[k] == 0 )
                  {
                  if( fabs( a[j][k] ) >= big )
                     {
                     big = fabs( a[j][k] ) ;
                     irow = j ;
                     icol = k ;
                     }
                  }

               else if( ipiv[k] > 1 )
                  {
                  printf( "\nGAUSSJ -> SINGULAR MATRIX (1) !\n" ) ;
                  return( -1 ) ;
                  }
               }


      ++(ipiv[ icol ]) ;


      if( irow != icol )
         {
         for( l = 1; l <= n; l++ )
            SWAP( a[irow][l], a[icol][l] )
         for( l = 1; l <= m; l++ )
            SWAP( b[irow][l], b[icol][l] )
         }



      indxr[i] = irow ;
      indxc[i] = icol ;

      if( a[icol][icol] == 0.0 )
         {
         printf( "\nGAUSSJ -> SINGULAR MATRIX (2) !\n" ) ;
         return( -1 ) ;
         }



      pivinv = 1.0 / a[icol][icol] ;
      a[icol][icol] = 1.0 ;

      for( l = 1; l <= n; l++ )
         a[icol][l] *= pivinv ;

      for( l = 1; l <= m; l++ )
         b[icol][l] *= pivinv ;

      for( ll = 1; ll <= n; ll++ )
         if( ll != icol )
            {

            dum = a[ll][icol] ;
            a[ll][icol] = 0.0 ;

            for( l = 1; l <= n; l++ )
               a[ll][l] -= a[icol][l] * dum ;
            for( l = 1; l <= m; l++ )
               b[ll][l] -= b[icol][l] * dum ;
            }
      }




   for( l = n; l >= 1; l-- )
      {
      if( indxr[l] != indxc[l] )
         for( k = 1; k <= n; k++ )
      SWAP( a[k][indxr[l]], a[k][indxc[l]] )
      }




/*FREE SPACE*/
   LFree1Dim( &indxc ) ;
   LFree1Dim( &indxr ) ;
   LFree1Dim( &ipiv ) ;


/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*LU DECOMPOSITION -> SOLUTION TO LINEAR INDEPENDENT SET OF EQUATIONS*/
short ludcmp( a, n, indx, d )
   long n, *indx ;
   double **a, *d ;
   {
   long i, imax, j, k ;
   double big, dum, sum, temp ;
   double *vv ;


/*ALLOCATE SPACE*/
   vv = NULL ;
   if( DAlloc1Dim( &vv, n + 1 ) == -1 )
      {
      printf( "\nLUDCMP (VV) -> CAN'T ALLOCATE ENOUGH DOUBLE SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }



   *d = 1.0 ;

   for( i = 1; i <= n; i++ )
      {
      big = 0.0 ;

      for( j = 1; j <= n; j++ )
         {
         if( (temp = fabs( a[i][j] )) > big )
            big = temp ;
         }


      if( big == 0.0 )
         {
         printf( "\nLUDCMP -> SINGULAR MATRIX !\n" ) ;
         return( -1 ) ;
         }


      vv[i] = 1.0 / big ;
      }



   for( j = 1; j <= n; j++ )
      {

      for( i = 1; i < j; i++ )
         {
         sum = a[i][j] ;

         for( k = 1; k < i; k++ )
            sum -= a[i][k] * a[k][j] ;

         a[i][j] = sum ;
         }

      big = 0.0 ;

      for( i = j; i <= n; i++ )
         {
         sum = a[i][j] ;

         for( k = 1; k < j; k++ )
            sum -= a[i][k] * a[k][j] ;

         a[i][j] = sum ;

         if( (dum = vv[i] * fabs( sum )) >= big )
            {
            big = dum ;
            imax = i ;
            }
         }


      if( j != imax )
         {

         for( k = 1; k <= n; k++ )
            {
            dum = a[imax][k] ;
            a[imax][k] = a[j][k] ;
            a[j][k] = dum ;
            }

         *d = -(*d) ;
         vv[imax] = vv[j] ;
         }

      indx[j] = imax ;

      if( a[j][j] == 0.0 )
         a[j][j] = TINY ;

      if( j != n )
         {
         dum = 1.0 / (a[j][j]) ;
         for( i = j+1; i <= n; i++ )
            a[i][j] *= dum ;
         }
      }



/*FREE SPACE*/
   DFree1Dim( &vv ) ;

   return( 1 ) ;
   }









/*LU DECOMPOSITION -> INVERSE OF MATRIX A COLUMN AT A TIME*/
short lubksb( a, n, indx, b )
   double **a, b[] ;
   long n, *indx ;
   {
   long i, ii = 0, ip, j ;
   double sum ;


   for( i = 1; i <= n; i++ )
      {
      ip = indx[i] ;
      sum = b[ip] ;
      b[ip] = b[i] ;

      if( ii )
         for( j = ii; j <= i - 1; j++ )
            sum -= a[i][j] * b[j] ;

      else if( sum )
         ii = i ;

      b[i] = sum ;
      }



   for( i = n; i >= 1; i-- )
      {
      sum = b[i] ;

      for( j = i + 1; j <= n; j++ )
         sum -= a[i][j] * b[j] ;

      b[i] = sum / a[i][i] ;
      }

   return( 1 ) ;
   }









/*TRIDIAGONALIZATION ROUTINE FOR REAL SYMMETRIC MATRICIES*/
short tred2( a, n, d, e )
   double **a, d[], e[] ;
   long n ;
   {
   long l, k, j, i ;
   double scale, hh, h, g, f ;


   for( i = n; i >= 2; i-- )
      {

      l = i - 1 ;
      h = scale = 0.0 ;

      if( l > 1 )
         {
         for( k = 1; k <= l; k++ )
            scale = scale + fabs( a[i][k] ) ;

         if( scale == 0.0 )
            e[i] = a[i][l] ;

         else
            {
            for( k = 1; k <= l; k++ )
               {
               a[i][k] = a[i][k] / scale ;
               h = h + a[i][k] * a[i][k] ;
               }



            f = a[i][l] ;
            g = f > 0 ? -sqrt( h ): sqrt( h ) ;
            e[i] = scale * g ;
            h = h - f * g ;
            a[i][l] = f - g ;
            f = 0.0 ;



            for( j = 1; j <= l; j++ )
               {
               a[j][i] = a[i][j] / h ;
               g = 0.0 ;

               for( k = 1; k <= j; k++ )
                  g = g + a[j][k] * a[i][k] ;
               for( k = j + 1; k <= l; k++ )
                  g = g + a[k][j] * a[i][k] ;

               e[j] = g / h ;
               f = f + e[j] * a[i][j] ;
               }

            hh = f / (h + h) ;

            for( j = 1; j <= l; j++ )
               {
               f = a[i][j] ;
               e[j] = g = e[j] - hh * f ;

               for( k = 1; k <= j; k++ )
                  a[j][k] = a[j][k] - (f * e[k] + g * a[i][k]) ;
               }
            }
         }

      else
         e[i] = a[i][l] ;

      d[i] = h ;
      }
         



   d[1] = 0.0 ;
   e[1] = 0.0 ;

   for( i = 1; i <= n; i++ )
      {
      l = i - 1 ;


      if( d[i] )
         {

         for( j = 1; j <= l; j++ )
            {
            g = 0.0 ;

            for( k = 1; k <= l; k++ )
               g = g + a[i][k] * a[k][j] ;

            for( k = 1; k <= l; k++ )
               a[k][j] = a[k][j] - g * a[k][i] ;
            }
         }


      d[i] = a[i][i] ;
      a[i][i] = 1.0 ;

      for( j = 1; j <= l; j++ )
         a[j][i] = a[i][j] = 0.0 ;
      }

   return( 1 ) ;
   }









/*DIAGONALIZATION OF A TRIDIAGONAL MATRIX ROUTINE*/
short tqli( d, e, n, z )
   double d[], e[], **z ;
   long n ;
   {
   long m, l, iter, i, k, ii, j ;
   double s, r, p, g, f, dd, c, b ;

   for( i = 2; i <= n; i++ )
      e[i - 1] = e[i] ;

   e[n] = 0.0 ;

   for( l = 1; l <= n; l++ )
      {
      iter = 0 ;

      do
         {
         for( m = l; m <= n - 1; m++ )
            {
            dd = fabs( d[m] ) + fabs( d[m + 1] ) ;

            if( fabs( e[m] ) + dd == dd )
               break ;
            }

         if( m != l )
            {
            if( iter++ == 30 )
               printf( "TOO MANY ITERATIONS IN TQLI\n" ) ;

            g = (d[l + 1] - d[l]) / (2.0 * e[l]) ;
            r = sqrt((g * g) + 1.0) ;
            g = d[m] - d[l] + e[l] / (g + SIGN(r, g)) ;
            s = c = 1.0 ;
            p = 0.0 ;

            for( i = m - 1; i >= l; i-- )
               {
               f = s * e[i] ;
               b = c * e[i] ;

               if( fabs( f ) >= fabs( g ) )
                  {
                  c = g / f ;
                  r = sqrt( (c * c) + 1.0 ) ;
                  e[i + 1] = f * r ;
                  c = c * (s = 1.0 / r) ;
                  }

               else
                  {
                  s = f / g ;
                  r = sqrt( (s * s) + 1.0 ) ;
                  e[i + 1] = g * r ;
                  s = s * (c = 1.0 / r) ;
                  }

               g = d[i + 1] - p ;
               r = (d[i] - g) * s + 2.0 * c * b ;
               p = s * r ;
               d[i + 1] = g + p ;
               g = c * r - b ;

               for( k = 1; k <= n; k++ )
                  {
                  f = z[k][i + 1] ;
                  z[k][i + 1] = s * z[k][i] + c * f ;
                  z[k][i] = c * z[k][i] - s * f ;
                  }
               }

            d[l] = d[l] - p ;
            e[l] = g ;
            e[m] = 0.0 ;
            }
         }

      while( m != l ) ;
      }


/*ORDER EIGENVALUES AND VECTORS*/
   for( ii = 2; ii <= n; ii++ )
      {
      i = ii - 1 ;
      k = i ;
      p = d[i] ;
      for( j = ii; j <= n; j++ )
         {
         if( d[j] < p )
            {
            k = j ;
            p = d[j] ;
            }
         }

      if( k != i )
         {
         d[k] = d[i] ;
         d[i] = p ;
         for( j = 1; j <= n; j++ )
            {
            p = z[j][i] ;
            z[j][i] = z[j][k] ;
            z[j][k] = p ;
            }
         }
      }

   return( 1 ) ;
   }
  








/*SORT ACCORDING INDEX ARRAY IN ASCENDING ORDER -> UNIT OFFSET ARRAYS NEEDED*/
short LldSort123( n, idx, is1, is2, ds3 )
   long n, idx[], is1[], is2[] ;
   double ds3[] ;
   {
   long j, *iwksp ;
   double *dwksp ;



/*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
   iwksp = NULL ;
   if( LAlloc1Dim( &iwksp, n + 1 ) == -1 )
      {
      printf( "\nIIISORT123 (IWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }

/*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
   dwksp = NULL ;
   if( DAlloc1Dim( &dwksp, n + 1 ) == -1 )
      {
      printf( "\nIIISORT123 (DWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
      fflush( stdout ) ;
      return( -1 ) ;
      }




   for( j = 1; j <= n; j++ )
      iwksp[ j ] = is1[ j ] ;

   for( j = 1; j <= n; j++ )
      is1[ j ] = iwksp[ idx[ j ] ] ;


   if( is2 != NULL )
      {
      for( j = 1; j <= n; j++ )
         iwksp[ j ] = is2[ j ] ;

      for( j = 1; j <= n; j++ )
         is2[ j ] = iwksp[ idx[ j ] ] ;
      }


   if( ds3 != NULL )
      {
      for( j = 1; j <= n; j++ )
         dwksp[ j ] = ds3[ j ] ;

      for( j = 1; j <= n; j++ )
         ds3[ j ] = dwksp[ idx[ j ] ] ;
      }



/*FREE SPACE*/
   LFree1Dim( &iwksp ) ;

/*FREE SPACE*/
   DFree1Dim( &dwksp ) ;


/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*SORT ACCORDING INDEX ARRAY IN DECENDING ORDER -> UNIT OFFSET ARRAYS NEEDED*/
short Csi6sfdrSort( n, idx, cs, ss, is, s6s, fs, ds )
   char cs[] ;
   short ss[], s6s[] ;
   long n, idx[], is[] ;
   float fs[] ;
   double ds[] ;
   {
   long j, rj ;
   char *cwksp ;
   short *swksp ;
   long *iwksp ;
   float *fwksp ;
   double *dwksp ;
   



/*SORT CHARACTER ARRAY*/
   if( cs != NULL )
      {

   /*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
      cwksp = NULL ;
      if( CAlloc1Dim( &cwksp, n + 1 ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (CWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= n; j++ )
         cwksp[ j ] = cs[ j ] ;

      for( rj = n, j = 1; j <= n; j++, rj-- )
         cs[ rj ] = cwksp[ idx[ j ] ] ;


   /*FREE SPACE*/
      CFree1Dim( &cwksp ) ;
      }


/*SORT SHORT ARRAY*/
   if( ss != NULL )
      {

   /*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
      swksp = NULL ;
      if( SAlloc1Dim( &swksp, n + 1 ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (SWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= n; j++ )
         swksp[ j ] = ss[ j ] ;

      for( rj = n, j = 1; j <= n; j++, rj-- )
         ss[ rj ] = swksp[ idx[ j ] ] ;


   /*FREE SPACE*/
      SFree1Dim( &swksp ) ;
      }



/*SORT INTEGER ARRAY*/
   if( is != NULL )
      {

   /*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
      iwksp = NULL ;
      if( LAlloc1Dim( &iwksp, n + 1 ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (IWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= n; j++ )
         iwksp[ j ] = is[ j ] ;

      for( rj = n, j = 1; j <= n; j++, rj-- )
         is[ rj ] = iwksp[ idx[ j ] ] ;


   /*FREE SPACE*/
      LFree1Dim( &iwksp ) ;
      }



/*SORT INTEGER ARRAY*/
   if( s6s != NULL )
      {

   /*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
      swksp = NULL ;
      if( SAlloc1Dim( &swksp, 6 * (n + 1) ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (SWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= n; j++ )
         {
         swksp[ j * 6 ] = s6s[ j * 6 ] ;
         swksp[ j * 6 + 1 ] = s6s[ j * 6 + 1 ] ;
         swksp[ j * 6 + 2 ] = s6s[ j * 6 + 2 ] ;
         swksp[ j * 6 + 3 ] = s6s[ j * 6 + 3 ] ;
         swksp[ j * 6 + 4 ] = s6s[ j * 6 + 4 ] ;
         swksp[ j * 6 + 5 ] = s6s[ j * 6 + 5 ] ;
         }


      for( rj = n, j = 1; j <= n; j++, rj-- )
         {
         s6s[ rj * 6 ] = swksp[ idx[ j ] * 6 ] ;
         s6s[ rj * 6 + 1 ] = swksp[ idx[ j ] * 6 + 1 ] ;
         s6s[ rj * 6 + 2 ] = swksp[ idx[ j ] * 6 + 2 ] ;
         s6s[ rj * 6 + 3 ] = swksp[ idx[ j ] * 6 + 3 ] ;
         s6s[ rj * 6 + 4 ] = swksp[ idx[ j ] * 6 + 4 ] ;
         s6s[ rj * 6 + 5 ] = swksp[ idx[ j ] * 6 + 5 ] ;
         }


   /*FREE SPACE*/
      SFree1Dim( &swksp ) ;
      }



/*SORT FLOAT ARRAY*/
   if( fs != NULL )
      {

   /*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
      fwksp = NULL ;
      if( FAlloc1Dim( &fwksp, n + 1 ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (FWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= n; j++ )
         fwksp[ j ] = fs[ j ] ;

      for( rj = n, j = 1; j <= n; j++, rj-- )
         fs[ rj ] = fwksp[ idx[ j ] ] ;


   /*FREE SPACE*/
      FFree1Dim( &fwksp ) ;
      }



/*SORT DOUBLE ARRAY*/
   if( ds != NULL )
      {

   /*ALLOCATE TEMPORARY SPACE FOR SWAPPING*/
      dwksp = NULL ;
      if( DAlloc1Dim( &dwksp, n + 1 ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (DWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= n; j++ )
         dwksp[ j ] = ds[ j ] ;

      for( rj = n, j = 1; j <= n; j++, rj-- )
         ds[ rj ] = dwksp[ idx[ j ] ] ;


   /*FREE SPACE*/
      DFree1Dim( &dwksp ) ;
      }


/*RETURN SUCCESS*/
   return( 1 ) ;
   }









/*INTEGER SORTING ROUTINE*/
short LIndexx( n, arrin, indx )
   long n, *arrin, *indx ;
   {
   long l, j, ir, indxt, i ;
   long q ;


   for( j = 1; j <= n; j++ )
      indx[ j ] = j ; 

   l = (n >> 1) + 1 ;
   ir = n ;

   for( ;; )
      {
      if( l > 1 )
         q = arrin[ (indxt = indx[ --l ]) ] ;
      else
         {
         q = arrin[(indxt = indx[ ir ])] ;
         indx[ ir ] = indx[ 1 ] ;
         if( --ir == 1 )
            {
            indx[ 1 ] = indxt ;
            return( 1 ) ;
            }
         }


      i = l ;
      j = l << 1 ;

      while( j <= ir )
         {
         if( j < ir && arrin[ indx[ j ] ] < arrin[ indx[ j + 1 ] ] )
            j++ ;

         if( q < arrin[ indx[ j ] ] )
            {
            indx[ i ] = indx[ j ] ;
            j += (i = j) ;
            }

         else
            j = ir + 1 ;
         }

      indx[ i ] = indxt ;
      }
   }









/*DOUBLE SORTING ROUTINE*/
short DIndexx( n, arrin, indx )
   long n, indx[] ;
   double arrin[] ;
   {

   long l, j, ir, indxt, i ;
   double q ;


   for( j = 1; j <= n; j++ )
      indx[ j ] = j ; 

   l = (n >> 1) + 1 ;
   ir = n ;

   for( ;; )
      {
      if( l > 1 )
         q = arrin[(indxt = indx[ --l ])] ;
      else
         {
         q = arrin[(indxt = indx[ ir ])] ;
         indx[ ir ] = indx[ 1 ] ;
         if( --ir == 1 )
            {
            indx[ 1 ] = indxt ;
            return( 1 ) ;
            }
         }


      i = l ;
      j = l << 1 ;

      while( j <= ir )
         {
         if( j < ir && arrin[ indx[ j ] ] < arrin[ indx[ j + 1 ] ] )
            j++ ;

         if( q < arrin[ indx[ j ] ] )
            {
            indx[ i ] = indx[ j ] ;
            j += (i = j) ;
            }

         else
            j = ir + 1 ;
         }

      indx[ i ] = indxt ;
      }
   }










/*
/SORT INTEGER ARRAY/
   if( s6s != NULL )
      {

   /ALLOCATE TEMPORARY SPACE FOR SWAPPING/
      swksp = NULL ;
      if( SAlloc1Dim( &swksp, 6 * (n + 1) ) == -1 )
         {
         printf( "\nCSI6SFDRSORT (SWKSP) -> CAN'T ALLOCATE ENOUGH SPACE !\n" ) ;
         fflush( stdout ) ;
         return( -1 ) ;
         }

      for( j = 1; j <= 6 * n; j += 6 )
         {
         swksp[ j ] = s6s[ j ] ;
         swksp[ j + 1 ] = s6s[ j + 1 ] ;
         swksp[ j + 2 ] = s6s[ j + 2 ] ;
         swksp[ j + 3 ] = s6s[ j + 3 ] ;
         swksp[ j + 4 ] = s6s[ j + 4 ] ;
         swksp[ j + 5 ] = s6s[ j + 5 ] ;
         }


      for( rj = 6 * n - 5, jj = 6 * idx[ 1 ] - 5, j = 1; j <= n; j++, jj = 6 * idx[ j ] - 5, rj -= 6 )
         {
         s6s[ rj ] = swksp[ jj ] ;
         s6s[ rj + 1 ] = swksp[ jj + 1 ] ;
         s6s[ rj + 2 ] = swksp[ jj + 2 ] ;
         s6s[ rj + 3 ] = swksp[ jj + 3 ] ;
         s6s[ rj + 4 ] = swksp[ jj + 4 ] ;
         s6s[ rj + 5 ] = swksp[ jj + 5 ] ;
         }


   /FREE SPACE/
      SFree1Dim( &swksp ) ;
      }

*/

