/**************************************************************************
 *
 *  Copyright (C) 1998 Neil Hart (nhart@magma.ca)
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Id: kprintadmin.cpp,v 1.16 1998/05/06 11:36:51 nhart Exp $
 *                                                                   
 **************************************************************************/

#include "kprintadmin.h"
#include "kprinterrs.h"

static bool GUI;

#include <kprocess.h>
#include <qmsgbox.h>
#include <qregexp.h>

#include <iostream.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

static char* RCSinfo = "$Id: kprintadmin.cpp,v 1.16 1998/05/06 11:36:51 nhart Exp $";

// note: these shouldn't be hardcoded, but should be determined by the tool
char SPOOL[] = "/var/spool/lpd/";

KPrinterAdmin::KPrinterAdmin( KPrinterApplication* k, QWidget* parent, const char* name, bool init )
	: KConfigWidget ( parent, name )
{
	kpa = k;

	if (init)
		GUI = FALSE;
	else
		GUI = TRUE;

	if (GUI)
	{
		printerListBox = new QListBox( this, "printerlist" );
		printerListBox->setGeometry( 20, 20, 400, 100 );
		refreshPrinterListBox();

		addLocalButton =
			new QPushButton( klocale->translate("Add Local Printer"), this );
		addLocalButton->setGeometry( 20, 130, 150, 30 );
		connect(addLocalButton,SIGNAL(clicked()),SLOT(addLocalPrinter()));

		addRemoteButton =
			new QPushButton( klocale->translate("Add Remote Printer"), this );
		addRemoteButton->setGeometry( 180, 130, 150, 30 );
		connect(addRemoteButton,SIGNAL(clicked()),SLOT(addRemotePrinter()));

		editButton =
			new QPushButton( klocale->translate("Edit Selected Printer"), this );
		editButton->setGeometry( 20, 170, 150, 30 );
		connect( editButton, SIGNAL(clicked()), SLOT(editPrinter()) );

		removeButton =
			new QPushButton( klocale->translate("Remove Selected Printer"), this );
		removeButton->setGeometry( 180, 170, 150, 30 );
		connect(removeButton,SIGNAL(clicked()),SLOT(removePrinter()));

		setTabOrder( addLocalButton, addRemoteButton );
		setTabOrder( addRemoteButton, editButton );
		setTabOrder( editButton, removeButton );
	}

}

KPrinterAdmin::~KPrinterAdmin()
{
}

void KPrinterAdmin::addLocalPrinter()
{
	if (getuid() != 0)
	{
		QMessageBox::information( this, klocale->translate("Printer Control"),
			klocale->translate("Only the superuser can add printers") );
		return;
	}

	// create a default printer name
	QString prName;
	if (!kpa->defaultPrinterName(&prName))
	{
		QMessageBox::information( this, klocale->translate("Printer Control"),
			klocale->translate("Unable to create default printer name") );
		return;
	}

	// create a default spool directory using the default printer name
	char defaultSpool[128];
	strcpy( defaultSpool, SPOOL );
	strcat( defaultSpool, prName );
	// ensure we don't include alternate names in the spool directory name
	char* bar = strchr( defaultSpool, '|' );
	if ( bar != NULL )
		*bar = '\0';

	KLocalPrinterDialog* kpad =
		new KLocalPrinterDialog ( this, "kpad", prName, defaultSpool, NULL,
			0, NULL, false, kpa );

	if (kpad->exec())
	{
		QString name(kpad->printerName());
		QString model(kpad->printerType());
		QString spooldir(kpad->printerSpool());
		int printerIndex = kpad->printerTypeIndex();
		QString* filter;
		kpa->getPrinterGsFilter(printerIndex, &filter);
		QString device(kpad->printerDevice());
		QString note(kpad->printerNote());
		switch ( kpa->addNewLocalPrinter(name, model, spooldir, *filter,
			device, note) )
		{
			case RETURN_OK :
				refreshPrinterListBox(); break;
			case SPOOLDIR_ALREADY_EXISTS :
				QMessageBox::information( this,
					klocale->translate("Printer Control"),
					klocale->translate("Spool directory already exists") );
					break;
			case CANT_CREATE_SPOOLDIR :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Unable to create spool directory") );
					break;
			case CANT_CHOWN_SPOOLDIR :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Unable to chown spool directory") );
					break;
			case CANT_CREATE_SEQ :
			case CANT_CREATE_ERRS :
			case CANT_CREATE_STATUS :
			case CANT_CREATE_LOCK :
			case CANT_CREATE_FILTER :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Unable to create files in spool directory") );
				break;
			case CANT_WRITE_PRINTCAP :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Unable to write to \"/etc/printcap\"") );
				break;
		}
	}
}


void KPrinterAdmin::addRemotePrinter()
{
	if (getuid() != 0)
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Only the superuser can add printers") );
		return;
	}

	// create a default printer name
	QString prName;
	if (!kpa->defaultPrinterName(&prName))
		return;

	// create a default spool directory using the default printer name
	char defaultSpool[128];
	strcpy( defaultSpool, SPOOL );
	strcat( defaultSpool, prName );
	// ensure we don't include alternate names in the spool directory name
	char* bar = strchr( defaultSpool, '|' );
	if ( bar != NULL )
		*bar = '\0';

	KRemotePrinterDialog* kpad =
		new KRemotePrinterDialog ( this, "kpad", prName, defaultSpool, NULL,
			NULL, NULL, false, kpa);

	if (kpad->exec())
	{
		QString name(kpad->printerName());
		QString spooldir(kpad->printerSpool());
		QString host(kpad->printerHost());
		QString remqueue(kpad->printerQueue());
		QString note(kpad->printerNote());
		switch ( kpa->addNewRemotePrinter(name, spooldir, host, remqueue,
			note) )
		{
			case RETURN_OK :
				refreshPrinterListBox(); break;
			case SPOOLDIR_ALREADY_EXISTS :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Spool directory already exists") );
					break;
			case CANT_CREATE_SPOOLDIR :
				QMessageBox::information( this,
					klocale->translate("Printer Control"),
					klocale->translate("Unable to create spool directory")) ;
					break;
			case CANT_CHOWN_SPOOLDIR :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Unable to chown spool directory") );
					break;
			case CANT_CREATE_SEQ :
			case CANT_CREATE_ERRS :
			case CANT_CREATE_STATUS :
			case CANT_CREATE_LOCK :
			case CANT_CREATE_FILTER :
				QMessageBox::information( this,
					klocale->translate("Printer Control"),
					klocale->translate("Unable to create files in spool directory") );
				break;
			case CANT_WRITE_PRINTCAP :
				QMessageBox::information( this, 
					klocale->translate("Printer Control"),
					klocale->translate("Unable to write to \"/etc/printcap\"") );
				break;
		}
	}
}

bool alsoRemoveDirs( const char* message1, bool* yesOrNo );

void KPrinterAdmin::editPrinter()
{
	if (getuid() != 0)
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Only the superuser can edit printers") );
		return;
	}
	if (printerListBox->currentItem() == -1)
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Please select a printer to edit, then press the Edit button") );
		return;
	}

	// which printer are we editing?
	int printerIndex = printerListBox->currentItem();
	// where is it?
	bool printerIsLocal;
	if ( (!kpa->getPrinterLocation( printerIndex, &printerIsLocal ) ) )
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Can't find that printer in /etc/printcap") );
		return;
	}

	// get details for this printer from printcap
	QString prName(256);
	QString filter(256);
	QString spooldir(256);
	QString device(256);
	QString host(256);
	QString remqueue(256);
	QString note(256);
	int     modelIndex;

	if ( printerIsLocal )
	{
		if ( (!kpa->getPrinterName( printerIndex, &prName )) ||
		     (!kpa->getPrinterFilter( printerIndex, &filter )) ||
		     (!kpa->getPrinterSpooldir( printerIndex, &spooldir )) ||
		     (!kpa->getPrinterDevice( printerIndex, &device )) ||
		     (!kpa->getPrinterNote( printerIndex, &note )) ||
		     (!kpa->getIndexFromFilter( &filter, &modelIndex )) )
		{
			QMessageBox::information( this, 
				klocale->translate("Printer Control"),
				klocale->translate("Can't find details for that printer in /etc/printcap") );
			return;
		}
	}
	else
	{
		if ( (!kpa->getPrinterName( printerIndex, &prName )) ||
		     (!kpa->getPrinterSpooldir( printerIndex, &spooldir )) ||
		     (!kpa->getPrinterHost( printerIndex, &host )) ||
		     (!kpa->getPrinterQueue( printerIndex, &remqueue )) ||
		     (!kpa->getPrinterNote( printerIndex, &note )) )
		{
			QMessageBox::information( this, 
				klocale->translate("Printer Control"),
				klocale->translate("Can't find details for that printer in /etc/printcap") );
			return;
		}
	}

	// invoke necessary dialog to edit details
	if ( printerIsLocal )
	{
		KLocalPrinterDialog* klpd =
			new KLocalPrinterDialog ( this, "kpad", prName, spooldir, device,
				modelIndex, note, true, kpa );
		if (klpd->exec())
		{
			// now modify details in printcap
			if ( prName != klpd->printerName() )
				kpa->changePrinterName( printerIndex, klpd->printerName() );
			if ( modelIndex != klpd->printerTypeIndex() )
			{
				QString* newModel;
				if ( kpa->getPrinterGsType( klpd->printerTypeIndex(),
						&newModel ) )
					kpa->changePrinterModel( printerIndex, newModel->data() );

				QString* newFilter;
				if ( kpa->getPrinterGsFilter( klpd->printerTypeIndex(),
						&newFilter ) )
					kpa->changePrinterFilter( printerIndex, newFilter->data() );
			}
			if ( spooldir != klpd->printerSpool() )
			{
				bool yesOrNo;
				if ( alsoRemoveDirs( 
					klocale->translate("This will create a new spool directory."),
						&yesOrNo ) )
					kpa->changePrinterSpooldir( printerIndex,
						klpd->printerSpool(), yesOrNo );
			}
			if ( device != klpd->printerDevice() )
				kpa->changePrinterDevice( printerIndex, klpd->printerDevice() );
			if ( note != klpd->printerNote() )
				kpa->changePrinterNote( printerIndex, klpd->printerNote() );
		}
	}
	else
	{
		KRemotePrinterDialog* krpd =
			new KRemotePrinterDialog ( this, "kpad", prName, spooldir, host,
				remqueue, note, true, kpa);
		if (krpd->exec())
		{
			// now modify details in printcap
			if ( prName != krpd->printerName() )
				kpa->changePrinterName( printerIndex, krpd->printerName() );
			if ( spooldir != krpd->printerSpool() )
			{
				bool yesOrNo;
				if ( alsoRemoveDirs( 
					klocale->translate("This will create a new spool directory."),
						&yesOrNo ) )
					kpa->changePrinterSpooldir( printerIndex,
						krpd->printerSpool(), yesOrNo );
			}
			if ( host != krpd->printerHost() )
				kpa->changePrinterHost( printerIndex, krpd->printerHost() );
			if ( remqueue != krpd->printerQueue() )
				kpa->changePrinterQueue( printerIndex, krpd->printerQueue() );
			if ( note != krpd->printerNote() )
				kpa->changePrinterNote( printerIndex, krpd->printerNote() );
		}
	}

	// update list box
	refreshPrinterListBox();
}


// return false for cancel, true for yes/no, putting value into second param
bool alsoRemoveDirs( const char* message1, bool* yesOrNo )
{
	char message[256];
	strcpy( message, message1 );
	strcat( message, 
		klocale->translate("\n\nDo you want kcmprinter to also remove the spool directory?\n" ) );
	QMessageBox mb( "Printer Control", message,
	                QMessageBox::Warning,
	                QMessageBox::Yes,
	                QMessageBox::No | QMessageBox::Default,
	                QMessageBox::Cancel | QMessageBox::Escape );
	switch( mb.exec() )
	{
		case QMessageBox::Yes:
			// delete printcap record and spool directory
			*yesOrNo = true;
			return true;
			break;
		case QMessageBox::No:
			// only delete printcap record
			*yesOrNo = false;
			return true;
			break;
		case QMessageBox::Cancel:
			return false;
			break;
		default:
			return false;
			break;
	}
}

void KPrinterAdmin::removePrinter()
{
	if (getuid() != 0)
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Only the superuser can remove printers") );
		return;
	}
	if (printerListBox->currentItem() == -1)
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Please select a printer to remove, then press the Remove button") );
		return;
	}
	QMessageBox mb( "Printer Control",
		klocale->translate("This will remove references to the selected printer\nfrom the /etc/printcap file.\n\nDo you want kcmprinter to also remove the spool directory?\n"),
	                QMessageBox::Warning,
	                QMessageBox::Yes,
	                QMessageBox::No | QMessageBox::Default,
	                QMessageBox::Cancel | QMessageBox::Escape );
	bool removeDirectories = false;
	switch( mb.exec() )
	{
		case QMessageBox::Yes:
			// delete printcap record and spool directory
			removeDirectories = true;
			break;
		case QMessageBox::No:
			// only delete printcap record
			break;
		case QMessageBox::Cancel:
			return;
			break;
	}

	if (!kpa->removePrinter(printerListBox->currentItem(), removeDirectories))
	{
		QMessageBox::information( this, 
			klocale->translate("Printer Control"),
			klocale->translate("Unable to remove printer record\n") );
		return;
	}
	if (!removeDirectories)
		QMessageBox::information( this,
			klocale->translate("Printer Control"),
			klocale->translate("Remember to remove the spool directory\n") );
		
	// update list box
	refreshPrinterListBox();

}

void KPrinterAdmin::loadSettings()
{
}

void KPrinterAdmin::applySettings()
{
}

void KPrinterAdmin::refreshPrinterListBox()
{
	printerListBox->clear();
	
	int printerIndex;
	for (printerIndex = 0; printerIndex < kpa->numberOfPrinters(); printerIndex++)
	{
		QString desc(256);
		kpa->getPrinterDescription(printerIndex, &desc);
		printerListBox->insertItem( desc );
	}
}


#include "kprintadmin.moc"
