//////////////////////////////////////////////////////////////////////////////
// This software is distributed under the terms of the General Public License.
//
// Program : kless
// Author  : Norbert Drees & Ralf Haferkamp
// E-Mail  : norbert@et-inf.fho-emden.de
//           hafer@et-inf.fho-emden.de
//////////////////////////////////////////////////////////////////////////////

#include "main.h"
#include "about.h"
#include "tab.h"
#include "input.h"

extern KApplication *app;
extern QString ConfigEditor;
extern QString ConfigPrinter;
extern QString ConfigWidth;
extern QString	ConfigHeight;
extern QString	ConfigTab;

bool config_changed = false;

void KLessWindow::dropEvent( KDNDDropZone *dropZone ) {

	name = dropZone->getData();
	getFile();
}

void KLessWindow::slotKFMJobDone() {

	delete kfmConnection;
	kfmConnection = NULL;
	readFile( CopyToHomeDir );
}

void KLessWindow::Load() {

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::Load()" << endl;
		cout << "========================" << endl;
	#endif

	QString path;
	if( !name.isNull() ) {
		KURL u( name );
		if( strcmp( u.protocol(), "file" ) == 0 ) {
			path.setStr( u.directory() );
		} else {
			path.setStr( HomeDir );
		}
	} else {
		path.setStr( HomeDir );
	}

	QString new_name;							// run the dialog
	new_name.setStr( KFileDialog::getOpenFileURL() );
	
	if( new_name.isNull() ) {				// dialog canceled
		return;
	}
	
	name.setStr( new_name );				// make a copy of new_name for method access

	KURL u( name );

	#ifdef KLESSDEBUG
		cout << "* u.protocol() : " << u.protocol() << endl;
		cout << "* u.directory() : " << u.directory() << endl;
		cout << "* u.filename() : " << u.filename() << endl;
	#endif

	if( !strcmp( u.filename(), "" ) ) {
		#ifdef KLESSDEBUG
			cout << "* No filename ..." << endl;
		#endif
		return;
	}

	QFileInfo info( u.path() );
	if( info.isDir() ) {
		#ifdef KLESSDEBUG
			cout << "* is directory ..." << endl;
		#endif
		return;
	}

	getFile();
}

void KLessWindow::Reload() {

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::Reload()" << endl;
		cout << "==========================" << endl;
	#endif

	if( name != NULL && strcmp( name, "READ_DATA_VIA_STDIN" ) ) {
		getFile();
		return;
	}
	
	if( !strcmp( name, "READ_DATA_VIA_STDIN" ) ) {
		return;
	}
}

void KLessWindow::SaveAs() {

	if( name == NULL && strcmp( name, "READ_DATA_VIA_STDIN" ) ) {
		return;
	}

	bool selected = hasSelectedText();
	Save( selected );
}

void KLessWindow::Edit() {

	if( ConfigEditor == NULL ) { return; }

	if( name == NULL && strcmp( name, "READ_DATA_VIA_STDIN" ) ) {
		return;
	}

	if( hasSelectedText() == true ) {
		Save( true ); 
	}

	KURL u( name );
	runCommand( ConfigEditor, u.path() );
}

void KLessWindow::Print() {

	if( ConfigPrinter == NULL ) { return; }

	if( name == NULL && strcmp( name, "READ_DATA_VIA_STDIN" ) ) {
		return;
	}

	if( hasSelectedText() == true ) {
		Save( true ); 
	}

	KURL u( name );
	runCommand( ConfigPrinter, u.path() );
}

void KLessWindow::NewWindow() {

	KLessWindow *klw = new KLessWindow ();
	klw->show ();
	windowList.append( klw );
}

void KLessWindow::Close() {

	if ( windowList.count() > 1 ) {
		windowList.remove( this );
		delete this;
	} else {
		Quit();
	}
}

void KLessWindow::Quit() {

 	QString s;

	// Save the window size
	KConfig *conf = app->getConfig();
	conf->setGroup( "Window" );

 	s.sprintf("%d", width() );
	ConfigWidth = conf->writeEntry( "WinWidth", s );
	s.sprintf("%d", height() );
 	ConfigHeight = conf->writeEntry( "WinHeight", s );
 	
	conf->sync();

	// configuration changed ?
	if( config_changed ) {
		QMessageBox yesno(
			"Options",
			klocale->translate( "Would you like to save\nthe changed Options ?" ),
			QMessageBox::NoIcon,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::No,
			0
		);
		yesno.setButtonText( QMessageBox::Yes, klocale->translate( "Yes" ) );
		yesno.setButtonText( QMessageBox::No, klocale->translate( "No" ) );

		if ( yesno.exec() == QMessageBox::Yes ) saveOptions();
	}
	
	app->quit();
	return;
}

void KLessWindow::Search() {

	if( searchdlg_open == true ) {
		return;
	}

	if( name == NULL ) {
		return;
	}

	multiedit->clearFocus();
	searchdlg_open = true;

	searchdlg = new SearchDialog( sense, searchText, data_path, this );
	connect (
		searchdlg,
		SIGNAL( searchFirst( QString, bool ) ),
		this,
		SLOT( SearchFirst( QString, bool ) )
	);
	connect( searchdlg, SIGNAL( searchNext() ), this, SLOT( SearchNext() ) );
	connect( searchdlg, SIGNAL( SearchClosed() ), this, SLOT( SearchClosed() ) );
	connect( multiedit, SIGNAL( Found() ), searchdlg, SLOT( setNext() ) );
	connect( multiedit, SIGNAL( notFound() ), searchdlg, SLOT( unsetNext() ) );
	searchdlg->show();

	resizeEvent( NULL );
}

void KLessWindow::SearchClosed() {

	disconnect( searchdlg, SIGNAL( searchNext() ), this, SLOT( SearchNext() ) );
	disconnect( searchdlg, SIGNAL( SearchClosed() ), this, SLOT( SearchClosed() ) );
	disconnect( multiedit, SIGNAL( Found() ), searchdlg, SLOT( setNext() ) );
	disconnect( multiedit, SIGNAL( notFound() ), searchdlg, SLOT( unsetNext() ) );

	delete searchdlg;
	searchdlg_open = false;
	resizeEvent( NULL );
}

void KLessWindow::SearchFirst( QString txt, bool casesense ) {

	searchText = txt;
	sense = casesense;

	if( searchText == 0 ) {
		return;
	}

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::SearchFirst(QString txt, bool casesense)" << endl;
		cout << "==========================================================" << endl;
		cout << "SearchFirst: " << searchText << endl;
	#endif

	multiedit->searchString( searchText, false, sense );
}


void KLessWindow::SearchNext() {

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::SearchNext()" << endl;
		cout << "==============================" << endl;
		cout << "SearchText: " << searchText << endl;
		cout << "name: " << name << endl;
	#endif

	if( name == NULL ) {
		return;
	}

	if( searchText == NULL ) {
		return;
	}

	multiedit->searchString( searchText, true, sense );
}

void KLessWindow::About() {

	QString s;
	s = "This is KLess " KLESS_VERSION " 

(c) 1997-1998 Norbert Drees & Ralf Haferkamp

E-Mail :  norbert@et-inf.fho-emden.de
E-Mail : hafer@et-inf.fho-emden.de
Website : http://snoopy.et-inf.fho-emden.de/~norbert
RPMs : ftp://ftp.sonic.net/pub/users/tengel/kless

This program is free software; you can
redistribute it and/or modify it under
the terms of the
GNU General Public License.

Special thanks to ...
Pascal Esche for regular expressions.
Troy Engel for Makefile patches and RPMs.
Everyboy who send us a bug report
and/or
ideas for new features.
Everybody who has done translation for KLess.";

	AboutWindow *dlg = new AboutWindow(
		klocale->translate( "About KLess" ),
		s, 16, 50, 300, 70, about_picture
	);
	dlg->exec();
	delete dlg;
}

void KLessWindow::Help() {

	app->invokeHTMLHelp( "", "" );
}

void KLessWindow::Editor() {

	InputWindow *dlg = new InputWindow(
		klocale->translate( "Enter your favorite text editor" ),
		ConfigEditor
	);
	dlg->exec();
	
	QString str = dlg->getText();
	delete dlg;

	if( strcmp( ConfigEditor, str ) && strcmp( str, "" ) ) {
		ConfigEditor = str;
		config_changed = true;
	}

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::Editor()" << endl;
		cout << "==========================" << endl;
		cout << "* ConfigEditor : " << ConfigEditor << endl;
	#endif
}

void KLessWindow::Printer() {

	InputWindow *dlg = new InputWindow(
		klocale->translate( "Enter the print command" ),
		ConfigPrinter
	);
	dlg->exec();
	
	QString str = dlg->getText();
	delete dlg;

	if( strcmp( ConfigPrinter, str ) && strcmp( str, "" ) ) {
	   ConfigPrinter = str;
		config_changed = true;
	}
	
	#ifdef KLESSDEBUG
		cout << "void KLessWindow::Printer()" << endl;
		cout << "===========================" << endl;
		cout << "* ConfigPrinter : " << ConfigPrinter << endl;
	#endif
}

void KLessWindow::Tab() {

	TabWindow *dlg = new TabWindow( tab_width );
	dlg->exec();

	int tab = dlg->getTab();
	delete dlg;

	if( tab != -1 && tab != tab_width ) { 
		tab_width = tab;
		config_changed = true;
		// Reload will check the name und STDIN too, but
		// the requester is troublesome user
		if( name != NULL && strcmp( name, "READ_DATA_VIA_STDIN" ) ) {
			Reload();
		}
	}
}

void KLessWindow::Font() {

	QFont font = multiedit->font();
	QFont old_font = font;

	KFontDialog::getFont( font );

	if( font != old_font ) config_changed = true;

	multiedit->setFont( font );
	
	#ifdef KLESSDEBUG
		cout << "void KLessWindow::Font()" << endl;
		cout << "===========================" << endl;
		cout << "* ConfigFont : " << font.key() << endl;
	#endif
}

void KLessWindow::bgColor() {

	QColor bgcolor = multiedit->backgroundColor();
	QColor old_bgcolor = bgcolor; 

	if( ( KColorDialog::getColor( bgcolor ) ) != ( QDialog::Accepted ) ){
		return;
	}

	if( bgcolor != old_bgcolor ) config_changed = true;

	QColorGroup cgrp = ( (multiedit->palette() ).copy() ).normal();
	QColorGroup newgrp(
		multiedit->foregroundColor(),
		cgrp.background(),
		cgrp.light(),
		cgrp.dark(),
		cgrp.mid(),
		multiedit->foregroundColor(),
		bgcolor
	);

	QPalette newpalette;
	newpalette.setNormal( newgrp );
	newpalette.setDisabled( newgrp );
	newpalette.setActive( newgrp );
	multiedit->setPalette( newpalette );
}

void KLessWindow::fgColor(){

	QColor fgcolor = multiedit->foregroundColor();
	QColor old_fgcolor = fgcolor;

	if( ( KColorDialog::getColor( fgcolor ) ) != ( QDialog::Accepted ) ) {
		return;
	}

	if( fgcolor != old_fgcolor ) config_changed = true;

	QColorGroup cgrp = ( (multiedit->palette() ).copy() ).normal();
	QColorGroup newgrp(
		fgcolor,cgrp.background(),
		cgrp.light(),
		cgrp.dark(),
		cgrp.mid(),
		fgcolor,
		multiedit->backgroundColor()
	);

	QPalette newpalette;
	newpalette.setNormal( newgrp );
	newpalette.setDisabled( newgrp );
	newpalette.setActive( newgrp );
	multiedit->setPalette( newpalette );
}

void KLessWindow::menubarMoved() {

	if( menubar->menuBarPos() == KMenuBar::Top ) {
		menubar->resize( width(), 0 );
	}

	resizeEvent( NULL );
}

void KLessWindow::saveOptions() {
 	QString s;

	KConfig *conf = app->getConfig();
	conf->setGroup( "Options" );
	ConfigEditor = conf->writeEntry( "Editor", ConfigEditor );
	ConfigPrinter = conf->writeEntry( "Printer", ConfigPrinter );
	s.sprintf("%d", tab_width );
	ConfigTab = conf->writeEntry( "Tab", s );
	conf->writeEntry( "ForeGround", multiedit->foregroundColor() );
	conf->writeEntry( "BackGround", multiedit->backgroundColor() );
	conf->writeEntry( "Font", multiedit->font() );

	conf->sync();

	config_changed = false;
}
