/***************************************************************************
                          knd_confmon.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include	"knd_confmon.h"


static	void	WrToLog
	(	FILE		*logfd,
		LogItem		what,
		const	char	*value
	)
{
	int	w	= what ;
	int	len	;

	len	= strlen (value) + 1 ;
	fwrite	(&w,   sizeof (int),   1, logfd) ;
	fwrite	(&len, sizeof (int),   1, logfd) ;
	fwrite	(value,          1,  len, logfd) ;
}

/*  KNDConfMon								*/
/*  KNDConfMon	: Constructor for configuration and control object	*/
/*  view	: KNDView *	: Parent object				*/
/*  capture	: KNDPCap *	: Associated network monitor		*/
/*  graphic	: KNDGraphic *	: Associated graphic display object	*/
/*  packets	: KNDpacket *	: Associated packet display object	*/
/*  multi	: KNDMulti *	: Multiple graph object			*/
/*  slot	: int		: Unique slot number			*/
/*  name	: char *	: Initial monitor name			*/
/*  iflist	: QArray<char*>*: Array of interface names		*/
/*  eplist	: QArray<char*>*: Array of ethernet protocol names	*/
/*  iplist	: QArray<char*>*: Array of internet protocol names	*/
/*  svlist	: QArray<char*>*: Array of service names		*/
/*  config	: KConfig *	: Initial configuration or NULL if none	*/
/*  (returns)	: KNDConfMon	:					*/

KNDConfMon::KNDConfMon
	(	KNDView		*view,
		KNDPCap		*capture,
		KNDGraphic	*graphic,
		KNDPacket	*packets,
		KNDMulti	*multi,
		int		slot,
		char		*name,
		QArray<char *>	*iflist,
		QArray<char *>	*eplist,
		QArray<char *>	*iplist,
		QArray<char *>	*svlist,
		KConfig		*config
	)
	:
	KNDConf	(view, graphic, packets, multi, slot, name),
	capture	(capture),
	c_prog	(this),
	l_if	(this),	s_if	(this),
	l_proto	(this),	s_epro	(this),	s_ipro	(this),
	s_host1	(this),	t_host1	(this),	s_port1	(true, this),
	s_host2	(this),	t_host2	(this),	s_port2	(true, this),
	s_andor	(this),
	t_prog	(this),
	t_logt	(this),	t_logb	(this),	s_log	(this)
{
	QToolTip::add (&t_name, i18n ("Monitor name")) ;
	QToolTip::add (&b_go,   i18n ("Start/stop monitoring network")) ;

	l_if   .setGeometry	  ( 10,  55,  60,  25) ;
	l_if   .setText		  ("Interface") ;
	s_if   .setGeometry	  ( 80,  55, 100,  25) ;
	v_if	= "" ;
	QToolTip::add (&s_if, i18n ("Network interface to be monitored")) ;

	c_prog .setGeometry	  (320, 105,  90,  25) ;
	c_prog .setText		  ("Use program") ;
	c_prog .setChecked	  (false ) ;
	b_prog	= false	;
	connect(&c_prog, SIGNAL(toggled(bool)), this, SLOT(clickProg(bool))) ;
	QToolTip::add (&c_prog, i18n ("Monitor with user-supplied filter program")) ;

	for (unsigned int ifno = 0 ; ifno < iflist->size() ; ifno += 1)
		s_if   .insertItem ((*iflist)[ifno]) ;

	l_proto.setGeometry	  ( 10, 105,  60,  25) ;
	l_proto.setText		  ("Protocols") ;
	s_epro .setGeometry	  ( 80, 105,  80,  25) ;
	s_ipro .setGeometry	  (170, 105,  80,  25) ;
	v_epro	= "" ;
	v_ipro	= "" ;

	for (unsigned int epno = 0 ; epno < eplist->size() ; epno += 1)
		s_epro.insertItem ((*eplist)[epno]) ;
	for (unsigned int ipno = 0 ; ipno < iplist->size() ; ipno += 1)
		s_ipro.insertItem ((*iplist)[ipno]) ;

	QToolTip::add (&s_epro, i18n ("Select network protocol")) ;
	QToolTip::add (&s_ipro, i18n ("Select internet protocol")) ;

	connect	(&s_epro, SIGNAL(activated(int)), this, SLOT(deltaEProto(int))) ;

	s_host1.setGeometry	  ( 10, 135,  60,  25) ;
	s_host1.insertItem	  ("Host", (int)Host ) ;
	s_host1.insertItem	  ("Srce", (int)Srce ) ;
	s_host1.insertItem	  ("Dest", (int)Dest ) ;
	t_host1.setGeometry	  ( 80, 135, 260,  25) ;
	s_port1.setGeometry	  (350, 135,  60,  25) ;
	v_host1	= "" 	;
	m_host1	= Host	;
	QToolTip::add (&s_host1, i18n ("Endpoint is source, destination, or either")) ;
	QToolTip::add (&t_host1, i18n ("Name or address of endpoint")) ;
	QToolTip::add (&s_port1, i18n ("Service name or port number of endpoint")) ;

	s_host2.setGeometry	  ( 10, 165,  60,  25) ;
	s_host2.insertItem	  ("Host", (int)Host ) ;
	s_host2.insertItem	  ("Srce", (int)Srce ) ;
	s_host2.insertItem	  ("Dest", (int)Dest ) ;
	t_host2.setGeometry	  ( 80, 165, 260,  25) ;
	s_port2.setGeometry	  (350, 165,  60,  25) ;
	v_host2	= ""	;
	m_host2	= Host	;
	QToolTip::add (&s_host2, i18n ("Endpoint is source, destination, or either")) ;
	QToolTip::add (&t_host2, i18n ("Name or address of endpoint")) ;
	QToolTip::add (&s_port2, i18n ("Service name or port number of endpoint")) ;

	for (unsigned int svno = 0 ; svno < svlist->size() ; svno += 1)
	{	s_port1.insertItem ((*svlist)[svno]) ;
		s_port2.insertItem ((*svlist)[svno]) ;
	}

	s_port1.setInsertionPolicy (QComboBox::AtTop) ;
	s_port2.setInsertionPolicy (QComboBox::AtTop) ;
	QToolTip::add (&s_andor, i18n ("Logical combination of endpoints")) ;

	s_andor.setGeometry	  (260, 105,  50,  25) ;
	s_andor.insertItem	  ("And", (int)And)    ;
	s_andor.insertItem	  ("Or",  (int)Or )    ;
	m_andor	= And	;

	t_prog .setGeometry	  ( 10, 200, 400,  25) ;
	t_prog .setEnabled	  (false  ) ;
	v_prog	= "" ;
	QToolTip::add (&t_prog, i18n ("Explicitely entered filter program")) ;

#if	0
	g_pkt  .setGeometry	  (510, 120, 140, 115) ;
	g_pkt  .setTitle	  ("Packet monitor") ;

	c_time .setGeometry	  (515, 140,  50,  25) ;
	c_time .setText		  ("Time" ) ;
	c_time .setChecked	  (true   ) ;
	d_time 	= true	;
	QToolTip::add (&c_time, i18n ("Show time in packet display")) ;

	c_proto.setGeometry	  (585, 140,  50,  25) ;
	c_proto.setText		  ("Proto") ;
	c_proto.setChecked	  (true   ) ;
	d_proto	= true	;
	QToolTip::add (&c_proto, i18n ("Show protocol in packet display")) ;

	c_srce .setGeometry	  (515, 170,  50,  25) ;
	c_srce .setText		  ("Srce" ) ;
	c_srce .setChecked	  (true   ) ;
	d_srce	= true	;
	QToolTip::add (&c_srce, i18n ("Show source in packet display")) ;

	c_dest .setGeometry	  (585, 170,  50,  25) ;
	c_dest .setText		  ("Dest" ) ;
	c_dest .setChecked	  (true   ) ;
	d_dest	= true	;
	QToolTip::add (&c_dest, i18n ("Show destination in packet display")) ;

	c_info .setGeometry	  (515, 200,  50,  25) ;
	c_info .setText		  ("Info" ) ;
	c_info .setChecked	  (true   ) ;
	d_info = true	;
	QToolTip::add (&c_info, i18n ("Show other information in packet display")) ;

	c_size .setGeometry	  (585, 200,  50,  25) ;
	c_size .setText		  ("Size" ) ;
	c_size .setChecked	  (true   ) ;
	d_size	= true	;
	QToolTip::add (&c_size, i18n ("Show size in packet display")) ;

	g_gra  .setGeometry	  (655, 120, 130, 115) ;
	g_gra  .setTitle	  ("Graphic display") ;

	l_load1.setGeometry	  (660, 140,  60,  25) ;
	l_load1.setText		  ("Histogram") ;
	s_load1.setGeometry	  (730, 140,  50,  25) ;
	s_load1.setRange	  ( 1,   50) ;
	s_load1.setValue	  ( 1) ;
	v_load1	=  1	;
	QToolTip::add (&s_load1, i18n ("Set update interval for graphic load histogram")) ;

	l_bar1 .setGeometry	  (660, 170,  60,  25) ;
	l_bar1 .setText		  ("Bar 1") ;
	s_bar1 .setGeometry	  (730, 170,  50,  25) ;
	s_bar1 .setRange	  ( 1,  250) ;
	s_bar1 .setValue	  ( 5) ;
	v_bar1	=  5	;
	QToolTip::add (&s_bar1, i18n ("Set period for first graphic load-average display")) ;

	l_bar2 .setGeometry	  (660, 200,  60,  25) ;
	l_bar2 .setText		  ("Bar 2") ;
	s_bar2 .setGeometry	  (730, 200,  50,  25) ;
	s_bar1 .setRange	  ( 1,  250) ;
	s_bar2 .setValue	  (20) ;
	v_bar1	= 20	;
	QToolTip::add (&s_bar2, i18n ("Set period for second graphic load-average display")) ;
#endif

	s_log  .setGeometry	  (515,  25, 120,  25) ;
	t_logt .setGeometry	  (515,  55, 260,  25) ;
	t_logb .setGeometry	  (515,  85, 260,  25) ;

	g_log  .setTitle	  ("Logging") ;
	s_log  .insertItem 	  ("Off",    	    (int)LogNone      ) ;
	s_log  .insertItem 	  ("Record Text",   (int)LogRecordText) ;
	s_log  .insertItem 	  ("Record Binary", (int)LogRecordBin ) ;
	QToolTip::add (&s_log, i18n ("Set logging mode or disable logging")) ;

	t_logt .setEnabled	  (false)     ;
	t_logb .setEnabled	  (false)     ;
	b_log  .setEnabled	  (false)     ;
	t_logt .setText		  (v_logt = v_name + ".kndt") ;
	t_logb .setText		  (v_logb = v_name + ".kndb") ;
	m_log	= LogNone ;
	QToolTip::add (&t_logt, i18n ("Name of text logging file")) ;
	QToolTip::add (&t_logb, i18n ("Name of binary logging file")) ;

	connect	(&s_log, SIGNAL(activated(int)), this, SLOT(deltaLog(int))) ;

	going	= false	;
	frozen	= false ;
	logfd	= NULL	;

	if (config != NULL)
	{
		b_prog	 = config->readBoolEntry ("UseProgramText",	true) ;
		v_if	 = config->readEntry	 ("InterFace",		(*iflist)[0]) ;
		v_epro	 = config->readEntry	 ("EtherProto",		(*eplist)[0]) ;
		v_ipro	 = config->readEntry	 ("InternetProto",	(*iplist)[0]) ;
		v_host1	 = config->readEntry	 ("Host1_Name",		""  ) ;
		v_host2	 = config->readEntry	 ("Host2_Name",		""  ) ;
		v_port1	 = config->readEntry	 ("Host1_Port",		(*svlist)[0]) ;
		v_port2	 = config->readEntry	 ("Host2_Port",		(*svlist)[0]) ;
		v_prog	 = config->readEntry	 ("ProgramText",	""  ) ;
		d_time	 = config->readBoolEntry ("DisplayTime",	true) ;
		d_proto	 = config->readBoolEntry ("DisplayProto",	true) ;
		d_srce   = config->readBoolEntry ("DisplaySrce",	true) ;
		d_dest	 = config->readBoolEntry ("DisplayDest",	true) ;
		d_info	 = config->readBoolEntry ("DisplayInfo",	true) ;
		d_size	 = config->readBoolEntry ("DisplaySize",	true) ;
		d_multi	 = config->readBoolEntry ("ShowParallel",	true) ;
		v_load1	 = config->readNumEntry  ("LoadHistInterval",	1   ) ;
		v_bar1	 = config->readNumEntry  ("LoadBar1Interval",	5   ) ;
		v_bar2	 = config->readNumEntry  ("LoadBar2Interval",	20  ) ;

		m_host1	 = (HostMode)config->readNumEntry ("Host1_Mode",  (int)Host) ;
		m_host2	 = (HostMode)config->readNumEntry ("Host2_Mode",  (int)Host) ;
		m_andor	 = (JoinMode)config->readNumEntry ("Hosts_AndOr", (int)And ) ;

		clickReset () ;
	}
}

/*  KNDConfMon								*/
/*  ~KNDConfMon	: Destructor for configuration object			*/
/*  (returns)	: void		:					*/

KNDConfMon::~KNDConfMon ()
{
	halt () ;
}

/*  KNDConfMon								*/
/*  halt	: Halt monitor if going					*/
/*  (returns)	: void		:					*/

void	KNDConfMon::halt ()
{
	if (going)
	{
		capture->execute   (false)    ;
		multi  ->unRegister(slot )    ;

		b_go    .setText   ("Go")     ;
		b_freeze.setText   ("Freeze") ;
		b_freeze.setEnabled(false)    ;

		if (logfd != NULL)
		{	fclose	(logfd) ;
			logfd	= NULL	  ;
		}

		frozen	= false	;
		going	= false	;
	}
}

/*  KNDConfMon								*/
/*  addHost	: Add a host to the monitor program			*/
/*  prog	: QString &	: Program under construction		*/
/*  mode	: HostMode	: Host mode (host/srce/dest)		*/
/*  host	: QString &	: Host name				*/
/*  port	: QString &	: Port name				*/
/*  (returns)	: void		:					*/

void	KNDConfMon::addHost
	(	QString	 &prog,
		HostMode mode,
		QString	 &host,
		QString	 &port
	)
{
	bool	both	= !host.isEmpty() && !port.isEmpty () ;

	if (both) prog += "( " ;

	if (!host.isEmpty()) 
	{
		switch (mode)
		{	case Host : prog += "host " ; break ;
			case Srce : prog += "src "  ; break ;
			case Dest : prog += "dst "  ; break ;
			default	  :		      break ;
		}

		prog	+= host ;
		prog	+= " "	;
	}

	if (both) prog += "and " ;

	if (!port.isEmpty())
	{
		switch (mode)
		{	case Host :		      break ;
			case Srce : prog += "src "  ; break ;
			case Dest : prog += "dst "  ; break ;
			default	  :		      break ;
		}

		prog	+= "port " ;
		prog	+= port	   ;
		prog	+= " "	   ;
	}

	if (both) prog += ") " ;
}

/*  KNDConfMon								*/
/*  clickSet	: Handle a click of the set button			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickSet ()
{
	QString	program	;
	bool	any	;
	int	options	;

	v_if	= s_if   .currentText() ;
	v_epro	= s_epro .currentText() ;
	v_ipro	= s_ipro .currentText() ;
	v_name	= t_name .text	     () ;
	v_host1	= t_host1.text	     () ;
	v_host2	= t_host2.text	     () ;
	v_port1	= s_port1.currentText() ;
	v_port2	= s_port2.currentText() ;
	v_prog	= t_prog .text	     () ;
	d_time	= c_time .isChecked  () ;
	d_proto	= c_proto.isChecked  () ;
	d_srce	= c_srce .isChecked  () ;
	d_dest	= c_dest .isChecked  () ;
	d_info	= c_info .isChecked  () ;
	d_size	= c_size .isChecked  () ;
	d_multi = c_multi.isChecked  () ;
	v_load1	= s_load1.value	     () ;
	v_bar1 	= s_bar1 .value	     () ;
	v_bar2 	= s_bar2 .value	     () ;
	v_logt	= t_logt .text 	     () ;
	v_logb	= t_logb .text	     () ;

	m_host1	= (HostMode)s_host1.currentItem() ;
	m_host2	= (HostMode)s_host2.currentItem() ;
	m_andor	= (JoinMode)s_andor.currentItem() ;

	view->setName (v_name) ;
	halt	() ;

	program	= ""	;
	any	= false	;

	if (!b_prog)
	{
		if ((strcmp (v_epro, "") != 0) && (strcmp (v_epro, "ip") != 0))
		{	program += v_epro	;
			program += " "		;
			any	 = true		;
		}
		else if (strcmp (v_ipro, "") != 0)
		{	program += "proto \\"	;
			program += v_ipro	;
			program += " "		;
			any	 = true		;
		}

		bool	end1	= !v_host1.isEmpty() || !v_port1.isEmpty() ;
		bool	end2	= !v_host2.isEmpty() || !v_port2.isEmpty() ;

		if	(end1 && end2)
		{
			if (any) program += "and " ;

			program	+= "( " ;
			addHost (program, m_host1, v_host1, v_port1) ;

			switch (m_andor)
			{	case And : program += "and " ; break ;
				case Or	 : program += "or "  ; break ;
				default	 :		       break ;
			}

			addHost	(program, m_host2, v_host2, v_port2) ;
			program	+= ") " ;
		}
		else if (end1)
		{
			if (any) program += "and " ;
			addHost	(program, m_host1, v_host1, v_port1) ;
		}
		else if (end2)
		{
			if (any) program += "and " ;
			addHost	(program, m_host2, v_host2, v_port2) ;
		}

		t_prog.setText (v_prog = program) ;
	}
	else	program	= v_prog ;

	/* Pass the packet display options down to the packet object,	*/
	/* and the histogram and load bar time constants to the graphic	*/
	/* object.							*/
	options	= (d_time  ? PO_TIME  : 0) |
		  (d_proto ? PO_PROTO : 0) |
		  (d_srce  ? PO_SRC   : 0) |
		  (d_dest  ? PO_DST   : 0) |
		  (d_info  ? PO_INFO  : 0) |
		  (d_size  ? PO_SIZE  : 0) ;

	capture->setOptions (options) ;
	packets->setOptions (options) ;
	graphic->setOptions (v_load1, v_bar1, v_bar2) ;

	b_go    .setEnabled (capture->setProg (v_if, program)) ;
}

/*  KNDConfMon								*/
/*  clickReset	: Handle a click of the reset button			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickReset ()
{
	/* Note that this button simply restores the configuration to	*/
	/* the current values, so the running/frozen state is not	*/
	/* affected.							*/
	t_name .setText	(v_name ) ;
	t_host1.setText	(v_host1) ;
	t_host2.setText	(v_host2) ;

	s_host1.setCurrentItem((int)m_host1) ;
	s_host2.setCurrentItem((int)m_host2) ;
	s_andor.setCurrentItem((int)m_andor) ;
	s_port1.setCurrentItem(QCBFindItem(s_port1, v_port1, 0)) ;
	s_port2.setCurrentItem(QCBFindItem(s_port2, v_port2, 0)) ;
	s_if   .setCurrentItem(QCBFindItem(s_if,    v_if,    0)) ;
	s_epro .setCurrentItem(QCBFindItem(s_epro,  v_epro,  0)) ;
	s_ipro .setCurrentItem(QCBFindItem(s_ipro,  v_ipro,  0)) ;

	c_time .setChecked	(d_time ) ;
	c_proto.setChecked	(d_proto) ;
	c_srce .setChecked	(d_srce ) ;
	c_dest .setChecked	(d_dest ) ;
	c_info .setChecked	(d_info ) ;
	c_size .setChecked	(d_size ) ;
	c_multi.setChecked	(d_multi) ;

	s_log  .setCurrentItem	(m_log  ) ;
	t_logt .setText		(v_logt ) ;
	t_logb .setText		(v_logb ) ;

	deltaEProto(s_epro.currentItem()) ;
}

/*  KNDConfMon								*/
/*  clickClear	: Handle a click of the clear button			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickClear ()
{
	t_host1.setText	  	("")	  ;
	t_host2.setText	  	("")	  ;
	c_time .setChecked	(true)	  ;
	c_proto.setChecked	(true)	  ;
	c_srce .setChecked	(true)	  ;
	c_dest .setChecked	(true)	  ;
	c_info .setChecked	(true)	  ;
	c_size .setChecked	(true)	  ;
	c_multi.setChecked	(true)	  ;
	s_log  .setCurrentItem	(LogNone) ;
	t_logt .setEnabled	(false)	  ;
	t_logb .setEnabled	(false)	  ;
	b_log  .setEnabled	(false)	  ;

	s_host1.setCurrentItem	((int)Host) ;
	s_host2.setCurrentItem	((int)Host) ;
	s_andor.setCurrentItem	((int)And ) ;
	s_port1.setCurrentItem	(0)	  ;
	s_port2.setCurrentItem	(0)	  ;
	s_ipro .setCurrentItem  (0)	  ;
	s_epro .setCurrentItem  (0)	  ;
	s_epro .setEnabled	(true)	  ;

	s_load1.setValue	( 1) ;
	s_bar1 .setValue	( 5) ;
	s_bar2 .setValue	(20) ;

	halt	() ;
}

/*  KNDConfMon								*/
/*  clickGo	: Handle a click of the go button			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickGo ()
{
	int	ferr	;

	/* If we are currently running, then all we need to do is to	*/
	/* halt the monitor.						*/
	if (going)
	{
		halt () ;
		return	;
	}

	/* Open the logging file, or not as appropriate, and note the	*/
	/* error code in case this fails.				*/
	switch (s_log.currentItem())
	{
		case LogRecordText :
			logfd	= fopen (v_logt, "w") ;
			ferr	= errno	;
			break	;

		case LogRecordBin  :
			logfd	= fopen (v_logb, "w") ;
			ferr	= errno	;
			break	;

		default	:
			logfd	= NULL	;
			ferr	= 0	;
			break	;
	}

	/* If the log file is not open and the error code is non-zero	*/
	/* then there was an error, which we duly report (but allow	*/
	/* monitoring to start anyway).					*/
	if ((logfd == NULL) && (ferr != 0))
		Error	("Ksnuffle file error", strerror(ferr),
						QMessageBox::Warning) ;


	/* If there is an open logging file then write a header which	*/
	/* contains the inteface name and the filter. Exactly how this	*/
	/* is done depends on whether we are logging textually or as a	*/
	/* binary dump.							*/
	if (logfd != NULL)
	{
		/* If we are running as root but the real UID is not	*/
		/* root then change the logging file ownership to be	*/
		/* that of the invoking user.				*/
		if ((geteuid () == 0) && (getuid () != 0))
			fchown (fileno(logfd), getuid(), getgid()) ;

		if (s_log.currentItem () == (int)LogRecordText)
		{
			fprintf	(logfd, "; Interface %s\n", (const char *)v_if  ) ;
			fprintf	(logfd, "; Filter    %s\n", (const char *)v_prog) ;
			fprintf	(logfd, ";\n") ;
		}
		else
		{	int l = ItemLast ;
			WrToLog	(logfd, ItemMagic,  MAGIC ) ;
			WrToLog	(logfd, ItemProg,   v_prog) ;
			WrToLog	(logfd, ItemIface,  v_if  ) ;
			fwrite	(&l, sizeof(int), 1, logfd) ;
		}
	}

	/* Start the capture object executing. The result is true if	*/
	/* all is goind OK, or false otherwise. Hence, reset the going	*/
	/* flag to the result.						*/
	going	= capture->execute (true,
				    logfd,
				    s_log.currentItem() == (int)LogRecordBin) ;

	/* If the capture object is running then similarly start the	*/
	/* packet and graphic display objects, and suitably alter the	*/
	/* Start/Stop and Freeze/Resume buttons.			*/
	if (going)
	{
		if (d_multi) multi->Register (slot, v_name) ;

		packets->execute    (true) ;
		graphic->execute    (true) ;

		b_go    .setText    ("Stop"  ) ;
		b_freeze.setEnabled (true    ) ;
		b_freeze.setText    ("Freeze") ;
	}
	else	/* If we failed to start capture and there is an open	*/
		/* logging file then close it.				*/
		if (logfd != NULL)
		{	fclose	(logfd) ;
			logfd	= NULL	;
		}
}

/*  KNDConfMon								*/
/*  clickFreeze	: Handle a click of the freeze button			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickFreeze ()
{
	frozen = !frozen ;
	packets->freeze  (frozen) ;
	graphic->freeze  (frozen) ;
	b_freeze.setText (frozen ? "Resume" : "Freeze") ;
}

/*  KNDConfMon								*/
/*  clickProg	: Handle change in program checkbox			*/
/*  on		: bool		: TRUE if checkbox is checked		*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickProg
	(	bool	on
	)
{
	b_prog = on ;
	t_prog .setEnabled ( on) ;
	s_epro .setEnabled (!on) ;
	s_ipro .setEnabled (!on) ;
	t_host1.setEnabled (!on) ;
	t_host2.setEnabled (!on) ;
	s_host1.setEnabled (!on) ;
	s_host2.setEnabled (!on) ;
	s_port1.setEnabled (!on) ;
	s_port2.setEnabled (!on) ;
	s_andor.setEnabled (!on) ;
}

/*  KNDConfMon								*/
/*  clickLog	: Handle log browse button click			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::clickLog ()
{
	char	*filter	= (LogMode)s_log.currentItem() == LogRecordBin ?
			  "*.kndb" :
			  "*.kndt" ;

	QString	name	= KFileDialog::getSaveFileName (NULL, filter, this) ;

	if (!name.isNull() && !name.isEmpty())
		if (s_log.currentItem() == (int)LogRecordText)
			t_logt.setText (name) ;
		else	t_logb.setText (name) ;
}


/*  KNDConfMon								*/
/*  deltaLog	: Handle change of log setting				*/
/*  index	: int		: Selected index			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::deltaLog
	(	int	index
	)
{
	t_logt.setEnabled (index == (int)LogRecordText) ;
	t_logb.setEnabled (index == (int)LogRecordBin ) ;
	b_log .setEnabled (index != 0) ;
}

/*  KNDConfMon								*/
/*  deltaEProto	: Handle change of ethernet protocol setting		*/
/*  index	: int		: Selected index			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::deltaEProto
	(	int	index
	)
{
	QString	eproto	= s_epro.currentText () ;
	s_ipro.setEnabled ( (strcmp (eproto, ""  ) == 0) ||
			    (strcmp (eproto, "ip") == 0) ) ;
}

/*  KNDConfMon								*/
/*  monitorStop	: Stop monitor execution				*/
/*  (returns)	: void		:					*/

void	KNDConfMon::monitorStop ()
{
	if ( going) clickGo () ;
}

/*  KNDConfMon								*/
/*  monitorStart: Start monitor execution				*/
/*  (returns)	: void		:					*/

void	KNDConfMon::monitorStart ()
{
	if (!going)
	{	clickSet () ;
		clickGo  () ;
	}
}

/*  KNDConfMon								*/
/*  monitorFreeze: Freeze monitor execution				*/
/*  (returns)	 : void		:					*/

void	KNDConfMon::monitorFreeze ()
{
	if (!frozen) clickFreeze () ;
}

/*  KNDConfMon								*/
/*  monitorResume: Resumemonitor execution				*/
/*  (returns)	 : void		:					*/

void	KNDConfMon::monitorResume ()
{
	if ( frozen) clickFreeze () ;
}

/*  KNDConfMon								*/
/*  writeConfig	: Write monitor configuration				*/
/*  config	: KConfig *	: Configuration object			*/
/*  (returns)	: void		:					*/

void	KNDConfMon::writeConfig
	(	KConfig	*config
	)
{
	config->writeEntry ("MonitorName",	v_name	) ;
	config->writeEntry ("ProgramText",	v_prog  ) ;
	config->writeEntry ("InterFace",	v_if	) ;
	config->writeEntry ("EtherProto",	v_epro	) ;
	config->writeEntry ("InternetProto",	v_ipro	) ;
	config->writeEntry ("Host1_Name",	v_host1 ) ;
	config->writeEntry ("Host2_Name",	v_host2	) ;
	config->writeEntry ("Host1_Port",	v_port1 ) ;
	config->writeEntry ("Host2_Port",  	v_port1 ) ;

	config->writeEntry ("Host1_Mode",	(int)m_host1) ;
	config->writeEntry ("Host2_Mode", 	(int)m_host2) ;
	config->writeEntry ("Hosts_AndOr",  	(int)m_andor) ;

	config->writeEntry ("UseProgramText",	b_prog	) ;
	config->writeEntry ("DisplayTime",	d_time	) ;
	config->writeEntry ("DisplayProto",	d_proto	) ;
	config->writeEntry ("DisplaySrce",	d_srce	) ;
	config->writeEntry ("DisplayDest",	d_dest	) ;
	config->writeEntry ("DisplayInfo",	d_info	) ;
	config->writeEntry ("DisplaySize",	d_size	) ;
	config->writeEntry ("ShowParallel",	d_multi	) ;
	config->writeEntry ("LoadHistInterval",	v_load1	) ;
	config->writeEntry ("LoadBar1Interval",	v_bar1	) ;
	config->writeEntry ("LoadBar2Interval",	v_bar2	) ;
}
