#include <qtimer.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qsize.h>
#include <qapplication.h>
#include <qpainter.h>
#include <qimage.h>
#include <qpoint.h>
#include <qpopupmenu.h>
#include <qfont.h>

#include <kurl.h>
#include <klocale.h>

#include "slideshow.h"

SlideShow::SlideShow(const KFileItemList& itemList,
                     int timeDelay, bool transitionEffect,
                     int fileNamePos,
                     WFlags fl):
    QWidget(0,0,fl),
    mItemList(itemList)
{
    mTimeDelay = timeDelay;
    mTransitionEffect = transitionEffect;
    mFileNamePosition = fileNamePos;

    mTimer = new QTimer;
    mTimerSlide = new QTimer;

    mIntArray = 0;
    mEffectRunning = false;
    mPaused = false;
    mPopMenu = 0;
    mCompleted = false;
    mShowingLastImage = false;

    connect(mTimer, SIGNAL(timeout()),
            this, SLOT(slot_timeout()));
    connect(mTimerSlide, SIGNAL(timeout()),
            this, SLOT(slot_timeoutSlide()));

    move(0, 0);
    mDesktopSize = QSize(QApplication::desktop()->size());
    mScaledSize = mDesktopSize;
    resize(mScaledSize);


    mCurrentPixmap.resize(mScaledSize);
    mCurrentPixmap.fill(Qt::black);
    setPaletteBackgroundPixmap(mCurrentPixmap);
    resize(mScaledSize);

    mNextPixmap.resize(mScaledSize);

    mCurrentItem = mItemList.first();
    if (mCurrentItem) {
        mItemList.remove(mCurrentItem);
        loadNextImage(mCurrentItem);
    }

    slot_timeout();
}

SlideShow::~SlideShow()
{

    mTimerSlide->stop();
    mTimer->stop();

    delete mTimer;
    delete mTimerSlide;

}

void SlideShow::loadNextImage(const KFileItem* fileItem)
{
    if (!fileItem) return;

    QImage image(fileItem->url().path());
    int w, h, x, y;
    if (image.width() >= image.height()) {
        w = mScaledSize.width();
        h = int(double(w)/double(image.width())
                * double(image.height()));
        x = 0;
        y = int(double(mScaledSize.height())/2.0 - double(h)/2.0);
    }
    else {
        h = mScaledSize.height();
        w = int(double(h)/double(image.height())
                * double(image.width()));
        x = int(double(mScaledSize.width())/2.0 - double(w)/2.0);
        y = 0;
    }
    image = image.scale(w,h,QImage::ScaleMin);

    mNextPixmap.fill(Qt::black);

    QPainter p;
    p.begin(&mNextPixmap);
    p.drawImage(x, y, image);

    p.setPen(Qt::yellow);
    p.setFont(QFont("Verdana",12,QFont::Bold));

    switch (mFileNamePosition) {
    case AtTop: {
        x = 5;
        y = 10;
        p.drawText(x,y,fileItem->url().filename());
        break;
    }
    case AtBottom: {
        x = 5;
        y = mDesktopSize.height()-5;
        p.drawText(x,y,fileItem->url().filename());
        break;
    }
    default:
        break;
    }


    p.end();

}

void SlideShow::loadLastPixmap()
{
    // SlideShow completed. Load one last pixmap
    mNextPixmap.fill(Qt::black);
    QPainter p;
    p.begin(&mNextPixmap);
    p.setPen(Qt::white);
    p.setFont(QFont("Verdana",20,QFont::Bold));
    int x = 100;
    int y = 100;
    p.drawText(x, y, i18n("SlideShow Completed. Click to Exit"));
    p.end();
}

bool SlideShow::runEffect(bool next) {


    int w = width();
    int h = height();
    int dx = 4;
    int dy = 16;
    int ix = w / dx;

    if (next) {

        if (mIntArray)
            delete [] mIntArray;
        mIntArray = 0;

        mIntArray = new int[ix];
        for (int i=ix-1; i>=0; i--)
            mIntArray[i] = 0;

    }

    int x,y;

    bool complete = true;
    mEffectRunning = true;

    for (int i=0,x=0; i<ix; i++,x+=dx)
    {
        y = mIntArray[i];
        if (y >= h) continue;
        complete = false;
        bitBlt(this, x, y+dy, this, x, y, dx, h-y-dy, Qt::CopyROP, true);
        bitBlt(this, x, y, &mCurrentPixmap, x, y, dx, dy, Qt::CopyROP, true);
        mIntArray[i] += dy;
    }


    if (complete) {
        delete [] mIntArray;
        mIntArray = 0;
    }

    return complete;
}

void SlideShow::slot_timeoutSlide()
{
    bool complete;


    if (mEffectRunning)
        complete = runEffect(false);
    else
        complete = runEffect(true);

    if (complete) {
        mTimerSlide->stop();
        mEffectRunning = false;
    }
    mPrevPixmap = mCurrentPixmap;

}

void SlideShow::slot_timeout()
{

    if (!mTransitionEffect) {

        mCurrentPixmap = mNextPixmap;

        if (!mCurrentPixmap.isNull())
            setPaletteBackgroundPixmap(mCurrentPixmap);

        mCurrentItem = mItemList.first();
        if (mCurrentItem) {
            mItemList.remove(mCurrentItem);
            loadNextImage(mCurrentItem);
        }
        else {
            loadLastPixmap();
        }

    }
    else {

        if (mEffectRunning) {
            mTimer->start(mTimeDelay,true);
            return;
        }


        mCurrentPixmap = mNextPixmap;

        mCurrentItem = mItemList.first();
        if (mCurrentItem) {
            mItemList.remove(mCurrentItem);
            loadNextImage(mCurrentItem);
        }
        else {
            loadLastPixmap();
        }

        if (!mCurrentPixmap.isNull()) {
            if (!mEffectRunning) {
                mTimerSlide->start(10,false);
            }
        }


    }

    if (mCurrentItem)
        mTimer->start(mTimeDelay,true);
    else if (!mCompleted) {
        mCompleted = true;
        mTimer->start(mTimeDelay,true);
    }
    else {
        mShowingLastImage = true;
    }
}

void SlideShow::mouseReleaseEvent(QMouseEvent *event)
{
    if (mShowingLastImage) {
        close();
        return;
    }

    if (mPopMenu) {
        event->ignore();
        return;
    }

    QPoint pos(event->pos());
    mPopMenu = new QPopupMenu();
    if (mPaused)
        mPopMenu->insertItem(i18n("Resume"), 10);
    else
        mPopMenu->insertItem(i18n("Pause"), 11);
    mPopMenu->insertItem(i18n("Exit"), 12);

    pause();

    bool shouldResume = true;

    int id = mPopMenu->exec(pos);
    switch(id) {

    case 10:
        resume();
        break;

    case 11:
        pause();
        shouldResume = false;
        break;

    default:
        break;
    }


    delete mPopMenu;
    mPopMenu = 0;

    if (id == 12) {
        close();
    }

    if (shouldResume)
        resume();

}

void SlideShow::pause()
{
    mPaused = true;
    mTimer->stop();
    bitBlt(this, 0, 0, &mPrevPixmap, 0, 0,
           width(), height(), Qt::CopyROP);
}

void SlideShow::resume()
{
    mPaused = false;
    bitBlt(this, 0, 0, &mPrevPixmap, 0, 0,
           width(), height(), Qt::CopyROP);
    slot_timeout();
}

