/*
 * SPDX-FileCopyrightText: 2017-2019 Red Hat Inc
 * SPDX-FileCopyrightText: 2020 Harald Sitter <sitter@kde.org>
 *
 * SPDX-License-Identifier: LGPL-2.0-or-later
 *
 * SPDX-FileCopyrightText:  2017-2019 Jan Grulich <jgrulich@redhat.com>
 */

#include "appchooserdialog.h"

#include <QQmlContext>
#include <QQmlEngine>
#include <QQuickItem>
#include <QQuickWidget>

#include <QDir>
#include <QSettings>
#include <QStandardPaths>

#include <KApplicationTrader>
#include <KLocalizedString>
#include <KProcess>
#include <KService>

AppChooserDialog::AppChooserDialog(const QStringList &choices, const QString &defaultApp, const QString &fileName, const QString &mimeName, QObject *parent)
    : QuickDialog(parent)
    , m_model(new AppModel(this))
    , m_mimeName(mimeName)
{
    m_model->setPreferredApps(choices);

    QVariantMap props = {
        {"title", i18n("Select an application to open '%1'", fileName)},
    };
    AppFilterModel *filterModel = new AppFilterModel(this);
    filterModel->setSourceModel(m_model);

    auto *data = new AppChooserData(this);
    data->setFileName(fileName);
    data->setDefaultApp(defaultApp);

    qmlRegisterSingletonInstance<AppFilterModel>("org.kde.xdgdesktopportal", 1, 0, "AppModel", filterModel);
    qmlRegisterSingletonInstance<AppChooserData>("org.kde.xdgdesktopportal", 1, 0, "AppChooserData", data);

    create(QStringLiteral("qrc:/AppChooserDialog.qml"), props);

    connect(data, &AppChooserData::openDiscover, this, &AppChooserDialog::onOpenDiscover);
    connect(data, &AppChooserData::applicationSelected, this, &AppChooserDialog::onApplicationSelected);
}

QString AppChooserDialog::selectedApplication() const
{
    return m_selectedApplication;
}

void AppChooserDialog::onApplicationSelected(const QString &desktopFile)
{
    m_selectedApplication = desktopFile;
    accept();
}

void AppChooserDialog::onOpenDiscover()
{
    QStringList args;
    if (!m_mimeName.isEmpty()) {
        args << QStringLiteral("--mime") << m_mimeName;
    }
    KProcess::startDetached(QStringLiteral("plasma-discover"), args);
}

void AppChooserDialog::updateChoices(const QStringList &choices)
{
    m_model->setPreferredApps(choices);
}

ApplicationItem::ApplicationItem(const QString &name, const QString &icon, const QString &desktopFileName)
    : m_applicationName(name)
    , m_applicationIcon(icon)
    , m_applicationDesktopFile(desktopFileName)
    , m_applicationCategory(AllApplications)
{
}

QString ApplicationItem::applicationName() const
{
    return m_applicationName;
}

QString ApplicationItem::applicationIcon() const
{
    return m_applicationIcon;
}

QString ApplicationItem::applicationDesktopFile() const
{
    return m_applicationDesktopFile;
}

void ApplicationItem::setApplicationCategory(ApplicationItem::ApplicationCategory category)
{
    m_applicationCategory = category;
}

ApplicationItem::ApplicationCategory ApplicationItem::applicationCategory() const
{
    return m_applicationCategory;
}

bool ApplicationItem::operator==(const ApplicationItem &item) const
{
    return item.applicationDesktopFile() == applicationDesktopFile();
}

AppFilterModel::AppFilterModel(QObject *parent)
    : QSortFilterProxyModel(parent)
{
    setDynamicSortFilter(true);
    setFilterCaseSensitivity(Qt::CaseInsensitive);
    sort(0, Qt::DescendingOrder);
}

AppFilterModel::~AppFilterModel()
{
}

void AppFilterModel::setShowOnlyPrefferedApps(bool show)
{
    m_showOnlyPreferredApps = show;

    invalidate();
}

bool AppFilterModel::showOnlyPreferredApps() const
{
    return m_showOnlyPreferredApps;
}

void AppFilterModel::setFilter(const QString &text)
{
    m_filter = text;

    invalidate();
}

QString AppFilterModel::filter() const
{
    return m_filter;
}

bool AppFilterModel::filterAcceptsRow(int source_row, const QModelIndex &source_parent) const
{
    const QModelIndex index = sourceModel()->index(source_row, 0, source_parent);

    ApplicationItem::ApplicationCategory category =
        static_cast<ApplicationItem::ApplicationCategory>(sourceModel()->data(index, AppModel::ApplicationCategoryRole).toInt());
    QString appName = sourceModel()->data(index, AppModel::ApplicationNameRole).toString();

    if (m_filter.isEmpty()) {
        return m_showOnlyPreferredApps ? category == ApplicationItem::PreferredApplication : true;
    }

    return m_showOnlyPreferredApps ? category == ApplicationItem::PreferredApplication && appName.toLower().contains(m_filter)
                                   : appName.toLower().contains(m_filter);
}

bool AppFilterModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
    ApplicationItem::ApplicationCategory leftCategory =
        static_cast<ApplicationItem::ApplicationCategory>(sourceModel()->data(left, AppModel::ApplicationCategoryRole).toInt());
    ApplicationItem::ApplicationCategory rightCategory =
        static_cast<ApplicationItem::ApplicationCategory>(sourceModel()->data(right, AppModel::ApplicationCategoryRole).toInt());
    QString leftName = sourceModel()->data(left, AppModel::ApplicationNameRole).toString();
    QString rightName = sourceModel()->data(right, AppModel::ApplicationNameRole).toString();

    if (leftCategory < rightCategory) {
        return false;
    } else if (leftCategory > rightCategory) {
        return true;
    }

    return QString::localeAwareCompare(leftName, rightName) > 0;
}

QString AppChooserData::defaultApp() const
{
    return m_defaultApp;
}

void AppChooserData::setDefaultApp(const QString &defaultApp)
{
    m_defaultApp = defaultApp;
    Q_EMIT defaultAppChanged();
}

AppChooserData::AppChooserData(QObject *parent)
    : QObject(parent)
{
}

QString AppChooserData::fileName() const
{
    return m_fileName;
}

void AppChooserData::setFileName(const QString &fileName)
{
    m_fileName = fileName;
    Q_EMIT fileNameChanged();
}

AppModel::AppModel(QObject *parent)
    : QAbstractListModel(parent)
{
    loadApplications();
}

AppModel::~AppModel()
{
}

void AppModel::setPreferredApps(const QStringList &list)
{
    for (ApplicationItem &item : m_list) {
        bool changed = false;

        // First reset to initial type
        if (item.applicationCategory() != ApplicationItem::AllApplications) {
            item.setApplicationCategory(ApplicationItem::AllApplications);
            changed = true;
        }

        if (list.contains(item.applicationDesktopFile())) {
            item.setApplicationCategory(ApplicationItem::PreferredApplication);
            changed = true;
        }

        if (changed) {
            const int row = m_list.indexOf(item);
            if (row >= 0) {
                QModelIndex index = createIndex(row, 0, AppModel::ApplicationCategoryRole);
                Q_EMIT dataChanged(index, index);
            }
        }
    }
}

QVariant AppModel::data(const QModelIndex &index, int role) const
{
    const int row = index.row();

    if (row >= 0 && row < m_list.count()) {
        ApplicationItem item = m_list.at(row);

        switch (role) {
        case ApplicationNameRole:
            return item.applicationName();
        case ApplicationIconRole:
            return item.applicationIcon();
        case ApplicationDesktopFileRole:
            return item.applicationDesktopFile();
        case ApplicationCategoryRole:
            return static_cast<int>(item.applicationCategory());
        default:
            break;
        }
    }

    return QVariant();
}

int AppModel::rowCount(const QModelIndex &parent) const
{
    return parent.isValid() ? 0 : m_list.count();
}

QHash<int, QByteArray> AppModel::roleNames() const
{
    return {
        {ApplicationNameRole, QByteArrayLiteral("applicationName")},
        {ApplicationIconRole, QByteArrayLiteral("applicationIcon")},
        {ApplicationDesktopFileRole, QByteArrayLiteral("applicationDesktopFile")},
        {ApplicationCategoryRole, QByteArrayLiteral("applicationCategory")},
    };
}

void AppModel::loadApplications()
{
    const KService::List appServices = KApplicationTrader::query([](const KService::Ptr &service) -> bool {
        return service->isValid() && !service->noDisplay() /* includes platform and desktop considerations */;
    });
    for (const KService::Ptr &service : appServices) {
        const QString fullName = service->property(QStringLiteral("X-GNOME-FullName"), QVariant::String).toString();
        const QString name = fullName.isEmpty() ? service->name() : fullName;
        ApplicationItem appItem(name, service->icon(), service->desktopEntryName());
        if (!m_list.contains(appItem)) {
            m_list.append(appItem);
        }
    }
}
