/*
  This file is part of the Groupware/KOrganizer integration.

  Requires the Qt and KDE widget libraries, available at no cost at
  http://www.trolltech.com and http://www.kde.org respectively

  Copyright (c) 2002 Klarlvdalens Datakonsult AB

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef KOGROUPWARE_H
#define KOGROUPWARE_H

#include <libkcal/icalformat.h>
#include <libkcal/freebusy.h>
#include <qstring.h>
#include <qobject.h>
#include <qdatetime.h>

#include <kio/job.h>

#include "notes/konotesview.h"

using namespace KCal;

class KCal::Calendar;
class KCal::Event;
class CalendarView;
class KOGroupware;
class KOEventEditor;
class KOTodoEditor;


/**
 * Class for downloading FreeBusy Lists
 */
class FBDownloadJob : public QObject {
  Q_OBJECT
public:
  FBDownloadJob( const QString& email, const KURL& url, KOGroupware* kogroupware, const char* name = 0 );

  virtual ~FBDownloadJob();
protected slots:
  void slotResult( KIO::Job* );
  void slotData(  KIO::Job*, const QByteArray &data );
signals:
  void fbDownloaded( const QString&, FreeBusy*);

private:
  KOGroupware* mKogroupware;
  QString  mEmail;

  QCString mFBData;
};

class KOGroupware : public QObject
{
  Q_OBJECT
  public:
    static KOGroupware* create( QObject* kmailTarget, CalendarView*,
                                KCal::Calendar* );
    static KOGroupware* instance();

    /** Send iCal messages after asking the user
	 Returns false if the user cancels the dialog, and true if the
	 user presses Yes og or No.
    */
    bool sendICalMessage( QWidget* parent, KCal::Scheduler::Method method,
			  Incidence* incidence, bool isDeleting = false );

    // THIS IS THE ACTUAL KM/KO API
    enum EventState { Accepted, ConditionallyAccepted, Declined, Request };

    // Event initiated by somebody else, coming into KO from KM, returning resulting state
    EventState incomingEventRequest( EventState, const QCString& receiver,
				     const QString& vCalIn,
                                     bool& vCalInOK, QString& vCalOut,
                                     bool& vCalOutOK );
    void incomingResourceRequest( const QValueList<QPair<QDateTime, QDateTime> >& busy,
                                  const QCString& resource,
                                  const QString& vCalIn,
                                  bool& vCalInOK,
                                  QString& vCalOut,
                                  bool& vCalOutOK,
                                  bool& isFree,
                                  QDateTime& start, QDateTime& end );

    // ANswer to invitation
    bool incidenceAnswer( const QCString& sender, const QString& vCalIn, QString& vCalOut );

    // KMail loads all notes into the calendar.
    void refreshNotes( const QStringList& );

    // KOrganizer publishes the free/busy list
    void publishFreeBusy();

    // Get the free/busy list as a string
    QString getFreeBusyString();

    /** KOrganizer downloads somebody else's free/busy list
	The call is asynchronous, and upon download, the
	receivers slot specified by member will be called.
	The slot should be of type "member(const QString&, KCal::FreeBusy*)"

	Return true if a download is initiated, and false otherwise
    */
    bool downloadFreeBusyData( const QString& email, QObject* receiver, const char* member );
    KCal::FreeBusy* parseFreeBusy( const QCString& data );

    // KOrganizer makes the calendar data available for syncing.
    void pullSyncData( const QString& filename );

    // KOrganizer reloads the calendar data after a syncing.
    void pushSyncData( const QString& filename );

    // A note is deleted from the outside
    void noteDeleted( const QString& uid );

    // A note is added from the outside
    bool noteAdded( const QString& note );


    // END OF THE ACTUAL KM/KO API

    bool canEditEvent() const { return mCanEditEvents; }
    bool canEditTodo() const { return mCanEditTodos; }

    void editingEvent( KOEventEditor* );
    void editingTodo( KOTodoEditor* );

    QValueList<Note> notes() const;

  public slots:
    void slotNoteNewOrUpdated( const Note& note );
    void slotNoteDeleted( const Note& note );

    // A note is added from the outside
    void slotNoteAdded( const QString& note, bool& accepted );
    void slotNoteRemoved( const QString& uid );

    // When something changed in the calendar, we get this called
    void slotPerhapsUploadFB();

    void slotSyncRunning( const QString& type, bool running );

  signals:
    void newOrUpdatedNote( const QString& id, const QString& geometry,
			   const QColor& color, const QString& text, bool );
    void deletedNote( const QString& id );
    void notesUpdated();

    void doSendICalMessage( const QString& methodStr, const QString& subject,
			    const QStringList& receiver,
			    const QString& messageText );

    void dialogsUp( const QString& type, bool up );
    void conflict( const QString& );
    void notesSync( bool );
    void getStripAlarmsForSending( bool& );

  protected:
    KOGroupware( QObject* kmailTarget, CalendarView*,
                 KCal::Calendar* );

    void timerEvent( QTimerEvent* );

  private slots:
    void slotUploadFreeBusyResult( KIO::Job* );
    void slotClosingEventDialog( KOEventEditor* );
    void slotClosingTodoDialog( KOTodoEditor* );

  private:
    static KOGroupware* mInstance;
    KCal::ICalFormat mFormat;
    CalendarView* mView;
    KCal::Calendar* mCalendar;
    QValueList<Note> mNotes;

    // Free/Busy uploading
    QDateTime mNextUploadTime;
    int mTimerID;
    bool mUploadingFreeBusy;
    bool mCanEditEvents;
    bool mCanEditTodos;
    int mEventDialogCount;
    int mTodoDialogCount;
};

#endif
