/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A simple period selector.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgsimpleperiodedit.h"
#include <skgservices.h>


SKGSimplePeriodEdit::SKGSimplePeriodEdit(QWidget* iParent)
    : SKGComboBox(iParent), m_Mode(PREVIOUS_MONTHS) {}

SKGSimplePeriodEdit::~SKGSimplePeriodEdit() {}

QDate SKGSimplePeriodEdit::firstDate() const
{
    return m_FirstDate;
}

void SKGSimplePeriodEdit::setFirstDate(const QDate& iDate)
{
    m_FirstDate = iDate;

    refreshList();
}
SKGSimplePeriodEdit::Mode SKGSimplePeriodEdit::mode() const
{
    return m_Mode;
}
void SKGSimplePeriodEdit::setMode(SKGSimplePeriodEdit::Mode iMode)
{
    m_Mode = iMode;
    refreshList();
}

QString SKGSimplePeriodEdit::period() const
{
    QString period = currentData().toString();
    if (period.isEmpty() == 1) {
        // Specific month
        period = text();
    }
    return period;
}

void SKGSimplePeriodEdit::refreshList()
{
    QDate today = QDate::currentDate();
    QDate c = SKGServices::periodToDate(SKGServices::dateToPeriod(m_FirstDate, "M"));
    if (!c.isValid()) {
        c = today;
    }
    c = c.addDays(1 - c.day());

    QString smonth = SKGServices::dateToPeriod(today, "M");
    QString squarter = SKGServices::dateToPeriod(today, "Q");
    QString ssemester = SKGServices::dateToPeriod(today, "S");
    QString syear = SKGServices::dateToPeriod(today, "Y");

    // Build the list
    QStringList list;
    do {
        QString cmonth = SKGServices::dateToPeriod(c, "M");
        QString cquarter = SKGServices::dateToPeriod(c, "Q");
        QString csemester = SKGServices::dateToPeriod(c, "S");
        QString cyear = SKGServices::dateToPeriod(c, "Y");
        if (cmonth != smonth || (m_Mode == PREVIOUS_AND_CURRENT_MONTHS || m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) {
            list.insert(0, cmonth);
        }

        if ((c.month() % 3 == 0 && cquarter != squarter && (m_Mode == PREVIOUS_PERIODS || m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) ||
            (cmonth == smonth && m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) {
            list.insert(0, cquarter);
        }
        if ((c.month() % 6 == 0 && csemester != ssemester && (m_Mode == PREVIOUS_PERIODS || m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) ||
            (cmonth == smonth && m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) {
            list.insert(0, csemester);
        }
        if ((c.month() == 12 && cyear != syear && (m_Mode == PREVIOUS_PERIODS || m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) ||
            (cmonth == smonth && m_Mode == PREVIOUS_AND_CURRENT_PERIODS)) {
            list.insert(0, cyear);
        }

        if (cmonth == smonth || c >= today) {
            break;
        }
        c = c.addMonths(1);
    } while (true);

    // Set the list and the current item
    QString current = text();
    bool previous = blockSignals(true);
    clear();

    // Add dynamic items
    if (list.contains(smonth) && m_Mode == PREVIOUS_AND_CURRENT_PERIODS) {
        addItem(i18nc("A period including all dates", "All dates"), "ALL");
    }

    if (list.contains(smonth)) {
        addItem(i18nc("The current month", "Current month"), smonth);
    }

    if (list.contains(squarter)) {
        addItem(i18nc("The current quarter", "Current quarter"), squarter);
    }

    if (list.contains(ssemester)) {
        addItem(i18nc("The current semester", "Current semester"), ssemester);
    }

    if (list.contains(syear)) {
        addItem(i18nc("The current year", "Current year"), syear);
    }

    QString period = SKGServices::getNeighboringPeriod(smonth);
    if (list.contains(period)) {
        addItem(i18nc("The month before the current month", "Last month"), period);
    }

    period = SKGServices::getNeighboringPeriod(squarter);
    if (list.contains(period)) {
        addItem(i18nc("The quarter before the current quarter", "Last quarter"), period);
    }

    period = SKGServices::getNeighboringPeriod(ssemester);
    if (list.contains(period)) {
        addItem(i18nc("The semester before the current semester", "Last semester"), period);
    }

    period = SKGServices::getNeighboringPeriod(syear);
    if (list.contains(period)) {
        addItem(i18nc("The year before the current year", "Last year"), period);
    }
    addItems(list);

    if (!current.isEmpty()) {
        setText(current);
    } else {
        setCurrentIndex(0);
    }
    blockSignals(previous);
}

