/***************************************************************************
                          completedwulist.cpp  -  description
                             -------------------
    begin                : Tue Nov 23 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>

#include <qheader.h>
#include <qcursor.h>

#include <math.h>

#include "ksetiwatch.h"
#include "completedwulist.h"
#include "csvdatabase.h"

// extern globals
extern QList<SetiLoc> SetiList;
extern bool HMS;
extern bool	SKYMAP_LOADED;
extern bool	SKYMAP_VISIBLE;
extern SkyMap *SKYMAP;
extern WUScore Record;
extern bool DRAWGRID;

TotalListViewItem::TotalListViewItem(QListView* parent):QListViewItem(parent)
{
}

TotalListViewItem::TotalListViewItem(QListView* parent, QListViewItem* after):QListViewItem(parent, after)
{
}

TotalListViewItem::~TotalListViewItem()
{
}

void TotalListViewItem::paintCell(QPainter* p, const QColorGroup& cg, int column, int width, int align)
{
QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(0, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
  }
}


WUListViewItem::WUListViewItem(QListView* parent):QListViewItem(parent)
{
}

WUListViewItem::~WUListViewItem()
{
}

/** Returns a string which is used for sorting the items. */
QString WUListViewItem::key(int column, bool ascending) const
{
// return value must be static
static QString ret;
QString val;
double tmp;

if(column == 2)
	{
	int colon;
	int d(0), h(0), m(0), s(0);
	
	val = text(column);
	colon = val.contains(':');
	if(colon == 2)
		{
		sscanf(text(column), "%d:%d:%d", &h, &m, &s);
		}
	else if(colon == 3)
		{
		sscanf(text(column), "%dd:%d:%d:%d", &d, &h, &m, &s);
		}
	tmp = 86400*d + 3600*h + 60*m + s;
	ret.sprintf("%032d", (int)tmp);
	}
else if(column == 3 || column == 4 || column == 10)
	{
	val = text(column);
	tmp = val.toDouble();
	val.sprintf("%f", tmp);
	ret = val.rightJustify(32, '0');
	}
else if(column >= 5 && column <= 7)
	{
	if(text(column).isEmpty())
		tmp = 0.0;
	else
		sscanf(text(column), " scr=%lf", &tmp);
	val.sprintf("%f", tmp);
	ret = val.rightJustify(32, '0');
	}
else
	{
	ret = text(column);
	}
	
return(ret);
}

void WUListViewItem::paintCell(QPainter* p, const QColorGroup& cg, int column, int width, int align)
{
QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(0, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
  }
}

/***************************************************************************/
CompletedWUList::CompletedWUList(QPixmap *starmap, QWidget *parent, const char *name) : QWidget(parent,name)
{
TableColumn complwu[] = {
											{i18n("Date Logged"), -1, AlignLeft},
									 		{i18n("Work Unit Name"), -1, AlignRight},
									 		{i18n("CPU Time"), -1, AlignRight},
									 		{i18n("% Per Hour"), -1, AlignRight},
									 		{i18n("Peak"), 55, AlignRight},
									 		{i18n("Strongest Gaussian"), -1, AlignLeft},
									 		{i18n("Strongest Pulse"), -1, AlignLeft},
									 		{i18n("Strongest Triplet"), -1, AlignLeft},
									 		{i18n("From"), -1, AlignRight},
									 		{i18n("Recorded On"), -1, AlignRight},
									 		{i18n("Base Frequency"), -1, AlignRight}
									 	};
const int wulist_columns(11);									 	

TableColumn totwu[] = {
											{i18n("Location"), -1, AlignLeft},
									 		{i18n("Work Units"), -1, AlignRight},
									 		{i18n("Total CPU Time"), -1, AlignRight},
									 		{i18n("Average CPU Time"), -1, AlignRight},
									 		{i18n("Average % / Hour"), -1, AlignRight},
									 		{i18n("Record Peak"), -1, AlignRight},
									 		{i18n("Record Gaussian"), -1, AlignLeft},
									 		{i18n("Record Pulse"), -1, AlignLeft},
									 		{i18n("Record Triplet"), -1, AlignLeft}
										 	};
const int totallist_columns(9);									 	

sm = starmap;

icons = new KIconLoader();
folder_icon = icons->loadIcon("folder_open", KIcon::Small);
dish_icon = icons->loadIcon("mini-seti", KIcon::User);

totallist = 0;
wulist = 0;
totalview = false;

splitview = new QSplitter(this);
splitview->move(5,1);

treelist = new popListView(splitview);
treelist->setRootIsDecorated(true);
treelist->addColumn("", -1);
treelist->header()->hide();
treelist->insertPopupItem(i18n("Show on Sky Map"), ShowSkymap);
treelist->insertPopupItem(i18n("Reload Locations"), ReloadLocations);
treelist->setPopupStyle(popListView::OverItem);
connect(treelist->popupMenu(), SIGNAL(activated(int)), SLOT(handleTreeListPopupCommand(int)));

totallist = new popListView(splitview);
createList(totallist, &totwu[0], totallist_columns);

wulist = new popListView(splitview);
createList(wulist, &complwu[0], wulist_columns);
connect(wulist->header(), SIGNAL(sectionClicked(int)), SLOT(toggleSorting(int)));
sortorder = true;
wulist->insertPopupItem(i18n("Show on Sky Map"), ShowSkymap);
wulist->setPopupStyle(popListView::OverItem);
connect(wulist->popupMenu(), SIGNAL(activated(int)), SLOT(handleWUListPopupCommand(int)));

// this makes for better splitting
splitview->setResizeMode(treelist, QSplitter::FollowSizeHint);

refreshTreeList();
}

CompletedWUList::~CompletedWUList()
{
}

void CompletedWUList::resizeEvent(QResizeEvent *e)
{
splitview->resize(this->width()-10, this->height()-5);
}

void CompletedWUList::refreshTreeList()
	{
  SetiLoc *loc;
	
  if(treelist)
  	{
		disconnect(treelist, SIGNAL(selectionChanged(QListViewItem*)), this, SLOT(slotSelect(QListViewItem*)));
    treelist->clear();
  	
  	QListViewItem *ri = treelist->insertRootItem(i18n("SETI Locations"), &folder_icon);
  	ri->setOpen(true);
  	treelist->setSelected(ri, true);
  	for( loc=SetiList.first(); loc != 0; loc=SetiList.next() )
  		{
  		treelist->insertChildItem(loc->location(), &dish_icon);
  		}
		connect(treelist, SIGNAL(selectionChanged(QListViewItem*)), SLOT(slotSelect(QListViewItem*)));
		slotSelect( ri );
  	}
	}
		
void CompletedWUList::createList(popListView* list, TableColumn *tc, int nrcol)
{

for(int i=0;i<nrcol;i++)
	{
	TableColumn t = tc[i];
	list->addColumn(t.text,-1);
 	if(t.width != -1) list->setColumnWidth(i, t.width);  	
  list->setColumnAlignment(i, t.alignment);
 	}
}

void CompletedWUList::slotSelect(QListViewItem *item)
{
// if the user has clicked white space, leave this place
if(item == 0) return;

for(SetiLoc* loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	if(loc->location() == treelist->currentItem()->text(0))
		{
		if(totalview)
			{
			totalview = false;
			totallist->hide();
			}
		fillWUList(loc);
		wulist->show();
		break;
		}
	}
if(QString(item->text(0)) == QString("SETI Locations"))
	{
 	totalview = true;
	wulist->hide();
 	fillTotalList();
 	totallist->show();
	}
}

void CompletedWUList::fillWUList(SetiLoc *loc)
{
QString csvfile = loc->directory() + "/SETILog.csv";
QString tmp;
QString entry, date;
QString from;
CSVDataBase csv((const char *)csvfile);
double x, y, z;
int h, m, s;
WUListViewItem *wuit;

wulist->clear();

// switch off automatic sorting
wulist->setSorting(-1, false);

if(csv.open(IO_ReadOnly))
	{
	KApplication::setOverrideCursor(WaitCursor);
	int i(1);
	do
		{
		date = csv.readItem("date", i);
		if(!date.isEmpty())
			{
			wuit = new WUListViewItem(wulist);
			if(wuit != 0)
				{
				// log date
  			wuit->setText(0, date);
  			// WU name
    		entry = csv.readItem("name", i);
    		wuit->setText(1, entry);
    		// CPU time
    		entry = csv.readItem("cpu", i);
    		x = entry.toDouble();
    		entry = SetiLoc::convertTime(x, HMS);
        wuit->setText(2, entry);
        // % per hour
    		entry = csv.readItem("prog", i);
    		y = entry.toDouble();
     		z = (3600.0/x)*y*100.0;
        wuit->setText(3, entry.setNum(z, 'f', 5));
        // spike
    		entry = csv.readItem("bs_power", i);
        x = entry.toDouble();
     		if(x > 1.0e6)
     			entry.setNum(x, 'e', 8);
     		else    				
      		entry.setNum(x, 'f', 3);
        wuit->setText(4, entry);
        // gaussian
    		entry = csv.readItem("bg_score", i);
        x = entry.toDouble();
        if(x > 0.0)
        	{
	    		entry = csv.readItem("bg_power", i);
  	      y = entry.toDouble();
    			entry = csv.readItem("bg_chisq", i);
      	  z = entry.toDouble();
	     		entry.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f",x, y, z);
	     		}
	     	else
	     		entry = "";	     		
        wuit->setText(5, entry);
        // pulse
    		entry = csv.readItem("bp_score", i);
        x = entry.toDouble();
        if(x > 0.0)
        	{
	    		entry = csv.readItem("bp_power", i);
  	      y = entry.toDouble();
    			entry = csv.readItem("bp_period", i);
        	z = entry.toDouble();
     			entry.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f",x, y, z);
     			}
     		else
     			entry = "";
        wuit->setText(6, entry);
        // triplet
    		entry = csv.readItem("bt_score", i);
        x = entry.toDouble();
        if(x > 0.0)
        	{
	    		//entry = csv.readItem("bt_power", i);
  	      //y = entry.toDouble();
    			entry = csv.readItem("bt_period", i);
      	  z = entry.toDouble();
     			entry.sprintf(" scr=%.4f, prd=%.4f",x, z);
     			}
     		else
     			entry = "";
        wuit->setText(7, entry);
        // from
       	entry = csv.readItem("start_ra", i);
       	x = entry.toDouble();
       	h = static_cast<int>( x );
       	m = static_cast<int>( (x - h)*60 );
       	s = static_cast<int>( (x - h)*3600 - m*60 );
       	from.sprintf(" %d hr %d min %d sec RA ", h, m, s);
       	entry = csv.readItem("start_dec", i);
       	x = entry.toDouble();
       	if(x >= 0.0) from += '+'; else from += '-';
       	y = fabs(x);
       	h = static_cast<int>( y );
       	m = static_cast<int>( (y - h)*60 );
       	s = static_cast<int>( (y - h)*3600 - m*60 );
       	from += tmp.sprintf(" %d deg %d' %d'' Dec ", h, m, s);
       	wuit->setText(8, from);
       	// time recorded
      	entry = csv.readItem("time_recorded", i);
      	h = entry.find('(');
      	m = entry.find(')');
      	entry = entry.mid(h+1, m-h-1) + " ";
      	wuit->setText(9, entry);
      	// base frequency
     		entry = csv.readItem("subband_base", i);
     		x = entry.toDouble()/1.0e9;
     		wuit->setText(10, entry.setNum(x, 'f', 9)+" GHz");
     		}
   		}
		i++;
	  } while(!csv.atEnd());
	csv.close();
	KApplication::restoreOverrideCursor();
	}
}
		
void CompletedWUList::fillTotalList()
{
QString entry;
QListViewItem *totit,*grandtotal;
double x, totaltime, totaltime_total;
int nwus, nwus_total;
WUScore current, max, record;

if(totalview)
	{
	KApplication::setOverrideCursor(WaitCursor);
	totallist->clear();
	totallist->setSorting(-1);
	totallist->header()->setClickEnabled(false, -1);
	totaltime_total = 0.0;
	nwus_total = 0;
	Ksetiwatch::initWUScore(&record);
	totit = 0;
	for(SetiLoc* loc=SetiList.first(); loc != 0; loc=SetiList.next() )
		{
		QString csvfile = loc->directory() + "/SETILog.csv";
		CSVDataBase csv((const char *)csvfile);
		// prevent sorting by adding one after another
		if(totit)
			totit = new TotalListViewItem(totallist, totit);
		else
			totit = new TotalListViewItem(totallist);			
		totit->setText(0, loc->location());
		
		Ksetiwatch::initWUScore(&max);
		totaltime = 0.0;
		nwus = 0;
		int i(1);
		if(csv.open(IO_ReadOnly))
			{
			do
				{
	  		entry = csv.readItem("bs_power", i);
	  		if(!entry.isEmpty())
	  			{
	  			// spike
		  		current.spike = entry.toDouble();
		  		if(current.spike > max.spike)
		  			{
		  			max.spike = current.spike;
		  			}
		  		if(max.spike > record.spike)
		  			{
		  			record.spike           = max.spike;
		  			entry                  = csv.readItem("bs_chirp_rate", i);
		  			record.spike_chirprate = entry.toDouble();
		  			record.spike_wu_name   = csv.readItem("name", i);
		  			}
		  		// gaussian
		  		entry = csv.readItem("bg_score", i);
		  		current.gaussian = entry.toDouble();
		  		if(current.gaussian > max.gaussian)
	  				{
	  				max.gaussian       = current.gaussian;
		  			entry              = csv.readItem("bg_power", i);
		  			max.gaussian_power = entry.toDouble();
		  			entry              = csv.readItem("bg_chisq", i);
	  				max.gaussian_chisq = entry.toDouble();
	  				}
	  			if(max.gaussian > record.gaussian)
		  			{
		  			record.gaussian           = max.gaussian;
	  				record.gaussian_power     = max.gaussian_power;
	  				record.gaussian_chisq     = max.gaussian_chisq;
		  			entry                     = csv.readItem("bg_chirp_rate", i);
		  			record.gaussian_chirprate = entry.toDouble();		  			
		  			record.gaussian_wu_name   = csv.readItem("name", i);
	  				}
	  			// pulse
	  			entry         = csv.readItem("bp_score", i);
	  			current.pulse = entry.toDouble();
	  			if(current.pulse > max.pulse)
	  				{
	  				max.pulse        = current.pulse;
		  			entry            = csv.readItem("bp_power", i);
		  			max.pulse_power  = entry.toDouble();
		  			entry            = csv.readItem("bp_period", i);
	  				max.pulse_period = entry.toDouble();
	  				}
	  			if(max.pulse > record.pulse)
		  			{
		  			record.pulse           = max.pulse;
	  				record.pulse_power     = max.pulse_power;
	  				record.pulse_period    = max.pulse_period;
		  			entry                  = csv.readItem("bp_chirp_rate", i);
		  			record.pulse_chirprate = entry.toDouble();		  			
		  			record.pulse_wu_name   = csv.readItem("name", i);
	  				}
		  		// triplet
	  			entry           = csv.readItem("bt_score", i);
	  			current.triplet = entry.toDouble();
	  			if(current.triplet > max.triplet)
	  				{
	  				max.triplet        = current.triplet;
		  			//entry              = csv.readItem("bt_power", i);
		  			//max.triplet_power  = entry.toDouble();
		  			entry              = csv.readItem("bt_period", i);
	  				max.triplet_period = entry.toDouble();
	  				}
	  			if(max.triplet > record.triplet)
		  			{
		  			record.triplet           = max.triplet;
	  				record.triplet_power     = max.triplet_power;
	  				record.triplet_period    = max.triplet_period;
		  			entry                    = csv.readItem("bt_chirp_rate", i);
		  			record.triplet_chirprate = entry.toDouble();		  			
		  			record.triplet_wu_name   = csv.readItem("name", i);
	  				}
	  			// sum up CPU time
		  		entry = csv.readItem("cpu", i);
		  		totaltime += entry.toDouble();
		  		// count WUs
		  		nwus++;
		  		}
				i++; // next entry
			  } while(!csv.atEnd());
			csv.close();
	    }
		nwus_total += nwus;
		totit->setText(1, entry.setNum(nwus));
 		
 		totaltime_total += totaltime;
 		entry = SetiLoc::convertTime(totaltime, HMS);
    totit->setText(2, entry);
 		
 		x = (nwus > 0) ? totaltime/nwus : 0.0;
 		entry = SetiLoc::convertTime(x, HMS);
 		totit->setText(3, entry);
		
		x = (x > 0.0) ? 360000.0/x : 0.0;
 		totit->setText(4, entry.setNum(x, 'f', 5));

 		if(max.spike > 1.0e6)
		  totit->setText(5, entry.setNum(max.spike, 'e', 8));
 		else    				
		  totit->setText(5, entry.setNum(max.spike, 'f', 3));
 		// display the strongest gaussian of this SETI location
 		if(max.gaussian > 0.0)
 			{
	 		entry.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f",max.gaussian,
 									max.gaussian_power, max.gaussian_chisq);
	 		totit->setText(6, entry);
	 		}
 		// display the strongest pulse of this SETI location
 		if(max.pulse > 0.0)
 			{
	 		entry.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f",max.pulse,
 									max.pulse_power, max.pulse_period);
	 		totit->setText(7, entry);
	 		}
 		// display the strongest triplet of this SETI location
 		if(max.triplet > 0.0)
 			{
	 		entry.sprintf(" scr=%.4f, prd=%.4f",max.triplet, max.triplet_period);
	 		totit->setText(8, entry);
	 		}
	  }
	// generate the Grand Total entry
	grandtotal = new TotalListViewItem(totallist, totit);
	grandtotal->setText(0, i18n("Grand Total"));
	grandtotal->setText(1, entry.setNum(nwus_total));
	entry = SetiLoc::convertTime(totaltime_total, HMS);
	grandtotal->setText(2, entry);
	x = (nwus_total > 0) ? totaltime_total/nwus_total : 0.0;
	entry = SetiLoc::convertTime(x, HMS);
	grandtotal->setText(3, entry);
	x = (x > 0.0) ? 360000.0/x : 0.0;
	grandtotal->setText(4, entry.setNum(x, 'f', 5));
	if(record.spike > 1.0e6)
		grandtotal->setText(5, entry.setNum(record.spike, 'e', 8));
	else    				
		grandtotal->setText(5, entry.setNum(record.spike, 'f', 3));
	if(record.gaussian > 0.0)
		{
		entry.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f",record.gaussian,
								record.gaussian_power, record.gaussian_chisq);
		grandtotal->setText(6, entry);
		}
	if(record.pulse > 0.0)
		{
		entry.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f",record.pulse,
								record.pulse_power, record.pulse_period);
		grandtotal->setText(7, entry);
		}
	if(record.triplet > 0.0)
		{
		entry.sprintf(" scr=%.4f, prd=%.4f",record.triplet, record.triplet_period);
		grandtotal->setText(8, entry);
		}
	// update current records if necessary
	if(record.spike > Record.spike)
		{
		Record.spike = record.spike;
		Record.spike_chirprate = record.spike_chirprate;
		Record.spike_wu_name = record.spike_wu_name;
		}
	if(record.gaussian > Record.gaussian)
		{
		Record.gaussian = record.gaussian;
		Record.gaussian_power = record.gaussian_power;
		Record.gaussian_chisq = record.gaussian_chisq;
		Record.gaussian_chirprate = record.gaussian_chirprate;
		Record.gaussian_wu_name = record.gaussian_wu_name;
		}
	if(record.pulse > Record.pulse)
		{
		Record.pulse           = record.pulse;
		Record.pulse_power     = record.pulse_power;
		Record.pulse_period    = record.pulse_period;
		Record.pulse_chirprate = record.pulse_chirprate;
		Record.pulse_wu_name   = record.pulse_wu_name;
		}
	if(record.triplet > Record.triplet)
		{
		Record.triplet           = record.triplet;
		Record.triplet_power     = record.triplet_power;
		Record.triplet_period    = record.triplet_period;
		Record.triplet_chirprate = record.triplet_chirprate;
		Record.triplet_wu_name   = record.triplet_wu_name;
		}		
	KApplication::restoreOverrideCursor();
	}
}

/** creates new or activate old skymap window */
SkyMap* CompletedWUList::openSkymapWindow()
{
if(SKYMAP_LOADED == true)
	{
 	if(sm->isNull() == false && SKYMAP_VISIBLE == false)
 		{
 		SKYMAP = new SkyMap(sm);
		SKYMAP_VISIBLE = true;		 		
		SKYMAP->show();
		}
	else if(SKYMAP->isActiveWindow() == false)
		{
		SKYMAP->show();
		SKYMAP->raise();
		}
	}
else
 	{
 	KMessageBox::sorry(topLevelWidget(), i18n("Couldn't load the pixmap file."));
	SKYMAP = 0;
 	}

return(SKYMAP);
}


/** processes the commands from the right-click popup menu */
void CompletedWUList::handleTreeListPopupCommand(int id)
{
QListViewItem *it = treelist->currentItem();
SetiLoc *loc = 0;

// browse through the list
for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	if(loc->location() == it->text(0)) break;
	}

switch(id)
	{
	case ShowSkymap: // show on sky map
		{
		if(it)
			{
			SKYMAP = openSkymapWindow();
			if(SKYMAP_LOADED)
				{
				SKYMAP->clearMap();
   			// identify corresponding SetiLoc item
				if(it->text(0) == QString("SETI Locations"))
					addAllLoggedWU();
				else
					if(loc) addLoggedWU(loc);
				// update the skymap with added work units
				SKYMAP->update();
				}
			}
		break;
		}
	case ReloadLocations:
		{
		if(it)
			{
 			// identify corresponding SetiLoc item
			if(it->text(0) == QString("SETI Locations"))
				fillTotalList();
			else
				if(loc) fillWUList(loc);				
			}
		break;
		}
	}
}

/**  */
void CompletedWUList::refreshList(SetiLoc* loc)
{
if(totalview)
	fillTotalList();
else
	{
	QListViewItem *it = treelist->currentItem();
	if(it)
		{
		if(loc->location() == it->text(0)) fillWUList(loc);
		}	
	}
}


/** adds all logged WUs of a specific SETI location to the skymap */
void CompletedWUList::addLoggedWU(SetiLoc *loc)
{
QString csv_file(loc->directory() + "/SETILog.csv");
CSVDataBase csv((const char *)csv_file);

if(csv.open(IO_ReadOnly))
	{
	KApplication::setOverrideCursor(WaitCursor);
	int i(1);
	QString entry;
	double ra(0.0), dec(0.0);
	int h, m;
	do
		{
		entry = csv.readItem("start_ra", i);
		if(!entry.isEmpty())
			{
			entry = csv.readItem("start_ra", i);
			ra = entry.toDouble();
			entry = csv.readItem("start_dec", i);
			dec = entry.toDouble();
			entry = csv.readItem("time_recorded", i);
			h = entry.find('(');
			m = entry.find(')');
			entry = entry.mid(h+1, m-h-1);
			}
		SKYMAP->addLocation(loc, (const char*)entry, ra, dec);
		i++;
		} while(!csv.atEnd());
	csv.close();
	KApplication::restoreOverrideCursor();
	}
}

/** adds all logged WUs of all SETI locations to the skymap */
void CompletedWUList::addAllLoggedWU()
{
SetiLoc *loc;
for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	addLoggedWU(loc);
	}
}

/** handles right-click mouse clicks in WU list */
void CompletedWUList::handleWUListPopupCommand(int id)
{
QListViewItem *tree_it = treelist->currentItem();
QListViewItem *wu_it = wulist->currentItem();
int a, b, c, d, e, f;
char sign;
double ra(0.0), dec(0.0);

SetiLoc *loc = 0;

// browse through the list
for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	if(loc->location() == tree_it->text(0)) break;
	}

if(loc)
	{
	sscanf(wu_it->text(8),"%d hr %d min %d sec RA %c %d deg %d' %d''",
					&a, &b, &c, &sign, &d, &e, &f);
	ra = (double)(a + b/60.0 + c/3600.0);
	dec = (double)(d + e/60.0 + f/3600.0);
	if(sign == '-') dec = -dec;
	SKYMAP = openSkymapWindow();
	if(SKYMAP_LOADED)
		{
		SKYMAP->addLocation(loc, wu_it->text(9), ra, dec);
		SKYMAP->repaint();
		}
	}
}

/**  */
void CompletedWUList::toggleSorting(int column)
{
if(!totalview)
	{
	sortorder = !sortorder;
	if(wulist) wulist->setSorting(column, sortorder);
	}
else
	{
	if(totallist) totallist->setSorting(-1, false);
	}
}
