/***************************************************************************
                          completedwulist.cpp  -  description
                             -------------------
    begin                : Tue Nov 23 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <qsplitter.h>
#include <qheader.h>
#include <qcursor.h>
#include <qtextstream.h>

#include <math.h>

#include "setiloc.h"
#include "completedwulist.h"
#include "csvdatabase.h"
#include "poplistview.h"
#include "skymap.h"

// extern globals
extern QList<SetiLoc> SetiList;
extern bool HMS;
extern bool	SKYMAP_LOADED;
extern bool	SKYMAP_VISIBLE;
extern SkyMap *SKYMAP;
extern WUScore Record;
extern bool DRAWGRID;

/*------------------------------------------------------------------------ */
TotalListViewItem::TotalListViewItem(QListView* parent)
                  : QListViewItem(parent)
{
}

/*------------------------------------------------------------------------ */
TotalListViewItem::TotalListViewItem(QListView* parent, QListViewItem* after)
                  : QListViewItem(parent, after)
{
}

/*------------------------------------------------------------------------ */
TotalListViewItem::~TotalListViewItem()
{
}

/*------------------------------------------------------------------------ */
void TotalListViewItem::paintCell(QPainter* p, const QColorGroup& cg,
                                  int column, int width, int align)
{
QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(0, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
  }
}

/*------------------------------------------------------------------------ */
SumListViewItem::SumListViewItem(QListView* parent)
                  : QListViewItem(parent)
{
}

/*------------------------------------------------------------------------ */
SumListViewItem::SumListViewItem(QListView* parent, QListViewItem* after)
                : QListViewItem(parent, after)
{
}

/*------------------------------------------------------------------------ */
SumListViewItem::~SumListViewItem()
{
}

/*------------------------------------------------------------------------ */
void SumListViewItem::paintCell(QPainter* p, const QColorGroup& cg,
                                int column, int width, int align)
{
QColorGroup myColorGroup(cg);
//QFont boldFont = p->font();

// use bold font
//boldFont.setBold(true);
//p->setFont(boldFont);

// draw text in blue color
myColorGroup.setColor(QColorGroup::Text, Qt::blue);

QListViewItem::paintCell(p, myColorGroup, column, width, align);
//QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(0, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
	QPen pen;
	pen.setWidth(2);
	p->setPen(pen);
	p->moveTo(0, 0);
	p->lineTo(width-1, 0);
  }
}

/*------------------------------------------------------------------------ */
SigSumItem::SigSumItem(QListView* parent, QListViewItem* after)
          : QListViewItem(parent, after)
{
}

/*------------------------------------------------------------------------ */
SigSumItem::~SigSumItem()
{
}

/*------------------------------------------------------------------------ */
void SigSumItem::paintCell(QPainter* p, const QColorGroup& cg,
                           int column, int width, int align)
{
QColorGroup myColorGroup(cg);
myColorGroup.setColor(QColorGroup::Text, Qt::blue);

QListViewItem::paintCell(p, myColorGroup, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(0, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
	QPen pen;
	pen.setWidth(2);
	p->setPen(pen);
	p->moveTo(0, 0);
	p->lineTo(width-1, 0);
  }
}


/*------------------------------------------------------------------------ */
WUListViewItem::WUListViewItem(QListView* parent):QListViewItem(parent)
{
}

/*------------------------------------------------------------------------ */
WUListViewItem::~WUListViewItem()
{
}

/*------------------------------------------------------------------------ */
/** Returns a string which is used for sorting the items. */
QString WUListViewItem::key(int column, bool ascending) const
{
// return value must be static
static QString ret;
QString val;
double tmp;

if(column == 2)
	{
	int colon;
	int d(0), h(0), m(0), s(0);
	
	val = text(column);
	colon = val.contains(':');
	if(colon == 2)
		{
		sscanf(text(column), "%d:%d:%d", &h, &m, &s);
		}
	else if(colon == 3)
		{
		sscanf(text(column), "%dd:%d:%d:%d", &d, &h, &m, &s);
		}
	tmp = 86400*d + 3600*h + 60*m + s;
	ret.sprintf("%032d", (int)tmp);
	}
else if(column == 3 || column == 4 || column == 10)
	{
	val = text(column);
	tmp = val.toDouble();
	val.sprintf("%f", tmp);
	ret = val.rightJustify(32, '0');
	}
else if(column >= 5 && column <= 7)
	{
	if(text(column).isEmpty())
		tmp = 0.0;
	else
		sscanf(text(column), " scr=%lf", &tmp);
	val.sprintf("%f", tmp);
	ret = val.rightJustify(32, '0');
	}
else
	{
	ret = text(column);
	}
	
return(ret);
}

/*------------------------------------------------------------------------ */
void WUListViewItem::paintCell(QPainter* p, const QColorGroup& cg,
                               int column, int width, int align)
{
QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(-10, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
  }
}

/*------------------------------------------------------------------------ */
SigListViewItem::SigListViewItem(QListView* parent):QListViewItem(parent)
{
}

/*------------------------------------------------------------------------ */
SigListViewItem::~SigListViewItem()
{
}

/*------------------------------------------------------------------------ */
/** Returns a string which is used for sorting the items. */
QString SigListViewItem::key(int column, bool ascending) const
{
// return value must be static
static QString ret;
QString val;
double tmp;

switch(column)
  {
  case 2:
  case 3:
  case 4:
  case 5:
    tmp = text(column).toInt();
    ret.sprintf("%032d", (int)tmp);
    break;
  default:
    ret = text(column);
    break;
  }
	
return(ret);
}

/*------------------------------------------------------------------------ */
void SigListViewItem::paintCell(QPainter* p, const QColorGroup& cg,
                               int column, int width, int align)
{
QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(-10, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
  }
}

/***************************************************************************/
CompletedWUList::CompletedWUList(QPixmap *starmap, QWidget *parent,
                                 const char *name) : QWidget(parent,name)
{
TableColumn wuList[] = {
                        {i18n("Date Logged"), -1, AlignLeft},
                        {i18n("Work Unit Name"), -1, AlignRight},
                        {i18n("CPU Time"), -1, AlignRight},
                        {i18n("%/Hour"), -1, AlignRight},
                        {i18n("Peak"), 55, AlignRight},
                        {i18n("Strongest Gaussian"), -1, AlignLeft},
                        {i18n("Strongest Pulse"), -1, AlignLeft},
                        {i18n("Strongest Triplet"), -1, AlignLeft},
                        {i18n("From"), -1, AlignRight},
                        {i18n("Recorded On"), -1, AlignRight},
                        {i18n("Base Frequency"), -1, AlignRight},
                        {i18n("Angle Range"), 90, AlignRight},
                        {i18n("TeraFlops"), 80, AlignRight}
                        };
const int wuListCols(13);									 	

TableColumn wuSum[] = {
                      {i18n("Location"), -1, AlignLeft},
                      {i18n("WUs"), -1, AlignRight},
                      {i18n("Total CPU Time"), -1, AlignRight},
                      {i18n("Avg. CPU Time"), -1, AlignRight},
                      {i18n("Avg. %/Hour"), -1, AlignRight},
                      {i18n("Record Peak"), -1, AlignRight},
                      {i18n("Record Gaussian"), -1, AlignLeft},
                      {i18n("Record Pulse"), -1, AlignLeft},
                      {i18n("Record Triplet"), -1, AlignLeft}
                      };
const int wuSumCols(9);									 	

TableColumn sigList[] = {
                       {i18n("Date Logged"), 150, AlignLeft},
                       {i18n("Work Unit Name"), -1, AlignRight},
                       {i18n("Spikes"), -1, AlignRight},
                       {i18n("Gauss'ns"), -1, AlignRight},
                       {i18n("Pulses"), -1, AlignRight},
                       {i18n("Triplets"), -1, AlignRight}
                       };
const int sigListCols(6);									 	

TableColumn sigSum[] = {
                       {i18n("Location"), -1, AlignLeft},
                       {i18n("WUs"), -1, AlignRight},
                       {i18n("Total Spikes"), -1, AlignRight},
                       {i18n("Total Gauss'ns"), -1, AlignRight},
                       {i18n("Total Pulses"), -1, AlignRight},
                       {i18n("Total Triplets"), -1, AlignRight},
                       {i18n("Spikes/WU"), -1, AlignRight},
                       {i18n("Gauss'ns/WU"), -1, AlignRight},
                       {i18n("Pulses/WU"), -1, AlignRight},
                       {i18n("Triplets/WU"), -1, AlignRight}
                      };
const int sigSumCols(10);									 	

sm = starmap;

icons = new KIconLoader();
folder_icon = icons->loadIcon("folder_open", KIcon::Small);
dish_icon = icons->loadIcon("mini-seti", KIcon::User);

cwuList[WUSummary]     = 0;
cwuList[WUList]        = 0;
cwuList[SignalSummary] = 0;
cwuList[SignalList]    = 0;
currentView            = WUSummary;

splitview = new QSplitter(this);
splitview->move(5,1);

// set up the tree list
treelist = new popListView(splitview);
treelist->setRootIsDecorated(true);
treelist->addColumn("", -1);
treelist->header()->hide();
treelist->insertPopupItem(i18n("Show on Sky Map"), ShowSkymap);
treelist->insertPopupItem(i18n("Reload Locations"), ReloadLocations);
treelist->setPopupStyle(popListView::OverItem);
// this results in better splitting
splitview->setResizeMode(treelist, QSplitter::FollowSizeHint);
connect(treelist->popupMenu(), SIGNAL(activated(int)),
                         this, SLOT(handleTreeListPopupCommand(int)));

// set up the WU summary list
cwuList[WUSummary] = new popListView(splitview);
createList(cwuList[WUSummary], &wuSum[0], wuSumCols);

// set up the WU list
cwuList[WUList] = new popListView(splitview);
createList(cwuList[WUList], &wuList[0], wuListCols);
connect(cwuList[WUList]->header(), SIGNAL(sectionClicked(int)),
                             this, SLOT(toggleSorting(int)));

// set up the signal summary list
cwuList[SignalSummary] = new popListView(splitview);
createList(cwuList[SignalSummary], &sigSum[0], sigSumCols);

// set up the signal list
cwuList[SignalList] = new popListView(splitview);
createList(cwuList[SignalList], &sigList[0], sigListCols);
connect(cwuList[SignalList]->header(), SIGNAL(sectionClicked(int)),
                                 this, SLOT(toggleSorting(int)));

sortorder = true;
cwuList[WUList]->setShowSortIndicator(true);
cwuList[WUList]->insertPopupItem(i18n("Show on Sky Map"), ShowSkymap);
cwuList[WUList]->setPopupStyle(popListView::OverItem);
connect(cwuList[WUList]->popupMenu(), SIGNAL(activated(int)),
                       this, SLOT(handleWUListPopupCommand(int)));

// hide all list views at the beginning
for(int i=0; i<4; i++) cwuList[i]->hide();

refreshTreeList();
}

/*------------------------------------------------------------------------ */
CompletedWUList::~CompletedWUList()
{
}

/*------------------------------------------------------------------------ */
void CompletedWUList::resizeEvent(QResizeEvent *e)
{
splitview->resize(this->width()-10, this->height()-5);
}

/*------------------------------------------------------------------------ */
void CompletedWUList::refreshTreeList()
{
SetiLoc* loc;
	
if(treelist)
  {
  disconnect(treelist, SIGNAL(selectionChanged(QListViewItem*)),
                 this, SLOT(slotSelect(QListViewItem*)));
  for(loc=SetiList.first(); loc != 0; loc=SetiList.next())
    {
    disconnect(loc, SIGNAL(workUnitLogged(SetiLoc*)),
              this, SLOT(refreshList(SetiLoc*)));
    }
  treelist->clear();
  treelist->setSorting(-1);
  	
  QListViewItem* sl = treelist->insertRootItem(i18n("SETI Locations"),
                                               &folder_icon);
  QListViewItem* rs = treelist->insertRootItem(i18n("Returned Signals"),
                                               &folder_icon);
  sl->setOpen(true);
  treelist->setSelected(sl, true);
  for(loc=SetiList.first(); loc != 0; loc=SetiList.next())
    {
    treelist->insertChildItem(loc->description(), &dish_icon, sl);
    treelist->insertChildItem(loc->description(), &dish_icon, rs);
    connect(loc, SIGNAL(workUnitLogged(SetiLoc*)),
           this, SLOT(refreshList(SetiLoc*)));
    }
  sl->sortChildItems(0, true);
  rs->sortChildItems(0, true);
  connect(treelist, SIGNAL(selectionChanged(QListViewItem*)),
              this, SLOT(slotSelect(QListViewItem*)));


  slotSelect(sl);
  }
}
		
/*------------------------------------------------------------------------ */
void CompletedWUList::createList(popListView* list, TableColumn *tc,
                                 int nrcol)
{

for(int i=0;i<nrcol;i++)
	{
	TableColumn t = tc[i];
	list->addColumn(t.text,-1);
 	if(t.width != -1) list->setColumnWidth(i, t.width);  	
  list->setColumnAlignment(i, t.alignment);
 	}
}

/*------------------------------------------------------------------------ */
void CompletedWUList::slotSelect(QListViewItem *item)
{
// if the user has clicked white space, leave this place
if(item == 0) return;

if(item->text(0) == i18n("SETI Locations"))
  {
  cwuList[currentView]->hide();
 	currentView = WUSummary;
 	fillTotalList();
 	cwuList[WUSummary]->show();
 	return;
  }

if(item->text(0) == i18n("Returned Signals"))
  {
  cwuList[currentView]->hide();
 	currentView = SignalSummary;
 	fillSigSumList();
 	cwuList[SignalSummary]->show();
 	return;
  }

for(SetiLoc* loc=SetiList.first(); loc != 0; loc=SetiList.next() )
  {
  if(loc->description() == treelist->currentItem()->text(0))
    {
    cwuList[currentView]->hide();
    QListViewItem* p = treelist->currentItem()->parent();
    if(p)
      {
      if(p->text(0) == i18n("SETI Locations"))
        {
        fillWUList(loc);
        currentView = WUList;
        cwuList[WUList]->show();
        break;
        }
      if(p->text(0) == i18n("Returned Signals"))
        {
        fillSigList(loc);
        currentView = SignalList;
        cwuList[SignalList]->show();
        break;
        }
      }
    }
  }
}

/*------------------------------------------------------------------------ */
void CompletedWUList::fillWUList(SetiLoc *loc)
{
QString csvfile = loc->logDirectory() + "/SETILog.csv";
QString tmp;
QString entry, date;
QString from;
CSVDataBase csv((const char *)csvfile);
double x, y, z;
int h, m, s, vrs;
WUListViewItem *wuit;

cwuList[WUList]->clear();

// switch off automatic sorting
cwuList[WUList]->setSorting(-1, false);

if(csv.open(IO_ReadOnly))
	{
	KApplication::setOverrideCursor(WaitCursor);
	int i(1);
	do
		{
		date = csv.readItem("date", i);
		if(!date.isEmpty())
			{
			// the version number
			entry = csv.readItem("version", i);
			// HACK: multiply with 100.01 to prevent rounding error
			vrs = (int)(entry.toDouble()*100.01);
			
			wuit = new WUListViewItem(cwuList[WUList]);
			//wuit->setExpandable(true);
			if(wuit != 0)
				{
				// log date
  			wuit->setText(0, date);
  			// WU name
    		entry = csv.readItem("name", i);
    		wuit->setText(1, entry);
    		// CPU time
    		entry = csv.readItem("cpu", i);
    		x = entry.toDouble();
    		entry = SetiLoc::convertTime(x, HMS);
        wuit->setText(2, entry);
        // % per hour
    		entry = csv.readItem("prog", i);
    		y = entry.toDouble();
     		z = (x > 0.0) ? (3600.0/x)*y*100.0 : 0.0;
        wuit->setText(3, entry.setNum(z, 'f', 5));
        // spike
    		entry = csv.readItem("bs_power", i);
        x = entry.toDouble();
     		if(x > 1.0e6)
     			entry.setNum(x, 'e', 8);
     		else    				
      		entry.setNum(x, 'f', 3);
        wuit->setText(4, entry);
        // gaussian
    		entry = csv.readItem("bg_score", i);
        x = entry.toDouble();
        if(x > 0.0)
        	{
	    		entry = csv.readItem("bg_power", i);
  	      y = entry.toDouble();
    			entry = csv.readItem("bg_chisq", i);
      	  z = entry.toDouble();
	     		entry.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f",x, y, z);
	     		}
	     	else
	     		entry = "";	     		
        wuit->setText(5, entry);
        // pulse
    		entry = csv.readItem("bp_score", i);
        x = entry.toDouble();
        if(x > 0.0)
        	{
	    		entry = csv.readItem("bp_power", i);
  	      y = entry.toDouble();
    			entry = csv.readItem("bp_period", i);
        	z = entry.toDouble();
     			entry.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f",x, y, z);
     			}
     		else
     			entry = "";
        wuit->setText(6, entry);
        // triplet
    		entry = csv.readItem("bt_score", i);
        x = entry.toDouble();
        if(x > 0.0)
        	{
	    		//entry = csv.readItem("bt_power", i);
  	      //y = entry.toDouble();
    			entry = csv.readItem("bt_period", i);
      	  z = entry.toDouble();
     			entry.sprintf(" scr=%.4f, prd=%.4f",x, z);
     			}
     		else
     			entry = "";
        wuit->setText(7, entry);
        // from
       	entry = csv.readItem("start_ra", i);
       	x = entry.toDouble();
       	h = static_cast<int>( x );
       	m = static_cast<int>( (x - h)*60 );
       	s = static_cast<int>( (x - h)*3600 - m*60 );
       	from.sprintf(" %d hr %d min %d sec RA ", h, m, s);
       	entry = csv.readItem("start_dec", i);
       	x = entry.toDouble();
       	if(x >= 0.0) from += '+'; else from += '-';
       	y = fabs(x);
       	h = static_cast<int>( y );
       	m = static_cast<int>( (y - h)*60 );
       	s = static_cast<int>( (y - h)*3600 - m*60 );
       	from += tmp.sprintf(" %d deg %d' %d'' Dec ", h, m, s);
       	wuit->setText(8, from);
       	// time recorded
      	entry = csv.readItem("time_recorded", i);
      	h = entry.find('(');
      	m = entry.find(')');
      	entry = entry.mid(h+1, m-h-1) + " ";
      	wuit->setText(9, entry);
      	// base frequency
     		entry = csv.readItem("subband_base", i);
     		x = entry.toDouble()/1.0e9;
     		wuit->setText(10, entry.setNum(x, 'f', 9)+" GHz");
     		// angle range
     		entry = csv.readItem("angle_range", i);
     		wuit->setText(11, entry);
     		// TeraFlops
     		double ar = entry.toDouble();
     		entry.setNum(loc->teraFlops(ar, vrs), 'f', 4);
     		wuit->setText(12, entry);
     		}
   		}
		i++;
	  } while(!csv.atEnd());
	csv.close();
	KApplication::restoreOverrideCursor();
	}
}
		
/*------------------------------------------------------------------------ */
void CompletedWUList::fillTotalList()
{
QString entry;
QListViewItem *totit,*grandtotal;
double x, totaltime, totaltime_total;
int nwus, nwus_nz, nwus_total, nwus_nz_total;
WUScore current, max, record;

if(currentView == WUSummary)
	{
	KApplication::setOverrideCursor(WaitCursor);
	
	// clear the list and enable sorting
	cwuList[WUSummary]->clear();
	cwuList[WUSummary]->setSorting(0);
	
	// prevent the user from fiddling around with the header
	cwuList[WUSummary]->header()->setClickEnabled(false, -1);
	
	// init variables for Grand Total
	totaltime_total = 0.0;
	nwus_total      = 0;
	nwus_nz_total   = 0;
	totit           = 0;
	SetiContainer::initWUScore(&record);
	// make a copy of the location list
  QList<SetiLoc> SetiList2 = SetiList;	
	// now fill the list
	for(SetiLoc* loc=SetiList.first(); loc != 0; loc=SetiList.next() )
		{
		// data are read from this log file
		QString csvfile = loc->logDirectory() + "/SETILog.csv";
		CSVDataBase csv((const char*)csvfile);
		
		// create the list view item
		totit = new TotalListViewItem(cwuList[WUSummary]);			
		totit->setText(0, loc->description());
		
    // see if log files are redirected
    bool redir = false;
    for(SetiLoc* loc2=SetiList2.first(); loc2 != 0; loc2=SetiList2.next() )
      {
      if(loc != loc2)
        {
        if(loc->logDirectory() == loc2->directory())
          {
          totit->setText(1, i18n("See %1").arg(loc2->description()));
          redir = true;
          }
        }
      }
    // don't fill if the location has its logs redirected; continue with next
    if(redir) continue;
		
		// init variables for this location
		SetiContainer::initWUScore(&max);
		totaltime = 0.0;
		// counts all WUs
		nwus = 0;
		// counts only WUs with non-zero processing time
		nwus_nz = 0;
		// line in SETILog.csv
		int i = 1;
		if(csv.open(IO_ReadOnly))
			{
			do
				{
	  		entry = csv.readItem("date", i);
	  		if(!entry.isEmpty())
	  			{
	  			// spike
  	  		entry = csv.readItem("bs_power", i);
		  		current.spike.power = entry.toDouble();
		  		if(current.spike.power > max.spike.power)
		  			{
		  			max.spike.power = current.spike.power;
		  			}
		  		if(max.spike.power > record.spike.power)
		  			{
		  			record.spike.power     = max.spike.power;
		  			entry                  = csv.readItem("bs_chirp_rate", i);
		  			record.spike.chirprate = entry.toDouble();
		  			record.spike.wu_name   = csv.readItem("name", i);
		  			}
		  		// gaussian
		  		entry = csv.readItem("bg_score", i);
		  		current.gaussian.score = entry.toDouble();
		  		if(current.gaussian.score > max.gaussian.score)
	  				{
	  				max.gaussian.score = current.gaussian.score;
		  			entry              = csv.readItem("bg_power", i);
		  			max.gaussian.power = entry.toDouble();
		  			entry              = csv.readItem("bg_chisq", i);
	  				max.gaussian.chisq = entry.toDouble();
	  				}
	  			if(max.gaussian.score > record.gaussian.score)
		  			{
		  			record.gaussian.score     = max.gaussian.score;
	  				record.gaussian.power     = max.gaussian.power;
	  				record.gaussian.chisq     = max.gaussian.chisq;
		  			entry                     = csv.readItem("bg_chirp_rate", i);
		  			record.gaussian.chirprate = entry.toDouble();		  			
		  			record.gaussian.wu_name   = csv.readItem("name", i);
	  				}
	  			// pulse
	  			entry         = csv.readItem("bp_score", i);
	  			current.pulse.score = entry.toDouble();
	  			if(current.pulse.score > max.pulse.score)
	  				{
	  				max.pulse.score  = current.pulse.score;
		  			entry            = csv.readItem("bp_power", i);
		  			max.pulse.power  = entry.toDouble();
		  			entry            = csv.readItem("bp_period", i);
	  				max.pulse.period = entry.toDouble();
	  				}
	  			if(max.pulse.score > record.pulse.score)
		  			{
		  			record.pulse.score     = max.pulse.score;
	  				record.pulse.power     = max.pulse.power;
	  				record.pulse.period    = max.pulse.period;
		  			entry                  = csv.readItem("bp_chirp_rate", i);
		  			record.pulse.chirprate = entry.toDouble();		  			
		  			record.pulse.wu_name   = csv.readItem("name", i);
	  				}
		  		// triplet
	  			entry           = csv.readItem("bt_score", i);
	  			current.triplet.score = entry.toDouble();
	  			if(current.triplet.score > max.triplet.score)
	  				{
	  				max.triplet.score    = current.triplet.score;
		  			//entry              = csv.readItem("bt_power", i);
		  			//max.triplet.power  = entry.toDouble();
		  			entry              = csv.readItem("bt_period", i);
	  				max.triplet.period = entry.toDouble();
	  				}
	  			if(max.triplet.score > record.triplet.score)
		  			{
		  			record.triplet.score     = max.triplet.score;
	  				//record.triplet.power     = max.triplet.power;
	  				record.triplet.period    = max.triplet.period;
		  			entry                    = csv.readItem("bt_chirp_rate", i);
		  			record.triplet.chirprate = entry.toDouble();		  			
		  			record.triplet.wu_name   = csv.readItem("name", i);
	  				}
	  			// sum up CPU time
		  		entry = csv.readItem("cpu", i);
		  		double prctime = entry.toDouble();
		  		totaltime += prctime;
		  		// count WUs
		  		nwus++;
		  		if(prctime > 0.0) nwus_nz++;
		  		}
				i++; // next entry
			  } while(!csv.atEnd());
			csv.close();
	    }
	  // sum up for Grand Total
		nwus_total      += nwus;
		nwus_nz_total   += nwus_nz;
 		totaltime_total += totaltime;
		
		// number of WUs
		totit->setText(1, entry.setNum(nwus));
 		 		
 		// total CPU time
 		entry = SetiLoc::convertTime(totaltime, HMS);
    totit->setText(2, entry);
 		
    // average time per WU
 		x = (nwus > 0) ? totaltime/nwus_nz : 0.0;
 		entry = SetiLoc::convertTime(x, HMS);
 		totit->setText(3, entry);
		
 		// average % per hour
		x = (x > 0.0) ? 360000.0/x : 0.0;
 		totit->setText(4, entry.setNum(x, 'f', 5));

 		// max spike
 		if(max.spike.power > 1.0e6)
		  totit->setText(5, entry.setNum(max.spike.power, 'e', 8));
 		else    				
		  totit->setText(5, entry.setNum(max.spike.power, 'f', 3));
 		
 		// max gaussian
 		if(max.gaussian.score > 0.0)
 			{
	 		entry.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f",max.gaussian.score,
 									max.gaussian.power, max.gaussian.chisq);
	 		totit->setText(6, entry);
	 		}
 		
 		// max pulse
 		if(max.pulse.score > 0.0)
 			{
	 		entry.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f",max.pulse.score,
 									max.pulse.power, max.pulse.period);
	 		totit->setText(7, entry);
	 		}
 		
 		// max triplet
 		if(max.triplet.score > 0.0)
 			{
	 		entry.sprintf(" scr=%.4f, prd=%.4f",max.triplet.score, max.triplet.period);
	 		totit->setText(8, entry);
	 		}
	  }
	
	// create the Grand Total entry
	// make sure that it's the last item
	QListViewItem* last(0);
	QListViewItem* li(cwuList[WUSummary]->firstChild());
	while(li != 0)
	  {
  	last = li;
	  li = last->itemBelow();
	  }
	// switch sorting off; otherwise "Grand Total" would be sorted into the list
	cwuList[WUSummary]->setSorting(-1);

  if(last)
    {
    grandtotal = new SumListViewItem(cwuList[WUSummary], last);
  	
    grandtotal->setText(0, i18n("Grand Total"));
  	
    // number of WUs
    grandtotal->setText(1, entry.setNum(nwus_total));
  	
    // total CPU time
    entry = SetiLoc::convertTime(totaltime_total, HMS);
    grandtotal->setText(2, entry);
  	
    // average CPU time
    x = (nwus_total > 0) ? totaltime_total/nwus_nz_total : 0.0;
    entry = SetiLoc::convertTime(x, HMS);
    grandtotal->setText(3, entry);
  	
    // average % per hour
    x = (x > 0.0) ? 360000.0/x : 0.0;
    grandtotal->setText(4, entry.setNum(x, 'f', 5));
  	
    // record peak
    if(record.spike.power > 1.0e6)
      grandtotal->setText(5, entry.setNum(record.spike.power, 'e', 8));
    else    				
      grandtotal->setText(5, entry.setNum(record.spike.power, 'f', 3));
  	
    // record gaussian
    if(record.gaussian.score > 0.0)
      {
      entry.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f",record.gaussian.score,
                    record.gaussian.power, record.gaussian.chisq);
      grandtotal->setText(6, entry);
      }
  	
    // record pulse
    if(record.pulse.score > 0.0)
      {
      entry.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f",record.pulse.score,
                    record.pulse.power, record.pulse.period);
      grandtotal->setText(7, entry);
      }
  	
    // record triplet
    if(record.triplet.score > 0.0)
      {
      entry.sprintf(" scr=%.4f, prd=%.4f",record.triplet.score,
                    record.triplet.period);
      grandtotal->setText(8, entry);
      }
    }
	
	// update current records if necessary
	if(record.spike.power > Record.spike.power)
		{
		Record.spike = record.spike;
		}
	if(record.gaussian.score > Record.gaussian.score)
		{
		Record.gaussian = record.gaussian;
		}
	if(record.pulse.score > Record.pulse.score)
		{
		Record.pulse = record.pulse;
		}
	if(record.triplet.score > Record.triplet.score)
		{
		Record.triplet = record.triplet;
		}		
	KApplication::restoreOverrideCursor();
	}
}

/*------------------------------------------------------------------------ */
/** Fills the signal list with values. */
void CompletedWUList::fillSigList(SetiLoc* loc)
{
QFile log(loc->logDirectory() + "/SETIResult.log");

// Needed to get the log time. Maybe do it in a more clever way later.
fillWUList(loc),

cwuList[SignalList]->clear();

// switch off automatic sorting
cwuList[SignalList]->setSorting(-1, false);

if(log.open(IO_ReadOnly))
  {
  KApplication::setOverrideCursor(WaitCursor);
  QTextStream t(&log);
  QString s, wu;
  s = t.readLine();
  while( !t.atEnd() )
    {
    if(s[0] == '[')
      {
      // found a WU marker; add it to the tree
      wu = s.mid(1, s.length()-2);
      QListViewItem* it = new SigListViewItem(cwuList[SignalList]);
      it->setText(1, wu);
      // now determine the log date
      QListViewItem* wit;
      for(wit=cwuList[WUList]->firstChild(); wit!=0; wit=wit->nextSibling())
        {
        if(wit->text(1) == wu) break;
        }
      if(wit) it->setText(0, wit->text(0));
      // count the signals
      int spikes    = 0;
      int gaussians = 0;
      int pulses    = 0;
      int triplets  = 0;
      do
        {
        s = t.readLine();
        if(s[0] == '[') break;  // outta here
        if(s.startsWith("spike")) spikes++;
        if(s.startsWith("gaussian")) gaussians++;
        if(s.startsWith("pulse")) pulses++;
        if(s.startsWith("triplet")) triplets++;
        } while(!t.atEnd());
      // set the columns
      it->setText(2, QString::number(spikes));
      it->setText(3, QString::number(gaussians));
      it->setText(4, QString::number(pulses));
      it->setText(5, QString::number(triplets));
      }
    }
  log.close();	
	KApplication::restoreOverrideCursor();
  }
else
  {
  QListViewItem* it = new SigListViewItem(cwuList[SignalList]);
  it->setText(0, i18n("None logged."));
  }
}

/*------------------------------------------------------------------------ */
/** Fills the signal summary list with values. */
void CompletedWUList::fillSigSumList()
{

KApplication::setOverrideCursor(WaitCursor);

// clear the list and enable sorting
cwuList[SignalSummary]->clear();
cwuList[SignalSummary]->setSorting(0);	

// prevent the user from fiddling around with the header
cwuList[SignalSummary]->header()->setClickEnabled(false, -1);

// make a copy of the location list
QList<SetiLoc> SetiList2 = SetiList;
// init ints for counting
int total_spikes    = 0;
int total_gaussians = 0;
int total_pulses    = 0;
int total_triplets  = 0;
int total_nrWU      = 0;
for(SetiLoc* loc=SetiList.first(); loc != 0; loc=SetiList.next() )
  {
  QFile log(loc->logDirectory() + "/SETIResult.log");

  // create list view entry
  TotalListViewItem* it = new TotalListViewItem(cwuList[SignalSummary]);
  // set the first column
  it->setText(0, loc->description());

  // see if log files are redirected
  bool redir = false;
  for(SetiLoc* loc2=SetiList2.first(); loc2 != 0; loc2=SetiList2.next() )
    {
    if(loc != loc2)
      {
      if(loc->logDirectory() == loc2->directory())
        {
        it->setText(1, i18n("See %1").arg(loc2->description()));
        redir = true;
        }
      }
    }
  // don't fill if the location has its logs redirected; continue with next
  if(redir) continue;

  int loc_spikes    = 0;
  int loc_gaussians = 0;
  int loc_pulses    = 0;
  int loc_triplets  = 0;
  int loc_nrWU      = 0;
  if(log.open(IO_ReadOnly))
    {
    QTextStream t(&log);
    QString s;
    s = t.readLine();
    while( !t.atEnd() )
      {
      if(s[0] == '[')
        {
        // found a WU marker; count the signals
        loc_nrWU++;
        do
          {
          s = t.readLine();
          if(s[0] == '[') break;  // outta here
          if(s.startsWith("spike")) loc_spikes++;
          if(s.startsWith("gaussian")) loc_gaussians++;
          if(s.startsWith("pulse")) loc_pulses++;
          if(s.startsWith("triplet")) loc_triplets++;
          } while(!t.atEnd());
        }
      }
    log.close();
    }
  // sum up totals
  total_spikes    += loc_spikes;
  total_gaussians += loc_gaussians;
  total_pulses    += loc_pulses;
  total_triplets  += loc_triplets;
  total_nrWU      += loc_nrWU;
  if(loc_nrWU > 0)
    {
    it->setText(1, QString::number(loc_nrWU));
    it->setText(2, QString::number(loc_spikes));
    it->setText(3, QString::number(loc_gaussians));
    it->setText(4, QString::number(loc_pulses));
    it->setText(5, QString::number(loc_triplets));
    it->setText(6, QString::number((double)(loc_spikes)/loc_nrWU, 'f', 2));
    it->setText(7, QString::number((double)(loc_gaussians)/loc_nrWU, 'f', 2));
    it->setText(8, QString::number((double)(loc_pulses)/loc_nrWU, 'f', 2));
    it->setText(9, QString::number((double)(loc_triplets)/loc_nrWU, 'f', 2));
    }
  else
    it->setText(1, i18n("None logged."));
  }

// create the Grand Total entry
// make sure that it's the last item
QListViewItem* last = 0;
QListViewItem* li(cwuList[SignalSummary]->firstChild());
while(li != 0)
  {
	last = li;
  li = last->itemBelow();
  }
// switch sorting off; otherwise "Grand Total" would be sorted into the list
cwuList[SignalSummary]->setSorting(-1);

if(last)
  {
  SigSumItem* grandtotal = new SigSumItem(cwuList[SignalSummary], last);
  	
  grandtotal->setText(0, i18n("Grand Total"));
  if(total_nrWU > 0)
    {
    grandtotal->setText(1, QString::number(total_nrWU));
    grandtotal->setText(2, QString::number(total_spikes));
    grandtotal->setText(3, QString::number(total_gaussians));
    grandtotal->setText(4, QString::number(total_pulses));
    grandtotal->setText(5, QString::number(total_triplets));
    grandtotal->setText(6,
      QString::number((double)(total_spikes)/total_nrWU, 'f', 2));
    grandtotal->setText(7,
      QString::number((double)(total_gaussians)/total_nrWU, 'f', 2));
    grandtotal->setText(8,
      QString::number((double)(total_pulses)/total_nrWU, 'f', 2));
    grandtotal->setText(9,
      QString::number((double)(total_triplets)/total_nrWU, 'f', 2));
    }
  else
    grandtotal->setText(1, i18n("None logged."));
  }

KApplication::restoreOverrideCursor();
}

/*------------------------------------------------------------------------ */
/** creates new or activate old skymap window */
SkyMap* CompletedWUList::openSkymapWindow()
{
if(SKYMAP_LOADED == true)
	{
 	if(sm->isNull() == false && SKYMAP_VISIBLE == false)
 		{
 		SKYMAP = new SkyMap(sm);
		SKYMAP_VISIBLE = true;		 		
		SKYMAP->show();
		}
	else if(SKYMAP->isActiveWindow() == false)
		{
		if(SKYMAP->isMinimized())
		  {
  		SKYMAP->hide();
      SKYMAP->showNormal();
      }
		SKYMAP->raise();
		}
	}
else
 	{
 	KMessageBox::sorry(topLevelWidget(), i18n("Couldn't load the pixmap file."));
	SKYMAP = 0;
 	}

return(SKYMAP);
}


/*------------------------------------------------------------------------ */
/** processes the commands from the right-click popup menu */
void CompletedWUList::handleTreeListPopupCommand(int id)
{
QListViewItem *it = treelist->currentItem();
SetiLoc *loc = 0;

// browse through the list
for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	if(loc->description() == it->text(0)) break;
	}

switch(id)
	{
	case ShowSkymap: // show on sky map
		{
		if(it)
			{
			SKYMAP = openSkymapWindow();
			if(SKYMAP_LOADED)
				{
				SKYMAP->clearMap();
   			// identify corresponding SetiLoc item
				if(it->text(0) == QString("SETI Locations"))
					addAllLoggedWU();
				else
					if(loc) addLoggedWU(loc);
				// update the skymap with added work units
				SKYMAP->update();
				}
			}
		break;
		}
	case ReloadLocations:
		{
		if(it)
			{
 			// identify corresponding SetiLoc item
      if(it->text(0) == i18n("SETI Locations"))
        fillTotalList();
      else
        if(loc) fillWUList(loc);
      }
		break;
		}
	}
}

/*------------------------------------------------------------------------ */
void CompletedWUList::refreshList(SetiLoc* loc)
{
switch(currentView)
  {
  case WUSummary:
  	fillTotalList();
  	break;
  case SignalSummary:
    fillSigSumList();
    break;
  default:
    QListViewItem* it = treelist->currentItem();
    if(it)
      {
      if(loc->description() == it->text(0))
        {
        QListViewItem* p = it->parent();
        if(p->text(0) == i18n("SETI Locations")) fillWUList(loc);
        if(p->text(0) == i18n("Returned Signals")) fillSigList(loc);
        }
      }
    break;
  }

emit workUnitLogged();
}

/*------------------------------------------------------------------------ */
/** adds all logged WUs of a specific SETI location to the skymap */
void CompletedWUList::addLoggedWU(SetiLoc *loc)
{
QString csv_file(loc->logDirectory() + "/SETILog.csv");
CSVDataBase csv((const char *)csv_file);

if(csv.open(IO_ReadOnly))
	{
	KApplication::setOverrideCursor(WaitCursor);
	int i(1);
	QString entry;
	double ra(0.0), dec(0.0);
	int h, m;
	do
		{
		entry = csv.readItem("start_ra", i);
		if(!entry.isEmpty())
			{
			entry = csv.readItem("start_ra", i);
			ra = entry.toDouble();
			entry = csv.readItem("start_dec", i);
			dec = entry.toDouble();
			entry = csv.readItem("time_recorded", i);
			h = entry.find('(');
			m = entry.find(')');
			entry = entry.mid(h+1, m-h-1);
			}
		SKYMAP->addLocation(loc, entry, ra, dec);
		i++;
		} while(!csv.atEnd());
	csv.close();
	KApplication::restoreOverrideCursor();
	}
}

/*------------------------------------------------------------------------ */
/** adds all logged WUs of all SETI locations to the skymap */
void CompletedWUList::addAllLoggedWU()
{
SetiLoc *loc;
for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	addLoggedWU(loc);
	}
}

/*------------------------------------------------------------------------ */
/** handles right-click mouse clicks in WU list */
void CompletedWUList::handleWUListPopupCommand(int id)
{
QListViewItem *tree_it = treelist->currentItem();
QListViewItem *wu_it = cwuList[WUList]->currentItem();
int a, b, c, d, e, f;
char sign;
double ra(0.0), dec(0.0);

SetiLoc *loc = 0;

// browse through the list
for(loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	if(loc->description() == tree_it->text(0)) break;
	}

if(loc)
	{
	sscanf(wu_it->text(8),"%d hr %d min %d sec RA %c %d deg %d' %d''",
					&a, &b, &c, &sign, &d, &e, &f);
	ra = (double)(a + b/60.0 + c/3600.0);
	dec = (double)(d + e/60.0 + f/3600.0);
	if(sign == '-') dec = -dec;
	SKYMAP = openSkymapWindow();
	if(SKYMAP_LOADED)
		{
		SKYMAP->addLocation(loc, wu_it->text(9), ra, dec);
		SKYMAP->repaint();
		}
	}
}

/*------------------------------------------------------------------------ */
/**  */
void CompletedWUList::toggleSorting(int column)
{
switch(currentView)
  {
  case WUList:
    sortorder = !sortorder;
    if(cwuList[WUList]) cwuList[WUList]->setSorting(column, sortorder);
    break;
  case SignalList:
    sortorder = !sortorder;
    if(cwuList[SignalList]) cwuList[SignalList]->setSorting(column, sortorder);
    break;
  default:
    if(cwuList[currentView]) cwuList[currentView]->setSorting(-1, false);
    break;
  }
}

#include "completedwulist.moc"
