/* 
 *  Copyright (C) 1999-2001 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program may be distributed under the terms of the Q Public
 * License as defined by Trolltech AS of Norway and appearing in the
 * file LICENSE.QPL included in the packaging of this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */


#include <qdir.h>
#include <stdlib.h>
#include <fnmatch.h>
#include "misc.h"

#include "cvsdir.h"


class CvsIgnoreList : public QStrList
{
public:
    CvsIgnoreList(const QDir &dir);

    void addEntriesFromString(const char *str);
    void addEntriesFromFile(const char *name);
    bool matches(QFileInfo *fi);
};


CvsIgnoreList::CvsIgnoreList(const QDir &dir)
    : QStrList(true) // always make deep copies
{
    static const char *ignorestr = ". .. core RCSLOG tags TAGS RCS SCCS .make.state\
.nse_depinfo #* .#* cvslog.* ,* CVS CVS.adm .del-* *.a *.olb *.o *.obj\
*.so *.Z *~ *.old *.elc *.ln *.bak *.BAK *.orig *.rej *.exe _$* *$";

    addEntriesFromString(ignorestr);
    // TODO?: addEntriesFromFile($CVSROOT/CVSROOT/cvsignore)
    addEntriesFromFile(QDir::homeDirPath() + "/.cvsignore");
    addEntriesFromString(getenv("CVSIGNORE"));
    addEntriesFromFile(dir.absPath() + "/.cvsignore");
}


void CvsIgnoreList::addEntriesFromString(const char *str)
{
    QStrList tokens = splitLine(str);

    QStrListIterator it(tokens);
    for (; it.current(); ++it)
	{
	    if (qstrcmp(it.current(), "!") == 0)
		clear();
	    else
		append(it.current());
	}
}


void CvsIgnoreList::addEntriesFromFile(const char *name)
{
    char buf[512];
    FILE *f = fopen(name, "r");
    if (!f)
	return;

    while (fgets(buf, sizeof buf, f))
	{
	    QString line = buf;
	    addEntriesFromString(buf);
	}
    fclose(f);
}


bool CvsIgnoreList::matches(QFileInfo *fi)
{
    // Directories e.g. with the name core never match
    if (fi->isFile())
        return false;
    
    QStrListIterator it(*this);
    for (; it.current(); ++it)
	{
	    if (fnmatch(it.current(), fi->fileName(), FNM_PATHNAME) == 0)
		return true;
	}
    
    return false;
}


const QFileInfoList *CvsDir::entryInfoList() const
{
    CvsIgnoreList ignorelist(*this);
    const QFileInfoList *fulllist = QDir::entryInfoList();
    if (!fulllist)
        return 0;
    
    QFileInfoList &elist = const_cast<QFileInfoList&>(entrylist);
    elist.clear();

    QFileInfoListIterator it(*fulllist);
    for (; it.current(); ++it)
	if (!ignorelist.matches(it.current()))
	    elist.append(it.current());

    return &entrylist;
}


// Local Variables:
// c-basic-offset: 4
// End:
