/* 
 *  Copyright (C) 1999-2001 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program may be distributed under the terms of the Q Public
 * License as defined by Trolltech AS of Norway and appearing in the
 * file LICENSE.QPL included in the packaging of this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */


#include <qstrlist.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qstack.h>
#include <kapp.h>
#include <klocale.h>
#include "misc.h"
#include "cvsdir.h"

#include <sys/stat.h>
#include <unistd.h>
#include <time.h>

#include "updateview.h"
#include "updateview.moc"


class UpdateDirItem : public QListViewItem
{
public:
    UpdateDirItem( QListView *parent, QString dirname );
    UpdateDirItem( UpdateDirItem *parent, QString dirname );
 
    QString dirPath();
    void syncWithDirectory();
    void syncWithEntries();
    void updateItem(QString name, UpdateView::Status status, bool isdir);
    void updateEntriesItem(QString name, UpdateView::Status status,
                           QString rev, QString tagname);
#if QT_VERSION >= 200
    virtual QString key(int col, bool) const;
    virtual QString text(int col) const;
#else
    virtual const char *key(int col, bool) const;
    virtual const char *text(int col) const;
#endif
    virtual void setOpen(bool o);
    virtual void setup();
 
    void maybeScanDir(bool recursive);

private:
    void scanDirectory();
    QString m_dirname;
};


class UpdateViewItem : public QListViewItem
{
public:
    UpdateViewItem( QListViewItem *parent, QString filename );

    QString filePath();
    QString revision()
        { return m_revision; }
    
#if QT_VERSION >= 200
    virtual QString key(int col, bool) const;
    virtual QString text(int col) const;
#else
    virtual const char *key(int col, bool) const;
    virtual const char *text(int col) const;
#endif
    virtual void paintCell(QPainter *p, const QColorGroup &cg,
			   int col, int width, int align);

    void setStatus(UpdateView::Status status);
    void updateStatus(UpdateView::Status status);
    void setRevTag(QString rev, QString tag);
    void markUpdated(bool laststage, bool success);
    
private:
    QString m_filename;
    QString m_revision;
    QString m_tag;
    UpdateView::Status m_status;
};


UpdateDirItem::UpdateDirItem( UpdateDirItem *parent, QString dirname )
    : QListViewItem(parent)
{
    m_dirname = dirname;
}
 
 
UpdateDirItem::UpdateDirItem( QListView *parent, QString dirname )
    : QListViewItem( parent )
{
    m_dirname = dirname;
}


QString UpdateDirItem::dirPath()
{
    UpdateDirItem *diritem = static_cast<UpdateDirItem*>(parent());

    return parent()? diritem->dirPath() + m_dirname + "/" : QString("");
}                      


/**
 * Update the status of an item; if it doesn't exist yet, create new one
 */
void UpdateDirItem::updateItem(QString name, UpdateView::Status status, bool isdir)
{
    for (QListViewItem *item = firstChild(); item;
	 item = item->nextSibling() )
	{
	    if (item->text(0) == name)
		{
		    if (UpdateView::isDirItem(item))
                        ; // ignore
		    else
                        static_cast<UpdateViewItem*>(item)->setStatus(status);
		    return;
		}
	}

    // Not found, make new entry
    if (isdir)
        (void) new UpdateDirItem(this, name);
    else
        ( new UpdateViewItem(this, name) )->setStatus(status);
}


/**
 * Update the revision and tag of an item. Use status only to create
 * new items and for items which were NotInCVS.
 */
void UpdateDirItem::updateEntriesItem(QString name, UpdateView::Status status,
                                      QString rev, QString tagname)
{
    for (QListViewItem *item = firstChild(); item;
	 item = item->nextSibling() )
	{
	    if (item->text(0) == name)
		{
		    if (UpdateView::isDirItem(item))
                        ; // ignore
		    else
                        {
                            UpdateViewItem *viewitem = static_cast<UpdateViewItem*>(item);
                            viewitem->updateStatus(status);
                            viewitem->setRevTag(rev, tagname);
                        }
		    return;
		}
	}

    ( new UpdateViewItem(this, name) )->setStatus(status);
}


void UpdateDirItem::scanDirectory()
{
    if (!dirPath().isEmpty() && !QFile::exists(dirPath()))
        return;

    CvsDir dir( dirPath() );
    
    const QFileInfoList *files = dir.entryInfoList();
    if (files)
	{
	    QFileInfoListIterator it(*files);
	    for (; it.current(); ++it)
		{
		    if ( it.current()->fileName() == "."
			 || it.current()->fileName() == ".." )
			continue;
		    if ( it.current()->isDir() )
			(void) new UpdateDirItem(this, it.current()->fileName());
		    else
			(void) new UpdateViewItem(this, it.current()->fileName());
		}
	}
}


static const char *lastModifiedStr(const char *fname)
{
    struct stat st;
    if (lstat(fname, &st) != 0)
	return "";
    struct tm *tm_p = gmtime(&st.st_mtime);
    char *p = asctime(tm_p);
    p[24] = '\0';
    return p;
}


// Format of the CVS/Entries file:
//   /NAME/REVISION/TIMESTAMP[+CONFLICT]/OPTIONS/TAGDATE

void UpdateDirItem::syncWithEntries()
{
    char buf[512];
    QString name, rev, timestamp, options, tagdate;
    UpdateView::Status status;

    //    DEBUGOUT( "Reading entries" );
    FILE *f = fopen(dirPath() + "CVS/Entries", "r");
    if (!f)
	return;

    while (fgets(buf, sizeof buf, f))
	{
	    char *p, *nextp;
	    if (buf[0] != '/')
		continue;
	    if ( (nextp = strchr(buf+1, '/')) == 0)
		continue;
	    *nextp = '\0';
	    name = QString(buf+1); p = nextp+1;
	    //	    DEBUGOUT( "Name: " << name );
	    if ( (nextp = strchr(p, '/')) == 0)
		continue;
	    *nextp = '\0';
	    rev = QString(p); p = nextp+1;
	    //	    DEBUGOUT( "Rev: " << rev );
	    if ( (nextp = strchr(p, '/')) == 0)
		continue;
	    *nextp = '\0';
	    timestamp = QString(p); p = nextp+1;
	    //	    DEBUGOUT( "Timestamp: " << timestamp );
	    if ( (nextp = strchr(p, '/')) == 0)
		continue;
	    *nextp = '\0';
	    options = QString(p); p = nextp+1;
	    //	    DEBUGOUT( "Opt: " << options );
	    if ( (nextp = strchr(p, '\n')) == 0)
		continue;
	    *nextp = '\0';
	    tagdate = QString(p); p = nextp+1;
	    //	    DEBUGOUT( "Tagdate: " << tagdate );

	    if (rev == "0")
		status = UpdateView::LocallyAdded;
	    else if (rev.length() > 2 && rev[0] == '-')
                {
                    status = UpdateView::LocallyRemoved;
                    rev.remove(0, 1);
                }
	    else if (timestamp.find('+') != -1)
		status = UpdateView::Conflict;
	    else if (timestamp != lastModifiedStr(dirPath() + name))
		status = UpdateView::LocallyModified;
	    else
		status = UpdateView::Unknown;
            //	    	    DEBUGOUT( "Timestamp: " << timestamp );
            //	    	    DEBUGOUT( "Mod. date: " << (lastModifiedStr(dirPath() + '/' + name)) );
            //	    DEBUGOUT( "Adding from Entries: " << name );

	    updateEntriesItem(name, status, rev, tagdate);
	}
    fclose(f);
}


void UpdateDirItem::syncWithDirectory()
{
    // Do not use CvsDir here, because CVS/Entries may
    // contain files which are in .cvsignore (stupid
    // idea, but that's possible...)
    QDir dir( dirPath(), 0, QDir::Name, QDir::All|QDir::Hidden );
    //    DEBUGOUT("Syncing" << dirPath().data() );

    const QFileInfoList *files = dir.exists()? dir.entryInfoList() : 0;

    for (QListViewItem *item = firstChild(); item;
         item = item->nextSibling() )
        {
            // Look if file still exists. We never remove directories!
            bool exists = false;
            if (UpdateView::isDirItem(item))
                exists = true;
            else if (files)
                {
                    QFileInfoListIterator it(*files);
                    for ( ; it.current(); ++it)
                        {
                            //                            DEBUGOUT("Against: " << it.current()->fileName());
                        if (it.current()->fileName() == item->text(0))
                            {
                                exists = true;
                                break;
                            }
                        }
                }
            if (!exists)
                {
                    UpdateViewItem *viewitem = static_cast<UpdateViewItem*>(item);
                    viewitem->setStatus(UpdateView::Removed);
                    viewitem->setRevTag("", "");
                }
        }
}


/**
 * Read in the content of the directory. If recursive is false, this
 * is shallow, otherwise all child directories are scanned recursively.
 */
void UpdateDirItem::maybeScanDir(bool recursive)
{
    if (!firstChild())
        {
            scanDirectory();
            syncWithEntries();
        }

    if (recursive)
        {
            for ( QListViewItem *item = firstChild(); item;
                  item = item->nextSibling() )
                if (UpdateView::isDirItem(item))
                    static_cast<UpdateDirItem*>(item)->maybeScanDir(true);
        }
}


void UpdateDirItem::setOpen(bool o)
{
    if ( o )
        maybeScanDir(false);
    
    QListViewItem::setOpen( o );
}
 

#if QT_VERSION >= 200
QString UpdateDirItem::key(int col, bool) const
#else
const char *UpdateDirItem::key(int col, bool) const
#endif
{
    static QString tmp;
    switch (col)
	{
	case 0: //return m_dirname;
	case 1: return tmp = QString("0") + m_dirname;
        default:return "";
	}
}


#if QT_VERSION >= 200
QString UpdateDirItem::text(int col) const
#else
const char *UpdateDirItem::text(int col) const
#endif
{
    switch (col)
	{
	case 0:  return m_dirname;
	default: return "";
	}
}       


void UpdateDirItem::setup()
{
    setExpandable(true);
    QListViewItem::setup();
}
 

UpdateViewItem::UpdateViewItem( QListViewItem *parent, QString filename )
    : QListViewItem(parent, i18n("Unknown")) 
{
    m_status = UpdateView::NotInCVS;
    m_filename = filename;
    m_revision = "";
    m_tag = "";
}


QString UpdateViewItem::filePath()
{
    UpdateDirItem *diritem = static_cast<UpdateDirItem*>(parent());
    return diritem->dirPath() + m_filename;
}                      


void UpdateViewItem::setStatus(UpdateView::Status newstatus)
{
    if (newstatus != m_status)
        {
            m_status = newstatus;
            repaint();
        }
}


void UpdateViewItem::updateStatus(UpdateView::Status newstatus)
{
    // Heuristics...
    if (m_status == UpdateView::NotInCVS ||
        m_status == UpdateView::LocallyRemoved ||
        newstatus == UpdateView::LocallyAdded ||
        newstatus == UpdateView::LocallyRemoved ||
        newstatus == UpdateView::Conflict)
        {
            m_status = newstatus;
            repaint();
        }
}


void UpdateViewItem::setRevTag(QString rev, QString tag)
{
    m_revision = rev;

    if (tag.length() == 20 && tag[0] == 'D' && tag[5] == '.'
        && tag[8] == '.' && tag[11] == '.' && tag[14] == '.'
        && tag[17] == '.')
        {
            m_tag = tag.mid(1, 4);
            m_tag += "/";
            m_tag += tag.mid(6, 2);
            m_tag += "/";
            m_tag += tag.mid(9, 2);
            m_tag += " ";
            m_tag += tag.mid(12, 2);
            m_tag += ":";
            m_tag += tag.mid(15, 2);
            m_tag += ":";
            m_tag += tag.mid(18, 2);
        }
    else if (tag.length() > 1 && tag[0] == 'T')
        m_tag = tag.mid(1, tag.length()-1);
    else
        m_tag = tag;

    widthChanged();
    repaint();
}


void UpdateViewItem::markUpdated(bool laststage, bool success)
{
    UpdateView::Status newstatus = m_status;
    
    if (laststage)
        {
            if (m_status == UpdateView::Indefinite)
                newstatus = success? UpdateView::UpToDate : UpdateView::Unknown ;
        }
    else
        newstatus = UpdateView::Indefinite;

    setStatus(newstatus);
}


#if QT_VERSION >= 200
QString UpdateViewItem::key(int col, bool) const
#else
const char *UpdateViewItem::key(int col, bool) const
#endif
{
    static QString tmp;
    QString prefix;

    switch (col)
	{
	case 0:
            // Put ordinary files behind all directories
            return tmp = QString("1") + m_filename;
#if 0
            if (isExpandable())
                prefix = "1";
            else
                prefix = "2";
	    return tmp = prefix + m_filename;
#endif
	case 1:
	    // We want to have a kind of priority order when
	    // sorted by 'status'
	    switch (m_status)
		{
		case UpdateView::Conflict:
		    prefix = "1"; break;
		case UpdateView::LocallyAdded:
		    prefix = "2"; break;
		case UpdateView::LocallyRemoved:
		    prefix = "3"; break;
		case UpdateView::LocallyModified:
		    prefix = "4"; break;
		case UpdateView::Updated:
		case UpdateView::NeedsUpdate:
		case UpdateView::Patched:
                case UpdateView::Removed:
		case UpdateView::NeedsPatch:
		case UpdateView::NeedsMerge:
		    prefix = "5"; break;
		case UpdateView::NotInCVS:
		    prefix = "6"; break;
		default:  prefix = "7"; 
		}
	    return tmp = prefix + m_filename;
        case 2:
            return m_revision; // First approximation...
        case 3:
            return m_tag;
	default:
	    return "";
	}
}


#if QT_VERSION >= 200
QString UpdateViewItem::text(int col) const
#else
const char *UpdateViewItem::text(int col) const
#endif
{
    switch (col)
	{
	case 0:
	    return m_filename;
	case 1:
	    switch (m_status)
		{
		case UpdateView::LocallyModified:
		    return i18n("Locally Modified"); 
		case UpdateView::LocallyAdded:
		    return i18n("Locally Added"); 
		case UpdateView::LocallyRemoved:
		    return i18n("Locally Removed"); 
		case UpdateView::NeedsUpdate:
		    return i18n("Needs Update"); 
		case UpdateView::NeedsPatch:
		    return i18n("Needs Patch"); 
		case UpdateView::NeedsMerge:
		    return i18n("Needs Merge"); 
		case UpdateView::UpToDate:
		    return i18n("Up to date");
		case UpdateView::Conflict:
		    return i18n("Conflict");
		case UpdateView::Updated:
		    return i18n("Updated"); 
		case UpdateView::Patched:
		    return i18n("Patched"); 
		case UpdateView::Removed:
		    return i18n("Removed"); 
		case UpdateView::NotInCVS:
		    return i18n("Not in CVS");
		default:  return i18n("Unknown");
		}
	case 2:
	    return m_revision;
        case 3:
            return m_tag;
	default:
	    return "";
	}
}       



void UpdateViewItem::paintCell(QPainter *p, const QColorGroup &cg,
			       int col, int width, int align)
{
    QColor color =
	(m_status == UpdateView::Conflict)? QColor(255, 100, 100)
	: (m_status == UpdateView::LocallyModified ||
	   m_status == UpdateView::LocallyAdded  ||
	   m_status == UpdateView::LocallyRemoved)? QColor(190, 190, 237)
	: (m_status == UpdateView::Patched ||
	   m_status == UpdateView::Updated ||
           m_status == UpdateView::Removed ||
	   m_status == UpdateView::NeedsPatch ||
	   m_status == UpdateView::NeedsUpdate)? QColor(255, 240, 190)
	: cg.base();
#if QT_VERSION >= 200
    QColorGroup mycg(cg);
    mycg.setBrush(QColorGroup::Base, color);
#else
    QColorGroup mycg(cg.foreground(), cg.background(), cg.light(), cg.dark(),
		     cg.mid(), cg.text(), color);
#endif
    QListViewItem::paintCell(p, mycg, col, width, align);
}


UpdateView::UpdateView(QWidget *parent, const char *name)
    : QListView(parent, name)
{
    setAllColumnsShowFocus(true);
#if QT_VERSION >= 200
    setSelectionMode(Extended);
#endif
    setMultiSelection(true);
    addColumn(i18n("File name"));
    addColumn(i18n("Status"));
    addColumn(i18n("Revision"));
    addColumn(i18n("Tag/Date"));

    QFontMetrics fm( fontMetrics() );

    int width = 0;
    width = QMAX(width, fm.width(i18n("Updated")));
    width = QMAX(width, fm.width(i18n("Patched")));
    width = QMAX(width, fm.width(i18n("Removed")));
    width = QMAX(width, fm.width(i18n("Needs Update")));
    width = QMAX(width, fm.width(i18n("Needs Patch")));
    width = QMAX(width, fm.width(i18n("Needs Merge")));
    width = QMAX(width, fm.width(i18n("Locally Added")));
    width = QMAX(width, fm.width(i18n("Locally Removed")));
    width = QMAX(width, fm.width(i18n("Locally Modified")));
    width = QMAX(width, fm.width(i18n("Up to date")));
    width = QMAX(width, fm.width(i18n("Conflict")));
    width = QMAX(width, fm.width(i18n("Not in CVS")));
    width = QMAX(width, fm.width(i18n("Unknown")));
    
    setColumnWidth(1, width+5);
}


bool UpdateView::isDirItem(QListViewItem *item)
{
    return qstrcmp(item->text(1), "") == 0;
}


bool UpdateView::hasSingleSelection()
{
    bool selfound = false;
    QStack<QListViewItem> s;

    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
	    if (item->firstChild())
		    s.push(item->firstChild());

	    if (item->isSelected())
		{
		    if (selfound || item->isExpandable())
			return false;
		    selfound = true;
		}
	}
    return selfound;
}


void UpdateView::getSingleSelection(QString *filename, QString *revision)
{
    QStack<QListViewItem> s;

    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
	    if (item->firstChild())
		s.push(item->firstChild());
	    else if (item->isSelected())
                {
                    UpdateViewItem *viewitem = static_cast<UpdateViewItem*>(item);
                    *filename = viewitem->filePath();
                    if (revision)
                        *revision = viewitem->revision();
                }
	}
}


QStrList UpdateView::multipleSelection()
{
    QStrList res(true);
    QStack<QListViewItem> s;

    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
	    if (item->firstChild())
		s.push(item->firstChild());

	    if (item->isSelected())
		{
		    if (isDirItem(item))
			{
			    QString dirpath
				= static_cast<UpdateDirItem*>(item)->dirPath();
			    if (!dirpath.isEmpty())
				dirpath.truncate(dirpath.length()-1);
                            else
                                dirpath = ".";
			    res.append(dirpath);
			}
		    else
			res.append( static_cast<UpdateViewItem*>(item)->filePath() );
		}
	}
    return res;
}


void UpdateView::deselectAll()
{
    QStack<QListViewItem> s;
    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
	    if (item->firstChild())
		s.push(item->firstChild());
	    item->setSelected(false);
	}
    
    triggerUpdate();
}


void UpdateView::unfoldTree()
{
    QApplication::setOverrideCursor(waitCursor);
    
    QStack<QListViewItem> s;
    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
	    if (isDirItem(item))
                item->setOpen(true);
            if (item->firstChild())
		s.push(item->firstChild());
            qApp->processEvents();
	}
    
    QApplication::restoreOverrideCursor();
}


/**
 * Clear the tree view and insert the directory dirname
 * into it as the new root item
 */
void UpdateView::openDirectory(QString dirname)
{
    clear();
    ( new UpdateDirItem(this, dirname) )->setOpen(true);
}


/**
 * Start a job. We want to be able to change the status field
 * correctly afterwards, so we have to remember the current
 * selection (which the user may change during the update).
 * In the recursive case, we collect all relevant directories.
 * Furthermore, we have to change the status fields to 'Unknown'.
 */
void UpdateView::prepareJob(bool recursive, Action action)
{
    act = action;

    // Scan recursively all entries - there's no way around this here
    if (recursive)
        static_cast<UpdateDirItem*>(firstChild())->maybeScanDir(true);
    
    rememberSelection(recursive);
    if (act != Add)
        markUpdated(false, false);
}


/**
 * Finishes a job. What we do depends a bit on
 * whether the command was successful or not.
 */
void UpdateView::finishJob(bool success)
{
    if (act != Add)
        markUpdated(true, success);
    syncSelection();
}


/**
 * Marking non-selected items in a directory updated (as a consequence
 * of not appearing in 'cvs update' output) is done in two steps: In the
 * first, they are marked as 'indefinite', so that their status on the screen
 * isn't misrepresented. In the second step, they are either set
 * to 'UpToDate' (success=true) or 'Unknown'.
 */
void UpdateView::markUpdated(bool laststage, bool success)
{
    QListIterator<QListViewItem> it(relevantSelection);
    for ( ; it.current(); ++it)
        if (isDirItem(it.current()))
            {
                for (QListViewItem *item = it.current()->firstChild(); item;
                     item = item->nextSibling() )
                    if (!isDirItem(item))
                        static_cast<UpdateViewItem*>(item)->markUpdated(laststage, success);
            }
        else
            static_cast<UpdateViewItem*>(it.current())->markUpdated(laststage, success);
}


/**
 * Remember the selection, see prepareJob()
 */
void UpdateView::rememberSelection(bool recursive)
{
    // Collect all selected dir and file items into relevantSelection

    QList<QListViewItem> shallowItems, deepItems;

    QStack<QListViewItem> s;
    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
            if (item->firstChild())
                s.push(item->firstChild());
            if (isSelected(item))
                shallowItems.append(item);
	}

    // In the recursive case, we add all directories from the hierarchies
    // under the selected directories.
    
    if (recursive)
        {
            QListIterator<QListViewItem> it(shallowItems);
            for ( ; it.current(); ++it)
                if (isDirItem(it.current()))
                    for ( QListViewItem *item = it.current()->firstChild(); item;
                          item = item->nextSibling()? item->nextSibling() : s.pop() )
                        {
                            if (item->firstChild())
                                s.push(item->firstChild());
                            if (isDirItem(item))
                                deepItems.append(item);
                        }
        }

    // Collect everything together, and avoid duplicates:
    
    relevantSelection.clear();
    QListIterator<QListViewItem> it1(shallowItems);
    for ( ; it1.current(); ++it1)
        if (!relevantSelection.contains(it1.current()))
            relevantSelection.append(it1.current());
    QListIterator<QListViewItem> it2(deepItems);
    for ( ; it2.current(); ++it2)
        if (!relevantSelection.contains(it2.current()))
            relevantSelection.append(it2.current());
}


/**
 * Use the remembered selection to resynchronize
 * with the actual directory and Entries content.
 */
void UpdateView::syncSelection()
{
    QList<UpdateDirItem> dirs;
    
    QListIterator<QListViewItem> it1(relevantSelection);
    for ( ; it1.current(); ++it1)
	{
            UpdateDirItem *diritem = 0;
            if (isDirItem(it1.current()))
                diritem = static_cast<UpdateDirItem*>(it1.current());
            else if (it1.current()->parent())
                diritem = static_cast<UpdateDirItem*>(it1.current()->parent());
            if (diritem && !dirs.contains(diritem))
                dirs.append(diritem);
        }

    QApplication::setOverrideCursor(waitCursor);
    
    QListIterator<UpdateDirItem> it2(dirs);
    for ( ; it2.current(); ++it2)
	{
            it2.current()->syncWithDirectory();
            it2.current()->syncWithEntries();
            qApp->processEvents();
        }
    
    QApplication::restoreOverrideCursor();
}


/**
 * Process one line from the output of 'cvs update'. If parseAsStatus
 * is true, it is assumed that the output is from a command
 * 'cvs update -n', i.e. cvs actually changes no files.
 */
void UpdateView::processUpdateLine(QString str)
{
    if (str.length() > 2 && str[1] == ' ')
        {
#if QT_VERSION >= 200
            QChar statuschar = str[0];
#else
            char statuschar = str[0];
#endif
            Status status = UpdateView::Unknown;
            if (statuschar == 'C')
                status = UpdateView::Conflict;
            else if (statuschar == 'A')
                status = UpdateView::LocallyAdded;
            else if (statuschar == 'R')
                status = UpdateView::LocallyRemoved;
            else if (statuschar == 'M')
                status = UpdateView::LocallyModified;
            else if (statuschar == 'U')
                status = (act==UpdateNoAct)?
                    UpdateView::NeedsUpdate : UpdateView::Updated;
            else if (statuschar == 'P')
                status = (act==UpdateNoAct)?
                    UpdateView::NeedsPatch : UpdateView::Patched;
            else if (statuschar == '?')
                status = UpdateView::NotInCVS;
            else
                return;
            updateItem(str.right(str.length()-2), status, false);
        }
    else if (str.left(21) == "cvs server: Updating " ||
             str.left(21) == "cvs update: Updating ")
        updateItem(str.right(str.length()-21), Indefinite, true);
}


void UpdateView::updateItem(QString name, Status status, bool isdir)
{
    // bla -> dirpath = "", filename = "bla"
    // bla/foo -> dirpath = "bla/", filename = "foo"

    if (isdir && name == ".")
        return;
    
    QFileInfo fi(name);
    QString dirpath(fi.dirPath());
    QString fileName(fi.fileName());

    if (dirpath == ".") 
    	dirpath = "";
    else
	dirpath += '/';

    QStack<QListViewItem> s;
    for ( QListViewItem *item = firstChild(); item;
	  item = item->nextSibling()? item->nextSibling() : s.pop() )
	{
	    if (item->firstChild())
		{
		    UpdateDirItem *diritem = static_cast<UpdateDirItem*>(item);
                    //		    DEBUGOUT( "Compare " << diritem->dirPath() << "." );
		    if (diritem->dirPath() == dirpath)
			{
			    diritem->updateItem(fileName, status, isdir);
			    return;
			}
		    else
			s.push(item->firstChild());
		}
	}
    
}


// Local Variables:
// c-basic-offset: 4
// End:
