/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */

#include "kbabel.h"

#include "kbabelpref.h"

#include <qdragobject.h>
#include <qlineedit.h>
#include <qhbox.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qtimer.h>

#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <ktoolbar.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kaccelmenu.h>
#include <kurl.h>
#include <kdialogbase.h>
#include <khelpmenu.h>
#include <kprogress.h>
#include <kled.h>
#include <kmessagebox.h>

#include "resources.h"
#include "catalog.h"
#include "catalogmanager.h"

#include "version.h"

#define ID_STATUS_TOTAL 1
#define ID_STATUS_CURRENT 2
#define ID_STATUS_FUZZY 3
#define ID_STATUS_UNTRANS 4
#define ID_STATUS_STATUS 5
#define ID_STATUS_PROGRESS 6
#define ID_STATUS_PROGRESSLABEL 7
#define ID_STATUS_EDITMODE 8
#define ID_STATUS_READONLY 9

#define ID_NAVBAR 1

#define ID_FILE_SAVE 3
#define ID_FILE_SAVEAS 4

#define  ID_EDIT_CUT 0
#define  ID_EDIT_COPY 1
#define  ID_EDIT_PASTE 2
#define  ID_EDIT_SELECT 3
#define  ID_EDIT_FIND 5
#define  ID_EDIT_FINDNEXT 6
#define  ID_EDIT_REPLACE 7
#define  ID_EDIT_REMFUZZY 9
#define  ID_EDIT_HEADER 11

#define ID_SEARCH_CAT 0
#define ID_SEARCH_COM 1
#define ID_SEARCH_AUX 2
#define ID_SEARCH_SELECTION_CAT 4
#define ID_SEARCH_SELECTION_COM 5
#define ID_SEARCH_SELECTION_AUX 6

#define ID_GO_PREV 0
#define ID_GO_NEXT 1
#define ID_GO_GOTO 2
#define ID_GO_TOP 4
#define ID_GO_BOTTOM 5
#define ID_GO_PREVFUZUNTRANS 7
#define ID_GO_NEXTFUZUNTRANS 8
#define ID_GO_PREVFUZZY 9
#define ID_GO_NEXTFUZZY 10
#define ID_GO_PREVUNTRANS 11
#define ID_GO_NEXTUNTRANS 12
#define ID_GO_PREVERROR 14
#define ID_GO_NEXTERROR 15


#define ID_TOOLS_SYNTAXCHECK 3

#define ID_TB_OPEN 50
#define ID_TB_SAVE 51
#define ID_TB_CUT 52
#define ID_TB_COPY 53
#define ID_TB_PASTE 54
#define ID_TB_FIND 55
#define ID_TB_STOP 56
#define ID_TB_SEARCH 57
#define ID_TB_WHATSTHIS 58
#define ID_TB_CATMAN 59

#define ID_NB_PREV 60
#define ID_NB_NEXT 61
#define ID_NB_GOTO 62
#define ID_NB_TOP 63
#define ID_NB_BOTTOM 64
#define ID_NB_PREVFUZZY 65
#define ID_NB_NEXTFUZZY 66
#define ID_NB_PREVUNTRANS 67
#define ID_NB_NEXTUNTRANS 68
#define ID_NB_PREVFUZUNTRANS 69
#define ID_NB_NEXTFUZUNTRANS 70
#define ID_NB_PREVERROR 71
#define ID_NB_NEXTERROR 72


// maximum number of recent files
#define MAX_RECENT 5


QStringList KBabel::_recentFiles;
QList<KBabelPreferences> KBabel::prefDialogs;

CatalogManager* KBabel::_catalogManager=0;
CatManSettings KBabel::_catManSettings;



KBabel::KBabel()
{
   Catalog* catalog=new Catalog();
   init(catalog);
}

KBabel::KBabel(Catalog* catalog)
{
    init(catalog);
}

void KBabel::init(Catalog* catalog)
{
    _prefDialog=0;
    prefDialogs.setAutoDelete(true);

    m_view=new KBabelView(catalog,this);
    m_accelKeys=new KAccel(this);

    // accept dnd
    setAcceptDrops(true);

    // tell the KTMainWindow that this is indeed the main widget
    setView(m_view);

    // setup our menubars and toolbars
    setupMenuBars();
    setupToolBars();
    setupStatusBar();

    m_accelKeys->insertItem(i18n("Toggle Editmode"),"Toggle Editmode","Insert");
    m_accelKeys->connectItem("Toggle Editmode",this,SLOT(toggleEditMode()));

    m_accelKeys->readSettings();


    connect(catalog,SIGNAL(signalNumberOfFuzziesChanged(uint)),this
            ,SLOT(setNumberOfFuzzies(uint)));
    connect(catalog,SIGNAL(signalNumberOfUntranslatedChanged(uint)),this
            ,SLOT(setNumberOfUntranslated(uint)));
    connect(catalog,SIGNAL(signalTotalNumberChanged(uint)),this
            ,SLOT(setNumberOfTotal(uint)));
    connect(catalog,SIGNAL(signalProgress(int)),progressBar,SLOT(setValue(int)));
    connect(catalog,SIGNAL(signalClearProgressBar()),this,SLOT(clearProgressBar()));
    connect(catalog,SIGNAL(signalResetProgressBar(QString,int))
           ,this,SLOT(prepareProgressBar(QString,int)));
    connect(catalog,SIGNAL(signalFileOpened(bool)),this,SLOT(enableDefaults(bool)));
    connect(catalog,SIGNAL(signalFileOpened(bool)),m_view,SLOT(newFileOpened(bool)));

    if(!catalog->currentURL().isEmpty())
    {
       enableDefaults(catalog->isReadOnly());
    }


    // allow the view to change the statusbar and caption
    connect(m_view, SIGNAL(signalChangeStatusbar(const QString&)),
            this,   SLOT(changeStatusbar(const QString&)));
    connect(m_view, SIGNAL(signalChangeCaption(const QString&)),
            this,   SLOT(changeCaption(const QString&)));
    connect(m_view,SIGNAL(signalFirstDisplayed(bool)),this
           ,SLOT(firstEntryDisplayed(bool)));
    connect(m_view,SIGNAL(signalLastDisplayed(bool)),this
           ,SLOT(lastEntryDisplayed(bool)));
    connect(m_view,SIGNAL(signalFuzzyDisplayed(bool)),this
           ,SLOT(fuzzyDisplayed(bool)));
    connect(m_view,SIGNAL(signalUntranslatedDisplayed(bool)),this
           ,SLOT(untranslatedDisplayed(bool)));
    connect(m_view,SIGNAL(signalDisplayed(uint)),this
           ,SLOT(displayedEntryChanged(uint)));
    connect(m_view,SIGNAL(signalFuzzyAfterwards(bool)),this
            ,SLOT(hasFuzzyAfterwards(bool)));
    connect(m_view,SIGNAL(signalFuzzyInFront(bool)),this
            ,SLOT(hasFuzzyInFront(bool)));
    connect(m_view,SIGNAL(signalUntranslatedAfterwards(bool)),this
            ,SLOT(hasUntranslatedAfterwards(bool)));
    connect(m_view,SIGNAL(signalUntranslatedInFront(bool)),this
            ,SLOT(hasUntranslatedInFront(bool)));
    connect(m_view,SIGNAL(signalErrorAfterwards(bool)),this
            ,SLOT(hasErrorAfterwards(bool)));
    connect(m_view,SIGNAL(signalErrorInFront(bool)),this
            ,SLOT(hasErrorInFront(bool)));


    connect(m_view,SIGNAL(signalSearchActive(bool)),this,SLOT(enableStop(bool)));
    connect(m_view,SIGNAL(signalEnableCatSearch(bool)),this,SLOT(enableCatSearch(bool)));
    connect(m_view,SIGNAL(signalEnableComSearch(bool)),this,SLOT(enableComSearch(bool)));
    connect(m_view,SIGNAL(signalEnableAuxSearch(bool)),this,SLOT(enableAuxSearch(bool)));

    connect(m_view,SIGNAL(signalSearchResultsShown()),this,SLOT(checkMenuItemSearchResults()));
    connect(m_view,SIGNAL(signalProgress(int)),progressBar,SLOT(setValue(int)));
    connect(m_view,SIGNAL(signalClearProgressBar()),this,SLOT(clearProgressBar()));
    connect(m_view,SIGNAL(signalResetProgressBar(QString,int))
           ,this,SLOT(prepareProgressBar(QString,int)));

    restoreSettings();

    KConfig* config=KGlobal::config();

    KConfigGroupSaver saver(config,"KBabel");
    if(!config->hasKey("Version"))
    {
        KMessageBox::information(0,i18n("You haven't run KBabel before.\n\
To let KBabel work correctly you have to enter some\n\
information in the preferences dialog first."));

        QTimer::singleShot(1,this,SLOT(optionsPreferences()));
    }

    config->writeEntry("Version",VERSION);
}

KBabel::~KBabel()
{
   if(_prefDialog)
   {
      prefDialogs.remove(_prefDialog);
   }
}


void KBabel::setSettings(SearchSettings searchOpts, EditorSettings editOpts
                       , SaveSettings saveOpts,IdentitySettings idOpts)
{
   m_view->setSettings(searchOpts);
   m_view->setSettings(editOpts);
   m_view->catalog()->setSettings(saveOpts);
   m_view->catalog()->setSettings(idOpts);
}

void KBabel::setSettings(SearchSettings searchOpts, EditorSettings editOpts)
{
   m_view->setSettings(searchOpts);
   m_view->setSettings(editOpts);
}



void KBabel::setupMenuBars()
{
    m_file    = new KAccelMenu(m_accelKeys,this);
    m_edit    = new KAccelMenu(m_accelKeys,this);
    m_go      = new KAccelMenu(m_accelKeys,this);
    m_tools   = new KAccelMenu(m_accelKeys,this);
    m_search  = new KAccelMenu(m_accelKeys,this);
    m_options = new QPopupMenu(this);
    m_help    = new QPopupMenu(this);

    m_recentFiles = new QPopupMenu(this);
    connect(m_recentFiles,SIGNAL(activated(int)),this,SLOT(openRecent(int)));


    // first, the File menu
    m_file->insItem(BarIcon("fileopen"), i18n("&Open..."),"Open",
                            this, SLOT(fileOpen()),KAccel::Open);
    m_file->insertItem(i18n("O&pen Recent"),m_recentFiles);
    m_file->insertSeparator(-1);

    m_file->insItem(BarIcon("filefloppy"), i18n("&Save"),"Save",
                            this, SLOT(fileSave()),KAccel::Save);
    m_file->insertItem(i18n("S&ave As..."),
                       this, SLOT(fileSaveAs()));
    m_file->insertSeparator(-1);
    m_file->insertItem(i18n("New &View"),
                       this, SLOT(fileNewView()));
    m_file->insertItem(i18n("New &Window"),
                       this, SLOT(fileNewWindow()));

    m_file->insertSeparator(-1);
    m_file->insItem(BarIcon("exit"), i18n("&Quit"),"Quit",
                            this, SLOT(quit()),KAccel::Quit);

    // initaly deactivate some items
    m_accelKeys->setItemEnabled("Save",false);
    m_file->setItemEnabled(m_file->idAt(ID_FILE_SAVE),false);
    m_file->setItemEnabled(m_file->idAt(ID_FILE_SAVEAS),false);


    // next, the edit-menu
    m_edit->insItem(BarIcon("editcut"), i18n("C&ut"),"Cut",
                         m_view, SLOT(textCut()),KAccel::Cut);
    m_edit->insItem(BarIcon("editcopy"), i18n("&Copy"),"Copy",
                         m_view, SLOT(textCopy()),KAccel::Copy);
    m_edit->insItem(BarIcon("editpaste"), i18n("&Paste"),"Paste",
                         m_view, SLOT(textPaste()),KAccel::Paste);
    m_edit->insItem(i18n("Select &All"),"Select All",m_view,SLOT(selectAll()));
    m_edit->insertSeparator(-1);

    m_edit->insItem(BarIcon("find"),i18n("&Find..."),"Find",
                         this, SLOT(find()),KAccel::Find);
    m_edit->insItem(i18n("Find &Next"),"Find Next",
                         this, SLOT(findNext()),"F3");
    m_edit->insItem(i18n("&Replace..."),"Replace",
                         this, SLOT(replace()),KAccel::Replace);
    m_edit->insertSeparator(-1);
    m_edit->insItem(i18n("Un&set Fuzzy Status"),"Remove Fuzzy",
                    m_view,SLOT(removeFuzzyStatus()),"CTRL+U");
    m_edit->insertSeparator(-1);
    m_edit->insItem(i18n("&Edit Header..."),"Edit Header",
                          m_view, SLOT(editHeader()));

    // initaly deactivate some items
    m_accelKeys->setItemEnabled("Cut",false);
    m_accelKeys->setItemEnabled("Copy",false);
    m_accelKeys->setItemEnabled("Paste",false);
    m_accelKeys->setItemEnabled("Find",false);
    m_accelKeys->setItemEnabled("Find Next",false);
    m_accelKeys->setItemEnabled("Replace",false);
    m_accelKeys->setItemEnabled("Remove Fuzzy",false);

    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_CUT),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_COPY),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_PASTE),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_SELECT),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_FIND),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_FINDNEXT),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_REPLACE),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_REMFUZZY),false);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_HEADER),false);

    // next, the go-menu
    m_go->insItem(BarIcon("prev"),i18n("&Previous"),"Prior",
                  m_view,SLOT(gotoPrev()),KAccel::Prior);
    m_go->insItem(BarIcon("next"),i18n("&Next"),"Next",
                  m_view,SLOT(gotoNext()),KAccel::Next);
    m_go->insItem(BarIcon("page"),i18n("&Go To Entry..."),"Goto Entry",
                  m_view,SLOT(gotoEntry()),"CTRL+G");
    m_go->insertSeparator(-1);

    m_go->insItem(BarIcon("top"),i18n("&First Entry"),"Goto First Entry",
                  m_view,SLOT(gotoFirst()));
    m_go->insItem(BarIcon("bottom"),i18n("&Last Entry"),"Goto Last Entry",
                  m_view,SLOT(gotoLast()));
    m_go->insertSeparator(-1);

    m_go->insItem(BarIcon("prevfuzzyuntrans"),i18n("P&revious Fuzzy Or Untranslated")
                  ,"Goto Prev Fuzzy or Untrans",
                  m_view,SLOT(gotoPrevFuzzyOrUntrans()),"CTRL+ALT+Prior");
    m_go->insItem(BarIcon("nextfuzzyuntrans"),i18n("N&ext Fuzzy Or Untranslated")
                  ,"Goto Next Fuzzy or Untrans",
                  m_view,SLOT(gotoNextFuzzyOrUntrans()),"CTRL+ALT+Next");
    m_go->insItem(BarIcon("prevfuzzy"),i18n("Pre&vious Fuzzy"),"Goto Prev Fuzzy",
                  m_view,SLOT(gotoPrevFuzzy()),"CTRL+Prior");
    m_go->insItem(BarIcon("nextfuzzy"),i18n("Ne&xt Fuzzy"),"Goto Next Fuzzy",
                  m_view,SLOT(gotoNextFuzzy()),"CTRL+Next");
    m_go->insItem(BarIcon("prevuntranslated"),i18n("Prev&ious Untranslated"),"Goto Prev Untrans",
                  m_view,SLOT(gotoPrevUntranslated()),"ALT+Prior");
    m_go->insItem(BarIcon("nextuntranslated"),i18n("Nex&t Untranslated"),"Goto Next Untrans",
                  m_view,SLOT(gotoNextUntranslated()),"ALT+Next");
    m_go->insertSeparator();
    m_go->insItem(BarIcon("preverror"),i18n("Previ&ous Error"),"Goto Prev Error",
                  m_view,SLOT(gotoPrevError()),"SHIFT+Prior");
    m_go->insItem(BarIcon("nexterror"),i18n("Next E&rror"),"Goto Next Error",
                  m_view,SLOT(gotoNextError()),"SHIFT+Next");


    // initaly deactivate some items
    m_accelKeys->setItemEnabled("Prior",false);
    m_accelKeys->setItemEnabled("Next",false);
    m_accelKeys->setItemEnabled("Goto Entry",false);
    m_accelKeys->setItemEnabled("Goto First Entry",false);
    m_accelKeys->setItemEnabled("Goto Last Entry",false);
    m_accelKeys->setItemEnabled("Goto Prev Fuzzy or Untrans",false);
    m_accelKeys->setItemEnabled("Goto Next Fuzzy or Untrans",false);
    m_accelKeys->setItemEnabled("Goto Prev Fuzzy",false);
    m_accelKeys->setItemEnabled("Goto Next Fuzzy",false);
    m_accelKeys->setItemEnabled("Goto Prev Untrans",false);
    m_accelKeys->setItemEnabled("Goto Next Untrans",false);
    m_accelKeys->setItemEnabled("Goto Prev Error",false);
    m_accelKeys->setItemEnabled("Goto Next Error",false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_PREV),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_NEXT),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_GOTO),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_TOP),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_BOTTOM),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZUNTRANS),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZUNTRANS),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZZY),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZZY),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_PREVUNTRANS),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTUNTRANS),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_PREVERROR),false);
    m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTERROR),false);


    //  the search menu
    m_search->insItem(i18n("Search &In MO-Files"),"Search In MO",
                      m_view,SLOT(startCatSearch()),"F5");
    m_search->insItem(i18n("Search I&n Compendium File"),"Search In Compendium",
                      m_view,SLOT(startComSearch()),"F6");
    m_search->insItem(i18n("Search In &Auxiliary File"),"Search In Auxiliary",
                      m_view,SLOT(startAuxSearch()),"F7");
    m_search->insertSeparator(-1);
    m_search->insItem(i18n("Search &Selection in MO-Files"),"Search Selection in MO",
                      m_view,SLOT(startSelectionCatSearch()),"F8");
    m_search->insItem(i18n("Search S&election in Compendium File"),"Search Selection in Compendium",
                      m_view,SLOT(startSelectionComSearch()),"F9");
    m_search->insItem(i18n("Search Se&lection in Auxiliary File"),"Search Selection in Auxiliary",
                      m_view,SLOT(startSelectionAuxSearch()),"F10");


    m_accelKeys->setItemEnabled("Search In MO",false);
    m_accelKeys->setItemEnabled("Search In Compendium",false);
    m_accelKeys->setItemEnabled("Search In Auxiliary",false);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_CAT),false);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_COM),false);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_AUX),false);
    m_accelKeys->setItemEnabled("Search Selection in MO",false);
    m_accelKeys->setItemEnabled("Search Selection in Compendium",false);
    m_accelKeys->setItemEnabled("Search Selection in Auxiliary",false);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_SELECTION_CAT),false);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_SELECTION_COM),false);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_SELECTION_AUX),false);

    m_tools->insertItem(BarIcon("transsearch"),i18n("&Search"),m_search);
    m_tools->insertItem(BarIcon("catalogmanager"),i18n("&Catalog Manager...")
                       ,this,SLOT(openCatalogManager()));
    m_tools->insertSeparator();
    m_tools->insItem(i18n("C&heck Syntax"),"Check Syntax",m_view,SLOT(checkSyntax()),"CTRL+Y");

    m_accelKeys->setItemEnabled("Check Syntax",false);
    m_tools->setItemEnabled(m_tools->idAt(ID_TOOLS_SYNTAXCHECK),false);

    // next, the Options menu
    m_options->setCheckable(true);
    m_toolbarId = m_options->insertItem(i18n("Show &Toolbar"),
                                        this, SLOT(optionsShowToolbar()));
    m_options->setItemChecked(m_toolbarId, true);
    m_navbarId = m_options->insertItem(i18n("Show &Navigationbar"),
                                        this, SLOT(optionsShowNavBar()));
    m_options->setItemChecked(m_navbarId, true);
    m_statusbarId = m_options->insertItem(i18n("Show &Statusbar"),
                                          this, SLOT(optionsShowStatusbar()));
    m_options->setItemChecked(m_statusbarId, true);
    m_commentId = m_options->insertItem(i18n("Show &Comments"),this,SLOT(optionsShowComments()));
    m_options->setItemChecked(m_commentId,true);
    m_searchId = m_options->insertItem(i18n("Show &Searchresults"),this,SLOT(optionsShowSearchResults()));
    m_options->setItemChecked(m_searchId,true);
    m_options->insertSeparator(-1);

    m_options->insertItem(i18n("Configure &Key Bindings..."),
                          this, SLOT(optionsConfigure()));
    m_options->insertItem(i18n("&Preferences..."),
                          this, SLOT(optionsPreferences()));




    // finally, the Help menu
    m_help = helpMenu(i18n("KBabel Version %1\n\
Copyright 1999 by Matthias Kiefer <matthias.kiefer@gmx.de>\n\n\
This program is licensed under the terms of the GNU GPL.\n\n\
THIS IS ALPHA SOFTWARE - USE WITH CARE!!!\n\n\
Special thanks to Thomas Diehl for many hints to the gui\n\
and the behaviour of KBabel.\n\n\
Many good ideas, especially for the Catalog Manager are taken\n\
from KTranslator by Andrea Rizzi.").arg(VERSION));


    // now add all of the menus to the menu bar
    menuBar()->insertItem(i18n("&File"),    m_file);
    menuBar()->insertItem(i18n("&Edit"),    m_edit);
    menuBar()->insertItem(i18n("&Go"),    m_go);
    menuBar()->insertItem(i18n("&Tools"),    m_tools);
    menuBar()->insertItem(i18n("&Options"), m_options);
    menuBar()->insertItem(i18n("&Help"),    m_help);
}

void KBabel::setupToolBars()
{
    // this toolbar should mirror the menubar fairly closely

    toolBar()->insertButton(BarIcon("fileopen"), ID_TB_OPEN, SIGNAL(clicked()),
                            this, SLOT(fileOpen()), true, i18n("Open"));
    toolBar()->insertButton(BarIcon("filefloppy"), ID_TB_SAVE, SIGNAL(clicked()),
                            this, SLOT(fileSave()), false, i18n("Save"));
    toolBar()->insertSeparator();
    toolBar()->insertButton(BarIcon("editcut"), ID_TB_CUT, SIGNAL(clicked()),
                            m_view, SLOT(textCut()), false, i18n("Cut"));
    toolBar()->insertButton(BarIcon("editcopy"), ID_TB_COPY, SIGNAL(clicked()),
                            m_view, SLOT(textCopy()), false, i18n("Copy"));
    toolBar()->insertButton(BarIcon("editpaste"), ID_TB_PASTE, SIGNAL(clicked()),
                            m_view, SLOT(textPaste()), false, i18n("Paste"));
    toolBar()->insertButton(BarIcon("find"), ID_TB_FIND, SIGNAL(clicked()),
                            this, SLOT(find()), false, i18n("Find"));
    toolBar()->insertSeparator();

    toolBar()->insertButton(BarIcon("transsearch"),ID_TB_SEARCH,SIGNAL(clicked()),
                           m_view,SLOT(startSearch()),false,i18n("Search Translations"));
    toolBar()->setDelayedPopup(ID_TB_SEARCH,m_search);

    toolBar()->insertButton(BarIcon("stop"),ID_TB_STOP,SIGNAL(clicked()),
                            m_view, SLOT(stopSearch()),false,i18n("Stop"));
    toolBar()->insertSeparator();
    toolBar()->insertButton(BarIcon("catalogmanager"),ID_TB_CATMAN,SIGNAL(clicked()),
                            this, SLOT(openCatalogManager()),true,i18n("Catalog Manager..."));

    KHelpMenu *help = new KHelpMenu();
    toolBar()->insertButton(BarIcon("help"), -1, SIGNAL(clicked()),
                            help, SLOT(appHelpActivated()),
                            true, i18n("Help"));

    toolBar()->insertButton(BarIcon("context_help"),ID_TB_WHATSTHIS,SIGNAL(clicked())
                           ,this,SLOT(whatsThis()),true,i18n("What's this?"));

    // insert Whatsthis help
    QWhatsThis::add((QWidget*)toolBar()->getButton(ID_TB_OPEN),
       i18n("<qt><p><b>Open...</b><p>\n\
<p>Opens a file dialog where \
you can select the PO- or POT-file you want to open.</p></qt>"));

    QWhatsThis::add((QWidget*)toolBar()->getButton(ID_TB_SAVE),
       i18n("<qt><p><b>Save</b></p>\n\
<p>Saves your changes to the opened file.</p></qt>"));

    QWhatsThis::add((QWidget*)toolBar()->getButton(ID_TB_SEARCH),
       i18n("<qt><p><b>Search Translations</b></p>\n\
<p>Click on this button to start the search in \
the default searchmode, or keep it pressed for \
a short time and a menu will popup from which \
you can select the wanted searchmode.</p>\n\
<p>You can select the default searchmode in \
the <b>Search</b> section of the \
preferences dialog.</p></qt>"));


    toolBar()->setFullWidth(false);
    toolBar(ID_NAVBAR)->setFullWidth(false);


    toolBar(ID_NAVBAR)->insertButton(BarIcon("prev"),ID_NB_PREV,SIGNAL(clicked()),
                           m_view,SLOT(gotoPrev()),false,i18n("Previous Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("next"),ID_NB_NEXT,SIGNAL(clicked()),
                           m_view,SLOT(gotoNext()),false,i18n("Next Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("page"),ID_NB_GOTO,SIGNAL(clicked()),
                           m_view,SLOT(gotoEntry()),false,i18n("Goto Entry..."));
    //toolBar(ID_NAVBAR)->insertLineSeparator();
    toolBar(ID_NAVBAR)->insertButton(BarIcon("top"),ID_NB_TOP,SIGNAL(clicked()),
                           m_view,SLOT(gotoFirst()),false,i18n("First Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("bottom"),ID_NB_BOTTOM,SIGNAL(clicked()),
                           m_view,SLOT(gotoLast()),false,i18n("Last Entry"));
    //toolBar(ID_NAVBAR)->insertLineSeparator();
    toolBar(ID_NAVBAR)->insertButton(BarIcon("prevfuzzyuntrans"),ID_NB_PREVFUZUNTRANS,SIGNAL(clicked()),
                           m_view,SLOT(gotoPrevFuzzyOrUntrans()),false,i18n("Previous Fuzzy Or Untranslated Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("nextfuzzyuntrans"),ID_NB_NEXTFUZUNTRANS,SIGNAL(clicked()),
                           m_view,SLOT(gotoNextFuzzyOrUntrans()),false,i18n("Next Fuzzy Or Untranslated Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("prevfuzzy"),ID_NB_PREVFUZZY,SIGNAL(clicked()),
                           m_view,SLOT(gotoPrevFuzzy()),false,i18n("Previous Fuzzy Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("nextfuzzy"),ID_NB_NEXTFUZZY,SIGNAL(clicked()),
                           m_view,SLOT(gotoNextFuzzy()),false,i18n("Next Fuzzy Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("prevuntranslated"),ID_NB_PREVUNTRANS,SIGNAL(clicked()),
                           m_view,SLOT(gotoPrevUntranslated()),false,i18n("Previous Untranslated Entry"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("nextuntranslated"),ID_NB_NEXTUNTRANS,SIGNAL(clicked()),
                           m_view,SLOT(gotoNextUntranslated()),false,i18n("Next Untranslated Entry"));
    toolBar(ID_NAVBAR)->insertSeparator();
    toolBar(ID_NAVBAR)->insertButton(BarIcon("preverror"),ID_NB_PREVERROR,SIGNAL(clicked()),
                           m_view,SLOT(gotoPrevError()),false,i18n("Previous Entry with errors"));
    toolBar(ID_NAVBAR)->insertButton(BarIcon("nexterror"),ID_NB_NEXTERROR,SIGNAL(clicked()),
                           m_view,SLOT(gotoNextError()),false,i18n("Next Entry with errors"));

}

void KBabel::setupStatusBar()
{
    statusBar()->insertItem(i18n("Current: 0   "),ID_STATUS_CURRENT);
    statusBar()->insertItem(i18n("Total: 0    "),ID_STATUS_TOTAL);
    statusBar()->insertItem(i18n("Fuzzy: 0    "),ID_STATUS_FUZZY);
    statusBar()->insertItem(i18n("Untranslated: 0    "),ID_STATUS_UNTRANS);

    QHBox* box = new QHBox(statusBar(),"statusBox");
    box->setSpacing(2);
    new QLabel(i18n("Status: "),box);
    _fuzzyLed = new KLed(Qt::red,KLed::Off,KLed::Sunken,KLed::Rectangular,box);
    _fuzzyLed->setFixedSize(15,12);
    new QLabel(i18n("fuzzy"),box);
    _untransLed = new KLed(Qt::red,KLed::Off,KLed::Sunken,KLed::Rectangular,box);
    _untransLed->setFixedSize(15,12);
    new QLabel(i18n("untranslated"),box);

    box->setFixedWidth(box->sizeHint().width());

    statusBar()->insertWidget(box,box->sizeHint().width(),ID_STATUS_STATUS);

    statusBar()->insertItem(" "+i18n("INS")+" ",ID_STATUS_EDITMODE);
    statusBar()->setAlignment(ID_STATUS_EDITMODE,Qt::AlignHCenter);

    statusBar()->insertItem(" "+i18n("RO")+" ",ID_STATUS_READONLY);
    statusBar()->setAlignment(ID_STATUS_READONLY,Qt::AlignHCenter);
    statusBar()->changeItem("",ID_STATUS_READONLY);


    statusBar()->insertItem(i18n("loading file"),ID_STATUS_PROGRESSLABEL);
    statusBar()->setAlignment(ID_STATUS_PROGRESSLABEL,Qt::AlignRight);
    statusBar()->changeItem("",ID_STATUS_PROGRESSLABEL);
    progressBar=new KProgress(statusBar(),"progressbar");
    statusBar()->insertWidget(progressBar,100,ID_STATUS_PROGRESS);


    QWhatsThis::add(statusBar(),
       i18n("<qt><p><b>Statusbar</b></p>\n\
<p>The statusbar displays some information about the opened file,\n\
like the total number of entries and the number of fuzzy and untranslated\n\
messages. Also the index and the status of the currently displayed entry is shown.</p></qt>"));

}


void KBabel::saveSettings()
{
    KConfig* config = KGlobal::config();

    {
        KConfigGroupSaver saver(config,"View");

        config->writeEntry("Toolbar",m_options->isItemChecked(m_toolbarId));
        config->writeEntry("Navigationbar",m_options->isItemChecked(m_navbarId));
        config->writeEntry("Statusbar",m_options->isItemChecked(m_statusbarId));
        config->writeEntry("Searchresults",m_options->isItemChecked(m_searchId));
        config->writeEntry("Comments",m_options->isItemChecked(m_commentId));

        config->writeEntry("PosToolbar",(int)(toolBar()->barPos()));
        config->writeEntry("PosNavigationbar",(int)(toolBar(ID_NAVBAR)->barPos()));

        config->writeEntry("Size",size());
    }

    {
        KConfigGroupSaver saver(config,"KBabel");

        config->writeEntry("RecentFiles",_recentFiles);
    }
    {
       KConfigGroupSaver cs(config,"CatalogManager");

       config->writeEntry("PoBaseDir",_catManSettings.poBaseDir);
       config->writeEntry("PotBaseDir",_catManSettings.potBaseDir);
       config->writeEntry("OpenWindow",_catManSettings.openWindow);

       config->writeEntry("KillCmdOnExit",_catManSettings.killCmdOnExit);

       config->writeEntry("DirCommands",_catManSettings.dirCommands);
       config->writeEntry("DirCommandNames",_catManSettings.dirCommandNames);
       config->writeEntry("FileCommands",_catManSettings.fileCommands);
       config->writeEntry("FileCommandNames",_catManSettings.fileCommandNames);
    }
}

void KBabel::restoreSettings()
{
    KConfig* config = KGlobal::config();

    {
       KConfigGroupSaver saver(config,"View");

       bool show=config->readBoolEntry("Toolbar",true);
       m_options->setItemChecked(m_toolbarId, show);
       if (show)
           toolBar()->enable(KToolBar::Show);
       else
           toolBar()->enable(KToolBar::Hide);

       show=config->readBoolEntry("Navigationbar",true);
       m_options->setItemChecked(m_navbarId, show);
       if (show)
           toolBar(ID_NAVBAR)->enable(KToolBar::Show);
       else
           toolBar(ID_NAVBAR)->enable(KToolBar::Hide);

       show=config->readBoolEntry("Statusbar",true);
       m_options->setItemChecked(m_statusbarId, show);
       if (show)
           statusBar()->enable(KStatusBar::Show);
       else
           statusBar()->enable(KStatusBar::Hide);

       show=config->readBoolEntry("Comments",true);
       m_options->setItemChecked(m_commentId, show);
       m_view->showComments(show);

       show=config->readBoolEntry("Searchresults",true);
       m_options->setItemChecked(m_searchId, show);
       m_view->showSearchResults(show);


       int pos=config->readNumEntry("PosToolbar",(int)KToolBar::Top);
       toolBar()->setBarPos((KToolBar::BarPosition)pos);
       pos=config->readNumEntry("PosNavigationbar",(int)KToolBar::Top);
       toolBar(ID_NAVBAR)->setBarPos((KToolBar::BarPosition)pos);

       QSize defaultSize(600,450);
       resize(config->readSizeEntry("Size",&defaultSize));

       m_view->restoreView();
    }

    {
       KConfigGroupSaver saver(config,"KBabel");

       m_recentFiles->clear();

       _recentFiles=config->readListEntry("RecentFiles");
       for(int i=0; i<(int)_recentFiles.count(); i++ )
       {
          m_recentFiles->insertItem(_recentFiles[i],i,i);
       }
    }
    if(memberList->count()==1) // if this is the first window, read settings
    {
       KConfigGroupSaver cs(config,"CatalogManager");

       _catManSettings.poBaseDir=config->readEntry("PoBaseDir",Defaults::CatalogManager::poBaseDir());
       _catManSettings.potBaseDir=config->readEntry("PotBaseDir",Defaults::CatalogManager::potBaseDir());
       _catManSettings.openWindow=config->readBoolEntry("OpenWindow",Defaults::CatalogManager::openWindow);

       _catManSettings.killCmdOnExit=config->readBoolEntry("KillCmdOnExit"
                         ,Defaults::CatalogManager::killCmdOnExit);

       if(config->hasKey("DirCommands"))
       {
          _catManSettings.dirCommands = config->readListEntry("DirCommands");
          _catManSettings.dirCommandNames = config->readListEntry("DirCommandNames");
       }
       else
       {
          _catManSettings.dirCommands = Defaults::CatalogManager::dirCommands();
          _catManSettings.dirCommandNames = Defaults::CatalogManager::dirCommandNames();
       }
       if(config->hasKey("FileCommands"))
       {
          _catManSettings.fileCommands = config->readListEntry("FileCommands");
          _catManSettings.fileCommandNames = config->readListEntry("FileCommandNames");
       }
       else
       {
          _catManSettings.fileCommands = Defaults::CatalogManager::fileCommands();
          _catManSettings.fileCommandNames = Defaults::CatalogManager::fileCommandNames();
       }
    }
}


void KBabel::saveProperties(KConfig *config)
{
   m_view->saveSession(config);
}

void KBabel::readProperties(KConfig *config)
{
   m_view->restoreSession(config);
}

bool KBabel::queryClose()
{
   if(m_view->isSearching())
   {
       connect(m_view,SIGNAL(signalSearchActive(bool)),this,SLOT(quit()));
       m_view->stopSearch();
       return false;
   }

   if(m_view->isLastView() && m_view->isModified())
   {
      switch(KMessageBox::warningYesNoCancel(this,
      i18n("The document contains unsaved changes.\n\
Do you want to save your changes or discard them?"),i18n("Warning"),
      i18n("&Save"),i18n("&Discard")))
      {
         case KMessageBox::Yes:
         {
            return m_view->saveFile();
         }
         case KMessageBox::No:
            return true;
         default:
            return false;
      }
   }

  return true;
}

bool KBabel::queryExit()
{
   saveSettings();
   m_view->saveSettings();
   m_view->catalog()->savePreferences();
   m_view->saveView();

   if(_catalogManager)
   {
      delete _catalogManager;
   }

   return true;
}

void KBabel::quit()
{
    close();
}


void KBabel::dragEnterEvent(QDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(QUriDrag::canDecode(event));
}

void KBabel::dropEvent(QDropEvent *event)
{
    QStrList uri;

    // see if we can decode a URI.. if not, just ignore it
    if (QUriDrag::decode(event, uri))
    {
       m_view->processUriDrop(uri,mapToGlobal(event->pos()));
    }
}



void KBabel::open(const QString& url,bool newWindow)
{
   if(newWindow)
   {
      fileNewWindow()->open(url,false);
   }
   else
   {
      m_view->open(url);
   }
}

void KBabel::openTemplate(const QString& openURL,const QString& saveURL,bool newWindow)
{
   if(newWindow)
   {
      fileNewWindow()->openTemplate(openURL,saveURL,false);
   }
   else
   {
      m_view->openTemplate(openURL,saveURL);
   }
}

void KBabel::fileOpen()
{
    m_view->open();
}

void KBabel::openRecent(int id)
{
   m_view->open(m_recentFiles->text(id));

   QString filename=m_recentFiles->text(id);
   m_recentFiles->insertItem(filename,-1,0);
   m_recentFiles->removeItem(id);
}

void KBabel::fileSave()
{
    // this slot is called whenever the File->Save menu is selected,
    // the Save shortcut is pressed (usually CTRL+S) or the Save toolbar
    // button is clicked

    if(!m_view->isModified())
    {
       statusBar()->message(i18n("There are no changes to save."),2000);
    }
    else
    {
       // save the current file
       m_view->saveFile();

       KURL url=m_view->currentURL();
       if(_catalogManager)
       {
          _catalogManager->updateFile(url.directory(false)+url.filename());
       }
    }
}

void KBabel::fileSaveAs()
{
    m_view->saveFileAs();

    KURL url=m_view->currentURL();
    if(_catalogManager)
    {
       _catalogManager->updateFile(url.directory(false)+url.filename());
    }
}


void KBabel::fileNewView()
{
   KBabel* b=new KBabel(m_view->catalog());
   b->setSettings(m_view->searchSettings(),m_view->editorSettings());
   b->show();
}

KBabel* KBabel::fileNewWindow()
{
   KBabel* b=new KBabel();
   b->setSettings(m_view->searchSettings(),m_view->editorSettings()
            ,m_view->catalog()->saveSettings(),m_view->catalog()->identitySettings());
   b->show();

   return b;
}


void KBabel::toggleEditMode()
{
   bool ovr=!m_view->isOverwriteMode();

   m_view->setOverwriteMode(ovr);

   if(ovr)
      statusBar()->changeItem(i18n("OVR"),ID_STATUS_EDITMODE);
   else
      statusBar()->changeItem(i18n("INS"),ID_STATUS_EDITMODE);

}






void KBabel::optionsShowToolbar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    bool show = m_options->isItemChecked(m_toolbarId);
    show = show ? false : true;

    m_options->setItemChecked(m_toolbarId, show);

    if (show)
        toolBar()->enable(KToolBar::Show);
    else
        toolBar()->enable(KToolBar::Hide);
}

void KBabel::optionsShowNavBar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    bool show = m_options->isItemChecked(m_navbarId);
    show = show ? false : true;

    m_options->setItemChecked(m_navbarId, show);

    if (show)
        toolBar(ID_NAVBAR)->enable(KToolBar::Show);
    else
        toolBar(ID_NAVBAR)->enable(KToolBar::Hide);
}

void KBabel::optionsShowStatusbar()
{
    // this is all very cut and paste code for showing/hiding the
    // statusbar
    bool show = m_options->isItemChecked(m_statusbarId);
    show = show ? false : true;

    m_options->setItemChecked(m_statusbarId, show);

    if (show)
        statusBar()->enable(KStatusBar::Show);
    else
        statusBar()->enable(KStatusBar::Hide);
}

void KBabel::optionsShowComments()
{
    bool show = m_options->isItemChecked(m_commentId);
    show = show ? false : true;

    m_options->setItemChecked(m_commentId, show);

    m_view->showComments(show);
}

void KBabel::optionsShowSearchResults()
{
    bool show = m_options->isItemChecked(m_searchId);
    show = show ? false : true;

    m_options->setItemChecked(m_searchId, show);

    m_view->showSearchResults(show);
}

void KBabel::checkMenuItemSearchResults()
{
   m_options->setItemChecked(m_searchId, true);
}

void KBabel::optionsConfigure()
{
    if(KKeyDialog::configureKeys(m_accelKeys))
       m_accelKeys->writeSettings();
}

void KBabel::optionsPreferences()
{
    if(!_prefDialog)
    {
        _prefDialog = new KBabelPreferences(m_view->searchSettings(),m_view->editorSettings(),
                      m_view->catalog()->saveSettings(),m_view->catalog()->identitySettings()
                      ,_catManSettings);
        prefDialogs.append(_prefDialog);

        connect(_prefDialog,SIGNAL(identityOptionsChanged(IdentitySettings))
                ,m_view->catalog(),SLOT(setSettings(IdentitySettings)));
        connect(_prefDialog,SIGNAL(saveOptionsChanged(SaveSettings))
                ,m_view->catalog(),SLOT(setSettings(SaveSettings)));
        connect(_prefDialog,SIGNAL(editorOptionsChanged(EditorSettings))
                ,m_view,SLOT(setSettings(EditorSettings)));
        connect(_prefDialog,SIGNAL(searchOptionsChanged(SearchSettings))
                ,m_view,SLOT(setSettings(SearchSettings)));

        connect(m_view->catalog(),SIGNAL(signalSettingsChanged(SaveSettings)),_prefDialog
                ,SLOT(updateSaveSettings(SaveSettings)));
        connect(m_view->catalog(),SIGNAL(signalSettingsChanged(IdentitySettings)),_prefDialog
                ,SLOT(updateIdentitySettings(IdentitySettings)));
        connect(m_view,SIGNAL(signalSearchSettingsChanged(SearchSettings)),_prefDialog
                ,SLOT(updateSearchSettings(SearchSettings)));

        connect(_prefDialog,SIGNAL(catManOptionsChanged(CatManSettings))
                  ,this,SLOT(setCatManSettings(CatManSettings)));

    }

    int prefHeight=_prefDialog->height();
    int prefWidth=_prefDialog->width();
    int width=this->width();
    int height=this->height();

    int x=width/2-prefWidth/2;
    int y=height/2-prefHeight/2;

    _prefDialog->move(mapToGlobal(QPoint(x,y)));

    if(!_prefDialog->isVisible())
    {
       _prefDialog->show();
    }

    _prefDialog->raise();
}


void KBabel::whatsThis()
{
   QWhatsThis::enterWhatsThisMode();
}

void KBabel::setCatManSettings(CatManSettings settings)
{
   if(_catalogManager)
   {
      _catalogManager->setSettings(settings);
   }

   updateCatManSettings(settings);
}

void KBabel::updateCatManSettings(CatManSettings settings)
{
   _catManSettings=settings;

   // update settings of the catalog manager in all preferences dialogs
   KBabelPreferences* dialog;
   for(dialog=prefDialogs.first(); dialog != 0 ; dialog=prefDialogs.next())
   {
       dialog->updateCatManSettings(settings);
   }
}


void KBabel::openCatalogManager()
{
    if(!_catalogManager)
    {
       _catalogManager= new CatalogManager(_catManSettings);
       connect(_catalogManager,SIGNAL(settingsChanged(CatManSettings))
               ,this,SLOT(updateCatManSettings(CatManSettings)));
    }

    int catManHeight=_catalogManager->height();
    int catManWidth=_catalogManager->width();
    int width=this->width();
    int height=this->height();

    int x=width/2-catManWidth/2;
    int y=height/2-catManHeight/2;

    _catalogManager->move(mapToGlobal(QPoint(x,y)));

    if(!_catalogManager->isVisible())
    {
       _catalogManager->show();
    }
    else
    {
       _catalogManager->raise();
    }

    _catalogManager->setPreferredWindow(this);
}






void KBabel::findNext()
{
}

void KBabel::find()
{
}

void KBabel::replace()
{
}


void KBabel::firstEntryDisplayed(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_TOP),!flag);
   m_go->setItemEnabled(m_go->idAt(ID_GO_PREV),!flag);

   m_accelKeys->setItemEnabled("Prior",!flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREV,!flag);
   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_TOP,!flag);
}

void KBabel::lastEntryDisplayed(bool flag)
{
   // enable next-button
   m_go->setItemEnabled(m_go->idAt(ID_GO_BOTTOM),!flag);
   m_go->setItemEnabled(m_go->idAt(ID_GO_NEXT),!flag);

   m_accelKeys->setItemEnabled("Next",!flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_BOTTOM,!flag);
   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXT,!flag);

}

void KBabel::fuzzyDisplayed(bool flag)
{
    if(flag)
    {
       if(_fuzzyLed->state()==KLed::Off)
       {
          _fuzzyLed->on();
       }
    }
    else
    {
       if(_fuzzyLed->state()==KLed::On)
           _fuzzyLed->off();
    }

    if(!m_view->isReadOnly())
    {
       m_accelKeys->setItemEnabled("Remove Fuzzy",flag);
       m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_REMFUZZY),flag);
    }
}

void KBabel::untranslatedDisplayed(bool flag)
{
    if(flag)
    {
       if(_untransLed->state()==KLed::Off)
          _untransLed->on();
    }
    else
    {
       if(_untransLed->state()==KLed::On)
          _untransLed->off();
    }
}


void KBabel::displayedEntryChanged(uint index)
{
   statusBar()->changeItem(i18n("Current: %1").arg(index),ID_STATUS_CURRENT);
}

void KBabel::setNumberOfTotal(uint number)
{
   statusBar()->changeItem(i18n("Total: %1").arg(number),ID_STATUS_TOTAL);
}

void KBabel::setNumberOfFuzzies(uint number)
{
   statusBar()->changeItem(i18n("Fuzzy: %1").arg(number),ID_STATUS_FUZZY);
}

void KBabel::setNumberOfUntranslated(uint number)
{
   statusBar()->changeItem(i18n("Untranslated: %1").arg(number),ID_STATUS_UNTRANS);
}

void KBabel::hasFuzzyAfterwards(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZZY),flag);

   m_accelKeys->setItemEnabled("Goto Next Fuzzy",flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTFUZZY,flag);

   // check if there is  a fuzzy or translated afterwards
   if(flag || m_go->isItemEnabled(m_go->idAt(ID_GO_NEXTUNTRANS)))
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZUNTRANS),true);

       m_accelKeys->setItemEnabled("Goto Next Fuzzy or Untrans",true);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTFUZUNTRANS,true);
   }
   else
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZUNTRANS),false);

       m_accelKeys->setItemEnabled("Goto Next Fuzzy or Untrans",false);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTFUZUNTRANS,false);
   }
}

void KBabel::hasFuzzyInFront(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZZY),flag);

   m_accelKeys->setItemEnabled("Goto Prev Fuzzy",flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVFUZZY,flag);

   // check if there is  a fuzzy or translated in front
   if(flag || m_go->isItemEnabled(m_go->idAt(ID_GO_PREVUNTRANS)))
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZUNTRANS),true);

       m_accelKeys->setItemEnabled("Goto Prev Fuzzy or Untrans",true);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVFUZUNTRANS,true);
   }
   else
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZUNTRANS),false);

       m_accelKeys->setItemEnabled("Goto Prev Fuzzy or Untrans",false);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVFUZUNTRANS,false);
   }
}

void KBabel::hasUntranslatedAfterwards(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTUNTRANS),flag);

   m_accelKeys->setItemEnabled("Goto Next Untrans",flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTUNTRANS,flag);

   // check if there is a fuzzy or translated afterwards
   if(flag || m_go->isItemEnabled(m_go->idAt(ID_GO_NEXTFUZZY)))
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZUNTRANS),true);

       m_accelKeys->setItemEnabled("Goto Next Fuzzy or Untrans",true);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTFUZUNTRANS,true);
   }
   else
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTFUZUNTRANS),false);

       m_accelKeys->setItemEnabled("Goto Next Fuzzy or Untrans",false);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTFUZUNTRANS,false);
   }
}

void KBabel::hasUntranslatedInFront(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_PREVUNTRANS),flag);

   m_accelKeys->setItemEnabled("Goto Prev Untrans",flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVUNTRANS,flag);

   // check if there is  a fuzzy or translated in front
   if(flag || m_go->isItemEnabled(m_go->idAt(ID_GO_PREVFUZZY)))
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZUNTRANS),true);

       m_accelKeys->setItemEnabled("Goto Prev Fuzzy or Untrans",true);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVFUZUNTRANS,true);
   }
   else
   {
       m_go->setItemEnabled(m_go->idAt(ID_GO_PREVFUZUNTRANS),false);

       m_accelKeys->setItemEnabled("Goto Prev Fuzzy or Untrans",false);

       toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVFUZUNTRANS,false);
   }
}



void KBabel::hasErrorAfterwards(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_NEXTERROR),flag);

   m_accelKeys->setItemEnabled("Goto Next Error",flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_NEXTERROR,flag);
}

void KBabel::hasErrorInFront(bool flag)
{
   m_go->setItemEnabled(m_go->idAt(ID_GO_PREVERROR),flag);

   m_accelKeys->setItemEnabled("Goto Prev Error",flag);

   toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_PREVERROR,flag);
}



void KBabel::prepareProgressBar(QString msg,int max)
{
   statusBar()->changeItem(msg,ID_STATUS_PROGRESSLABEL);
   progressBar->setRange(0,max);
   progressBar->setValue(0);

}

void KBabel::clearProgressBar()
{
   progressBar->setValue(0);
   statusBar()->changeItem("",ID_STATUS_PROGRESSLABEL);
}


void KBabel::changeStatusbar(const QString& text)
{
    // display the text on the statusbar
    statusBar()->message(text,10);
}

void KBabel::changeCaption(const QString& text)
{
    // display the text on the caption
    setCaption(text);


    QStringList::Iterator it=_recentFiles.find(text);

    if(it != _recentFiles.end())
    {
       _recentFiles.remove(it);
       _recentFiles.prepend(text);
    }
    else if( _recentFiles.count() < MAX_RECENT)
    {
       _recentFiles.prepend(text);
    }
    else
    {
       _recentFiles.remove(_recentFiles.last());
       _recentFiles.insert(_recentFiles.begin(),text);
    }

    // check if the filename is already in the menu and return
    for(int i=0;i<(int)m_recentFiles->count();i++)
    {
       if(m_recentFiles->text(m_recentFiles->idAt(i))==text)
          return;
    }

    if(m_recentFiles->count()==MAX_RECENT)
    {
       m_recentFiles->removeItemAt(MAX_RECENT-1);
    }

    m_recentFiles->insertItem(text,-1,0);
}


void KBabel::enableDefaults(bool readOnly)
{
    m_accelKeys->setItemEnabled("Save",true);
    m_file->setItemEnabled(m_file->idAt(ID_FILE_SAVE),true);
    m_file->setItemEnabled(m_file->idAt(ID_FILE_SAVEAS),true);

    m_accelKeys->setItemEnabled("Cut",!readOnly);
    m_accelKeys->setItemEnabled("Copy",true);
    m_accelKeys->setItemEnabled("Paste",!readOnly);
//    m_accelKeys->setItemEnabled("Find",true);
//    m_accelKeys->setItemEnabled("Find Next",true);
//    m_accelKeys->setItemEnabled("Replace",!readOnly);

    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_CUT),!readOnly);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_COPY),true);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_PASTE),!readOnly);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_SELECT),true);
//    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_FIND),true);
//    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_FINDNEXT),true);
//    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_REPLACE),!readOnly);
    m_edit->setItemEnabled(m_edit->idAt(ID_EDIT_HEADER),true);

    toolBar()->setItemEnabled(ID_TB_SAVE,true);
    toolBar()->setItemEnabled(ID_TB_CUT,!readOnly);
    toolBar()->setItemEnabled(ID_TB_COPY,true);
    toolBar()->setItemEnabled(ID_TB_PASTE,!readOnly);

    toolBar()->setItemEnabled(ID_TB_SEARCH,true);


    toolBar(ID_NAVBAR)->setItemEnabled(ID_NB_GOTO,true);
    m_go->setItemEnabled(m_go->idAt(ID_GO_GOTO),true);
    m_accelKeys->setItemEnabled("Goto Entry",true);


    m_tools->setItemEnabled(m_tools->idAt(ID_TOOLS_SYNTAXCHECK),true);
    m_accelKeys->setItemEnabled("Check Syntax",true);

    if(readOnly)
       statusBar()->changeItem(i18n("RO"),ID_STATUS_READONLY);
    else
       statusBar()->changeItem("",ID_STATUS_READONLY);
}


void KBabel::enableStop(bool flag)
{
   toolBar()->setItemEnabled(ID_TB_STOP,flag);
}

void KBabel::enableCatSearch(bool flag)
{
    m_accelKeys->setItemEnabled("Search In MO",flag);
    m_accelKeys->setItemEnabled("Search Selection in MO",flag);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_CAT),flag);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_SELECTION_CAT),flag);
}

void KBabel::enableComSearch(bool flag)
{
    m_accelKeys->setItemEnabled("Search In Compendium",flag);
    m_accelKeys->setItemEnabled("Search Selection in Compendium",flag);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_COM),flag);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_SELECTION_COM),flag);
}

void KBabel::enableAuxSearch(bool flag)
{
    m_accelKeys->setItemEnabled("Search In Auxiliary",flag);
    m_accelKeys->setItemEnabled("Search Selection in Auxiliary",flag);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_AUX),flag);
    m_search->setItemEnabled(m_search->idAt(ID_SEARCH_SELECTION_AUX),flag);
}
