/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */

#include "searchresultbox.h"
#include "settings.h"
#include "resources.h"
#include "catalog.h" // for Catalog::readHeader
#include "catalogitem.h"
#include "msgfmt.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qmultilineedit.h>
#include <qpushbutton.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qpopupmenu.h>
#include <qdragobject.h>

#include <kdialog.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kapp.h>
#include <kmessagebox.h>
#include <kurl.h>


#define ID_DROP_MO 1
#define ID_DROP_COM 2
#define ID_DROP_AUX 3
#define ID_DROP_MOFILES 4
#define ID_DROP_COMFILE 5
#define ID_DROP_AUXFILE 6


SearchResultBox::SearchResultBox(QWidget* parent,const char* name)
               : QWidget(parent,name)
{
    setAcceptDrops(true);

    _stopSearch=false;
    _searching=false;

    _catList.setAutoDelete(true);



    QVBoxLayout* layout = new QVBoxLayout(this);
    layout->setSpacing(KDialog::spacingHint());

    layout->addSpacing(KDialog::marginHint());

    QHBoxLayout* hb=new QHBoxLayout(layout);
    hb->addSpacing(KDialog::marginHint());
    QLabel* label=new QLabel(i18n("Total:"),this);
    hb->addWidget(label);
    _totalLabel=new QLabel(this);
    hb->addWidget(_totalLabel);
    hb->setStretchFactor(_totalLabel,1);
    label=new QLabel(i18n("Found in:"),this);
    hb->addWidget(label);
    _fileLabel = new QLabel(this);
    _fileLabel->setMinimumSize(50,_fileLabel->minimumHeight());
    hb->addWidget(_fileLabel);
    hb->setStretchFactor(_fileLabel,2);
    hb->addSpacing(KDialog::marginHint());


    _msgidBox = new QMultiLineEdit(this);
    _msgidBox->setAcceptDrops(false);
    layout->addWidget(_msgidBox);
    _msgidBox->setReadOnly(true);
    layout->setStretchFactor(_msgidBox,2);
    _msgstrBox = new QMultiLineEdit(this);
    _msgstrBox->setAcceptDrops(false);
    layout->addWidget(_msgstrBox);
    _msgstrBox->setReadOnly(true);
    layout->setStretchFactor(_msgstrBox,2);

    hb=new QHBoxLayout(layout);
    hb->addStretch(1);
    _prevButton=new QPushButton(i18n("<&Back"),this);
    hb->addWidget(_prevButton);
    _nextButton=new QPushButton(i18n("&Next>"),this);
    hb->addWidget(_nextButton);
    hb->addStretch(1);

    int prevSize=_prevButton->sizeHint().width();
    int nextSize=_nextButton->sizeHint().width();
    int size=QMAX(prevSize,nextSize);
    _prevButton->setFixedWidth(size);
    _nextButton->setFixedWidth(size);


    _prevButton->setAutoRepeat(true);
    _nextButton->setAutoRepeat(true);

    layout->addSpacing(KDialog::marginHint());


    connect(_prevButton,SIGNAL(clicked()),this,SLOT(prevResult()));
    connect(_nextButton,SIGNAL(clicked()),this,SLOT(nextResult()));

    _textDropMenu = new QPopupMenu(this);
    _textDropMenu->insertItem(i18n("Search Text in MO-Files"),ID_DROP_MO);
    _textDropMenu->insertItem(i18n("Search Text in Compendium File"),ID_DROP_COM);
    _textDropMenu->insertItem(i18n("Search Text in Auxiliary File"),ID_DROP_AUX);

    _urlDropMenu = new QPopupMenu(this);
    //_urlDropMenu->insertItem(i18n("Add to MO-files"),ID_DROP_COMFILE);
    _urlDropMenu->insertItem(i18n("Use as compendium file"),ID_DROP_COMFILE);
    _urlDropMenu->insertItem(i18n("Use as auxiliary file"),ID_DROP_AUXFILE);

    clearResults();

    readSettings();
}

SearchResultBox::~SearchResultBox()
{
   if(_auxFile.isOpen())
      _auxFile.close();
   if(_comFile.isOpen())
      _comFile.close();

   if(_searching)
      stopSearch();
}



void SearchResultBox::search(QString msgid,SearchMode mode)
{
    if(_searching)
    {
       stopSearch();

       // if searching, store searchmode and searchstring and wait till search stopped
       _searchMode=mode;
       _searchString=msgid;
       connect(this,SIGNAL(signalSearchActive(bool)),this,SLOT(startDelayedSearch()));
    }
    else
    {
      kDebugArea(KBABEL_SEARCH,"starting search...");

      switch(mode)
      {
         case CatalogSearch:
            catSearch(msgid);
            break;
         case CompendiumSearch:
            comSearch(msgid);
            break;
         case AuxiliarySearch:
            auxSearch(msgid);
            break;
         default:
            break;
      }
   }
}

void SearchResultBox::startDelayedSearch(QString msgid,SearchMode mode)
{
    // if searching, store searchmode and searchstring and wait till search stopped
    _searchMode=mode;
    _searchString=msgid;

    if(_searching)
    {
       stopSearch();

       connect(this,SIGNAL(signalSearchActive(bool)),this,SLOT(startDelayedSearch()));
    }
    else
    {
       QTimer::singleShot(5,this,SLOT(startDelayedSearch()));
    }
}

void SearchResultBox::startDelayedSearch()
{
    kDebugArea(KBABEL_SEARCH,"starting delayed search...");

    disconnect(this,SIGNAL(signalSearchActive(bool)),this,SLOT(startDelayedSearch()));

    switch(_searchMode)
    {
         case CatalogSearch:
            catSearch(_searchString);
            break;
         case CompendiumSearch:
            comSearch(_searchString);
            break;
         case AuxiliarySearch:
            auxSearch(_searchString);
            break;
         default:
            break;
    }

    _searchString="";
}

void SearchResultBox::catSearch(QString msgid)
{
   if(_catList.isEmpty())
   {
      /*
      KMessageBox::sorry(this,i18n("Search in MO-Files is not available!\n\n\
The file I tried to open was \n%1\n\n\
You may want to edit the \"Search->Files\"-Section in the preferences dialog."));
        */

      KMessageBox::sorry(this,i18n("Search in MO-Files is not implemented yet!"));

      emit signalEnableCatSearch(false);

      return;
   }

   _stopSearch=false;
   _searching=true;
   emit signalSearchActive(true);


   _searching=false;
   kDebugArea(KBABEL_SEARCH,"search ended");
   emit signalSearchActive(false);
}

void SearchResultBox::auxSearch(QString msgid)
{
   if(!_auxFile.isOpen())
   {
      KMessageBox::sorry(this,i18n("Auxiliary search is not available!\n\n\
The file I tried to open was \n%1\n\n\
You may want to edit the \"Search->Files\"-Section\nin the preferences dialog.").arg(_auxFile.name()));

      emit signalEnableAuxSearch(false);
      return;
   }

   _stopSearch=false;
   _searching=true;
   emit signalSearchActive(true);


   clearResults();

   emit signalResetProgressBar(i18n("searching"),_auxFile.size());

   // remember position in file, because it is set to the first entry
   int startIndex=_auxFile.at();

   CatalogItem entry;
   bool found=false;

   QTextStream stream(&_auxFile);
   while(!stream.eof() && !found && !_stopSearch)
   {
      entry.read(stream);

      if(entry.msgid()==msgid)
         found=true;

      emit signalProgress(_auxFile.at());

      kapp->processEvents();
   }

   if(found)
   {
       SearchResultItem item;
       item.item=entry;
       item.package=i18n("Auxiliary");

       _results.append(item);
       setTotalResults(1);
   }
   else
   {
      _msgidBox->setText(i18n("<none>"));
   }

   // set position in file to first entry
   _auxFile.at(startIndex);

   emit signalClearProgressBar();

   _searching=false;
   kDebugArea(KBABEL_SEARCH,"search ended");
   emit signalSearchActive(false);
}

void SearchResultBox::comSearch(QString msgid)
{
   if(!_comFile.isOpen())
   {
      KMessageBox::sorry(this,i18n("Compendium search is not available!\n\n\
The file I tried to open was \n%1\n\n\
You may want to edit the \"Search->Files\"-Section\nin the preferences dialog.").arg(_comFile.name()));

      emit signalEnableComSearch(false);

      return;
   }

   _stopSearch=false;
   _searching=true;
   emit signalSearchActive(true);

   clearResults();


   emit signalResetProgressBar(i18n("searching"),_comFile.size());

   // remember position in file, because it is set to the first entry
   int startIndex=_comFile.at();

   CatalogItem entry;
   int counter=0;

   QTextStream stream(&_comFile);
   while(!stream.eof() && !_stopSearch)
   {
      entry.read(stream);

      QString origStr=msgid;
      QString foundStr=entry.msgid();

      if(foundStr.isEmpty())
         continue;

      if(!_settings.caseSensitive)
      {
         origStr=origStr.lower();
         foundStr=foundStr.lower();
      }
      if(_settings.ignoreAmpersand)
      {
         origStr.replace(QRegExp("&"),"");
         foundStr.replace(QRegExp("&"),"");
      }
      if(_settings.ignoreNewline)
      {
         origStr.replace(QRegExp("\\\\n"),"");
         origStr.replace(QRegExp("\n")," ");
         foundStr.replace(QRegExp("\\\\n"),"");
         foundStr.replace(QRegExp("\n")," ");
      }
      if(_settings.ignoreDots)
      {
         origStr.replace(QRegExp("\\.")," ");
         foundStr.replace(QRegExp("\\.")," ");
      }
      if(_settings.ignoreOther && !_settings.otherIgnores.isEmpty())
      {
         origStr.replace(QRegExp("["+_settings.otherIgnores+"]"),"");
         foundStr.replace(QRegExp("["+_settings.otherIgnores+"]"),"");
      }

      if(foundStr.simplifyWhiteSpace().isEmpty())
         continue;

      bool found=false;
      if(_settings.exact)
      {
         if(origStr==foundStr)
            found=true;
      }
      if(!found && _settings.contains)
      {
         if(foundStr.contains(origStr))
            found=true;
      }
      if(!found && _settings.isContained)
      {
         if(origStr.contains(foundStr))
            found=true;
      }

      if(found)
      {
         counter++;

         SearchResultItem item;
         item.item=entry;
         item.package=i18n("Compendium");

         _results.append(item);
         setTotalResults(counter);
      }

      if(_comFile.at()%100==0)
         emit signalProgress(_comFile.at());

      kapp->processEvents();
   }

   if(counter==0)
   {
      _msgidBox->setText(i18n("<none>"));
   }

   // set position in file to first entry
   _comFile.at(startIndex);

   emit signalClearProgressBar();

   _searching=false;
   kDebugArea(KBABEL_SEARCH,"search ended");
   emit signalSearchActive(false);
}


void SearchResultBox::showResult(int index)
{
      if(index >= (int)_results.count()-1)
      {
         _nextButton->setEnabled(false);
      }
      else
      {
         _nextButton->setEnabled(true);
      }
      if(index==0)
      {
         _prevButton->setEnabled(false);
      }
      else
      {
         _prevButton->setEnabled(true);
      }

      _msgidBox->setText(_results[index].item.msgid());
      _msgstrBox->setText(_results[index].item.msgstr());
      _fileLabel->setText(_results[index].package);

      _currentResult=index;
}

void SearchResultBox::clearResults()
{
   _results.clear();
   _currentResult=-1;

   setTotalResults(0);
   _fileLabel->setText("");

   _prevButton->setEnabled(false);
   _nextButton->setEnabled(false);

   _msgstrBox->setText("");
   _msgidBox->setText("");
}


void SearchResultBox::setTotalResults(int total)
{
   if(total==1)
   {
      showResult(0);
   }

   if(_currentResult<total-1)
   {
      _nextButton->setEnabled(true);
   }
   else
   {
      _nextButton->setEnabled(false);
   }


   _totalLabel->setNum(total);
}

void SearchResultBox::setPackageName(const QString& packageName, const QString& dir)
{
   kDebugStringArea(KBABEL_SEARCH,( QString("setting package name to: ")+packageName ));

   _packageName=packageName;
   _packageDir=dir;

   // check what filenames depend on the package name or package dir
   // and reread if necessary

   QFileInfo info(_settings.moPath);
   bool isRelative=false;
   if(info.isRelative())
   {
      isRelative=true;
   }
   if(_catList.isEmpty() || _settings.moPath.contains("@PACKAGE@") || isRelative)
   {
       createCatList();
   }

   info.setFile(_settings.auxiliary);
   isRelative=false;
   if(info.isRelative())
   {
      isRelative=true;
      QString filename=_packageDir+"/"+_settings.auxiliary;
      info.setFile(filename);
   }
   if(_auxFile.name().isEmpty() || _settings.auxiliary.contains("@PACKAGE@") || info.isDir() || isRelative)
   {
      findAuxFile();
   }

   info.setFile(_settings.compendium);
   isRelative=false;
   if(info.isRelative())
   {
      isRelative=true;
      QString filename=_packageDir+"/"+_settings.compendium;
      info.setFile(filename);
   }
   if(_comFile.name().isEmpty() || _settings.compendium.contains("@PACKAGE@") || info.isDir() || isRelative)
   {
      findComFile();
   }

}


void SearchResultBox::setLanguage(const QString& lang)
{
   // only reread catList and/or filenames when they contain
   // @LANG@ because this is replaced with the language name
   _settings.language=lang;

   if(_catList.isEmpty() || _settings.moPath.contains("@LANG@"))
   {
      createCatList();
   }
   if(_auxFile.name().isEmpty() || _settings.auxiliary.contains("@LANG@"))
   {
      findAuxFile();
   }
   if(_comFile.name().isEmpty() || _settings.compendium.contains("@LANG@"))
   {
      findComFile();
   }
}

void SearchResultBox::nextResult()
{
   if(_currentResult >= (int)_results.count()-1)
      return;

   _currentResult++;
   showResult(_currentResult);
}


void SearchResultBox::prevResult()
{
   if(_currentResult<=0)
      return;

   _currentResult--;
   showResult(_currentResult);
}

void SearchResultBox::stopSearch()
{
   if(_searching)
   {
      _stopSearch=true;
   }

}



void SearchResultBox::createCatList()
{
   //kDebugArea(KBABEL_SEARCH,"creating list of catalogs");

   emit signalEnableCatSearch(false);
}

void SearchResultBox::findAuxFile()
{
   kDebugArea(KBABEL_SEARCH,"searching auxiliary file...");

   if(_auxFile.isOpen())
   {
      _auxFile.close();
   }

   QString filename=_settings.auxiliary;
   filename.replace(QRegExp("@PACKAGE@"),_packageName);
   filename.replace(QRegExp("@LANG@"),_settings.language);

   QFileInfo info(filename);
   if(info.isRelative())
   {
      filename=_packageDir+"/"+filename;
      info.setFile(filename);
   }

   if(info.isDir())
   {
      if(filename.right(1)!="/")
      {
         filename+="/";
      }
      filename+=_packageName+".po";
   }

   _auxFile.setName(filename);

   if(_auxFile.exists())
   {
      if(_auxFile.open(IO_ReadOnly))
      {
         QTextStream stream(&_auxFile);

         CatalogItem temp;
         if(Catalog::readHeader(stream,temp)!=Catalog::OK)
         {
             _auxFile.close();

             kDebugStringArea(KBABEL_SEARCH, (QString("no valid po-file given: ")+filename ));
         }
         else
         {
            kDebugStringArea(KBABEL_SEARCH, ( QString("opened auxiliary file: ")+filename ) );
         }
      }
      else
      {
         kDebugStringArea(KBABEL_SEARCH, (QString("failed to open ")+filename) );
      }

   }
   else
   {
      kDebugStringArea(KBABEL_SEARCH, (QString("file does not exist ")+filename ));
   }


   // always enable searchmodes, so the user can try to use it and only sees
   // the error message when trying to search.
   emit signalEnableAuxSearch(true);


}

void SearchResultBox::findComFile()
{
   kDebugArea(KBABEL_SEARCH,"searching compendium file");

   if(_comFile.isOpen())
   {
      _comFile.close();
   }

   QString filename=_settings.compendium;
   filename.replace(QRegExp("@PACKAGE@"),_packageName);
   filename.replace(QRegExp("@LANG@"),_settings.language);

   QFileInfo info(filename);
   if(info.isRelative())
   {
      filename=_packageDir+"/"+filename;
      info.setFile(filename);
   }


   _comFile.setName(filename);

   if(_comFile.exists() && !info.isDir())
   {
      if(_comFile.open(IO_ReadOnly))
      {
         QTextStream stream(&_comFile);

         CatalogItem temp;
         if(Catalog::readHeader(stream,temp)!=Catalog::OK)
         {
             _comFile.close();
             kDebugStringArea(KBABEL_SEARCH,(QString("no valid po-file given: ")+filename ));
         }
         else
         {
            kDebugStringArea(KBABEL_SEARCH,( QString("opened compendium file: ")+filename ));
         }
      }
      else
      {
         kDebugStringArea(KBABEL_SEARCH, (QString("failed to open %s")+filename) );
      }

   }
   else
   {
      kDebugStringArea(KBABEL_SEARCH, (QString("file does not exist %s")+filename) );
   }

   // always enable searchmodes, so the user can try to use it and only sees
   // the error message when trying to search.
   emit signalEnableComSearch(true);
}

void SearchResultBox::readSettings()
{
   KConfig* config=KGlobal::config();

   KConfigGroupSaver gs(config,"Search");

   _settings.caseSensitive=config->readBoolEntry("CaseSensitive"
                       ,Defaults::Search::caseSensitive);

   _settings.ignoreAmpersand=config->readBoolEntry("IgnoreAmpersand"
                       ,Defaults::Search::ignoreAmpersand);
   _settings.ignoreDots=config->readBoolEntry("IgnoreDots"
                       ,Defaults::Search::ignoreDots);
   _settings.ignoreNewline=config->readBoolEntry("IgnoreNewline"
                       ,Defaults::Search::ignoreNewline);
   _settings.ignoreOther=config->readBoolEntry("IgnoreOther"
                       ,Defaults::Search::ignoreOther);
   _settings.otherIgnores=config->readEntry("OtherIgnores"
                       ,Defaults::Search::otherIgnores);

   _settings.exact=config->readBoolEntry("Exact"
                       ,Defaults::Search::exact);
   _settings.contains=config->readBoolEntry("Contains"
                       ,Defaults::Search::contains);
   _settings.isContained=config->readBoolEntry("IsContained"
                       ,Defaults::Search::isContained);

   _settings.language=config->readEntry("DefaultLang",Defaults::Search::defaultLanguage());

   _settings.moPath=config->readEntry("MOPath",Defaults::Search::moPath);
   _settings.auxiliary=config->readEntry("Auxiliary",Defaults::Search::auxiliary);
   _settings.compendium=config->readEntry("Compendium",Defaults::Search::compendium);

   _settings.autoSearch=config->readBoolEntry("AutoSearch",Defaults::Search::autoSearch);

   _settings.defaultMode=(SearchMode)config->readNumEntry("DefaultMode",
                      (int)Defaults::Search::defaultMode);
}


void SearchResultBox::saveSettings()
{
   KConfig* config=KGlobal::config();

   KConfigGroupSaver gs(config,"Search");

   config->writeEntry("CaseSensitive",_settings.caseSensitive);

   config->writeEntry("IgnoreAmpersand",_settings.ignoreAmpersand);
   config->writeEntry("IgnoreDots",_settings.ignoreDots);
   config->writeEntry("IgnoreNewline",_settings.ignoreNewline);
   config->writeEntry("IgnoreOther",_settings.ignoreOther);
   config->writeEntry("OtherIgnores",_settings.otherIgnores);

   config->writeEntry("Exact",_settings.exact);
   config->writeEntry("Contains",_settings.contains);
   config->writeEntry("IsContained",_settings.isContained);

   config->writeEntry("DefaultLang",_settings.language);

   config->writeEntry("MOPath",_settings.moPath);
   config->writeEntry("Auxiliary",_settings.auxiliary);
   config->writeEntry("Compendium",_settings.compendium);

   config->writeEntry("AutoSearch",_settings.autoSearch);
   config->writeEntry("DefaultMode",(int)_settings.defaultMode);
}

void SearchResultBox::setSettings(SearchSettings settings)
{
   const SearchSettings oldSettings=_settings;
   _settings=settings;

   bool languageChanged=false;
   if(oldSettings.language != _settings.language) // if something changed we have to check the filenames
   {
      languageChanged=true;
   }

   if(oldSettings.moPath != _settings.moPath)
   {
      // if the mask for the catalogs changed, reread the list of catalogs
      if(!_packageName.isEmpty())
      {
         createCatList();
      }
   }
   else if(languageChanged && _settings.moPath.contains("@LANG@"))
   {
      if(!_packageName.isEmpty())
      {
         createCatList();
      }
   }

   if(oldSettings.auxiliary != _settings.auxiliary)
   {
      // if the mask for the auxiliary file changed, reopen the file
      if(!_packageName.isEmpty())
      {
         findAuxFile();
      }
   }
   else if(languageChanged && _settings.auxiliary.contains("@LANG@"))
   {
      if(!_packageName.isEmpty())
      {
         findAuxFile();
      }
   }

   if(oldSettings.compendium != _settings.compendium)
   {
      // if the mask for the compendium file changed, reopen the file
      if(!_packageName.isEmpty())
      {
         findComFile();
      }
   }
   else if(languageChanged && _settings.compendium.contains("@LANG@"))
   {
      if(!_packageName.isEmpty())
      {
         findComFile();
      }
   }
}


bool SearchResultBox::addToMOFiles(QString)
{
   return false;
}

bool SearchResultBox::setCompendiumFile(QString url)
{
   bool success=false;

   KURL tempURL(QDir::currentDirPath()+"/",url);

   if(!tempURL.isLocalFile())
   {
      KMessageBox::sorry(this,i18n("KBabel can only search in local files!"));
   }
   else
   {
      QFileInfo file(tempURL.path(0));

      if(!file.exists())
      {
         KMessageBox::sorry(this,i18n("The file does not exist:\n%1").arg(file.filePath()));
      }
      else if(!file.isReadable())
      {
         KMessageBox::sorry(this
            ,i18n("You don't have permissions to read file:\n%1").arg(file.filePath()));
      }
      else if(file.isDir())
      {
         KMessageBox::sorry(this
            ,i18n("You have specified a directory not a file:\n%1").arg(file.filePath()));
      }
      else
      {
         QString temp;
         Msgfmt msgfmt;
         Msgfmt::Status result=msgfmt.checkSyntax(file.filePath(),temp);

         if( result == Msgfmt::Ok || result == Msgfmt::NoExecutable)
         {
            _settings.compendium=file.filePath();
            emit signalSettingsChanged(_settings);

            if(_comFile.isOpen())
               _comFile.close();

            _comFile.setName(file.filePath());

            if(_comFile.open(IO_ReadOnly))
            {
               QTextStream stream(&_comFile);

               CatalogItem temp;
               if(Catalog::readHeader(stream,temp)!=Catalog::OK)
               {
                  _comFile.close();
                   KMessageBox::sorry(this
                      ,i18n("This is not a valid PO-file:\n%1").arg(file.filePath()));
               }
               else
               {
                  success=true;
                  emit signalEnableComSearch(true);
               }
            }
            else
            {
               KMessageBox::sorry(this
                  ,i18n("Error reading file:\n%1").arg(file.filePath()));
            }
         }
         else
         {
            KMessageBox::sorry(this
              ,i18n("This is not a valid PO-file:\n%1").arg(file.filePath()));
         }
      }
   }

   return success;
}

bool SearchResultBox::setAuxiliaryFile(QString url)
{
   bool success=false;

   KURL tempURL(QDir::currentDirPath()+"/",url);

   if(!tempURL.isLocalFile())
   {
      KMessageBox::sorry(this,i18n("KBabel can only search in local files!"));
   }
   else
   {
      QFileInfo file(tempURL.path(0));

      if(!file.exists())
      {
         KMessageBox::sorry(this,i18n("The file does not exist:\n%1").arg(file.filePath()));
      }
      else if(!file.isReadable())
      {
         KMessageBox::sorry(this
            ,i18n("You don't have permissions to read file:\n%1").arg(file.filePath()));
      }
      else if(file.isDir())
      {
         KMessageBox::sorry(this
            ,i18n("You have specified a directory not a file:\n%1").arg(file.filePath()));
      }
      else
      {
         QString temp;
         Msgfmt msgfmt;
         Msgfmt::Status result=msgfmt.checkSyntax(file.filePath(),temp);

         if( result == Msgfmt::Ok || result == Msgfmt::NoExecutable)
         {
            _settings.auxiliary=file.filePath();
            emit signalSettingsChanged(_settings);

            if(_auxFile.isOpen())
               _auxFile.close();

            _auxFile.setName(file.filePath());

            if(_auxFile.open(IO_ReadOnly))
            {
               QTextStream stream(&_auxFile);

               CatalogItem temp;
               if(Catalog::readHeader(stream,temp)!=Catalog::OK)
               {
                  _auxFile.close();
                   KMessageBox::sorry(this
                      ,i18n("This is not a valid PO-file:\n%1").arg(file.filePath()));
               }
               else
               {
                  success=true;
                  emit signalEnableAuxSearch(true);
               }
            }
            else
            {
               KMessageBox::sorry(this
                  ,i18n("Error reading file:\n%1").arg(file.filePath()));
            }
         }
         else
         {
            KMessageBox::sorry(this
              ,i18n("This is not a valid PO-file:\n%1").arg(file.filePath()));
         }
      }
   }

   return success;
}


void SearchResultBox::dragEnterEvent( QDragEnterEvent *event)
{
   if(QUriDrag::canDecode(event))
      event->accept(true);
   else if(QTextDrag::canDecode(event))
   {
      if(_catList.isEmpty() && !_comFile.isOpen() && ! _auxFile.isOpen())
         event->accept(false);
      else
         event->accept(true);
   }
   else
      event->accept(false);
}


void SearchResultBox::dropEvent( QDropEvent* event)
{
   QString text;
   QStrList uri;

   // see if we can decode a URI.. if not, just ignore it
   if (QUriDrag::decode(event, uri))
   {
       // okay, we have a URI.. process it
       QString url = uri.first();


       int result = _urlDropMenu->exec(mapToGlobal(event->pos()));
       switch(result)
       {
          case ID_DROP_MOFILES:
             addToMOFiles(url);
             break;
          case ID_DROP_COMFILE:
             setCompendiumFile(url);
             break;
          case ID_DROP_AUXFILE:
             setAuxiliaryFile(url);
             break;
       }

   }
   else if ( QTextDrag::decode(event, text) )
   {
        _textDropMenu->setItemEnabled(ID_DROP_MO,!_catList.isEmpty());
        _textDropMenu->setItemEnabled(ID_DROP_COM,_comFile.isOpen());
        _textDropMenu->setItemEnabled(ID_DROP_AUX,_auxFile.isOpen());

        int result=_textDropMenu->exec(mapToGlobal(event->pos()));

        switch(result)
        {
           case ID_DROP_MO:
              startDelayedSearch(text,CatalogSearch);
              break;
           case ID_DROP_COM:
              startDelayedSearch(text,CompendiumSearch);
              break;
           case ID_DROP_AUX:
              startDelayedSearch(text,AuxiliarySearch);
              break;
        }
   }

}
