/***************************************************************************
filter.cpp  -  description
-------------------
begin                : Sun Mar 19 11:30:14 MET 2000
copyright            : (C) 2001 by Martin Erdtmann  /  Stefan Okrongli
email                : martin.erdtmann@gmx.de  /  s_okrongli@gmx.net
***************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

// include files for QT
#include <qdir.h>
#include <qstrlist.h>
#include <qprinter.h>
#include <qpainter.h>

// include files for KDE
#include <kcolordlg.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kconfig.h>
#include <kstdaction.h>

// application specific includes
#include "kfilter.h"
#include "kfilterview.h"
#include "kfilterdoc.h"
#include "resource.h"
#include "colordialog.h"


FilterApp::FilterApp()
{
	m_pkconfigConfiguration=kapp->config();
	
	///////////////////////////////////////////////////////////////////
	// call inits to invoke all other construction parts
	InitializeMenuBar();
	InitializeToolBar();
	InitializeStatusBar();
	InitializeAccelerators();
	InitializeDocument();
	InitializeView();
   	
	ReadOptions();
	
	///////////////////////////////////////////////////////////////////
	// disable menu and toolbar items at startup
	DisableCommand(ID_FILE_SAVE);
	DisableCommand(ID_FILE_SAVE_AS);
	DisableCommand(ID_FILE_PRINT);
	
	DisableCommand(ID_EDIT_CUT);
	DisableCommand(ID_EDIT_COPY);
	DisableCommand(ID_EDIT_PASTE);
}

FilterApp::~FilterApp()
{
	
}

void FilterApp::InitializeAccelerators()
{
	m_pkaccelAccelerators = new KAccel(this);
	
	// fileMenu accelerators
	m_pkaccelAccelerators->connectItem(KStdAccel::New, this, SLOT(slotFileNew()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Open, this, SLOT(slotFileOpen()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Save, this, SLOT(slotFileSave()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Close, this, SLOT(slotFileClose()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Print, this, SLOT(slotFilePrint()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Quit, this, SLOT(slotFileQuit()));
	// editMenu accelerators
	m_pkaccelAccelerators->connectItem(KStdAccel::Cut, this, SLOT(slotEditCut()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Copy, this, SLOT(slotEditCopy()));
	m_pkaccelAccelerators->connectItem(KStdAccel::Paste, this, SLOT(slotEditPaste()));
	
	m_pkaccelAccelerators->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));
	
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuFileMenu, ID_FILE_NEW, KStdAccel::New);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuFileMenu, ID_FILE_OPEN, KStdAccel::Open);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuFileMenu, ID_FILE_SAVE, KStdAccel::Save);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuFileMenu, ID_FILE_CLOSE, KStdAccel::Close);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuFileMenu, ID_FILE_PRINT, KStdAccel::Print);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuFileMenu, ID_FILE_QUIT, KStdAccel::Quit);
	
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuEditMenu, ID_EDIT_CUT, KStdAccel::Cut);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuEditMenu, ID_EDIT_COPY, KStdAccel::Copy);
	m_pkaccelAccelerators->changeMenuAccel( m_pqpopupmenuEditMenu, ID_EDIT_PASTE, KStdAccel::Paste);
	
	m_pkaccelAccelerators->readSettings();	
}

void FilterApp::InitializeMenuBar()
{
	///////////////////////////////////////////////////////////////////
	// MENUBAR
	m_pqpopupmenuRecentFilesMenu = new QPopupMenu();
	
	///////////////////////////////////////////////////////////////////
	// menuBar entry fileMenu
	m_pqpopupmenuFileMenu = new QPopupMenu();
	m_pqpopupmenuFileMenu->insertItem(kapp->miniIcon(), i18n("New &Window"), ID_FILE_NEW_WINDOW);
	m_pqpopupmenuFileMenu->insertSeparator();
	m_pqpopupmenuFileMenu->insertItem(SmallIcon("filenew.xpm"), i18n("&New"), ID_FILE_NEW);
	m_pqpopupmenuFileMenu->insertItem(SmallIcon("fileopen.xpm"), i18n("&Open..."), ID_FILE_OPEN);
	m_pqpopupmenuFileMenu->insertItem(i18n("Open &recent"), m_pqpopupmenuRecentFilesMenu, \
		ID_FILE_OPEN_RECENT);
	connect( m_pqpopupmenuRecentFilesMenu, SIGNAL(activated(int)), SLOT(slotFileOpenRecent(int)));
	
	m_pqpopupmenuFileMenu->insertItem(i18n("&Close"), ID_FILE_CLOSE);
	m_pqpopupmenuFileMenu->insertSeparator();
	m_pqpopupmenuFileMenu->insertItem(SmallIcon("filefloppy.xpm") ,i18n("&Save"), ID_FILE_SAVE);
	m_pqpopupmenuFileMenu->insertItem(i18n("Save &As..."), ID_FILE_SAVE_AS);
	m_pqpopupmenuFileMenu->insertSeparator();
	m_pqpopupmenuFileMenu->insertItem(SmallIcon("fileprint.xpm"), i18n("&Print..."), ID_FILE_PRINT);
	m_pqpopupmenuFileMenu->insertSeparator();
	m_pqpopupmenuFileMenu->insertItem(i18n("E&xit"), ID_FILE_QUIT);
	
	///////////////////////////////////////////////////////////////////
	// menuBar entry editMenu
	m_pqpopupmenuEditMenu = new QPopupMenu();
	m_pqpopupmenuEditMenu->insertItem(SmallIcon("editcut.xpm"), i18n("Cu&t"), ID_EDIT_CUT);
	m_pqpopupmenuEditMenu->insertItem(SmallIcon("editcopy.xpm"), i18n("&Copy"), ID_EDIT_COPY);
	m_pqpopupmenuEditMenu->insertItem(SmallIcon("editpaste.xpm"), i18n("&Paste"), ID_EDIT_PASTE);
	
	///////////////////////////////////////////////////////////////////
	// menuBar entry viewMenu
	m_pqpopupmenuViewMenu = new QPopupMenu();
	m_pqpopupmenuViewMenu->setCheckable(true);
	m_pqpopupmenuViewMenu->insertItem(i18n("&Toolbar"), ID_VIEW_TOOLBAR);
	m_pqpopupmenuViewMenu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR);
	
	///////////////////////////////////////////////////////////////////
	
	m_pqpopupmenuDriverMenu = new QPopupMenu();
	m_pqpopupmenuDriverMenu->insertItem(SmallIcon("driver.xpm"), i18n("&Parameter"), ID_DRIVER_PARAM);
	m_pqpopupmenuDriverMenu->insertItem(SmallIcon("mini-network.xpm"), i18n("&Network"), ID_DRIVER_NETWORK);
	m_pqpopupmenuDriverMenu->insertItem(SmallIcon("mini-volume.xpm"), i18n("&Volume"), ID_DRIVER_VOLUME);
	///////////////////////////////////////////////////////////////////
	m_pqpopupmenuToolsMenu = new QPopupMenu();
	m_pqpopupmenuToolsMenu->insertItem(SmallIcon("filterwiz.xpm"), i18n("&FilterWizard"), ID_TOOLS_WIZARD);
	///////////////////////////////////////////////////////////////////
	m_pqpopupmenuOptionsMenu = new QPopupMenu();
	m_pqpopupmenuOptionsMenu->insertItem(SmallIcon("fill.png"), i18n("&Colors"), ID_OPTIONS_COLORS);
	///////////////////////////////////////////////////////////////////
	// menuBar entry helpMenu
	QString aboutstring=QString(kapp->name())+
		", a tool to design loudspeakers. Version "+VERSION+"\n"
		"created by Martin Erdtmann (martin.erdtmann@gmx.de)\n";
	
	m_pqpopupmenuHelpMenu = new QPopupMenu();
	m_pqpopupmenuHelpMenu = helpMenu(aboutstring, true);
	
	///////////////////////////////////////////////////////////////////
	// MENUBAR CONFIGURATION
	// insert your popup menus with the according menubar entries in the order
	// they will appear later from left to right
	//	KMenuBar *_menuBar = new KMenuBar();
	menuBar()->insertItem(i18n("&File"), m_pqpopupmenuFileMenu);
	menuBar()->insertItem(i18n("&Edit"), m_pqpopupmenuEditMenu);
	menuBar()->insertItem(i18n("&View"), m_pqpopupmenuViewMenu);
	menuBar()->insertItem(i18n("&Driver"), m_pqpopupmenuDriverMenu);
	menuBar()->insertItem(i18n("&Tools"), m_pqpopupmenuToolsMenu);
	menuBar()->insertItem(i18n("&Options"), m_pqpopupmenuOptionsMenu);
	
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), m_pqpopupmenuHelpMenu);
	
	///////////////////////////////////////////////////////////////////
	// CONNECT THE MENU SLOTS WITH SIGNALS
	// for execution slots and statusbar messages
	
	connect( m_pqpopupmenuFileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect( m_pqpopupmenuFileMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	
	connect( m_pqpopupmenuEditMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect( m_pqpopupmenuEditMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	
	connect( m_pqpopupmenuViewMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect( m_pqpopupmenuViewMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	
	connect( m_pqpopupmenuDriverMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect( m_pqpopupmenuDriverMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	
	connect( m_pqpopupmenuToolsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect( m_pqpopupmenuToolsMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	
	connect( m_pqpopupmenuOptionsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect( m_pqpopupmenuOptionsMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	
}

void FilterApp::InitializeToolBar()
{
	
	///////////////////////////////////////////////////////////////////
	// TOOLBAR
	toolBar()->insertButton(SmallIcon("filenew.xpm"), ID_FILE_NEW, true, i18n("New File"));
	toolBar()->insertButton(SmallIcon("fileopen.xpm"), ID_FILE_OPEN, true, i18n("Open File"));
	toolBar()->insertButton(SmallIcon("filesave.xpm"), ID_FILE_SAVE, true, i18n("Save File"));
	toolBar()->insertButton(SmallIcon("fileprint.xpm"), ID_FILE_PRINT, true, i18n("Print"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(SmallIcon("editcut.xpm"), ID_EDIT_CUT, true, i18n("Cut"));
	toolBar()->insertButton(SmallIcon("editcopy.xpm"), ID_EDIT_COPY, true, i18n("Copy"));
	toolBar()->insertButton(SmallIcon("editpaste.xpm"), ID_EDIT_PASTE, true, i18n("Paste"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(SmallIcon("help.xpm"), ID_HELP_CONTENTS, SIGNAL(clicked()),
		this, SLOT(appHelpActivated()), true,i18n("Help"));
	
	///////////////////////////////////////////////////////////////////
	// INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)
	
	///////////////////////////////////////////////////////////////////
	// CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
	// connect for invoking the slot actions
	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
	// connect for the status help on holing icons pressed with the mouse button
	connect(toolBar(), SIGNAL(pressed(int)), SLOT(statusCallback(int)));
	
}

void FilterApp::InitializeStatusBar()
{
	///////////////////////////////////////////////////////////////////
	// STATUSBAR
	// TODO: add your own items you need for displaying current application status.
	statusBar()->insertItem(i18n(IDS_STATUS_DEFAULT), ID_STATUS_MSG);
	statusBar()->insertItem(i18n(" Ready"), ID_STATUS_MSG);
}

void FilterApp::InitializeDocument()
{
	m_pfilterdocFilterDocument = new FilterDoc(this);
	m_pfilterdocFilterDocument->newDocument();
	QObject::connect(m_pfilterdocFilterDocument, SIGNAL(forceviewrefresh() ), this, SLOT( slotrefreshview() ) );
}

void FilterApp::InitializeView()
{ 
	////////////////////////////////////////////////////////////////////
	// create the main widget here that is managed by KTMainWindow's view-region and
	// connect the widget to your document to display document contents.
	
	m_pfilterviewFilterView = new FilterView(this);
	m_pfilterdocFilterDocument->addView(m_pfilterviewFilterView);
	setCentralWidget(m_pfilterviewFilterView);
	QString caption=kapp->caption();
	setCaption(caption+"KFilter: "+m_pfilterdocFilterDocument->getTitle());
	
}

void FilterApp::EnableCommand( int a_intCommandID )
{
	///////////////////////////////////////////////////////////////////
	// enable menu and toolbar functions by their ID's
	menuBar()->setItemEnabled( a_intCommandID, true );
	toolBar()->setItemEnabled( a_intCommandID, true );
}

void FilterApp::DisableCommand(int a_intCommandID )
{
	///////////////////////////////////////////////////////////////////
	// disable menu and toolbar functions by their ID's
	menuBar()->setItemEnabled( a_intCommandID, false );
	toolBar()->setItemEnabled( a_intCommandID, false );
}

void FilterApp::addRecentFile(const QString & a_qstringFileName )
{
	if( m_qstrlistRecentFiles.find( a_qstringFileName ) == -1 )
	{
		if( m_qstrlistRecentFiles.count() < 5)
		{
			m_qstrlistRecentFiles.insert( 0, a_qstringFileName );
		}
		else
		{
			m_qstrlistRecentFiles.remove( 4 );
			m_qstrlistRecentFiles.insert( 0, a_qstringFileName );
		}
		m_pqpopupmenuRecentFilesMenu->clear();
		for ( int intI =0 ; intI < (int) m_qstrlistRecentFiles.count(); intI++ )
		{
			m_pqpopupmenuRecentFilesMenu->insertItem( m_qstrlistRecentFiles.at( intI ), intI );
		}
	}
}

void FilterApp::openDocumentFile(const char* a_pcharCommandLine )
{
	slotStatusMsg(i18n("Opening file..."));
	
	m_pfilterdocFilterDocument->openDocument( a_pcharCommandLine );
	QString caption=kapp->caption();
	setCaption(caption+"KFilter: "+m_pfilterdocFilterDocument->getTitle());
	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}


FilterDoc *FilterApp::getDocument() const
{
	return m_pfilterdocFilterDocument;
}

void FilterApp::SaveOptions()
{	
	m_pkconfigConfiguration->setGroup("General Options");
	m_pkconfigConfiguration->writeEntry("Geometry", size());
	m_pkconfigConfiguration->writeEntry("Show Toolbar", toolBar()->isVisible());
	m_pkconfigConfiguration->writeEntry("Show Statusbar",statusBar()->isVisible());
	//  m_pkconfigConfiguration->writeEntry("MenuBarPos", (int) menuBar()->menuBarPos());
	m_pkconfigConfiguration->writeEntry("ToolBarPos", (int) toolBar()->barPos());
	m_pkconfigConfiguration->writeEntry("Recent Files", m_qstrlistRecentFiles);
	m_pkconfigConfiguration->writeEntry("BackgroundColor", m_pfilterviewFilterView->backgroundColor());
	m_pkconfigConfiguration->writeEntry("GridColor", m_pfilterviewFilterView->gridColor());
	m_pkconfigConfiguration->writeEntry("PressureColor", m_pfilterviewFilterView->pressureColor());
	m_pkconfigConfiguration->writeEntry("ImpedanceColor", m_pfilterviewFilterView->impedanceColor());
	m_pkconfigConfiguration->writeEntry("PressureSColor", m_pfilterviewFilterView->pressureSummaryColor());
	m_pkconfigConfiguration->writeEntry("ImpedanceSColor", m_pfilterviewFilterView->impedanceSummaryColor());
	m_pkconfigConfiguration->writeEntry("ScalarPressureSColor", m_pfilterviewFilterView->scalarPressureSummaryColor());
}


void FilterApp::ReadOptions()
{
	
	m_pkconfigConfiguration->setGroup("General Options");
	
	QColor qcolorBackgroundColor( 150, 52, 52 );
	qcolorBackgroundColor = m_pkconfigConfiguration->readColorEntry( "BackgroundColor", &qcolorBackgroundColor );
	m_pfilterviewFilterView->setBackgroundColor( qcolorBackgroundColor );
	
	QColor qcolorGridColor( black );
	qcolorGridColor = m_pkconfigConfiguration->readColorEntry( "GridColor", &qcolorGridColor );
	m_pfilterviewFilterView->setGridColor( qcolorGridColor );
	
	QColor qcolorPressureColor( white );
	qcolorPressureColor = m_pkconfigConfiguration->readColorEntry( "PressureColor", &qcolorPressureColor );
	m_pfilterviewFilterView->setPressureColor( qcolorPressureColor );
	
	QColor qcolorImpedanceColor( yellow );
	qcolorImpedanceColor = m_pkconfigConfiguration->readColorEntry( "ImpedanceColor", &qcolorImpedanceColor );
	m_pfilterviewFilterView->setImpedanceColor( qcolorImpedanceColor );
	
	QColor qcolorPressureSummaryColor( blue );
	qcolorPressureSummaryColor = m_pkconfigConfiguration->readColorEntry( "PressureSColor", &qcolorPressureSummaryColor );
	m_pfilterviewFilterView->setPressureSummaryColor( qcolorPressureSummaryColor );
	
	QColor qcolorImpedanceSummaryColor( green );
	qcolorImpedanceSummaryColor = m_pkconfigConfiguration->readColorEntry( "ImpedanceSColor", &qcolorImpedanceSummaryColor );
	m_pfilterviewFilterView->setImpedanceSummaryColor( qcolorImpedanceSummaryColor );
	
	QColor qcolorScalarPressureSummaryColor( red );
	qcolorScalarPressureSummaryColor = m_pkconfigConfiguration->readColorEntry( "ScalarPressureSColor",
		&qcolorScalarPressureSummaryColor );
	m_pfilterviewFilterView->setScalarPressureSummaryColor( qcolorScalarPressureSummaryColor );
	
	// bar status settings
	bool boolViewToolbar = m_pkconfigConfiguration->readBoolEntry( "Show Toolbar", true);
	m_pqpopupmenuViewMenu->setItemChecked( ID_VIEW_TOOLBAR, boolViewToolbar );
	if( !boolViewToolbar )
	{
		//     enableToolBar(KToolBar::Hide);
	}
	
	bool boolViewStatusbar = m_pkconfigConfiguration->readBoolEntry( "Show Statusbar", true);
	m_pqpopupmenuViewMenu->setItemChecked(ID_VIEW_STATUSBAR, boolViewStatusbar );
	if( !boolViewStatusbar )
	{
		//    enableStatusBar( KStatusBar::Hide );
	}
	
	// bar position settings
	/*  KMenuBar::menuPosition menupositionBarPosition;
	menupositionBarPosition=(KMenuBar::menuPosition) m_pkconfigConfiguration->readNumEntry("MenuBarPos", KMenuBar::Top);
	

	  KToolBar::BarPosition menupositionToolBarPos;
	  menupositionToolBarPos=(KToolBar::BarPosition) m_pkconfigConfiguration->readNumEntry("ToolBarPos", KToolBar::Top);
	  
		menuBar()->setMenuBarPos( menupositionBarPosition );
		
		  toolBar()->setBarPos( menupositionToolBarPos );
	*/	
	// initialize the recent file list
	m_qstrlistRecentFiles.setAutoDelete( TRUE );
	m_pkconfigConfiguration->readListEntry( "Recent Files", m_qstrlistRecentFiles );
	
	for (int intCount = 0; intCount < (int) m_qstrlistRecentFiles.count(); intCount++ )
	{
		m_pqpopupmenuRecentFilesMenu->insertItem( m_qstrlistRecentFiles.at( intCount ), intCount );
	}
	
	QSize qsizeSize = m_pkconfigConfiguration->readSizeEntry( "Geometry" );
	if( !qsizeSize.isEmpty() )
	{
		resize( qsizeSize );
	}
}

void FilterApp::saveProperties( KConfig * a_pkconfigConfiguration )
{
	if( m_pfilterdocFilterDocument->getTitle()!=i18n("Untitled") &&
		!m_pfilterdocFilterDocument->isModified() )
	{
		// saving to tempfile not necessary
		
	}
	else
	{
		QString qstringFilename = m_pfilterdocFilterDocument->getAbsFilePath();	
		a_pkconfigConfiguration->writeEntry( "filename", qstringFilename );
		a_pkconfigConfiguration->writeEntry( "modified", m_pfilterdocFilterDocument->isModified());
		
		QString qstringTempname = kapp->tempSaveName( qstringFilename );
		m_pfilterdocFilterDocument->saveDocument( qstringTempname );
	}
}


void FilterApp::readProperties( KConfig* a_pkconfigConfiguration )
{
	QString qstringFilename = a_pkconfigConfiguration->readEntry( "filename", "" );
	bool modified = a_pkconfigConfiguration->readBoolEntry("modified", false);
	if(modified)
	{
		bool boolCanRecover;
		QString qstringTempname = kapp->checkRecoverFile( qstringFilename, boolCanRecover );
		
		if( boolCanRecover )
		{
			m_pfilterdocFilterDocument->openDocument( qstringTempname );
			m_pfilterdocFilterDocument->setModified();
			QFileInfo info( qstringFilename );
			m_pfilterdocFilterDocument->setAbsFilePath( info.absFilePath() );
			m_pfilterdocFilterDocument->setTitle( info.fileName() );
			QFile::remove( qstringTempname );
		}
	}
	else
	{
		if( !qstringFilename.isEmpty() )
		{
			m_pfilterdocFilterDocument->openDocument( qstringFilename );
		}
	}
	
	QString qstringCaption = kapp->caption();	
	setCaption( qstringCaption + "KFilter: " + m_pfilterdocFilterDocument->getTitle() );
}		

bool FilterApp::queryClose()
{
	return m_pfilterdocFilterDocument->saveModified();
}

bool FilterApp::queryExit()
{
	SaveOptions();
	return true;

}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void FilterApp::slotFileNewWindow()
{
	slotStatusMsg(i18n("Opening a new application window..."));
	
	FilterApp *pfilterappNewWIndow = new FilterApp();
	pfilterappNewWIndow->show();
	
	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void FilterApp::slotFileNew()
{
	slotStatusMsg(i18n("Creating new document..."));
	
	if( !m_pfilterdocFilterDocument->saveModified() )
	{
		// here saving wasn't successful
		
	}
	else
	{	
		m_pfilterdocFilterDocument->newDocument();
		DisableCommand(ID_FILE_SAVE);
		DisableCommand(ID_FILE_SAVE_AS);
		
		QString qstringCaption = kapp->caption();	
		setCaption( qstringCaption + "KFilter: " + m_pfilterdocFilterDocument->getTitle() );
	}
	
	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void FilterApp::slotFileOpen()
{
	slotStatusMsg(i18n("Opening file..."));
	
	if(!m_pfilterdocFilterDocument->saveModified())
	{
		// here saving wasn't successful
		
	}
	else
	{	
		QString qstringFileName = KFileDialog::getOpenFileName( QDir::homeDirPath(),
			i18n( "*.kfp|KFilter project files\n*.*|All FIles"), this, i18n( "Open File..." ) );
		
		if( !qstringFileName.isEmpty() )
		{
			m_pfilterdocFilterDocument->openDocument( qstringFileName );
			QString qstringCaption = kapp->caption();	
			setCaption( qstringCaption + "KFilter: " + m_pfilterdocFilterDocument->getTitle() );
			addRecentFile( qstringFileName );
			EnableCommand( ID_FILE_SAVE );
			EnableCommand( ID_FILE_SAVE_AS );
		}
		
	}
	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotFileOpenRecent( int a_intFileIndex )
{
	// SO - funny bug: get index -16 !
	// maybe incompatible libraries
	if ( a_intFileIndex < 0 )
	{
		a_intFileIndex = -a_intFileIndex - 16;
	}
	slotStatusMsg( i18n("Opening file...") );
	
	if( !m_pfilterdocFilterDocument->saveModified() )
	{
		// here saving wasn't successful
	}
	else
	{
		m_pfilterdocFilterDocument->openDocument( m_qstrlistRecentFiles.at( a_intFileIndex ) );
		QString qstringCaption = kapp->caption();	
		setCaption( qstringCaption + "KFilter: " + m_pfilterdocFilterDocument->getTitle() );
	}
	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotFileSave()
{
	slotStatusMsg( i18n("Saving file...") );
	
	m_pfilterdocFilterDocument->saveDocument( m_pfilterdocFilterDocument->getAbsFilePath() );
	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotFileSaveAs()
{
	slotStatusMsg( i18n("Saving file with a new filename...") );
	
	QString qstringNewName = KFileDialog::getSaveFileName( QDir::currentDirPath(),
		i18n("*.kfp|KFilter project files\n*.*|All FIles"), this, i18n("Save as...") );
	
	if( !qstringNewName.isEmpty() )
	{
		// check for the right extension
		QString qstringExtension = ".kfp";
		uint uintPosition = qstringNewName.findRev( qstringExtension, -1, FALSE );
		if ( uintPosition != qstringNewName.length() - qstringExtension.length() )
		{
			// that means that the the extension is not the end of the filename
			qstringNewName += qstringExtension;
		}
		
		QFileInfo saveAsInfo( qstringNewName );
		
		m_pfilterdocFilterDocument->setTitle( saveAsInfo.fileName() );
		m_pfilterdocFilterDocument->setAbsFilePath( saveAsInfo.absFilePath() );
		m_pfilterdocFilterDocument->saveDocument( qstringNewName );
		addRecentFile( qstringNewName );
		EnableCommand(ID_FILE_SAVE);
		
		QString qstringCaption = kapp->caption();	
		setCaption( qstringCaption + "KFilter: " + m_pfilterdocFilterDocument->getTitle() );
	}
	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotFileClose()
{
	slotStatusMsg( i18n("Closing file...") );
	
	close();
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotFilePrint()
{
	slotStatusMsg( i18n("Printing...") );
	
	QPrinter qprinterPrinter;
	if ( qprinterPrinter.setup(this))
	{
		m_pfilterviewFilterView->print( &qprinterPrinter );
	}
	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotFileQuit()
{
	slotStatusMsg( i18n("Exiting...") );
	SaveOptions();
	// close the first window, the list makes the next one the first again.
	// This ensures that queryClose() is called on each window to ask for closing
	KMainWindow* pkmainwindowMainWindow;
	if( memberList )
	{
		for( pkmainwindowMainWindow = memberList->first();
		pkmainwindowMainWindow != NULL;
		pkmainwindowMainWindow = memberList->first() )
		{
			// only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
			// the window and the application stay open.
			if( !pkmainwindowMainWindow->close() )
				break;
		}
	}	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotEditCut()
{
	slotStatusMsg(i18n("Cutting selection..."));
	
	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void FilterApp::slotEditCopy()
{
	slotStatusMsg( i18n("Copying selection to clipboard...") );
	
	slotStatusMsg( i18n(IDS_STATUS_DEFAULT) );
}

void FilterApp::slotEditPaste()
{
	slotStatusMsg(i18n("Inserting clipboard contents..."));
	
	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void FilterApp::slotViewToolBar()
{
	slotStatusMsg( i18n("Toggle the toolbar...") );
	///////////////////////////////////////////////////////////////////
	// turn Toolbar on or off
	if( m_pqpopupmenuViewMenu->isItemChecked(ID_VIEW_TOOLBAR) )
	{
		m_pqpopupmenuViewMenu->setItemChecked( ID_VIEW_TOOLBAR, false );
		//    enableToolBar(KToolBar::Hide);
	toolBar()->hide();
	}
	else
	{
		m_pqpopupmenuViewMenu->setItemChecked( ID_VIEW_TOOLBAR, true );
		//    enableToolBar(KToolBar::Show);
	toolBar()->show();
	}		

	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void FilterApp::slotViewStatusBar()
{
	slotStatusMsg(i18n("Toggle the statusbar..."));
	///////////////////////////////////////////////////////////////////
	//turn Statusbar on or off
	if( m_pqpopupmenuViewMenu->isItemChecked( ID_VIEW_STATUSBAR ) )
	{
		m_pqpopupmenuViewMenu->setItemChecked( ID_VIEW_STATUSBAR, false );
		//    enableStatusBar(KStatusBar::Hide);
		statusBar()->hide();
	}
	else
	{
		m_pqpopupmenuViewMenu->setItemChecked( ID_VIEW_STATUSBAR, true );
		//    enableStatusBar(KStatusBar::Show);
    statusBar()->show();
	}

	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}


void FilterApp::slotStatusMsg( const QString & a_qstringStatusText )
{
	///////////////////////////////////////////////////////////////////
	// change status message permanently
	QStatusBar* pqstatusbarStatusBar = statusBar();
	// SO - if the statusBar() pointer is not checked, the app crashes on
	// shutdown!
	if ( pqstatusbarStatusBar != 0 )
	{
		pqstatusbarStatusBar->clear();
		//pqstatusbarStatusBar->changeItem( a_qstringStatusText, ID_STATUS_MSG );
	}
}


void FilterApp::slotStatusHelpMsg(const QString & a_qstringStatusHelpText )
{
	///////////////////////////////////////////////////////////////////
	// change status message of whole statusbar temporary (text, msec)
	QStatusBar* pqstatusbarStatusBar = statusBar();
	if ( pqstatusbarStatusBar != 0 )
	{
		pqstatusbarStatusBar->message( a_qstringStatusHelpText, 2000 );
	}
}

void FilterApp::slotDriverParam()
{
	m_pfilterdocFilterDocument->initParamDialog();
	EnableCommand( ID_FILE_SAVE_AS );
	EnableCommand( ID_FILE_SAVE );
	//disableCommand(ID_DRIVER_PARAM);
	m_pfilterdocFilterDocument->setModified();
}

void FilterApp::slotDriverNetwork()
{
	m_pfilterdocFilterDocument->initNetworkDialog();
	EnableCommand( ID_FILE_SAVE_AS );
	EnableCommand( ID_FILE_SAVE );
}

void FilterApp::slotDriverVolume()
{
	m_pfilterdocFilterDocument->initVolumeDialog();
	EnableCommand( ID_FILE_SAVE_AS );
	EnableCommand( ID_FILE_SAVE );
}

void FilterApp::slotToolsWizard()
{
	m_pfilterdocFilterDocument->initToolsWizard();
}

void FilterApp::slotOptionsColors()
{
	//QColor color = m_pfilterviewFilterView->backgroundColor();
	//KColorDialog::getColor(color);
	//m_pfilterviewFilterView->setBackgroundColor(color);
	ColorDialog* colordialogDialog = new ColorDialog( m_pfilterviewFilterView, i18n("Color Selection") );
	colordialogDialog->show();
}

void FilterApp::slotrefreshview()
{
	m_pfilterviewFilterView->repaint();
}


void FilterApp::commandCallback( int a_intCallbackID )
{
	switch ( a_intCallbackID )
	{
    case ID_FILE_NEW_WINDOW:
		slotFileNewWindow();
		break;
		
    case ID_FILE_NEW:
		slotFileNew();
		break;
		
    case ID_FILE_OPEN:
		slotFileOpen();
		break;
		
    case ID_FILE_SAVE:
		slotFileSave();
		break;
		
    case ID_FILE_SAVE_AS:
		slotFileSaveAs();
		break;
		
    case ID_FILE_CLOSE:
		slotFileClose();
		break;
		
    case ID_FILE_PRINT:
		slotFilePrint();
		break;
		
    case ID_FILE_QUIT:
		slotFileQuit();
		break;
		
    case ID_EDIT_CUT:
		slotEditCut();
		break;
		
    case ID_EDIT_COPY:
		slotEditCopy();
		break;
		
    case ID_EDIT_PASTE:
		slotEditPaste();
		break;
		
    case ID_VIEW_TOOLBAR:
		slotViewToolBar();
		break;
		
    case ID_VIEW_STATUSBAR:
		slotViewStatusBar();
		break;
		
    case ID_DRIVER_PARAM:
		slotDriverParam();
		break;
		
    case ID_DRIVER_NETWORK:
		slotDriverNetwork();
		break;
		
    case ID_DRIVER_VOLUME:
		slotDriverVolume();
		break;
	case ID_TOOLS_WIZARD:
		slotToolsWizard();
		break;
		
	case ID_OPTIONS_COLORS:
		slotOptionsColors();
		break;
		
    default:
		break;
	}
}

void FilterApp::statusCallback( int a_intCallbackID )
{
	switch ( a_intCallbackID )
	{
    case ID_FILE_NEW_WINDOW:
		slotStatusHelpMsg( i18n("Opens a new application window") );
		break;
		
    case ID_FILE_NEW:
		slotStatusHelpMsg( i18n("Creates a new document") );
		break;
		
    case ID_FILE_OPEN:
		slotStatusHelpMsg( i18n("Opens an existing document") );
		break;
		
    case ID_FILE_OPEN_RECENT:
		slotStatusHelpMsg( i18n("Opens a recently used file") );
		break;
		
    case ID_FILE_SAVE:
		slotStatusHelpMsg( i18n("Saves the actual document") );
		break;
		
    case ID_FILE_SAVE_AS:
		slotStatusHelpMsg( i18n("Saves the actual document as...") );
		break;
		
    case ID_FILE_CLOSE:
		slotStatusHelpMsg( i18n("Closes the actual document") );
		break;
		
    case ID_FILE_PRINT:
		slotStatusHelpMsg( i18n("Prints out the actual document") );
		break;
		
    case ID_FILE_QUIT:
		slotStatusHelpMsg( i18n("Quits the application") );
		break;
		
    case ID_EDIT_CUT:
		slotStatusHelpMsg( i18n("Cuts the selected section and puts it to the clipboard") );
		break;
		
    case ID_EDIT_COPY:
		slotStatusHelpMsg( i18n("Copies the selected section to the clipboard") );
		break;
		
    case ID_EDIT_PASTE:
		slotStatusHelpMsg( i18n("Pastes the clipboard contents to actual position") );
		break;
		
	case ID_DRIVER_PARAM:
		slotStatusHelpMsg( i18n("Specifies the Thiele Small parameters of the driver") );
		break;
		
	case ID_DRIVER_NETWORK:
		slotStatusHelpMsg( i18n("Specifies the frequency range network connected to the driver") );
		break;
		
	case ID_DRIVER_VOLUME:
		slotStatusHelpMsg( i18n("Specifies the technical data of the box volume") );
		break;
		
	case ID_OPTIONS_COLORS:
		slotStatusHelpMsg( i18n("Change of the output window colors") );
		break;
		
    case ID_VIEW_TOOLBAR:
		slotStatusHelpMsg( i18n("Enables/disables the toolbar") );
		break;
		
    case ID_VIEW_STATUSBAR:
		slotStatusHelpMsg(i18n("Enables/disables the statusbar"));
		break;
		
    default:
		break;
	}
}
