/***************************************************************************
                          imagemapeditor.cpp  -  description
                             -------------------
    begin                : Wed Apr 4 2001
    copyright            : (C) 2001 by Jan Schfer
    email                : j_schaef@informatik.uni-kl.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream.h>
// QT
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qvbox.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qpen.h>
#include <qbrush.h>
#include <qpalette.h>
#include <qcolor.h>
#include <qbitmap.h>
#include <qpointarray.h>
#include <qcombobox.h>
#include <qsplitter.h>
#include <qfileinfo.h>
#include <qmultilineedit.h>
#include <qtextstream.h>
#include <qpopupmenu.h>
#include <qdict.h>

// KDE
#include <kdebug.h>
#include <klocale.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kapp.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <klineeditdlg.h>
#include <kglobal.h>
#include <kstddirs.h>


// local
#include "kimagemapeditor.moc"
#include "drawzone.h"
#include "kimedialogs.h"
#include "qextfileinfo.h"

// I don't want to include stdlib.h for this function
inline int abs(int i) {
	if (i < 0)
		return -i;
	else
		return i;
}

Area::Area(ShapeType type)
{
	_type=type;
	_coords=new QPointArray();
	_selectionPoints= new SelectionPointList();
	_finished=false;
	_isSelected=false;
	_name=i18n("noname");
	_listViewItem=0L;
	switch (_type) {
		case Rectangle :
		case Circle : {
      	QRect *p = new QRect(0,0,5,5);
      	_selectionPoints->append(p);
      	p = new QRect(0,0,5,5);
      	_selectionPoints->append(p);
      	p = new QRect(0,0,5,5);
      	_selectionPoints->append(p);
      	p = new QRect(0,0,5,5);
      	_selectionPoints->append(p);
    	}
    	break;
    default: break;
  }
}

Area::Area(const Area & copy) {
	_listViewItem=0L;
	setArea(copy);
}

Area::~Area() {
	if (_listViewItem) {
		delete _listViewItem;
	}
}

void Area::setArea(const Area & copy) {
	_type=copy.type();
	_coords=new QPointArray(copy.coords()->copy());
	_selectionPoints= new SelectionPointList();
	// Need a deep copy of the list
	for (QRect *r=copy.selectionPoints()->first();r!=0L;r=copy.selectionPoints()->next())
		_selectionPoints->append(new QRect( r->topLeft(),r->bottomRight() ) );
	
	_finished=copy.finished();
	_isSelected=copy.isSelected();
	_rect=copy.rect();
	_name=copy.name();
	_href=copy.href();
}

void Area::setListViewItem(QListViewItem* item) {
	_listViewItem=item;
}


void Area::setRect(const QRect & r) {
	_rect=r;

	switch (_type) {
		case Rectangle : {
			updateSelectionPoints();
		break;			
		}
		// We don't want ellipses
		case Circle : {
			_rect.setHeight(_rect.width());				
			updateSelectionPoints();
			break;
		}
		default : break;
	}
}

void Area::moveBy(int dx, int dy) {
	_rect.moveBy(dx,dy);
	for (uint i=0;i<_coords->size();i++) {
		int newX=_coords->point(i).x()+dx;
		int newY=_coords->point(i).y()+dy;
		_coords->setPoint(i,newX,newY);
	}
	
	for (QRect *r=_selectionPoints->first();r!=0L;r=_selectionPoints->next()) {
		r->moveBy(dx,dy);
	}
}


void Area::moveTo(int x, int y) {
	int dx=_rect.left()-x;
	int dy=_rect.top()-y;
	moveBy(dx,dy);
}

void Area::addCoord(const QPoint & p) {
	_coords->resize(_coords->size()+1);
	_coords->setPoint(_coords->size()-1,p);
	
	QRect *r= new QRect(0,0,5,5);
	r->moveCenter(p);
	_selectionPoints->append(r);
	_rect=_coords->boundingRect();
}

bool Area::setCoords(const QString & s) {
	_finished=true;
	switch (_type) {
	case Circle: {
			QStringList list=QStringList::split(",",s);
   		bool ok=true;
   		QStringList::Iterator it = list.begin();
   		int x=(*it).toInt(&ok,10);it++;
   		int y=(*it).toInt(&ok,10);it++;
   		int rad=(*it).toInt(&ok,10);
   		if (!ok) return false;
   		QRect r;
   		r.setWidth(rad*2);
   		r.setHeight(rad*2);
   		r.moveCenter(QPoint(x,y));
   		setRect(r);
		break;
	}
	case Rectangle:	{
			QStringList list=QStringList::split(",",s);
			QRect r;
   		bool ok=true;
   		QStringList::Iterator it = list.begin();
			r.setLeft((*it).toInt(&ok,10));it++;
			r.setTop((*it).toInt(&ok,10));it++;
			r.setRight((*it).toInt(&ok,10));it++;
			r.setBottom((*it).toInt(&ok,10));
			if (ok) {
				setRect(r);
				return true;
			} else
				return false;
		break;
	}
	case Polygon: {
			QStringList list=QStringList::split(",",s);
			_coords=new QPointArray();
			_selectionPoints= new SelectionPointList();
    	for (QStringList::Iterator it = list.begin(); it !=list.end(); it++) {
    		bool ok=true;
    		int newXCoord=(*it).toInt(&ok,10);
    		if (!ok) return false;
    		it++;
    		if (it==list.end())	break;
    		int newYCoord=(*it).toInt(&ok,10);
    		if (!ok) return false;
    		addCoord(QPoint(newXCoord,newYCoord));
    	}
  	break;
  }
  default: return true;
  }
	return false;
}

void Area::setSelected(bool b) {
	_isSelected=b;
	if (_listViewItem) {
		_listViewItem->listView()->setSelected(_listViewItem,b);
//		_listViewItem->setSelected(b);
//		_listViewItem->repaint();
	}
}

QRect Area::selectionRect() const {
	QRect r = _rect;
	r.moveBy(-2,-2);
	r.setSize(r.size()+QSize(5,5));
	return r;
}

void Area::setFinished(bool b) {
	// The last Point is the same as the first
	// so delete it
	_coords->resize(_coords->size()-1);
	_selectionPoints->removeLast();
	_finished=b;
}

void Area::draw(QPainter & p) {
	
	p.setRasterOp(Qt::XorROP);
	p.setPen(QPen(QColor("white"),1));
	switch (_type) {
		case Rectangle :
			p.drawRect(_rect);
			break;
		case Circle :
			p.drawEllipse(_rect);
			break;
		case Polygon : {
				if (_coords->count()==0) return;
				p.moveTo(_coords->point(0));
				for (uint i=1;i<_coords->count();i++)
					p.lineTo(_coords->point(i));
									
				if (_finished)
					p.lineTo(_coords->point(0));
			}
			break;
		default : break;
	}
	if (_isSelected) {
		for (QRect *r=_selectionPoints->first();r!=0L;r=_selectionPoints->next()) {
  		p.fillRect(*r,QBrush("white"));
		}
	}
};

void Area::updateSelectionPoints() {
	QRect *r;
	switch (_type) {
		case Circle :
		case Rectangle :
			_selectionPoints->first()->moveCenter(_rect.topLeft());
			_selectionPoints->next()->moveCenter(_rect.topRight());
			_selectionPoints->next()->moveCenter(_rect.bottomLeft());
			_selectionPoints->next()->moveCenter(_rect.bottomRight());
		break;
		case Polygon :
			r=_selectionPoints->first();
			for (uint i=0;i<_coords->size();i++)	{
				r->moveCenter(_coords->point(i));				
				r=_selectionPoints->next();
			}
		break;
		default : break;
	}

}

QRect* Area::onSelectionPoint(const QPoint & p) const {
	for (QRect *r=_selectionPoints->first();r!=0L;r=_selectionPoints->next()) {
		if (r->contains(p))
			return r;
	}
	
	return 0L;
}

void Area::moveSelectionPoint(QRect* selectionPoint, const QPoint & p) {
	selectionPoint->moveCenter(p);
	switch (_type) {
		case Rectangle: {
			int i=0;
			for (QRect *r=_selectionPoints->first();r!=0L;r=_selectionPoints->next()) {
				if (r==selectionPoint)
					break;
				i++;
			}
			
			switch (i) {
				case 0 : _rect.setLeft(p.x());
								 _rect.setTop(p.y());
					break;
				case 1 : _rect.setRight(p.x());
				 				 _rect.setTop(p.y());
					break;
				case 2 : _rect.setLeft(p.x());
								 _rect.setBottom(p.y());
					break;
				case 3 : _rect.setRight(p.x());
			 					 _rect.setBottom(p.y());
					break;
			}			
			updateSelectionPoints();
		  break;
		}
		case Circle: {
			int i=0;
			for (QRect *r=_selectionPoints->first();r!=0L;r=_selectionPoints->next()) {
				if (r==selectionPoint)
					break;
				i++;
			}
			
			switch (i) {
				case 0 : _rect.setLeft(p.x());
								 _rect.setTop(p.y());
					break;
				case 1 : _rect.setRight(p.x());
				 				 _rect.setTop(p.y());
					break;
				case 2 : _rect.setLeft(p.x());
								 _rect.setBottom(p.y());
					break;
				case 3 : _rect.setRight(p.x());
			 					 _rect.setBottom(p.y());
					break;
			}			
			
			if (abs(_rect.height())>abs(_rect.width()))
				_rect.setHeight(_rect.width());
			else
				_rect.setWidth(_rect.height());
			
		
			}
			updateSelectionPoints();
		  break;
		case Polygon: {
  			int i=0;
  			for (QRect *r=_selectionPoints->first();r!=0L;r=_selectionPoints->next()) {
  				if (r==selectionPoint)
  					break;
  				i++;
  			}
  			_coords->setPoint(i,p);
  			_rect=_coords->boundingRect();
			}
		  break;
		default: break;
	}
}

bool Area::contains(const QPoint & p) const {
	switch (_type) {
		case Rectangle:
			return _rect.contains(p);
			break;
		case Circle: {
			QRegion r(_rect,QRegion::Ellipse);
			return r.contains(p);
			break;
		}
		case Polygon: {
			if (_coords->count() >2 ) {
				QRegion r(*_coords);
				return r.contains(p);
			} else
				return false;
			break;
		}
		default: return false;
	}
	return false;
}

QString Area::coordsToString() const {
	QString retStr;
	switch (_type) {
		case Rectangle:
				retStr=QString("%1,%2,%3,%4")
					.arg(_rect.left())
					.arg(_rect.top())
					.arg(_rect.right())
					.arg(_rect.bottom());
			break;
		case Circle:
				retStr=QString("%1,%2,%3")
					.arg(_rect.center().x())
					.arg(_rect.center().y())
					.arg(_rect.width()/2);
			break;
		case Polygon:
				for (uint i=0;i<_coords->count();i++) {
					retStr.append(QString("%1,%2,")
						.arg(_coords->point(i).x())
						.arg(_coords->point(i).y()));
				}
				retStr.remove(retStr.length()-1,1);
			break;
		default: break;
	}
	return retStr;
}

QPixmap Area::cutOut(const QImage & image) const {

	// Create a mask
	QBitmap mask(_rect.width(),_rect.height());
	mask.fill(Qt::color0);
	QPainter p(&mask);
	p.setBackgroundColor(Qt::color0);
	p.setPen(Qt::color1);
	p.setBrush(Qt::color1);
	switch (_type) {
		case Rectangle :
				mask.fill(Qt::color1);
			break;
		case Circle :
				p.drawPie(QRect(0,0,_rect.width(),_rect.height()),0,5760);
				p.flush();
			break;
		case Polygon : {
				p.setClipping(true);
				QRegion r(*_coords);
				r.translate(-_rect.left(),-_rect.top());
				p.setClipRegion(r);
				p.fillRect(QRect(0,0,_rect.width(),_rect.height()),Qt::color1);
				p.flush();
				break;
		}
		default : break;
	}
	p.end();
	// The rectangle which is part of the image
	QRect partOfImage=_rect;
	QRect partOfMask(0,0,mask.width(),mask.height());
	
	// If the area is outside of the image make the
	// preview smaller
	if ( (_rect.x()+_rect.width()) > image.width() ) {
		partOfImage.setWidth( image.width()-_rect.x() );
		partOfMask.setWidth(  image.width()-_rect.x() );
	}
		
	if ( (_rect.x() < 0) ) {
//		partOfImage.setWidth( _rect.width()+_rect.x() );
		partOfImage.setX(0);
		partOfMask.setX(abs(_rect.x()));
	}
	
	if ( (_rect.y()+_rect.height()) > image.height() ) {
		partOfImage.setHeight( image.height()-_rect.y() );
		partOfMask.setHeight ( image.height()-_rect.y() );
	}
	
	if ( (_rect.y() < 0) ) {
//		partOfImage.setHeight( _rect.height()+_rect.y() );
		partOfImage.setY(0);
		partOfMask.setY(abs(_rect.y()));
		
	}
		
	QImage tempImage=mask.convertToImage().copy(partOfMask);
	mask.convertFromImage(tempImage);
	QImage cut=image.copy(partOfImage);
	
	QPixmap pix;
	pix.convertFromImage(cut);
	QPixmap retPix(pix.width(),pix.height());
	QPainter p3(&retPix);
	
	// if transparent image fill the background
	// with gimp-like rectangles
	if (pix.mask()) {
  	QPixmap backPix(32,32);
  	
  	// Gimp like rectangle
  	QPainter p2(&backPix);
  	p2.fillRect(0,0,32,32,QColor(156,149,156));
  	p2.fillRect(0,16,16,16,QColor(98,105,98));
  	p2.fillRect(16,0,16,16,QColor(98,105,98));
  	p2.flush();
  	
  	p3.setPen(QPen());
  	p3.fillRect(0,0,pix.width(),pix.height(),QBrush(QColor("black"),backPix));
	}
	
	
	p3.drawPixmap(QPoint(0,0),pix);
	p3.flush();
	retPix.setMask(mask);
	
	
	return retPix;
}

QString Area::getHTMLCode() const {
	QString retStr;
	retStr+="<area ";
	switch (_type) {
		case Rectangle : retStr+="shape=\"rect\" ";break;
		case Circle : retStr+="shape=\"circle\" ";break;
		case Polygon : retStr+="shape=\"poly\" ";break;
		case Default : retStr+="shape=\"default\" ";break;
		default: break;
	}
	retStr+="href=\""+_href+"\" ";
	retStr+="alt=\""+_alt+"\" ";
	if (_type!=Default)
		retStr+="coords=\""+coordsToString()+"\" ";
	retStr+=">";
	return retStr;
}



KImageMapEditor::KImageMapEditor(QWidget* parent)
	: KMainWindow(parent)
{
	QSplitter *splitter= new QSplitter(this);
	
	listView = new QListView(splitter);
	listView->addColumn(i18n("Name"));
	listView->addColumn(i18n("Image"));
	listView->setMultiSelection(false);
	connect( listView, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(select(QListViewItem*)));
	connect( listView, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(showTagEditor(QListViewItem*)));
	connect( listView, SIGNAL(rightButtonPressed(QListViewItem*,const QPoint &,int)), this,
		SLOT(slotShowPopupMenu(QListViewItem*,const QPoint &)));
	drawZone = new DrawZone(splitter,this);
	
	// Shows the text:
	// "Drop an image or a html file"
//	QPixmap dropBackground(300,300);
//	dropBackground.fill(drawZone->viewport()->backgroundColor());
//	QPainter p(&dropBackground);
//	p.setFont(QFont( "helvetica", 30,QFont::Bold ));
//	p.drawText(50,150,"Drop an image or a html file");
//	p.end();
	QString path = KGlobal::dirs()->findResourceDir( "data", "kimagemapeditor/dropimage.png" ) + "kimagemapeditor/dropimage.png";
	drawZone->setPicture(QImage(path));
	
	splitter->setResizeMode(drawZone,QSplitter::Stretch);
	splitter->setResizeMode(listView,QSplitter::KeepSize);
	
	areas = new AreaList();
	currentSelected=0L;
	copyArea=0L;
	defaultArea=0L;
	_stdout=false;
	setupActions();
	setupStatusBar();
	createGUI("kimagemapeditorui.rc");
  setCentralWidget(splitter);
	readProperties(kapp->config());
	popupMenu = static_cast<QPopupMenu *>(factory()->container("popup_main", this));
}

KImageMapEditor::~KImageMapEditor(){
}

void KImageMapEditor::setupActions() {
	// File Quit
	(void)KStdAction::quit(kapp, SLOT(closeAllWindows()),actionCollection());

	// File Open
  KAction *temp=KStdAction::open(this, SLOT(slotOpenFile()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "openimage", SmallIcon("fileopen") );
	temp->setWhatsThis(i18n("<h3><img source=\"openimage\"> Open File</h3>Click this to <em>open a new picture</em>."));
	temp->setToolTip(i18n("Open new picture"));
	
	// File Open Recent
  recentFilesAction = KStdAction::openRecent(this, SLOT(slotOpenFile(const KURL&)),
                                      actionCollection());
	// File Save
  saveAction=KStdAction::save(this, SLOT(slotSaveFile()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "saveimage", SmallIcon("filesave") );
	saveAction->setWhatsThis(i18n("<h3><img source=\"saveimage\"> Save Imagemap</h3>Click this to <em>save the imagemap</em>."));
	saveAction->setToolTip(i18n("Save imagemap"));
	// File Save As
  (void)KStdAction::saveAs(this, SLOT(slotSaveFileAs()), actionCollection());

  // Edit Copy
  copyAction=KStdAction::copy(this, SLOT(slotCopy()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "editcopyimage", SmallIcon("editcopy") );
	copyAction->setWhatsThis(i18n("<h3><img source=\"editcopyimage\"> Copy</h3>"
													"Click this to <em>copy</em> the selected area."));
	copyAction->setEnabled(false);													
					
	// Edit Cut																					
  cutAction=KStdAction::cut(this, SLOT(slotCut()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "editcutimage", SmallIcon("editcut") );
	cutAction->setWhatsThis(i18n("<h3><img source=\"editcutimage\"> Cut</h3>"
													"Click this to <em>cut</em> the selected area."));
	cutAction->setEnabled(false);													
	
	// Edit Paste													
  pasteAction=KStdAction::paste(this, SLOT(slotPaste()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "editpasteimage", SmallIcon("editpaste") );
	pasteAction->setWhatsThis(i18n("<h3><img source=\"editpasteimage\"> Paste</h3>"
													"Click this to <em>paste</em> the copied area."));
	pasteAction->setEnabled(false);													

	// Edit Properties
	areaPropertiesAction= new KAction(i18n("Pr&operties"),0,this,SLOT(showTagEditor()),
		actionCollection(), "edit_properties");
	areaPropertiesAction->setEnabled(false);		
													
	// View Zoom In													
  zoomInAction=KStdAction::zoomIn(this, SLOT(slotZoomIn()), actionCollection());
	// View Zoom Out													
  zoomOutAction=KStdAction::zoomOut(this, SLOT(slotZoomOut()), actionCollection());
	
	// View Zoom
  zoomAction=new KSelectAction(i18n("Zoom"), 0,this,SLOT (slotZoom()),
      actionCollection(), "view_zoom");
	zoomAction->setWhatsThis(i18n("<h3>Zoom</h3>"
													"Choose the desired zoom level."));
	zoomAction->setItems(QStringList()
		 << i18n("25 %")
		 << i18n("50 %")
		 << i18n("100 %")
		 << i18n("150 %")
		 << i18n("200 %")
		 << i18n("250 %")
		 << i18n("300 %"));
		
	zoomAction->setCurrentItem(2);
	zoomAction->setComboWidth(50);

	temp= new KAction(i18n("Map &Name"),0,this,SLOT(slotEditMapName()),
		actionCollection(), "map_name");
	
	temp= new KAction(i18n("Edit &Default Area"),0,this,SLOT(slotEditDefaultArea()),
		actionCollection(), "map_defaultarea");
	
	temp= new KAction(i18n("Show &HTML"),0,this,SLOT(slotShowHTML()),
		actionCollection(), "map_showhtml");
	
	// Draw Nothing	
  arrowAction=new KRadioAction(i18n("Draw &Nothing"), "arrow",
             0,this,SLOT (slotDrawArrow()),
      actionCollection(), "draw_arrow");
  QMimeSourceFactory::defaultFactory()->setPixmap( "arrowimage", SmallIcon("arrow") );
	arrowAction->setWhatsThis(i18n("<h3><img source=\"arrowimage\"> Draw Nothing</h3>"
													"Click this to draw nothing."));
	arrowAction->setExclusiveGroup("drawing");
	arrowAction->setChecked(true);

	// Draw Circle
  circleAction=new KRadioAction(i18n("Draw &Circle"), "circle",
             0,this,SLOT (slotDrawCircle()),
      actionCollection(), "draw_circle");
  QMimeSourceFactory::defaultFactory()->setPixmap( "circleimage", SmallIcon("circle") );
	circleAction->setWhatsThis(i18n("<h3><img source=\"circleimage\"> Draw Circle</h3>"
													"Click this to start drawing a circle."));
	circleAction->setExclusiveGroup("drawing");
													
	// Draw Rectangle
  rectangleAction=new KRadioAction(i18n("Draw &Rectangle"), "rectangle",
             0,this,SLOT (slotDrawRectangle()),
      actionCollection(), "draw_rectangle");
  QMimeSourceFactory::defaultFactory()->setPixmap( "rectangleimage", SmallIcon("rectangle") );
	rectangleAction->setWhatsThis(i18n("<h3><img source=\"rectangleimage\"> Draw Rectangle</h3>"
													"Click this to start drawing a rectangle."));
	rectangleAction->setExclusiveGroup("drawing");

	// Draw Polygon
  polygonAction=new KRadioAction(i18n("Draw &Polygon"), "polygon",
             0,this,SLOT (slotDrawPolygon()),
      actionCollection(), "draw_polygon");
  QMimeSourceFactory::defaultFactory()->setPixmap( "polygonimage", SmallIcon("polygon") );
	polygonAction->setWhatsThis(i18n("<h3><img source=\"polygonimage\"> Draw Polygon</h3>"
													"Click this to start drawing a polygon."));
	polygonAction->setExclusiveGroup("drawing");
	
	(void)KStdAction::showToolbar(this, SLOT(slotToggleToolbar()), actionCollection());
  (void)KStdAction::showStatusbar(this, SLOT(slotToggleStatusbar()), actionCollection());
  (void)KStdAction::keyBindings(this, SLOT(slotEditKeys()), actionCollection());
	(void)KStdAction::configureToolbars(this, SLOT(slotEditToolbars()), actionCollection());	
	(void)KStdAction::preferences(this, SLOT(slotShowPreferences()), actionCollection());
}

void KImageMapEditor::setupStatusBar() {
	statusBar()->insertItem(i18n("Cursor")+" : x: 0 ,y: 0",STATUS_CURSOR);
	statusBar()->insertItem(i18n("Selection")+" : - ",STATUS_SELECTION);
}

void KImageMapEditor::slotShowPopupMenu(const QPoint & pos) {
	popupMenu->popup(pos);
}

void KImageMapEditor::slotShowPopupMenu(QListViewItem* item,const QPoint & p) {
	select(item);
	slotShowPopupMenu(p);
}

void KImageMapEditor::slotChangeStatusCoords(int x,int y) {
	statusBar()->changeItem(QString("Cursor : x: %1 ,y: %2").arg(x).arg(y),STATUS_CURSOR);
	if (selected()) {
		QRect r=selected()->rect();
		statusBar()->changeItem(
		QString("Selection : x: %1 ,y: %2, w: %3, h: %4").arg(r.left()).arg(r.top()).arg(r.width()).arg(r.height())
		,STATUS_SELECTION);
//		statusBar()->repaint();
		kapp->processEvents();
	} else
		statusBar()->changeItem("Selection : - ",STATUS_SELECTION);
}

void KImageMapEditor::addArea(Area* s) {
	areas->append(s);
	s->setListViewItem(new QListViewItem(listView,s->href()));
	s->listViewItem()->setPixmap(1,makeListViewPix(*s));
}

void KImageMapEditor::addAreaAndEdit(Area* s) {
	areas->prepend(s);
	s->setListViewItem(new QListViewItem(listView,s->href()));
	s->listViewItem()->setPixmap(1,makeListViewPix(*s));
	select(s);
	showTagEditor(s);
}

void KImageMapEditor::deleteArea(Area* s) {
	if (!s)
		return;
		
	Area a(*s);
	currentSelected=0L;
	areas->remove(s);
	delete s->listViewItem();
	drawZone->repaintArea(a);
	// Only to disable cut and copy actions
	if (areas->count()==0)
		deselectAll();
}

void KImageMapEditor::deleteAllAreas() {
	for (Area* a=areas->first();a!=0L;) {
		deleteArea(a);
		a=areas->first(); // because the current is deleted
	}
}

void KImageMapEditor::updateAllAreas() {
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		a->listViewItem()->setPixmap(1,makeListViewPix(*a));
	}
}


Area* KImageMapEditor::selected() const {
	return currentSelected;
}

void KImageMapEditor::select(Area* s) {
	
	if (currentSelected) {
		currentSelected->setSelected(false);
		drawZone->repaintArea(*currentSelected);
	}		
	currentSelected=s;
	
	if (s) {
		s->setSelected(true);
		drawZone->repaintArea(*s);
		copyAction->setEnabled(true);													
		cutAction->setEnabled(true);													
		areaPropertiesAction->setEnabled(true);
	} else {
		copyAction->setEnabled(false);													
		cutAction->setEnabled(false);		
		areaPropertiesAction->setEnabled(false);
	}
		
	
}

void KImageMapEditor::select(QListViewItem* item) {
	if (!item) return;
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		if (a->listViewItem()==item) {
			select(a);
			return;
		}
	}
}

void KImageMapEditor::slotAreaChanged(Area *a) {
	if (!a) return;
	drawZone->repaintArea(*a);
	if (a->listViewItem()) {
		a->listViewItem()->setText(0,a->href());
		a->listViewItem()->setPixmap(1,makeListViewPix(*a));
	}
}

void KImageMapEditor::deselectAll() {
	Area *a=0L;
	select(a);
}

Area* KImageMapEditor::onArea(const QPoint & p) const {
	for (Area* s=areas->first();s!=0L;s=areas->next()) {
		if (s->contains(p))
			return s;
	}
	return 0L;
}

void KImageMapEditor::slotShowPreferences() {
	PreferencesDialog *dialog = new PreferencesDialog(this);
	connect(dialog, SIGNAL(applyClicked()), this, SLOT(slotConfigChanged()));
	dialog->exec();
	delete dialog;
//	disconnect(dialog, SIGNAL(applyClicked()), this, SLOT(slotConfigChanged()));
}

void KImageMapEditor::showTagEditor(Area *a) {
	if (!a) return;
	AreaDialog *dialog= new AreaDialog(this,a);
	connect (dialog, SIGNAL(areaChanged(Area*)), this, SLOT(slotAreaChanged(Area*)));
	dialog->show();
}

void KImageMapEditor::showTagEditor(QListViewItem *item) {
	if (!item) return;
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		if (a->listViewItem()==item) {
			showTagEditor(a);
			return;
		}
	}
}

void KImageMapEditor::showTagEditor() {
	showTagEditor(selected());
}


QString KImageMapEditor::getHTMLImageMap() const {
	QString retStr;
	retStr+="<map "+QString("name=\"")+_mapName+"\">\n";
	
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		retStr+="  "+a->getHTMLCode()+"\n";
	}
	
	if (defaultArea)
		retStr+="  "+defaultArea->getHTMLCode()+"\n";
	
	retStr+="</map>\n";
	return retStr;
}

QPixmap KImageMapEditor::makeListViewPix(const Area & a) const
{
	QPixmap pix=a.cutOut(drawZone->picture());
	
	double shrinkFactor=1;

	// picture fits into max row height ?		
	if (maxAreaPreviewHeight < pix.height())
		shrinkFactor = ( (double) maxAreaPreviewHeight / pix.height() );
		
	QPixmap pix2(pix.width()*shrinkFactor, pix.height()*shrinkFactor);
	
	// Give all pixels a defined color
	pix2.fill(Qt::white);
	
	QPainter p(&pix2);
	
	p.scale(shrinkFactor,shrinkFactor);
	p.drawPixmap(0,0,pix);
	
	return pix2;
}


void KImageMapEditor::setPicture(const KURL & url) {
	_imageUrl=url;
	setPicture(QImage(url.path()));
}

void KImageMapEditor::setPicture(const QImage & pix) {
	drawZone->setPicture(pix);
}


void KImageMapEditor::slotDrawArrow() {
	_currentShapeType=Area::None;
	
}

void KImageMapEditor::slotDrawCircle() {
	_currentShapeType=Area::Circle;
	
}

void KImageMapEditor::slotDrawRectangle() {
	_currentShapeType=Area::Rectangle;
	
}

void KImageMapEditor::slotDrawPolygon() {
	_currentShapeType=Area::Polygon;
}



void KImageMapEditor::slotZoom() {

	int i=zoomAction->currentItem();
	switch (i) {
		case 0 : drawZone->setZoom(0.25);break;
		case 1 : drawZone->setZoom(0.5);break;
		case 2 : drawZone->setZoom(1);break;
		case 3 : drawZone->setZoom(1.5);break;
		case 4 : drawZone->setZoom(2.0);break;
		case 5 : drawZone->setZoom(2.5);break;
		case 6 : drawZone->setZoom(3);break;
	}
	if (i<6)
  	zoomInAction->setEnabled(true);
  else
  	zoomInAction->setEnabled(false);
  	
  if (i>0)
		zoomOutAction->setEnabled(true);
	else
		zoomOutAction->setEnabled(false);
}

void KImageMapEditor::slotZoomIn() {
	if (zoomAction->currentItem()==(int)(zoomAction->items().count()-1))
		return;
		
	zoomAction->setCurrentItem(zoomAction->currentItem()+1);
	slotZoom();
}

void KImageMapEditor::slotZoomOut() {
	if (zoomAction->currentItem()==0)
		return;

	zoomAction->setCurrentItem(zoomAction->currentItem()-1);
	slotZoom();
}

void KImageMapEditor::slotEditDefaultArea() {
	if (defaultArea)
		showTagEditor(defaultArea);
	else {
		defaultArea= new Area(Area::Default);
		showTagEditor(defaultArea);
	}
		
}

void KImageMapEditor::slotEditMapName() {
	_mapName = KLineEditDlg::getText(i18n("Enter the name of the map"),_mapName,0,this);	
}

void KImageMapEditor::slotShowHTML() {
	KDialogBase *dialog= new KDialogBase(this,"",true,i18n("HTML code of the map"),KDialogBase::Ok);
	QMultiLineEdit *edit = new QMultiLineEdit(dialog);
	
	edit->setText(getHTMLImageMap());
	edit->setReadOnly(true);
	dialog->setMainWidget(edit);
//  dialog->resize(dialog->calculateSize(edit->maxLineWidth(),edit->numLines()*));
//	dialog->adjustSize();
	dialog->resize(600,400);
	dialog->exec();
}


void KImageMapEditor::slotCut() {
	if (!currentSelected)
		return;
	if (copyArea)
		delete copyArea;
				
	copyArea=new Area(*currentSelected);
	pasteAction->setEnabled(true);	
	deleteArea(currentSelected);
}

void KImageMapEditor::slotCopy() {
	if (copyArea)
		delete copyArea;
	
	copyArea=new Area(*currentSelected);
	pasteAction->setEnabled(true);	
}

void KImageMapEditor::slotPaste() {
	if (!copyArea)
		return;
	copyArea->moveBy(5,5);
	Area *a=new Area(*copyArea);
	addAreaAndEdit(a);
}

void KImageMapEditor::slotOpenFile() {
	KURL url=KFileDialog::getOpenURL(QString::null,
					"*.png *.jpg *.jpeg *.gif *.htm *.html|Web file\n"
					"*.png *.jpg *.jpeg *.gif|Image\n"
					"*.htm *.html|HTML-File\n"
					"*.png|PNG-Image\n*.jpg *.jpeg|JPEG-Image\n*.gif|GIF-Image\n*|All files"
					,this,i18n("Choose a picture to open"));
	if (!url.isEmpty()) {
    slotOpenFile(url);
    recentFilesAction->addURL(url);
	}
}

void KImageMapEditor::slotOpenFile(const KURL & url) {
	QString ext=QFileInfo(url.path()).extension().lower();

  QFileInfo fileInfo(url.path());

  if ( !fileInfo.exists() )
  {
    	KMessageBox::information(this,
        i18n("The file <b> %1 </b> does not exists!").arg(fileInfo.fileName()),
        i18n("File does not exists!"));
      return;
  }
	
	if (ext=="html" || ext=="htm") {
		openHTMLFile(url);
	} else {
		setPicture(url);
		updateAllAreas();
	}
	drawZone->viewport()->repaint();
}

void KImageMapEditor::openHTMLFile(const KURL & url) {	
	QFile f(url.path());
	f.open(IO_ReadOnly);
	QTextStream s(&f);
	QString str;
	QChar w;
	QDict<QString> *attr=0L;
	QList<ImageTag> *images= new QList<ImageTag>;
	MapTag *map=0L;
	QList<MapTag> *maps = new QList<MapTag>;
	
	
	bool readMap=false;
	
	while (!s.atEnd()) {
		
 		s >> w;
		if (w=="<") {
			attr=new QDict<QString>(getTagAttributes(s));
			
			if (attr->find("tagname")) {
 				
  			if (attr->find("tagname")->lower()=="img") {
  				images->append(attr);
  				
  			} else
  			if (attr->find("tagname")->lower()=="map") {
  				map = new MapTag();
  				map->name=(*attr->find("name"));
  				readMap=true;
  			} else
  			if (attr->find("tagname")->lower()=="/map") {
  				readMap=false;
  				if (maps)
  					maps->append(map);
				} else
				if (readMap && attr->find("tagname")->lower()=="area") {
  				map->append(attr);
				}
			}
		}		
	}

	f.close();
	
	ImageMapChooseDialog *dialog = new ImageMapChooseDialog(this,maps,images,url);
	dialog->exec();
	map=dialog->currentMap;
	
	KURL imageUrl=dialog->pixUrl;
	
		
	if (map) {
		// Remove old areas only if a new map is loaded
		deleteAllAreas();
		
		setMapName(map->name);
  	for (AreaTag *tag=map->first();tag!=0L;tag=map->next()) {
  		QString shape="rect";
  		if (tag->find("shape"))
  			shape=*tag->find("shape");
  			
  		Area::ShapeType type=Area::Rectangle;
  		if (shape=="circle")
  			type=Area::Circle;
  		else if (shape=="poly")
  			type=Area::Polygon;
  		else if (shape=="default")
  			type=Area::Default;
  			
  		Area* a=new Area(type);

  		if (tag->find("href"))
  			a->setHREF(*tag->find("href"));
	  		
  		if (tag->find("alt"))
  			a->setAlt(*tag->find("alt"));
  		
  		if (type==Area::Default) {
  			defaultArea=a;
  			continue;
  		}
  		
  		if (tag->find("coords"))
  			a->setCoords(*tag->find("coords"));
  			
			addArea(a);  		
  	}
	}
	
	// Only update the image if an image was choosen
	if (!imageUrl.isEmpty()) {
		setPicture(imageUrl);
		updateAllAreas();		
	}

	
}

QDict<QString> KImageMapEditor::getTagAttributes(QTextStream & s) {
	QDict<QString> dict(17,false);
	// the "<" is already read
	QChar w;
	QString attr,value;
	
	// get the tagname
	while (!s.atEnd() && w!=" ") {
		s >> w;
		if (w==" " || w==">") {
			dict.insert("tagname",new QString(value));
			break;
		}
 		value+=w;
	}
	
	// do we have a comment ?
	// read the comment and return
	if (value.right(3)=="-->")
		return dict;
		
	if (value.startsWith("!--")) {
		while (!s.atEnd()) {
			s >> w;
			if (w=="-") {
				s >> w;
				if (w=="-") {
					s >> w;
					if (w==">")
						return dict;
				}
			}
		}
	}
	
	bool attrRead=true;	// currently reading an attribute ?
	bool equalSign=false; // an equalsign was read?
	bool valueRead=false; // currently reading a value ?
	bool quotation=false; // currently reading a value with quotation marks ?
	attr="";
	value="";
	
	//get the other attributes
	while (!s.atEnd() && w!=">") {
		s >> w;
		if (w==">") {
			if (valueRead)
				dict.insert(attr,new QString(value));
				return dict;
		} else
		// currently reading an attribute ?
		if (attrRead) {
			// if there is a whitespace the attributename has finished
			// possibly there isn't any value e.g. noshade
			if (w==" ")
				attrRead=false;
			else
			// an equal sign signals that the value follows
			if (w=="=") {
				attrRead=false;
				equalSign=true;
			} else
				attr+=w;
		} else
		// an equalsign was read ? delete every whitespace
		if (equalSign) {
			if (w!=" ") {
				equalSign=false;
				valueRead=true;
				if (w=="\"" || w=="'")
					quotation=true;
			}
		} else
		// currently reading the value
		if (valueRead) {
			// if value within quotation marks is read
			// only stop when another quotationmark is found
			if (quotation) {
				if (w!="\"" && w!="'") {
					value+=w;
				} else {
					quotation=false;
					valueRead=false;
					dict.insert(attr,new QString(value));
					attr="";value="";
					
				}
			} else
		  // a whitespace indicates that the value has finished
			if (w==" ") {
				valueRead=false;
				dict.insert(attr,new QString(value));
				attr="";value="";
			}
		} else {
			if (w!=" ") {
				attrRead=true;
				attr+=w;
			}
		}
	}

	return dict;		
	
}

void KImageMapEditor::slotSaveFile()  {
	if (_mapUrl.isEmpty())
		slotSaveFileAs();
	else
		saveImageMap(_mapUrl);
}

void KImageMapEditor::slotSaveFileAs()  {
  KURL url = KFileDialog::getSaveURL(0L,"*.htm *.html|"+i18n("HTML file")+"\n*.txt|"+
  						i18n("Text file")+"\n*|"+i18n("All Files"),this);
  if (url.isEmpty() || !url.isValid()) {
    return;
  }			


  QFileInfo fileInfo(url.path());

  if(fileInfo.exists() && !fileInfo.isWritable()) {
    KMessageBox::sorry(this, i18n("You do not have write permission to this file"));
    return;
  }

  if ( fileInfo.exists() )
  {
  	if (fileInfo.extension().lower()=="html" ||
  			fileInfo.extension().lower()=="htm")
  	{
    	if (KMessageBox::warningYesNo(this,
        i18n("<b>ATTENTION!</b> The file <b> %1 </b> already exists!<br>"
        		"Even saving to a HTML-file will overwrite it<br>"
        		"Saving to a HTML-file will be implemented later"
        		"Do you want to overwrite it?<br>").arg(fileInfo.fileName()),
        i18n("Really overwrite the HTML file ?"))==KMessageBox::No)
        return;
  	} else
  	if (KMessageBox::warningYesNo(this,
      i18n("The file <b> %1 </b> already exists!<br>Do you want to overwrite it?").arg(fileInfo.fileName()),
      i18n("Really overwrite the file ?"))==KMessageBox::No)
      return;
  }
  saveImageMap(url);
}

void KImageMapEditor::saveImageMap(const KURL & url) {
	QFileInfo fileInfo(url.path());
	QString ext=fileInfo.extension(false).lower();
	if (ext=="txt") {
		QFile file(url.path());
		file.open(IO_WriteOnly);
		QTextStream t(&file);
		t << getHTMLImageMap();
		file.close();
	} else
	if (ext=="html" || ext=="htm") {
  	if (_mapName.isEmpty()) {
  		slotEditMapName();
  	}
		QFile file(url.path());
		file.open(IO_WriteOnly);
		QTextStream t(&file);
		t << "<html>\n"
			<< "<head>\n"
			<< "  <title></title>\n"
			<< "</head>\n"
		 	<< "<body>\n"
		 	<< "  " << getHTMLImageMap()
		 	<< "\n"
		 	<< "  <img src=\"" << QExtFileInfo::toRelative(_imageUrl.path(),url.directory()) << "\""
		 	<< " usemap=\"#" << _mapName << "\""
		 	<< " width=\"" << drawZone->picture().width() << "\""
		 	<< " height=\"" << drawZone->picture().height() << "\">\n"
		 	<< "</body>";
		
		file.close();
	}
	
}

void KImageMapEditor::saveProperties(KConfig *config) {
	saveMainWindowSettings(config);
	saveWindowSize(config);
	recentFilesAction->saveEntries(config,"Data");
  config->sync();
}

void KImageMapEditor::readProperties(KConfig *config) {
	applyMainWindowSettings(config);
	restoreWindowSize(config);
  recentFilesAction->loadEntries(config,"Data");
  config->setGroup("Appearance");
	maxAreaPreviewHeight=config->readNumEntry("maximum-preview-height",50);
}

void KImageMapEditor::slotConfigChanged() {
  kapp->config()->setGroup("Appearance");
	int newHeight=kapp->config()->readNumEntry("maximum-preview-height",50);
	
	// if the image preview size changed update all images
	if (maxAreaPreviewHeight!=newHeight) {
		maxAreaPreviewHeight=newHeight;
		updateAllAreas();	
	}
}

bool KImageMapEditor::queryClose()  {
	saveProperties(kapp->config());
	if (_stdout) {
		cout << getHTMLImageMap() << endl;
	}
	return true;
}

void KImageMapEditor::slotEditKeys() {
  KKeyDialog::configureKeys(actionCollection(), xmlFile());
}

void KImageMapEditor::slotEditToolbars()
{
  KEditToolbar *dlg = new KEditToolbar(actionCollection());

  if (dlg->exec())
    createGUI("kimagemapeditorui.rc");

  delete dlg;
}

void KImageMapEditor::slotToggleToolbar() {
	if (toolBar()->isVisible())
		toolBar()->hide();
	else
		toolBar()->show();
}

void KImageMapEditor::slotToggleStatusbar() {
	if (statusBar()->isVisible())
		statusBar()->hide();
	else
		statusBar()->show();
}






