/*
    KMLOVoice

    A utility to process voice messages received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdio.h>
#include <unistd.h>

#include <qglobal.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qwhatsthis.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>

#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>

#include "kmlovoice.h"
#include "configure.h"
#include "voiceview.h"
#include "downloaddlg.h"
#include "playback.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "global.h"




KMLOVoice::KMLOVoice(QWidget *parent, const char *name) : KMainWindow(parent, name)
{
	config = kapp->config();

	initMenuBar();
	initToolBar();
	initStatusBar();
	initKeyAccel();
	
	configure = new Configure(this, "configure");
	CHECK_PTR(configure);

	calleraliases = new CallerAliases(this, "calleraliases");
	CHECK_PTR(calleraliases);

	view = new VoiceView(this, "voiceview");
	CHECK_PTR(view);
	setCentralWidget(view);
	
	initWhatsThis();
	
	downloaddlg = new DownloadDlg(this, "downloaddlg");
	CHECK_PTR(downloaddlg);
	playback = new PlayBack(0, "playback");
	CHECK_PTR(playback);
	exportdlg = new ExportDlg(this, "exportdlg");
	CHECK_PTR(exportdlg);
	maildlg = new MailDlg(this, "maildlg");
	CHECK_PTR(maildlg);
	
	readOptions();

	slotNewConfig();

	hasSelectionSlot(false);

	connect(configure, SIGNAL(newConfig()), SLOT(slotNewConfig()));

	connect(downloaddlg, SIGNAL(message(const QString &)), SLOT(slotStatusHelpMsg(const QString &)));
	connect(downloaddlg, SIGNAL(gotFile(const QString &, uchar, const QString &)), SLOT(fetchFile(const QString &, uchar, const QString &)));
	connect(downloaddlg, SIGNAL(done()), SLOT(fetchDone()));

	connect(view, SIGNAL(hasSelection(bool)), SLOT(hasSelectionSlot(bool)));
	connect(view, SIGNAL(command(int)), SLOT(commandCallback(int)));

	show();

	slotStatusHelpMsg(i18n("%1 File(s) in directory.").arg(view->loadDir()));
}


KMLOVoice::~KMLOVoice()
{
	delete playback;
}


void KMLOVoice::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void KMLOVoice::readOptions()
{
	config->setGroup("MainWindow Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		toolBar()->hide();
	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		statusBar()->hide();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	QSize geoSize(650, 400);
	resize(config->readSizeEntry("Geometry", &geoSize));

	view->readOptions();
}


void KMLOVoice::saveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());

	view->saveOptions();
}


bool KMLOVoice::queryClose()
{
	view->saveDir();

	return true;
}


void KMLOVoice::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "mainfilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(SmallIcon("filedownload"), i18n("&Download"), ID_FILE_DOWNLOAD);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("play"), i18n("&Playback..."), ID_FILE_PLAYBACK);
	fileMenu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_FILE_EXPORT);
	fileMenu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_FILE_MAIL);
	fileMenu->insertItem(SmallIcon("remove"), i18n("De&lete"), ID_FILE_DELETE);
	fileMenu->insertItem(i18n("Caller &Alias..."), ID_FILE_CALLER_ALIAS);
	fileMenu->insertSeparator();
	fileMenu->insertItem(i18n("P&urge Directory"), ID_FILE_PURGE_DIRECTORY);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("exit"), i18n("&Quit"), ID_FILE_QUIT);

	settingsMenu = new QPopupMenu(0, "mainsettingsmenu");
	CHECK_PTR(settingsMenu);
	settingsMenu->setCheckable(true);
	settingsMenu->insertItem(i18n("Show &Toolbar"), ID_SETTINGS_TOOLBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, true);
	settingsMenu->insertItem(i18n("Show &Statusbar"), ID_SETTINGS_STATUSBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, true);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Save Options"), ID_SETTINGS_SAVEOPTIONS);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Caller &Aliases..."), ID_SETTINGS_CALLER_ALIASES);
	settingsMenu->insertItem(SmallIcon("configure"), i18n("&Configure KMLOVoice..."), ID_SETTINGS_CONFIGURE);
	
	QString aboutstr = QString("KMLOVoice ") + VERSION + i18n(
		"\nCopyright %1 2000 Oliver Gantz <Oliver.Gantz@epost.de>\n\n" \
		"A utility to process voice messages received with the\n" \
		"ELSA MicroLink(tm) Office modem.\n\n" \
		"This program is free software and is distributed under the terms\n" \
		"of the GNU General Public License.\n\n" \
		"ELSA and MicroLink are trademarks of ELSA AG, Aachen.").arg('');
	
	helpMenu_ = helpMenu(aboutstr);
	
	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu_);

	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(settingsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void KMLOVoice::initToolBar()
{
	toolBar()->insertButton(BarIcon("filedownload"), ID_FILE_DOWNLOAD, true, i18n("Download Messages"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("play"), ID_FILE_PLAYBACK, true, i18n("Playback Message"));
	toolBar()->insertButton(BarIcon("filesave"), ID_FILE_EXPORT, true, i18n("Export Message"));
	toolBar()->insertButton(BarIcon("mail_send"), ID_FILE_MAIL, true, i18n("Mail Message"));
	toolBar()->insertButton(BarIcon("remove"), ID_FILE_DELETE, true, i18n("Delete Message"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("configure"), ID_SETTINGS_CONFIGURE, true, i18n("Configure KMLOVoice"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->insertButton(BarIcon("exit"), ID_FILE_QUIT, true, i18n("Quit"));
	toolBar()->alignItemRight(ID_FILE_QUIT, true);

	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void KMLOVoice::initStatusBar()
{
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);
}


void KMLOVoice::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Download Messages"), "DownloadFiles", CTRL+Key_D);
	
	keyAccel->insertItem(i18n("Playback Message"), "PlayFile", CTRL+Key_P);
	keyAccel->insertItem(i18n("Export Message"), "ExportFile", CTRL+Key_E);
	keyAccel->insertItem(i18n("Mail Message"), "MailFile", CTRL+Key_M);
	keyAccel->insertItem(i18n("Delete Message"), "DeleteFile", Key_Delete);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);

	keyAccel->readSettings();
	
	keyAccel->connectItem("DownloadFiles", this, SLOT(slotFileDownload()));
	keyAccel->connectItem("PlayFile", this, SLOT(slotFilePlay()));
	keyAccel->connectItem("ExportFile", this, SLOT(slotFileExport()));
	keyAccel->connectItem("MailFile", this, SLOT(slotFileMail()));
	keyAccel->connectItem("DeleteFile", this, SLOT(slotFileDelete()));
	keyAccel->connectItem(KStdAccel::Quit, this, SLOT(slotFileQuit()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotSettingsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotSettingsShowStatusbar()));
	
	keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));

	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DOWNLOAD, "DownloadFiles");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_PLAYBACK, "PlayFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_EXPORT, "ExportFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_MAIL, "MailFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DELETE, "DeleteFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KStdAccel::Quit);

	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_STATUSBAR, "ShowStatusbar");
}


void KMLOVoice::initWhatsThis()
{
	QWhatsThis::add(toolBar()->getButton(ID_FILE_DOWNLOAD),
		i18n("Click this button to download messages from your modem's flash memory.<br><br>" \
		"You can also select the <b>Download Command</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_PLAYBACK),
		i18n("Click this button to playback the message that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the <b>Playback Command</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_EXPORT),
		i18n("Click this button to export the message " \
		"that is currently selected in the list below into a WAVE file.<br><br>" \
		"You can also select the <b>Export Command</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_MAIL),
		i18n("Click this button to send the message that is " \
		"currently selected in the list below as email.<br><br>" \
		"You can also select the <b>Mail Command</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_DELETE),
		i18n("Click this button to delete the message that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the <b>Delete Command</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_SETTINGS_CONFIGURE),
		i18n("Opens the configure dialog.<br><br>" \
		"You can also select the <b>Configure KMLOVoice Command</b> from the \"Settings\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KMLOVoice help pages displayed.<br><br>" \
		"You can also select the <b>Contents Command</b> from the \"Help\" menu."));
	QWhatsThis::add(view,
		i18n("The list shows the messages you have formerly " \
		"received. The columns of this list have the following meaning:<br>" \
		"<dl compact>"
		"<dt>Name<dd>File name of the message." \
		"<dt>New<dd>Indicates if the file is new." \
		"<dt>Date (Time)<dd>Date and time when your modem received the message." \
		"<dt>Caller<dd>Caller ID of the sender where the message came from." \
		"<dt>Duration<dd>Length of the message in seconds." \
		"<dt>Size<dd>File size in bytes of the message file." \
		"<dt>File<dd>Indicates if the message file still exists on your local data directory." \
		"</dl>"));
}


void KMLOVoice::commandCallback(int id_)
{
	switch (id_) {
		case ID_FILE_DOWNLOAD:
			slotFileDownload();
			break;
		case ID_FILE_PLAYBACK:
			slotFilePlay();
			break;
		case ID_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_FILE_MAIL:
			slotFileMail();
			break;
		case ID_FILE_DELETE:
			slotFileDelete();
			break;
		case ID_FILE_CALLER_ALIAS:
			slotFileCallerAlias();
			break;
		case ID_FILE_PURGE_DIRECTORY:
			slotFilePurgeDir();
			break;
		case ID_FILE_QUIT:
			slotFileQuit();
			break;
		case ID_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_SETTINGS_SAVEOPTIONS:
			saveOptions();
			break;
		case ID_SETTINGS_CONFIGURE:
			slotSettingsConfigure();
			break;
		case ID_SETTINGS_CALLER_ALIASES:
			slotSettingsCallerAliases();
			break;
		default:
			break;
	}
}

		
void KMLOVoice::slotFileDownload()
{
	if (downloaddlg->startDownload()) {
		enableCommand(ID_FILE_DOWNLOAD, false);
		slotStatusHelpMsg(i18n("Looking for new messages..."));
	}
}


void KMLOVoice::slotFilePlay()
{
	if (!view->selectedMsgName().isEmpty()) {
		playback->playMessage(view->selectedMsgName(), view->selectedMsgType());
		view->setSelectedMsgNew(false);
	}
}


void KMLOVoice::slotFileExport()
{
	if (!view->selectedMsgName().isEmpty())
		exportdlg->exportMessage(view->selectedMsgName(), view->selectedMsgType());
}


void KMLOVoice::slotFileMail()
{
	if (!view->selectedMsgName().isEmpty())
		maildlg->mailMessage(view->selectedMsgName(), view->selectedMsgType());
}


void KMLOVoice::slotFileDelete()
{
	if (!view->selectedMsgName().isEmpty()) {
		if (view->selectedMsgNew()) {
			KMessageBox::sorry(this, i18n("Cannot delete new message file.\nPlease playback message first."), i18n("File Error"));
			return;
		}

		if (KMessageBox::questionYesNo(this, i18n("Really delete file\n\"%1\"?").arg(view->selectedMsgName()), i18n("Question"), i18n("&Delete"), i18n("&Cancel")) == KMessageBox::Yes) {
			QFile::remove(expandPath(view->selectedMsgName()));
			view->setSelectedMsgFile(false);
			hasSelectionSlot(true);
		}
	}
}


void KMLOVoice::slotFileCallerAlias()
{
	if (!view->selectedMsgName().isEmpty()) {
		calleraliases->setAlias(view->selectedMsgCaller());
	}
}


void KMLOVoice::slotFilePurgeDir()
{
	if (KMessageBox::questionYesNo(this, i18n("Really remove entries with deleted files?"), i18n("Question")) == KMessageBox::Yes)
		view->purgeDir();
}

/*
void KMLOVoice::slotFileRebuildDir()
{
	if (KMessageBox::questionYesNo(this, i18n("Really discard current directory and scan for existing message files?"), i18n("Question")) == KMessageBox::Yes) {
		slotStatusHelpMsg(i18n("%1 File(s) found.").arg(view->rebuildDir()));
	}
}
*/

void KMLOVoice::slotFileQuit()
{
	close();
}


void KMLOVoice::slotSettingsShowToolbar()
{
	bool toolBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_TOOLBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (toolBarOn)
		toolBar()->show();
	else
		toolBar()->hide();
}


void KMLOVoice::slotSettingsShowStatusbar()
{
	bool statusBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_STATUSBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (statusBarOn)
		statusBar()->show();
	else
		statusBar()->hide();
}


void KMLOVoice::slotSettingsConfigure()
{
	if (configure->isVisible())
		configure->raise();
	else
		configure->show();
}


void KMLOVoice::slotSettingsCallerAliases()
{
	if (calleraliases->isVisible())
		calleraliases->raise();
	else
		calleraliases->show();
}


void KMLOVoice::slotStatusMsg(const QString& text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOVoice::slotStatusHelpMsg(const QString& text)
{
	statusBar()->message(text, 2000);
}



void KMLOVoice::slotNewConfig()
{
	bool tooltips;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	tooltips = config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS);

	configure->addToolTips(tooltips);
	exportdlg->addToolTips(tooltips);
	maildlg->addToolTips(tooltips);
	calleraliases->addToolTips(tooltips);
}


void KMLOVoice::hasSelectionSlot(bool on)
{
	view->gotSelection(on);
	
	enableCommand(ID_FILE_CALLER_ALIAS, on);
	
	if (on)
		on = view->selectedMsgFile();

	enableCommand(ID_FILE_PLAYBACK, on);
	enableCommand(ID_FILE_EXPORT, on);
	enableCommand(ID_FILE_MAIL, on);
	enableCommand(ID_FILE_DELETE, on);
}


void KMLOVoice::fetchFile(const QString& name, uchar type, const QString& caller)
{
	slotStatusHelpMsg(i18n("Received file %1.").arg(name));
	view->addVoiceItem(name, type, caller);	
}


void KMLOVoice::fetchDone()
{
	enableCommand(ID_FILE_DOWNLOAD, true);
	view->saveDir();
}
