/***************************************************************************
                          KNCListViewKeyboard.cpp  -  description
                             -------------------
    begin                : Tue Aug 29 2000
    copyright            : (C) 2000 by Henrik Stormer
    email                : stormer@ifi.unizh.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <qpopupmenu.h>
#include <qdragobject.h>

#include <klocale.h>
#include <kdirsize.h>

#include "kncview.h"
#include "knclistviewitem.h"
#include "kncfilecommand.h"
#include "dirsizedialog.h"

KNCListViewItem* KNCListView::findItem(KFileItem* f)
{
  KNCListViewItem* Item = (KNCListViewItem*)firstChild();
  while (Item)
  {
    if (Item->getFileItem()->name() == f->name()) return Item;
    Item = (KNCListViewItem*)Item->itemBelow();
  }
  return NULL;
}

KNCListViewItem* KNCListView::findItem(QString Name)
{
 KNCListViewItem* Item = (KNCListViewItem*)firstChild();
 while (Item != NULL)
 {
   if (Item->getFileItem()->name() == Name) break;
   Item = (KNCListViewItem*)Item->itemBelow();
 }

 return Item;
}


void KNCListView::makeItemVisible(KNCListViewItem* Item)
{
 if (Item)
 {
   int dy =  contentsY() - Item->itemPos();
   if (dy > 0) scrollBy(0, -dy);

   dy = (Item->itemPos() + Item->height()) - (contentsY() + viewport()->height());
   if (dy > 0) scrollBy(0, dy);
 }
}


bool KNCListView::select(QRegExp RegExp, bool Selection)
{
  if (!RegExp.isValid()) return false;
  else
  {
    if (currentItem())
    {
      KNCListViewItem* Item = (KNCListViewItem*) firstChild();
      while (Item != NULL)
      {
        if (RegExp.match(Item->getFileItem()->name()) != -1)
        {
          setSelected(Item, Selection);
        }
        Item = (KNCListViewItem*)Item->itemBelow();
      }
    }
    return true;
  }
}


void KNCListView::slotUp ()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*)currentItem();

   if (_SubString != NULL) Item = findSubStringItem(false, _SubString);
   else Item = (KNCListViewItem*)Item->itemAbove();

   if (Item) setCurrentItem(Item);
   else Item = (KNCListViewItem*)currentItem();
   makeItemVisible(Item);
   emit cursorChanged();
 }
}

void KNCListView::slotDown()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*)currentItem();

   if (_SubString != NULL) Item = findSubStringItem(true, _SubString);
   else Item = (KNCListViewItem*)Item->itemBelow();

   if (Item) setCurrentItem(Item);
   else Item = (KNCListViewItem*)currentItem();
   makeItemVisible(Item);
   emit cursorChanged();
 }
}

void KNCListView::slotLeft()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*)currentItem();
   if (_SubString != NULL)
   {
     if (_SubString != "")
     {
       _SubString.truncate(_SubString.length()-1);
       Item->repaint();
     }
   }
   else scrollBy(-10,0);
 }
}

void KNCListView::slotRight()
{
if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*)currentItem();
   if (_SubString != NULL)
   {
     if (Item->getFileItem()->name().length() > _SubString.length())
     {
       _SubString += Item->getFileItem()->name()[_SubString.length()];
       Item->repaint();
     }
   }
   else scrollBy(10,0);

 }
}

void KNCListView::slotPageUp()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*) currentItem();
   int count = viewport()->height() / Item->height();
   while ((count-- > 0) && (Item->itemAbove() != NULL))
     Item = (KNCListViewItem*)Item->itemAbove();

   if (_SubString != NULL)
     if (!matchItem(Item, _SubString)) return;

   setCurrentItem(Item);
   makeItemVisible(Item);
   emit cursorChanged();
 }
}

void KNCListView::slotPageDown()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*) currentItem();
   int count = viewport()->height() / Item->height();
   while ((count-- > 0) && (Item->itemBelow() != NULL))
     Item = (KNCListViewItem*)Item->itemBelow();

   if (_SubString != NULL)
     if (!matchItem(Item, _SubString)) return;

   setCurrentItem(Item);
   makeItemVisible(Item);
   emit cursorChanged();
  }
}

void KNCListView::slotScrollBegin()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*)firstChild();
   slotResetSubString();
   setCurrentItem(Item);
   makeItemVisible(Item);
   emit cursorChanged();
 }
}

//es gibt leider keine lastChild Methode!
void KNCListView::slotScrollEnd()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*) currentItem();
   while (Item->itemBelow() != NULL) Item = (KNCListViewItem*)Item->itemBelow();

   slotResetSubString();
   setCurrentItem(Item);
   makeItemVisible(Item);
   emit cursorChanged();
 }
}

void KNCListView::slotDirSizeResult(KIO::Job* Job)
{
  KNCListViewItem* Item = _DirSizeJobMapping[Job];
  if (!Item) return;

  _DirSizeJobMapping.remove(Job);
  long NewSize = ((KDirSize*)Job)->totalSize();
  _SelectedSize += NewSize - Item->dirSize();
  Item->setDirSize(NewSize);
  Item->repaint();
  emit selectChanged();

  if (_DirSizeJobMapping.isEmpty())
    _DirSizeDialog->hide();
}

void KNCListView::slotCancelCalculateDirSize()
{
 for (QMap<KIO::Job*, KNCListViewItem*>::Iterator it = _DirSizeJobMapping.begin();it != _DirSizeJobMapping.end();++it)
   it.key()->kill();

 _DirSizeJobMapping.clear();
 _DirSizeDialog->hide();
}

void KNCListView::slotCalculateDirSize()
{
  _DirSizeDialog->show();
  QList<KNCListViewItem> ItemList = getSelectedViewItems();

  for (QListIterator<KNCListViewItem> it(ItemList);it.current();++it)
  {
    if (!(it.current())->getFileItem()->isDir()) continue;
    slotCalculateDirSize(it.current());
  }

}

void KNCListView::slotCalculateDirSize(KNCListViewItem* Item)
{
  if ((!Item->getFileItem()->isDir()) || (!Item->isSelected())) return;

  KDirSize* DirSizeJob =  KDirSize::dirSizeJob(Item->getFileItem()->url());
  _DirSizeJobMapping.insert(DirSizeJob, Item);
  connect (DirSizeJob, SIGNAL(result(KIO::Job*)), SLOT(slotDirSizeResult(KIO::Job*)));
}

void KNCListView::setSelected (QListViewItem* QItem, bool selected)
{
 KNCListViewItem* Item = (KNCListViewItem*) QItem;
 if ((Item->isSelected()) && (!selected))
 {
   if (Item->getFileItem()->isDir())
   {
     _NumSelectedDirs--;
     _SelectedSize -= Item->dirSize();
   }
   else
   {
     _NumSelectedFiles--;
     _SelectedSize -= Item->getFileItem()->size();
   }
 }
 if ((!Item->isSelected()) && (selected))
 {
   if (Item->getFileItem()->isDir())
   {
     _NumSelectedDirs++;
     _SelectedSize += Item->dirSize();
   }
   else
   {
     _NumSelectedFiles++;
     _SelectedSize += Item->getFileItem()->size();
   }
 }

 QListView::setSelected(QItem, selected);
 emit selectChanged();
}

void KNCListView::slotInvertSelectionAndCalculate()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*) currentItem();
   if (Item->getFileItem()->name() != "..")
   {
     setSelected(Item, !Item->isSelected());
     slotCalculateDirSize(Item);
     Item->repaint();
   }
   slotDown();
 }
}

void KNCListView::slotInvertSelection()
{
 if (currentItem())
 {
   KNCListViewItem* Item = (KNCListViewItem*) currentItem();
   if (Item->getFileItem()->name() != "..")
   {
     setSelected(Item, !Item->isSelected());
     Item->repaint();
   }
   slotDown();
 }
}

void KNCListView::slotDirectoryUp()
{
  QString DirName = _Dir.fileName();
  KURL Dir = _Dir.upURL();

  openURL(Dir, DirName);
}

void KNCListView::slotResetSubString()
{
  _SubString = (const char*) NULL;
  currentItem()->repaint();
}

void KNCListView::keyPressEvent ( QKeyEvent * e)
{
  if ( !e ) return;

  _KeyPressed = true;

  if ((!e->state()) || (e->state() & ShiftButton))
  {
    if (e->key() == Key_Up) {slotUp();return;}
    if (e->key() == Key_Down) {slotDown();return;}
    if (e->key() == Key_Left) {slotLeft();return;}
    if (e->key() == Key_Right) {slotRight();return;}
    if (e->key() == Key_PageUp) {slotPageUp();return;}
    if (e->key() == Key_PageDown) {slotPageDown();return;}
    if (e->key() == Key_Home) {slotScrollBegin();return;}
    if (e->key() == Key_End) {slotScrollEnd();return;}
    if (e->key() == Key_Space) {slotInvertSelection();return;}
    if (e->key() == Key_Escape) {slotResetSubString();return;}
    if (e->key() == Key_Return) {slotEnter();return;}
    if (e->key() == Key_Backspace) {slotDirectoryUp();return;}
    if (e->key() == Key_Insert) {slotInvertSelectionAndCalculate();return;}
    //normal key -> use SubString
    if ((e->ascii()) && (currentItem()))
    {
      KNCListViewItem* Item = (KNCListViewItem*)currentItem();
      QString text = Item->text(0);

      QString TSubString = _SubString + char(e->ascii());
      if (!matchItem(Item, TSubString))
        Item = findSubStringItem(true, TSubString);

      if (Item != NULL)
      {
        _SubString = TSubString;
        setCurrentItem( Item );
        makeItemVisible( Item );
        Item->repaint();
        emit cursorChanged();
      }
    }
  }
}

bool KNCListView :: matchItem(KNCListViewItem* Item, QString SubString)
{
  QString Text = Item->text(0);
  Text = Text.left(SubString.length());
  if (Text == SubString) return true;
  else return false;
}

KNCListViewItem* KNCListView::findSubStringItem(bool down, QString SubString)
{
  if ( !currentItem() )	return NULL;
  KNCListViewItem* Item = (KNCListViewItem*)currentItem();
  KNCListViewItem* StartItem = Item;
  while(1)
  {
    if (down) Item = (KNCListViewItem*)Item->itemBelow(); else Item = (KNCListViewItem*)Item->itemAbove();
    if (!Item)
    {
      if (down) Item = (KNCListViewItem*)firstChild();
      else //Troll forgot to make a lastChild Method :-(
      {
        Item = StartItem;
        while(Item->itemBelow()) Item = (KNCListViewItem*)Item->itemBelow();
      }
    }
    if (matchItem(Item, SubString)) return Item;

    if (Item == StartItem) return NULL;
  }
}

void KNCListView :: contentsMousePressEvent(QMouseEvent* e)
{
  if ( !e )
	return;
  KNCListViewItem* Item = (KNCListViewItem*) itemAt(contentsToViewport(e->pos()));
  if (!Item) return;

  if (e->button() == LeftButton)
  {
    if (_SubString != NULL) slotResetSubString();
    setCurrentItem(Item);
    _BeginDrag = true;
    emit cursorChanged();
  }
  if (e->button() == RightButton)
  {
    setSelected(Item,!Item->isSelected());
  }
}

void KNCListView :: contentsMouseReleaseEvent(QMouseEvent* e)
{
 if (_BeginDrag == true)
 {
   _BeginDrag = false;
 }
}

void KNCListView :: contentsMouseMoveEvent (QMouseEvent *e)
{
  if (_BeginDrag) startDrag();
  _BeginDrag = false;
}

void KNCListView :: startDrag()
{
 KNCListViewItem* Item = (KNCListViewItem*)firstChild();
 QPixmap pixmap;
 QStrList URLs;

 while (Item != NULL)
 {
   if (Item->isSelected())
   {
     URLs.append(Item->getFileItem()->url().url());
     if (pixmap.isNull())
       pixmap = Item->getFileItem()->pixmap(KIcon::SizeMedium);
   }
   Item = (KNCListViewItem*)Item->itemBelow();
 }

 if ((URLs.isEmpty()) && (currentItem()))
 {
   URLs.append(((KNCListViewItem*)currentItem())->getFileItem()->url().url());
   pixmap = ((KNCListViewItem*)currentItem())->getFileItem()->pixmap(KIcon::SizeMedium);
 }


 if (URLs.count() > 1)
   pixmap = DesktopIcon( "kmultiple", KIcon::SizeMedium );

 QPoint hotspot;
 QUriDrag *d = new QUriDrag( URLs, viewport() );

 if ( !pixmap.isNull())
 {
    hotspot.setX( pixmap.width() / 2 );
    hotspot.setY( pixmap.height() / 2 );
    d->setPixmap( pixmap, hotspot );
  }
  d->drag();
}


void KNCListView :: viewportDragEnterEvent (QDragEnterEvent *e)
{
  e->accept(QUriDrag::canDecode(e));
  KNCListViewItem* Item = (KNCListViewItem*) itemAt(e->pos());
  if (Item == _DragOverItem) return;
  if (Item && Item->getFileItem()->isDir())
  {
    if (_DragOverItem != NULL)
      setSelected(_DragOverItem, _DragOverItemSelected);

    _DragOverItem = Item;
    _DragOverItemSelected = Item->isSelected();
    setSelected(_DragOverItem, true);
  }
  else if (_DragOverItem)
  {
    setSelected(_DragOverItem, _DragOverItemSelected);
    _DragOverItem = NULL;
  }
}

void KNCListView :: viewportDragMoveEvent (QDragMoveEvent *e)
{
  e->accept(QUriDrag::canDecode(e));

  KNCListViewItem* Item = (KNCListViewItem*) itemAt(e->pos());
  if (Item == _DragOverItem) return;
  if (Item && Item->getFileItem()->isDir())
  {
    if (_DragOverItem != NULL)
      setSelected(_DragOverItem, _DragOverItemSelected);

    _DragOverItem = Item;
    _DragOverItemSelected = Item->isSelected();
    setSelected(_DragOverItem, true);
  }
  else if (_DragOverItem)
  {
    setSelected(_DragOverItem, _DragOverItemSelected);
    _DragOverItem = NULL;
  }
}

void KNCListView :: viewportDragLeaveEvent (QDragLeaveEvent *e)
{
  if (_DragOverItem)
  {
    setSelected(_DragOverItem, _DragOverItemSelected);
    _DragOverItem = NULL;
  }
}

void KNCListView :: viewportDropEvent (QDropEvent *e)
{
  KNCListViewItem* DragOverItem = _DragOverItem;
  QStrList lst;
  bool ret = QUriDrag::decode( e, lst );
  if (!ret) return;

  QPopupMenu Menu (this);
  Menu.insertItem(i18n("&Copy here"), 1);
  Menu.insertItem(i18n("&Move here"), 2);
  Menu.insertItem(i18n("&Link here"), 3);


  int res;
  if ((res = Menu.exec(mapToGlobal(e->pos()))) == -1)
  {
    if (DragOverItem)
    {
      setSelected(DragOverItem, _DragOverItemSelected);
      _DragOverItem = NULL;
    }
    return;
  }

  KURL TargetURL = getURL();
  if (DragOverItem)
  {
    TargetURL.cd(DragOverItem->getFileItem()->name());
    setSelected(DragOverItem, _DragOverItemSelected);
    _DragOverItem = NULL;
  }

  KURL::List URLList;
  for (QStrListIterator it(lst); *it; ++it)
    URLList.append(KURL(*it)); // *it is encoded already

  if (URLList.isEmpty()) return;

  if (res == 1) KNCFileCommand::instance().copy(URLList, TargetURL);
  if (res == 2) KNCFileCommand::instance().move(URLList, TargetURL);
  if (res == 3) KNCFileCommand::instance().link(URLList, TargetURL);
}
