// -*- C++ -*-
/*****************************************************************************
 * KWATCH --- KDE Log File Viewer
 *
 * $Id: prefs.cpp,v 0.9 2000/12/14 20:37:23 mj Exp $
 *
 * Class for kwatch preferences dialog
 *
 *****************************************************************************
 * Copyright (C) 2000
 *  _____ _____
 * |     |___  |   Martin Junius             Internet:  mailto:mj@m-j-s.net
 * | | | |   | |   Radiumstr. 18                        http://www.m-j-s.net/
 * |_|_|_|@home|   D-51069 Koeln, Germany
 *
 * Based on kwatch 0.3, with the following copyright notice:
 *
 * Kwatch is Copyright (C) 1997,1998 Ralph C. Weichert and can be
 * distributed freely under the terms of the GNU General Public License
 * (see file COPYING)
 *****************************************************************************
 * This file is part of KWATCH.
 *
 * KWATCH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * KWATCH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KWATCH; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/


#include <qpushbutton.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qvbox.h>

#include <kapp.h>
#include <kfontdialog.h>
#include <kcolordlg.h>
#include <ksimpleconfig.h>
#include <klocale.h>
#include <kfiledialog.h>

#include "prefs.h"



/*
 * Static member functions
 */
int kwatchPrefs::setWidth(QWidget *w, int wd)
{
    w->adjustSize();
    w->setMinimumSize( w->size() );
    w->setFixedHeight( w->height() );
    int w2 = w->width();
    return w2 > wd ? w2 : wd;
}



void kwatchPrefs::setFontFgBg(QWidget *w,
				   const QFont &font,
				   const QColor &fg, const QColor &bg)
{
    // set font, fg, bg colors
    w->setFont(font);
    QPalette    p = w->palette().copy();
    QColorGroup g = p.normal();
    QColorGroup n(fg, bg, g.light(), g.dark(), g.mid(),
		  fg, g.brightText(), bg, bg); 
    p.setNormal(n);
    p.setActive(n);
    p.setInactive(n);
    w->setPalette(p);
}



kwatchPrefs::kwatchPrefs(QWidget *parent, kwatchConf &cf)
    : KDialogBase(Tabbed, i18n("Preferences"), Ok|Cancel, Ok, parent),
      m_cf(cf), m_copy(cf)
{
    QVBox *page;

    /* Page Files ************************************************************/
    page = addVBoxPage(i18n("Files"));

    QWidget * filetab = new QWidget(page);

    m_files = new QListBox( filetab );

    QPushButton *newfile = new QPushButton( i18n( "Add..." ), filetab );
    connect( newfile, SIGNAL(clicked()), SLOT(addFile()) );

    QPushButton *delfile = new QPushButton( i18n( "Remove" ), filetab );
    connect( delfile, SIGNAL(clicked()), SLOT(delFile()) );

    int w = setWidth( newfile );
    w = setWidth( delfile, w );
    m_files->resize( 3*w, 100 );

    QHBoxLayout * hl = new QHBoxLayout( filetab, 10 );
    hl->addWidget( m_files, 1 );
    QVBoxLayout * vl = new QVBoxLayout();
    hl->addLayout(vl);
    vl->addWidget(newfile);
    vl->addWidget(delfile);
    vl->addStretch(1);

    hl->activate();

    /* Page Window ***********************************************************/
    page = addVBoxPage(i18n("Window"));

    QWidget * viewtab = new QWidget(page);

    m_preview = new QLabel( viewtab );
    m_preview->setText( i18n( "Sample\nText" ) );
    m_preview->setAlignment( AlignCenter );
    m_preview->setFrameStyle( QLabel::Panel | QLabel::Sunken );
    m_preview->setGeometry( 10, 10, 100, 100 );

    QPushButton * fnt = new QPushButton( i18n( "Font..." ), viewtab );
    fnt -> setGeometry( 130, 10, 80, 25 );
    connect( fnt, SIGNAL(clicked()), SLOT(chooseFont()) );

    QPushButton * fg = new QPushButton( i18n( "Foreground..." ), viewtab );
    fg -> setGeometry( 130, 50, 80, 25 );
    connect( fg, SIGNAL(clicked()), SLOT(chooseForeground()) );

    QPushButton * bg = new QPushButton( i18n( "Background..." ), viewtab );
    bg -> setGeometry( 130, 90, 80, 25 );
    connect( bg, SIGNAL(clicked()), SLOT(chooseBackground()) );

    w = setWidth( fnt );
    w = setWidth( fg, w );
    w = setWidth( bg, 0 );
    m_preview->resize( 2*w, 100 );

    hl = new QHBoxLayout( viewtab, 10 );
    hl->addWidget( m_preview, 1 );
    vl = new QVBoxLayout();
    hl->addLayout( vl );
    vl->addWidget( fnt );
    vl->addWidget( fg );
    vl->addWidget( bg );
    vl->addStretch( 1);

    hl->activate();

    /* Page Parameters *******************************************************/
    page = addVBoxPage(i18n("Parameters"));
    QWidget *paramtab = new QWidget(page);
    QGridLayout *g = new QGridLayout(paramtab, 3, 2, 10);
    
    QLabel *label;
    
    label = new QLabel(i18n("Number of lines:"), paramtab);
    label->setAlignment(AlignRight|AlignVCenter);
    m_param1 = new KIntNumInput(paramtab);
    g->addWidget(label, 0, 0);
    g->addWidget(m_param1, 0, 1);
    connect(m_param1, SIGNAL(valueChanged(int)), SLOT(changeParam1(int)));

    label = new QLabel(i18n("Buffer size:"), paramtab);
    label->setAlignment(AlignRight|AlignVCenter);
    m_param2 = new KIntNumInput(paramtab);
    g->addWidget(label, 1, 0);
    g->addWidget(m_param2, 1, 1);
    connect(m_param2, SIGNAL(valueChanged(int)), SLOT(changeParam2(int)));

    /*************************************************************************/
    updateConf();
}



kwatchPrefs::~kwatchPrefs() {
}


void kwatchPrefs::changeParam1(int val)
{
    fprintf(stderr, "kwatchPrefs::changeParam1(%d)\n", val);

    m_copy.maxlines = val;
}

void kwatchPrefs::changeParam2(int val)
{
    fprintf(stderr, "kwatchPrefs::changeParam2(%d)\n", val);

    m_copy.bufsize = val;
}



void kwatchPrefs::openPrefs()
{
    updateConf();
    show();
}



void kwatchPrefs::updateConf()
{
    m_copy = m_cf;

    m_files->clear();
    m_files->insertStringList(m_copy.logfiles);

    setFontFgBg(m_preview, m_copy.font, m_copy.fg, m_copy.bg);

    m_param1->setValue(m_copy.maxlines);
    m_param2->setValue(m_copy.bufsize);
}



void kwatchPrefs::addFile()
{
    QString s = KFileDialog::getOpenFileName(m_copy.logdir, QString::null,
					     this, i18n("Add Log File")   );
    fprintf(stderr, "kwatchPrefs::addFile() s=%s\n",
	    (const char *)s);

    if(!s.isNull() && !s.isEmpty()) 
    {
	if (m_copy.logfiles.find(s) == m_copy.logfiles.end()) {
	    m_files->insertItem(s);
	    m_copy.logfiles.append(s);
	}
    }
}



void kwatchPrefs::delFile() 
{
    int i = m_files->currentItem();
    if (i < 0)
	return;

    QString f = m_files->text(i);
    if (f.isNull() || f.isEmpty())
	return;

    m_copy.logfiles.remove(f);
    m_files->removeItem(i);
}



void kwatchPrefs::chooseFont() 
{
    KFontDialog::getFont(m_copy.font);
    setFontFgBg(m_preview, m_copy.font, m_copy.fg, m_copy.bg);
}



void kwatchPrefs::chooseForeground()
{
    KColorDialog::getColor(m_copy.fg);
    setFontFgBg(m_preview, m_copy.font, m_copy.fg, m_copy.bg);
}



void kwatchPrefs::chooseBackground() 
{
    KColorDialog::getColor(m_copy.bg);
    setFontFgBg(m_preview, m_copy.font, m_copy.fg, m_copy.bg);
}



void kwatchPrefs::setMainConf()
{
    m_cf = m_copy;
}



#include "prefs.moc"
