/**
 * kdesubsystem.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#include "kdesubsystem.h"

#include "configurationview.h"
#include "grouptoken.h"
#include "entrytoken.h"
#include "progressmanager.h"

#include <kio/netaccess.h>
#include <kio/job.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>
#include <kfileitem.h>

#include <kapplication.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kconfig.h>
#include <klocale.h>
#include <kdebug.h>

#include <qeventloop.h>
#include <qdom.h>
#include <qfile.h>
#include <qcstring.h>
#include <qpair.h>

using KIO::MultiGetJob;
using KIO::SimpleJob;
using KIO::UDSEntry;
using KIO::UDSEntryListConstIterator;

using namespace KConfigEditor;

const int MAXITERS = 10;

KDESubsystem::KDESubsystem( QObject *parent )
    : Subsystem( parent ), m_allItems( 0 ), m_parsedItems( 0 ),
      m_attaching( false ), m_attachIterator( m_topLevelGroup ),
      m_pendingJobs( 0 ), m_finishedParsing( false )
{
}

void KDESubsystem::setValue( const KURL &, const QVariant & )
{

}

void KDESubsystem::parse( ConfigurationView *view )
{
    m_view = view;
    m_pendingJobs = 0;
    QStringList kcfgPaths( KGlobal::dirs()->resourceDirs( "kcfg" ) );
    QStringList paths( KGlobal::dirs()->resourceDirs( "config" ) );
    for ( QStringList::Iterator it = kcfgPaths.begin(); it != kcfgPaths.end(); ++it ) {
        createListJob( *it, true );
    }
    for ( QStringList::Iterator it = paths.begin(); it != paths.end(); ++it ) {
        createListJob( *it );
    }
}

void KDESubsystem::saveConfig( GroupToken *app )
{
    Q_ASSERT( app->hasPendingChanges() );

    KConfig config( app->name() );

    QDict<EntryToken> &entries = app->childEntries();
    QDictIterator<EntryToken> itr( entries );
    while ( itr.current() ) {
        if ( itr.current()->hasPendingChanges() )
            writeEntry( config, itr.current() );
        ++itr;
    }

    QDict<GroupToken> &gentries = app->childGroups();
    QDictIterator<GroupToken> itr2( gentries );
    while ( itr2.current() ) {
        writeGroupEntry( config, itr2.current() );
        ++itr2;
    }
}

void KDESubsystem::writeGroupEntry( KConfig &config, GroupToken *token )
{
    if ( !token->hasPendingChanges() )
        return;

    config.setGroup( token->name() );

    QDict<EntryToken> &entries = token->childEntries();
    QDictIterator<EntryToken> itr( entries );
    while ( itr.current() ) {
        if ( itr.current()->hasPendingChanges() )
            writeEntry( config, itr.current() );
        ++itr;
    }
    token->prunePendingChanges();
    config.setGroup( QString::null );
}

void KDESubsystem::writeEntry( KConfig &config, EntryToken *token )
{
    config.writeEntry( token->name(), token->valueAsString() );
    token->prunePendingChanges();
}

void KDESubsystem::createListJob( const QString &url, bool kcfg )
{
    kdDebug()<<"List job with url = "<<url<<endl;
    KIO::SimpleJob* job = KIO::listDir( url, false );

    if ( kcfg )
        m_pendingKCFG.insert( job, job );

    ++m_pendingJobs;
    connect( job, SIGNAL(entries( KIO::Job*, const KIO::UDSEntryList&)),
             SLOT(slotEntries(KIO::Job*, const KIO::UDSEntryList&)) );
    connect( job, SIGNAL(result(KIO::Job *)),
             SLOT(slotListResult(KIO::Job *)) );
    connect( job, SIGNAL(canceled(KIO::Job*)),
             SLOT(slotListCanceled(KIO::Job *)) );
}

void KDESubsystem::slotListCanceled( KIO::Job *job )
{
    m_pendingKCFG.remove( job ); //we get this if an error occured
    --m_pendingJobs;
}

void KDESubsystem::slotEntries( KIO::Job *job, const KIO::UDSEntryList &lst )
{
    KIO::SimpleJob *sjob = static_cast<KIO::SimpleJob*>( job );
    KURL baseUrl( sjob->url() );
    kdDebug()<<"............"<<endl;
    m_allItems += lst.count();
    m_allItems += lst.count()/MAXITERS;//for attaching

    if ( m_pendingKCFG.find( job ) ) {
        for ( KIO::UDSEntryListConstIterator it = lst.begin(); it != lst.end(); ++it ) {
            m_kcfgURLS.append( qMakePair( *it, baseUrl ) );
        }
        m_pendingKCFG.remove( job );
    } else {
         for ( KIO::UDSEntryListConstIterator it = lst.begin(); it != lst.end(); ++it ) {
             m_kconfigURLS.append( qMakePair( *it, baseUrl ) );
         }
    }
    kdDebug()<<"XXXXXXXXXXXXXXXX"<<endl;
    --m_pendingJobs;
    //we need to get all kcfg files before starting parser or
    //we'll end up with conflicts between kconfig and kcfg files
    if ( !m_pendingJobs ) {
        progressItem()->setTotalItems( m_allItems );
        progressItem()->setLabel( i18n( "Parsing KDE files..." ) );
        progressItem()->setStatus( i18n( "Parsing..." ) );
        parseNextFile();
    }
}

void KDESubsystem::slotListResult( KIO::Job* job )
{
    m_pendingKCFG.remove( job ); //we get this if an error occured
    --m_pendingJobs;
}

void KDESubsystem::parseNextFile()
{
    QPair<KIO::UDSEntry, KURL> pair;
    bool kcfgFile = false;

    if ( !m_kcfgURLS.isEmpty() ) {
        pair = m_kcfgURLS.front();
        m_kcfgURLS.pop_front();
        kcfgFile = true;
    } else if ( !m_kconfigURLS.isEmpty() ) {
        pair = m_kconfigURLS.front();
        m_kconfigURLS.pop_front();
    } else {
        if ( m_kcfgURLS.isEmpty() && m_kconfigURLS.isEmpty() && m_pendingJobs &&
             !m_finishedParsing ) {
            //finished parsing
            m_finishedParsing = true;
            kdDebug()<<"X Finished parsing"<<endl;
            QTimer::singleShot( 0, this, SLOT(startAttaching()) );
        }

        return;
    }

    KFileItem fitem( pair.first, pair.second, false, true );

    if ( fitem.isFile() && fitem.size() > 0 && //skip non-files and empty files
         !fitem.name().endsWith( "~" ) ) { //skip backup files
        if ( kcfgFile )
            parseKCFGFile( fitem );
        else
            parseKConfigFile( fitem );
    }

    ++m_parsedItems;

    if ( m_kcfgURLS.isEmpty() && m_kconfigURLS.isEmpty() && m_pendingJobs &&
         !m_finishedParsing) {
        //finished parsing
        m_finishedParsing = true;
        kdDebug()<<"Finished parsing"<<endl;
        QTimer::singleShot( 0, this, SLOT(startAttaching()) );
    } else {
        progressItem()->setProgress( m_parsedItems*100 / m_allItems );
        QTimer::singleShot( 0, this, SLOT(parseNextFile()) );
    }
}

void KDESubsystem::parseKCFGFile( const KFileItem &fitem )
{
    QFile file( fitem.url().path() );
    if ( !file.open( IO_ReadOnly ) ) {
        return;
    }

    QByteArray ba = file.readAll();

    QDomDocument doc( "kcfg" );
     if ( !doc.setContent( ba ) ) {
        kdError() << "Unable to load document." << endl;
        return;
    }
    QDomElement cfgElement = doc.documentElement();
    if ( cfgElement.isNull() ) {
        kdError() << "No document in cfg file" << endl;
        return;
    }
    GroupToken *group = parseKCFGXML( cfgElement, fitem.name() );

    if ( group )
        group->propagateChanges( true );
    else
        kdWarning()<<"KCFG file \"" << fitem.url().url()<< "\" is invalid!!"<<endl;
}

GroupToken *KDESubsystem::parseKCFGXML( const QDomElement &el, const QString &fname )
{
    QDomNode n;
    GroupToken *app = 0;
    GroupToken *curGroup =0;

    for ( n = el.firstChild(); !n.isNull(); n = n.nextSibling() ) {
        QDomElement e = n.toElement();

        QString tag = e.tagName();
        if ( tag == "kcfgfile" ) {
            if ( !app ) {
                QString name = e.attribute( "name" );
                if ( name.isEmpty() ) {
                    name = fname;
                    if ( name.endsWith( ".kcfg" ) )
                        name.truncate( ( name.length() - 5 ) );
                    name += "rc";
                }
                if ( tokenExists( name ) ) {
                    kdWarning()<<"Duplicate config files for \""<< name
                               <<"\". This one will be skipped"<<endl;
                    return 0;
                }
                app = new GroupToken( protocol(), name );
                app->propagateChanges( false );//during the startup we disable it
                m_topLevelGroup.insert( app->url().url(), app );
                //app->attach( m_view );
            }
        } else if ( tag == "include" ) {
            //ignore
        } else if ( tag == "group" ) {
            QString group = e.attribute( "name" );
            if ( group.isEmpty() ) {
                kdError() << "Group without name" << endl;
                continue;
            }
            if ( !app ) {
                kdWarning()<<"The group appears before the kcfgfile entry"<<endl;
                return 0;
            }

            curGroup = app->group( group );
            if ( !curGroup ) {
                curGroup = app->createGroup( group );
                //curGroup->attach( m_view );
            }

            QDomNode n2;
            for( n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() ) {
                QDomElement e2 = n2.toElement();
                curGroup->propagateChanges( false );
                parseKCFGXMLEntry( e2, curGroup );
                curGroup->propagateChanges( true );
            }
        }
    }
    addKConfigValues( app );

    return app;
}

void KDESubsystem::addKConfigValues( GroupToken *parent )
{
    KConfig config( parent->name(), true, false );
    QStringList lst( config.groupList() );
    for ( QStringList::iterator it = lst.begin(); it != lst.end(); ++it ) {
        QMap<QString, QString> en = config.entryMap( *it );

        GroupToken* token = parent->group( *it );
        if ( !token ) {
            token = parent->createGroup( *it );
            //token->attach( m_view );
        }
        token->propagateChanges( false );
        QMap<QString, QString>::ConstIterator itr;
        for ( itr = en.begin(); itr != en.end(); ++itr ) {
            EntryToken* etoken = token->entry( itr.key() );
            if ( !etoken ) {
                etoken = token->createEntry( itr.key(), "String" );
                //etoken->attach( m_view );
            }
            etoken->setValueFromString( itr.data() );
        }
        token->propagateChanges( true );
    }
}

void KDESubsystem::parseKCFGXMLEntry( const QDomElement &element, GroupToken *group )
{
    QString label;
    QString defaultValue;
    double min = 0;
    double max = 0;
    EntryToken::Choice choice;
    QValueList<EntryToken::Choice> choiceList;
    QString whatsThis;
    QString code;
    QStringList choices;
    QStringList values;
    QString type = element.attribute( "type" );
    QString name = element.attribute( "name" );
    QString key  = element.attribute( "key" );
    bool hidden  = ( element.attribute( "hidden" ) == "true" );

    if ( name.isEmpty() && key.isEmpty() )
        return;//FIXME: report an error?

    QDomNode n;
    for ( n = element.firstChild(); !n.isNull(); n = n.nextSibling() ) {
        QDomElement e = n.toElement();
        QString tag = e.tagName();
        if ( tag == "label" ) label = e.text();
        else if ( tag == "whatsthis" ) whatsThis = e.text();
        else if ( tag == "min" ) {
            min = e.text().toDouble();
        } else if ( tag == "max" ) {
             max = e.text().toDouble();
        } else if ( tag == "code" ) code = e.text();
        else if ( tag == "values" ) {
            QDomNode n3;
            for ( n3 = e.firstChild(); !n3.isNull(); n3 = n3.nextSibling() ) {
                QDomElement e3 = n3.toElement();
                if ( e3.tagName() == "value" ) {
                    values.append( e3.text() );
                }
            }
        }
        else if ( tag == "parameter" )
        {
            //ignore
        }
        else if ( tag == "default" ) {
            defaultValue = e.text();
        } else if ( tag == "choices" ) {
            QDomNode n2;
            for( n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() ) {
                QDomElement e2 = n2.toElement();
                if ( e2.tagName() == "choice" ) {
                    QDomNode n3;
                    EntryToken::Choice choice;
                    choice.name = e2.attribute( "name" );
                    if ( choice.name.isEmpty() ) {
                        kdError() << "Tag <choice> requires attribute 'name'." << endl;
                    }
                    for( n3 = e2.firstChild(); !n3.isNull(); n3 = n3.nextSibling() ) {
                        QDomElement e3 = n3.toElement();
                        if ( e3.tagName() == "label" )
                            choice.label = e3.text();
                        if ( e3.tagName() == "whatsthis" )
                            choice.whatsThis = e3.text();
                    }
                    choiceList.append( choice );
                }
            }
        }
    }


    if ( !key.isEmpty() ) {
        name = key;
    }

    if ( type.isEmpty() ) type = "String";

    EntryToken* token = group->entry( name );
    if ( !token ) {
        token =  group->createEntry( name,
                                     type,
                                     hidden );
        Q_ASSERT( token );
        //token->attach( m_view );
    } else {
        //FIXME: might need to switch types here
        Q_ASSERT( token->typeName() == type );
        token->setHidden( hidden );
    }

    token->setLabel( label );
    token->setWhatsThis( whatsThis );
    token->setDefaultValue( defaultValue );
    if ( !defaultValue.isEmpty() )
        token->setValueFromString( defaultValue );
    token->setChoices( choiceList );
    token->setMax( max );
    token->setMin( min );
    token->setValues( values );
}

void KDESubsystem::parseKConfigFile( const KFileItem &fitem )
{
    if ( tokenExists( fitem.name() ) || skipFile( fitem.name() ) )
        return;

    GroupToken *group = new GroupToken( protocol(), fitem.name() );
    m_topLevelGroup.insert( group->url().url(), group );
    //group->attach( m_view );

    group->propagateChanges( false );
    addKConfigValues( group );
    group->propagateChanges( true );
}

bool KDESubsystem::tokenExists( const QString &name )
{
    KURL url;
    url.setProtocol( protocol() );
    url.addPath( name );

    return m_topLevelGroup.find( url.url() );
}

bool KDESubsystem::skipFile( const QString &fileName )
{
    if ( fileName.isEmpty() )
        return true;
    else if ( fileName == "kdebug.areas" )
        return true;
    else if ( fileName == "kxkb_groups" )
        return true;
    else if ( fileName == "gtkrc" )
        return true;

    return false;
}

void KDESubsystem::startAttaching()
{
    m_view->triggerUpdate();

    GroupToken *group = 0;
    if ( !m_attaching ) {
        m_attachIterator = QDictIterator<GroupToken>( m_topLevelGroup );
        m_attaching = true;
    }

    int i = 0;
    while ( ( group = m_attachIterator.current() ) ) {
        group->attach( m_view );

        QDict<GroupToken> &gentries = group->childGroups();
        QDictIterator<GroupToken> itr2( gentries );
        while ( itr2.current() ) {
            itr2.current()->attach( m_view );
            ++itr2;
        }
        ++i;
        ++m_attachIterator;
        if ( i == MAXITERS ) {
            i = 0;
            ++m_parsedItems;
            progressItem()->setProgress( m_parsedItems*100 / m_allItems );
            kapp->eventLoop()->processEvents( QEventLoop::ExcludeUserInput );
        }
    }

    //if !group we're done, otherwise schedule continuation
    if ( !group ) {
        progressItem()->setStatus( i18n( "Done" ) );
        progressItem()->setComplete();
        emit parsingFinished( this );
        m_attaching = false;
    } else {
        QTimer::singleShot( 0, this, SLOT(startAttaching()) );
    }
}

void KDESubsystem::saveToken( EntryToken *token )
{
    EntryToken *entry = static_cast<EntryToken*>( token );
    GroupToken *parent = entry->parent();
    QString group;
    if ( !parent->isTopLevel() ) {
        group = parent->name();
        parent = parent->parent();
    }

    KConfig config( parent->name() );
    config.setGroup( group );
    config.writeEntry( entry->name(), entry->valueAsString() );
}

#include "kdesubsystem.moc"
