#ifndef __KMTRACALL__H__
#define __KMTRACALL__H__

// Qt includes
#include <qobject.h>
#include <qstring.h>

/**
 * This class represends a function/method call in a stack trace. This
 * class is closely related to the class KMtraceLeak as a memory leak
 * consists of a stack trace (class KMtraceCallStack).
 *
 * @short The model of a function/method call
 * @author Andreas Baumann <andreas_baumann@users.sourceforge.net>
 * @see KMtraceModel, KMtraceLeak, KMtraceCallStack
 * @version $ProjectMajorVersion: 0 $
 */
class KMtraceCall : public QObject
{
   Q_OBJECT
   
   public:
      // constructors
      KMtraceCall( );
      KMtraceCall( QString address, QString module, int line, QString function,
                   QString offset );

      // inherit default copy constructor and destructor
      
      /**
       * Get the memory address of the function call.
       *
       * @returns the memory address in the format "0xXXXXXXXX"
       */
      QString getAddress( );

      /**
       * Get the name of the module containing the function. This
       * is normally the name of a C/C++ module (in case the function
       * belongs to code compiled with the gcc -g option). It can
       * also be the name of a library in case of library function,
       * e.g libc.so.6. If you ommit the -g gcc option the name of the
       * module results in the name of the binary being inspected.
       *
       * @returns the name of a source file, binary or a shared library
       *          which contains the function
       */
      QString getModule( );

      /**
       * Get the line where the function starts. This is >0 if
       * @ref getModule returns the name of a C/C++ source code file or
       * 0 in case the function is part of library.
       *
       * @returns the line of the module of the function or 0 in case
       *          case of a library
       */
      int getLine( );
      
      /**
       * Get the name of the function being called. This can be the
       * function/method name as used in the source code followed
       * optionally by an offset "+0xXX" (see @ref getOffset).
       *
       * @returns the name of the function being called
       */
      QString getFunction( );
      
      /**
       * Get the offset into the function. This can happend for
       * inlined code or in code within libraries.
       *
       * @returns an offset of the form "0xXX"
       */
      QString getOffset( );
      
   private:
      // the address of the call
      QString m_address;
      
      // module
      QString m_module;
      
      // line
      int m_line;
      
      // function
      QString m_function;
   
      // offset
      QString m_offset;
};

#endif
