// Qt includes
#include <qwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qstring.h>
#include <qobject.h>
#include <qobjectlist.h>

// KDE includes
#include <kdialogbase.h>
#include <klocale.h>
#include <klistview.h>

// kmtraceviewer includes
#include "KMtraceSuppressListDialog.hh"
#include "KMtraceSuppressListDialog.moc"
#include "KMtraceModel.hh"
#include "KMtraceSuppression.hh"
#include "KMtraceSuppressionViewItem.hh"

KMtraceSuppressListDialog::KMtraceSuppressListDialog( KMtraceModel *model,
                                                      QWidget *parent,
                                                      const char *name,
                                                      bool modal )
: KDialogBase( parent, name, modal, i18n( "List of all Suppressions" ),
               modal ? User2 | User1 | Cancel : User2 | User1 | Close,
               User1, false, i18n( "&Add" ), i18n( "&Delete" ) )
{
   QWidget *page = new QWidget( this );
   setMainWidget( page );
   QVBoxLayout *mainLayout = new QVBoxLayout( page, 0, spacingHint( ) );
   
   QLabel *topLabel = new QLabel( i18n( "The following leak types will be suppressed:" ), page );
   mainLayout->addWidget( topLabel, 1, 0 );
   
   m_listView = new KListView( page );
   m_listView->setAllColumnsShowFocus( true );
   m_listView->addColumn( i18n( "leak type" ) );
   m_listView->addColumn( i18n( "suppression method" ) );
   m_listView->addColumn( i18n( "where" ) );
   m_listView->setShowSortIndicator( true );
   
   mainLayout->addWidget( m_listView, 2, 0 );
   
   if( !modal ) {
      connect( this, SIGNAL( closeClicked( ) ), this, SLOT( slotCancel( ) ) );
   }

   setModel( model );

   // we want to get informed if the model changes (a suppression is added
   // or deleted)
   connect( m_model, SIGNAL( changed( ) ), this, SLOT( slotModelChanged( ) ) );

   // reset values for the suppress dialog
   m_suppressDialog = 0;
   m_suppressDialogRunning = false;
}

void KMtraceSuppressListDialog::slotModelChanged( )
{
   setModel( m_model );
}

// add
void KMtraceSuppressListDialog::slotUser1( )
{
   if( m_suppressDialogRunning ) {
      m_suppressDialog->setFocus( );
   } else {
      KMtraceLeak *leak = 0;
      if( m_suppressDialog == 0 ) {
         m_suppressDialog = new KMtraceSuppressDialog( leak, this );
         connect( m_suppressDialog, SIGNAL( suppress( KMtraceLeak *, QString ) ),
                  this, SLOT( slotSuppress( KMtraceLeak *, QString ) ) );
         connect( m_suppressDialog, SIGNAL( done( ) ), this, SLOT( slotAddSuppressionDone( ) ) );
      } else {
         m_suppressDialog->setLeak( leak );
      }
      m_suppressDialog->show( );
      m_suppressDialogRunning = true;
   }
}

// delete
void KMtraceSuppressListDialog::slotUser2( )
{
   QListViewItem *item = 0;
   KMtraceSuppressionViewItem *suppressionView = 0;
   KMtraceSuppression *suppressionModel = 0;

   item = m_listView->currentItem( );
   if( item != 0 ) {
      suppressionView = (KMtraceSuppressionViewItem *)item;
      suppressionModel = suppressionView->getModel( );
      if( m_model ) {
         m_model->deleteSuppression( suppressionModel );
      }
   }
}

void KMtraceSuppressListDialog::slotCancel( )
{
   emit done( );
   KDialogBase::slotCancel( );
}

void KMtraceSuppressListDialog::setModel( KMtraceModel *model )
{
   m_model = model;

   if( !m_model ) return;

   m_listView->clear( );
   
   QObjectList *suppressionList = m_model->getSuppressionList( );
   QObjectListIt suppressionIterator( *suppressionList );
   QObject *obj;
   
   while( ( obj = suppressionIterator.current( ) ) != 0 ) {
      KMtraceSuppression *suppression = (KMtraceSuppression *)obj;
      if( suppression->getMethod( ) == MatchByString ) {
         KMtraceSuppressionViewItem *item = new KMtraceSuppressionViewItem( m_listView,
                                                  suppression->getLeakType( ),
                                                  suppression->getMethodAsString( ),
                                                  suppression->getDescription( ) );
         item->setModel( suppression );
         suppression->setView( (QObject *)item );
      }
      ++suppressionIterator;
   }
   
}

void KMtraceSuppressListDialog::slotSuppress( KMtraceLeak *leak, QString string )
{
   if( m_suppressDialog == 0 ) {
      return;
   }

   // close the dialog
   m_suppressDialog->close( );
   m_suppressDialogRunning = false;

   // add the new suppression to the model, the model itself will
   // determine how many leaks match this new suppression and will
   // issue messages accordingly...
   KMtraceSuppression *suppression = new KMtraceSuppression( string, MatchByString );
   if( m_model ) {
      m_model->addSuppression( suppression );
   }
}

void KMtraceSuppressListDialog::slotAddSuppressionDone( )
{
   if( m_suppressDialog == 0 ) {
      return;
   }

   m_suppressDialogRunning = false;
}

