/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef COMMAND_H
#define COMMAND_H

// Qt includes
#include <qobject.h>
#include <qvaluelist.h>

// WorKflow includes
#include "value.h"

// forward declarations
class QWidget;
class QMimeSource;
class QDomElement;
class QDomDocument;

namespace WorKflow
{
  class Parameter;
  class Result;
  class CommandDescription;
  class Document;
  class Service;
}

namespace WorKflow
{
  /**
   * @short Represents a WorKflow command.
   *
   * @section overview Overview
   *
   * This class represents a command in a workflow. For actual functionality, it
   * needs to be subclassed. Future versions will provide means to implement a
   * command's functionality using scripting languages.
   *
   * You should not create instances of this class directly. Use a subclass of
   * the CommandFactoryBase class (e.g. CommandFactory) if you need to create
   * a command. A ready-made command factory is available through the command's
   * description object: CommandDescription::factory().
   *
   * @section param Parameters, Results, Input and Output
   *
   * Each command has zero or more parameters as well as zero or more results.
   * Every visible item in the command's GUI usually represents a parameter. In
   * addition to that, a command can get input from it's predecessor in the
   * workflow. This input is also represented by a parameter object. See the
   * Parameter class for details.
   *
   * At all times, a command has at most one assigned input parameter. Input
   * parameters can be those parameters which don't use the Parameter::Source
   * GuiSource (i.e. Parameter::ConnectionSource or Parameter::AllSources). Also,
   * there can be at most one assigned Result object. The value of this result
   * object is passed to the next command as input. The current input parameter
   * is returned by the input() method and set with the setInput() method. The
   * current active Result is returned by output() and set with setOutput().
   *
   * All parameters and results of a command (in fact, all values) are typed.
   * It is very common that the value of a parameter is passed unchanged to a
   * result (sometimes it's not even looked at). Even more often, the value
   * might be changed, but the type of a result depends on the type of a
   * parameter. In this case, it is possible to establish a "link" between
   * a parameter and a result. Then, the type of the result will be the same
   * as the type of the parameter, and the parameter's value will optionally be
   * copied to the result.
   *
   * @section impl Implementing a Custom Command
   *
   * To implement a command, you need to do at least three things:
   *
   * - Add the relevant information to your library's XML file, describing the
   *   command.
   * - Override the execute() method to implement actual functionality.
   * - Add a command factory for your command in your Library subclass.
   *
   * Additionally, it might be neccessary to do one or more of the following
   * things:
   *
   * - Implement a GUI for the command (either with Qt designer or manually).
   * - Reimplement the typeCheck() method.
   * - Manage the services provided by the command.
   *
   * @section xml The XML file
   *
   * The information you need to provide in the XML file is the command's id,
   * name, a description, some keywords, the command's categories, the parameters
   * and the results.
   *
   * Example:
   * @code
   * <command id="org.kde.Standard.ListCount" >
   *     <name>Count List Length</name>
   *     <description>Calculates the length of a list</description>
   *     <categories>
   *         <category id="org.kde.public.ListManipulation" />
   *     </categories>
   *     <keywords>
   *         <keyword>list</keyword>
   *         <keyword>count</keyword>
   *         <keyword>length</keyword>
   *     </keywords>
   *     <param source="connection" id="list" >
   *         <name>List</name>
   *         <description>The list which length is calculated</description>
   *         <types>
   *             <type id="org.kde.public.anything" />
   *         </types>
   *     </param>
   *     <result id="count" >
   *         <name>Count</name>
   *         <description>The number of items in the list</description>
   *         <types>
   *             <type id="org.kde.public.number" />
   *         </types>
   *     </result>
   * </command>
   * @endcode
   *
   * This information will be available at runtime through the
   * id(), originalTitle(), description() and keywords() methods. It will also
   * be accessible through the command's CommandDescription instance, which all
   * commands of a particular type share. The command's Parameter
   * and Result objects will be automatically created and initialized according
   * the the XML description. You can access the parameters and results using the
   * parameter() and result() methods. Additionally, to access a Parameter's
   * value directly, you can use the value() method; to set a Result's value,
   * use the setResult() method.
   *
   * @section xmlref XML Tag Reference
   *
   * The following XML tags and attributes are recognized for commands:
   *
   * - @c command The command tag, required. The required @c id attribute defines
   *   the command's ID. The id is accessible through the id() method.
   *   - @c name The command's name, in english. This will be translated to the
   *     user's current language. Use originalTitle() to get this translated text.
   *     Required.
   *   - @c description A short help text, in english. This will be translated to
   *     the user's current languate. Use description() to get this text. Required.
   *   - @c icon The icon to use. The @c name attribute contains the name of the
   *     icon. Optional.
   *   - @c categories Initiates the list of categories to which this command
   *     belongs.
   *     - @c category A category to which the command belongs. The @c id
   *       attribute is the category's ID.
   *   - @c keywords Initiates the list of keywords for the command. Optional.
   *     - @c keyword An english keyword which is associated to the command. This
   *       will be translated to the user's current language. You can use
   *       lower-case letters for keywords, since when searching for keywords, case
   *       is ignored.
   *   - @c param A tag describing a parameter, optional. Multiple @c param tags
   *     are allowed. See Parameter for details.
   *   - @c result A tag describing a result, optional. Multiple @c result tags
   *     are allowed. See Result for details.
   *
   * @section widget The Command's GUI
   *
   * For most commands, it will be useful to provide a GUI to the user so that
   * she can adjust the command's parameters. In most cases, this can be done
   * in Qt designer using the WorKflow widgets. The widget containing the GUI
   * elements should be a WorKflow::Widget. You should consider using the widgets
   * provided in the WorKflow widget library, since you can assign a command's
   * parameter to them. In that case, the command's parameters are always in
   * sync with the GUI, meaning that the following things are automatically
   * provided:
   *
   * - The WorKflow application is notified of command changes.
   * - Undo and Redo works when the command was modified.
   * - Loading and saving of the command from/to a @c .workflow file works.
   *
   * There are some cases when the provided widgets can't fit your needs. In
   * this case, you have two options:
   *
   * - Implement an own widget, deriving from WidgetInterface and implementing
   *   the needed virtual functions, which then keeps the GUI and the assigned
   *   parameter in sync.
   * - Reimplement the updateParameter() and updateWidget() methods to keep the
   *   GUI and the parameters in sync manually.
   *
   * Which option you should use depends on your case. In general, if the GUI
   * element you need can be used in several commands, it probably is better to
   * implement an own widget; otherwise you should probably use the second
   * option.
   *
   * If you should choose the second option, you need to make sure that the
   * WorKflow library will be informed by every change in the GUI. You do this
   * by emitting the WorKflow::Widget::changed() signal. Then, your
   * reimplemented updateParameters() method will be called. Here, you need to
   * set the values of the parameters which are not assigned to any GUI elements.
   *
   * When a parameter's value is changed by something other than the GUI (e.g.
   * the user undoes a change), the updateWidget() method is called. In your
   * reimplemented version, you should set the contents of the GUI elements
   * which have no parameter assigned.
   *
   * @section constructor The Command's Comstructor
   *
   * The constructor of your command subclass must take a pointer to the
   * document object to which the command is added, as well as a pointer to the
   * command's CommandDescription object. If your command provides a GUI for the
   * user to adjust its parameters, then you should call the setWidget() method
   * from your constructor. Otherwise, just don't call setWidget().
   *
   * Example:
   * @code
   * FilterListCommand::FilterListCommand(Document* parent, CommandDescription* description)
   *   : Command(parent, description)
   * {
   *   FilterListWidget* w = new FilterListWidget(this);
   *   setWidget(w);
   * }
   * @endcode
   *
   * @section typechecking Type Checking
   *
   * The WorKflow library will make sure that values of can only be assigned
   * if their types are equal or if the source type is convertible to the
   * destination type (see also the Conversion class for more information about
   * conversions, and the Datatype class for more information about types). The
   * default implementation of the typeCheck() method checks if the type of the
   * previous command's current output() can be assigned to your command's
   * current input. Additionally, for each result that is linked to a parameter,
   * it sets the result's type to the parameter's type.
   *
   * You can reimplement the typeCheck() method if you need to do more
   * sophisticated things. For example, the commands for variable handling
   * reimplement typeCheck(), because variables have different types, and it
   * can't be known in advance which type a variable will be of.
   *
   * Each command's typeCheck() method will be called whenever some commmand's
   * current input or output changed. You can force a type check by emitting
   * the typeCheckNeeded() signal.
   *
   * @section services Services
   *
   * Some commands provide "services", which are things they do that influence
   * the workflow when it will be executed. Such services are represented by
   * objects of subclasses of the Service class. Other commands can then query
   * at what point in the workflow the particular service will be provided, and
   * by which command. Of course, a service object may store more information
   * than that. Example: One command might store information in a variable.
   * Another command can then query at what point the variable will be defined,
   * and which type the variable's value will be. This allows for extensible
   * type checking. Another use is to check if a previous command executes a
   * certain application which is needed to be running by your command.
   *
   * To indicate that your command provides a certain service, use the
   * addService() method. If the service is not provided anymore, or if details
   * changed about the service, use the removeService() method to indicate that.
   * The services() method returns the current list of provided services.
   *
   * To query the services provided by other commands, use the ServiceInfo
   * class. This will be most useful in a reimplemented typeCheck() method. You
   * should never store a pointer to a service object you got from the
   * ServiceInfo class, as service objects might be deleted at any time.
   *
   * @section execute Command Execution
   *
   * The command is executed by the execute() method. For actual functionality,
   * you need to reimplement this method. The default implementation copies
   * parameters' values to their linked results; if you have linked
   * parameter/result pairs, you should call Command::execute() from your
   * reimplemented method.
   *
   * A typical command's execute method first gets the value of the input
   * parameter and then processes each item of the value, thereby possibly
   * calculating the results. Lastly, for each result, setResult() is called.
   *
   * The following simple example looks for all items in the input value which
   * match with a filter string or regular expression:
   *
   * @code
   * void FilterListCommand::execute()
   * {
   *   QStringList list = value("input").asStringList();
   *   QStringList res;
   *
   *   bool casesens = value("casesensitive").nextBoolean();
   *
   *   if (value("regexp").nextBoolean()) {
   *     QRegExp rx(value("filter").nextString(), casesens);
   *     res = list.grep(rx);
   *   } else {
   *     QString filter = value("filter").nextString();
   *     res = list.grep(filter, casesens);
   *   }
   *
   *   setResult("output", Value::stringList("org.kde.public.anything", res));
   * }
   * @endcode
   *
   * More time-consuming commands should provide progress information. This
   * should be done as each item is processed. To provide progress information,
   * first calculate how much of the execution is done, and then emit the
   * progress() signal, with a string describing the current task.
   *
   * Often, a command needs to wait for some background job to finish. In KDE,
   * when a job is finished, this is usually indicated by the emission of a
   * signal. To simplify waiting for a signal, you can use the waitFor() method
   * which enters the main loop and returns as soon as the signal is emitted.
   *
   * If your command encounters a fatal error or is canceled by the user, you
   * should emit the aborted() signal which terminates the workflow.
   *
   * @section drop Making Commands Drop-Aware
   *
   * Some commands bring "items" of a certain type into the workflow. For example,
   * the "Get Files" command allows the user to specify a list of files which is
   * then passed to the following command. In such cases, it might be a good idea
   * to allow the user to drag these "items" into WorKflow's view and to create
   * a command which then contains these items. Here, "items" would be files
   * which the user dragged from another application, e.g. Konqueror. To make
   * commands drop-aware, you need to specify the accepted drop type in the
   * command's XML description and reimplement the drop() method. In the drop()
   * method, you should add the dropped items to your command's GUI.
   *
   * @section title Title, Original Title and Row
   *
   * The user can change the command's title. The current title is returned by
   * the title() method. The command's original title, as described in the XML
   * file and returned by CommandDescription::name() is also returned by the
   * originalTitle() method.
   *
   * Each command has a unique position in the workflow. This position is
   * returned by the row() method. The setRow() method sets the command's
   * position. Note that setRow() does not actually change the order of the
   * commands in the document. You probably should never need to call setRow().
   *
   * @section state Saving and Loading the Command State
   *
   * The saveState() and loadState() save and load the command parameter's
   * contents to/from a QDataStream. This is neccessary to implement undo and
   * redo. The default implementations save and load the parameter's values.
   * Usually, this should be enough. If you need to store more information,
   * you can reimplement these methods.
   *
   * @sa CommandDescription, CommandFactory, CommandManager
   *
   * @author Thomas Kadauke
   */
  class Command : public QObject
  {
    Q_OBJECT

  public:
    /**
     * Typedef for a list of parameter slots.
     */
    typedef QValueList<Parameter*> ParameterList;

    /**
     * Typedef for a list of result slots.
     */
    typedef QValueList<Result*> ResultList;

    /**
     * Typedef for a list of service objects.
     */
    typedef QValueList<Service*> ServiceList;

    /**
     * Constructor.
     * @param parent The parent document.
     * @param description The command description shared by all commands of the
     *   same type.
     */
    Command(Document* parent, CommandDescription* description);

    /**
     * Destructor.
     */
    virtual ~Command();

    /**
     * Returns a pointer to the command's parent document.
     * @return The command's document.
     */
    Document* document() const;

    /**
     * Returns the command's factory's id.
     * @return The command's factory's id.
     */
    QString id();

    /**
     * Returns the command's current title. Initially this equals the return value
     * of the originalTitle() method, but the title can be changed by the user.
     * @return The command's current title.
     */
    QString title();

    /**
     * Returns the command's original title. If the user didn't change the
     * command's title, this equals the return value of title(). Otherwise, this
     * is the same string as returned by CommandDescription::name().
     * @return The command's original title, i18n'ed.
     */
    QString originalTitle();

    /**
     * Returns a short helpful description of the command.
     * @return The command's description, i18n'ed.
     */
    QString description();

    /**
     * Returns the command's list of parameters.
     * @return A @c const reference to the command's parameter list.
     * @sa addParameter()
     */
    const ParameterList& parameterList();

    /**
     * Returns the list of parameters which can receive input from the previous
     * command.
     * @return The list of connectable parameters.
     */
    ParameterList connectableParameters() const;

    /**
     * Returns the command's list of result slots.
     * @return A @c const reference to the command's result list.
     * @sa addResult()
     */
    const ResultList& resultList();

    /**
     * Finds a parameter by its ID.
     * @param id The parameter's ID.
     * @return The parameter with the given ID, or @c 0, if no such parameter
     *   exists.
     */
    Parameter* parameter(const QString& id);

    /**
     * Finds a result by its ID.
     * @param id The result's ID.
     * @return The result with the given ID, or @c 0, if no such result exists.
     */
    Result* result(const QString& id);

    /**
     * Returns the command's current input parameter, which receives input from
     * the previous command.
     * @return The current input parameter.
     */
    Parameter* input() const;

    /**
     * Returns the command's currently active result, which sends output to the
     * following command.
     * @return The active result.
     */
    Result* output() const;

    /**
     * Returns the current value of parameter @p paramId.
     * @param paramId The parameter's ID.
     * @returns The parameter's current value.
     */
    Value value(const QString& paramId);

    /**
     * Sets the value of result @p resultId to @p value.
     * @param resultId The result's id.
     * @param value The result's new value.
     */
    void setResult(const QString& resultId, const Value& value);

    /**
     * Returns whether or not the command can contain other commands.
     * @return @c true if the command is a container, false otherwise.
     */
    bool isContainer();

    /**
     * Returns the Document object containing the command's child commands.
     * @return The child document.
     */
    Document* childDocument();

    /**
     * Returns whether or not the command can be executed. The default
     * implementation returns @c true. Reimplement if you need control of that.
     * @return @c true if that command can be executed, @c false
     *   otherwise.
     */
    virtual bool canExecute();

    /**
     * Executes the command. You need to override this method in order to give
     * your command meaningful functionality.
     */
    virtual void execute();

    /**
     * The default implementation checks if the type of the previous command's
     * output is convertible to this command's input. Reimplement this method
     * if you need to check more, or if you need to assign types dynamically
     * to parameters or results.
     * @param prev The previous command's current output. Can be @c 0.
     */
    virtual void typeCheck(Result* prev);

    /**
     * Saves the command's state as a string containing XML tags. If you need
     * to store additional information in the command state, reimplement
     * writeXML() and readXML().
     * @return A string containing XML tags describing the command's state.
     */
    QString saveState();

    /**
     * Loads the command's state from a string containing XML tags. If you need
     * to store additional information in the command state, reimplement
     * readXML() and writeXML().
     * @param state A string containing XML tags describing the command's state.
     */
    void loadState(const QString& state);

    /**
     * Resets the command such that it can be executed again.
     */
    void reset();

    /**
     * Returns the widget associated with this command. The default
     * implementation returns the widget which was given to the setWidget()
     * method. Override this method if you need a more sophisticated widget
     * creation.
     */
    virtual QWidget* widget();

    /**
     * Adds a Parameter instance to the list of parameter slots. You should
     * not need to call this method, as this is done automatically according
     * to the XML library description file.
     * @param parameter The parameter to add.
     * @sa parameterList()
     */
    void addParameter(Parameter* parameter);

    /**
     * Adds a Result instance to the list of result slots. You should
     * not need to call this method, as this is done automatically according
     * to the XML library description file.
     * @param result The result to add.
     * @sa resultList()
     */
    void addResult(Result* result);

    /**
     * Returns the command's row number in the command grid.
     * @return The command's row number.
     */
    int row();

    /**
     * Adds a service that is provided by this command.
     * @param service The service to add.
     */
    void addService(Service* service);

    /**
     * Removes a service which is not anymore provided by this command. If a
     * previously provided service merely changes, you should call this method,
     * delete the service object and call addService() for a new service object.
     * @param service The service to remove.
     */
    void removeService(Service* service);

    /**
     * Returns the list of services the command currently provides. If you want
     * to know about services from other commands, use the ServiceInfo class.
     * @return The list of this command's services.
     */
    ServiceList services() const;

    /**
     * Used to handle drops in the WorKflow view window, which are not commands.
     * Reimplement this method in your command class, if your command can handle
     * drops. Example: the "Get Files" command accepts drops of type KUrlDrag,
     * i.e. drags from Konqueror. The default implementation returns @c false.
     * @return Whether or not the drag was really accepted. You should always
     *   return this value, however, if you return @c false, this indicates that
     *   you got a wrong drag type and points out a bug.
     */
    virtual bool drop(QMimeSource* source);

    /**
     * Waits for the emission of signal @p signal, emitted by object @p sender.
     * @param sender The object which will emit the signal.
     * @param signal The signal to wait for. Use the SIGNAL() macro.
     */
    void waitFor(QObject* sender, const char* signal);

    /**
     * Reads the command's state from an XML file. You should not need to call
     * this method. However, if you need to store additional information in the
     * XML file (or the internal state, e.g. for undo/redo), reimplement this
     * method and writeXML().
     * @param e The command's XML tag element.
     */
    virtual void readXML(const QDomElement& e);

    /**
     * Writes the command's state to an XML file. You should not need to call
     * this method. However, if you need to store additional information in the
     * XML file (or the internal state, e.g. for undo/redo), reimplement this
     * method and readXML().
     * @param doc The XML document object.
     * @param e The command's parent XML tag.
     */
    virtual void writeXML(QDomDocument& doc, QDomElement& e);

  public slots:
    /**
     * Sets the command's position in the command list. This method emits the
     * orderChanged() signal.
     * @param row The command's new row number.
     */
    void setRow(int row);

    /**
     * Updates the parameters' contents. You can connect UI elements' signals
     * to this slot, and reimplement it to update the Parameter instances in
     * case the user modified some UI element.
     *
     * If you prefer more fine-grained control over changes, you can use a
     * custom slot for parameter updates.
     */
    virtual void updateParameters();

    /**
     * Updates the widget's UI elements on a parameter change. You need to
     * reimplement this function in your subclass to keep the UI in sync with
     * the parameters, unless you only use parameter-aware widgets.
     */
    virtual void updateWidget();

    /**
     * Sets the command's current title and emits the guiUpdateNeeded() signal.
     * @param title The command's new title.
     */
    void setTitle(const QString& title);

    /**
     * Sets the command's current input parameter. Emits the typeCheckNeeded()
     * signal.
     * @param parameterId The new input parameter's ID.
     */
    void setInput(const QString& parameterId);

    /**
     * Sets the command's current output. Emits the typeCheckNeeded() signal.
     * @param resultId The new output's ID.
     */
    void setOutput(const QString& resultId);

  signals:
    /**
     * Indicates that the command's position in the command grid has changed.
     * This signal is emitted by the setOrder() method.
     * @param row The command's new row number.
     */
    void orderChanged(int row);

    /**
     * Indicates that the widget's size changed.
     */
    void sizeChanged();

    /**
     * Indicates that the command's content (i.e. it's parameters) were
     * modified.
     */
    void modified(Command* command);

    /**
     * Indicates that the GUI around the command might need updating (e.g. if
     * the command's title was changed)
     */
    void guiUpdateNeeded();

    /**
     * Indicates some type changed and that the workflow needs to be checked
     * for type mismatches.
     */
    void typeCheckNeeded();

    /**
     * Indicates that the command wishes to abort the executing workflow.
     */
    void aborted();

    /**
     * Updates the progress display.
     * @param prog The current progress. Must be between @c 0.0 and @c 1.0.
     * @param currentTask A string describing the current task.
     */
    void progress(double prog, const QString& currentTask);

  protected:
    /**
     * Sets the commands associated widget.
     * @param widget The widget.
     */
    void setWidget(QWidget* widget);

    /**
     * Makes this command a container, based on parameter @p container.
     * @param container If @c true, the command can contain other commands.
     */
    void setContainer(bool container);

  protected slots:
    /**
     * Notifies the WorKflow library of changes.
     */
    void notifyChange();

    /**
     * Exits the event loop entered by waitFor().
     */
    void doneWaiting();

  private:
    void fillResults();

    class Private;
    Private* d;
  };
}

#endif

