/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef COMMANDMANAGER_H
#define COMMANDMANAGER_H

// Qt includes
#include <qvaluelist.h>

// forward declarations
namespace WorKflow
{
  class Category;
  class CategoryDescription;
  class CommandDescription;
}

namespace WorKflow
{
  /**
   * @short Keeps track of commands and categories.
   *
   * This singleton class keeps track about command factories, command
   * descriptions and categories. Command descriptions and categories register
   * themselvers with this class, so you don't need to worry about registration.
   * The singleton instance is returned by the static self() method.
   *
   * The registered categories can be queried using the categories() method. The
   * command descriptions in a given category are returned by the commands()
   * method. The allCommands() method returns all available command factories.
   * To find a category by their unique ID, use the findCategory() method.
   *
   * @sa CommandDescription, Category
   *
   * @author Thomas Kadauke
   */
  class CommandManager
  {
  public:
    /**
     * Typedef for a list of command factories.
     */
    typedef QValueList<CommandDescription*> DescriptionList;

    /**
     * Typedef for a list of categories.
     */
    typedef QValueList<Category*> CategoryList;

    /**
     * Returns the one and only singleton instance of this class.
     * @return The only instance.
     */
    static CommandManager* self();

    /**
     * Registers a new category with the CommandManager. You should not need to
     * call this function.
     * @param cat The new category to register.
     */
    void registerCategory(Category* cat);

    /**
     * Removes a category from the list of registered categories. You should not
     * need to call this function.
     * @param cat The category to remove.
     */
    void unregisterCategory(Category* cat);

    /**
     * Returns a list of all registered categories.
     * @return A list of all categories.
     */
    CategoryList categories();

    /**
     * Returns a list of all command factories belonging to a given category.
     * @param cat The category's unique ID to which the returned command
     *   descriptions belong to.
     * @return A list of matching command factories.
     */
    DescriptionList commands(const QString& cat);

    /**
     * Returns a list of all command factories registered with the
     * CommandManager.
     * @return A list if all command factories.
     */
    DescriptionList allCommands();

    /**
     * Finds a command category by their given unique ID.
     * @param id The ID of the command category to look for.
     * @return The Category object with the given ID, or @c 0 if no such
     *   category exists.
     */
    Category* findCategory(const QString& id);

    /**
     * Registers a command description object with the command manager. You
     * should not need to call this function.
     * @param id The command's ID.
     * @param desc The description object.
     */
    void addCommandDescription(const QString& id, CommandDescription* desc);

    /**
     * Removes the command description object with the given @p id. You should
     * not need to call this function.
     * @param id The id of the command for which the description object is to be
     *   removed.
     */
    void removeCommandDescription(const QString& id);

    /**
     * Returns the description object of commands with the given @p id.
     * @param id The commands' id.
     * @return The description object.
     */
    CommandDescription* commandDescription(const QString& id);

    /**
     * Returns a list of command description objects which describe commands
     * that can handle drops (as in drag'n'drop) of the given type.
     * @param type The drop type, e.g. @c "text/plain".
     * @return A list of descriptions for commands which can handle drops of the
     *   given type.
     */
    DescriptionList queryForDropType(const QString& type);

  private:
    // disable constructor and destructor
    CommandManager();
    ~CommandManager();

    class Private;
    Private* d;
  };
}

#endif
