/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>
#include <kinputdialog.h>
#include <klocale.h>

// Qt includes
#include <qlistbox.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qlineedit.h>

// WorKflow includes
#include "command.h"
#include "parameter.h"
#include "datatype.h"

// debugger includes
#include "parameterinspector.h"

using namespace WorKflow;

typedef Command::ParameterList ParList;

ParameterInspector* ParameterInspector::s_self = 0;

ParameterInspector* ParameterInspector::self()
{
  if (!s_self) {
    s_self = new ParameterInspector();
  }
  return s_self;
}

ParameterInspector::ParameterInspector()
  : ParameterInspectorBase(0), m_command(0), m_parameter(0)
{
  m_parameterListBox->setEnabled(false);

  m_typeButtonGroup->setEnabled(false);
  m_valueGroupBox->setEnabled(false);
  m_infoGroupBox->setEnabled(false);

  m_updateInspectorButton->setEnabled(false);
  m_updateWidgetButton->setEnabled(false);
}

ParameterInspector::~ParameterInspector()
{
}

void ParameterInspector::setCommand(Command* command)
{
  m_command = command;
  m_parameter = 0;

  m_parameterListBox->clear();
  ParList parameters = m_command->parameterList();
  for (ParList::ConstIterator i = parameters.begin(); i != parameters.end(); ++i) {
    Parameter* param = *i;
    m_parameterListBox->insertItem(QString("%1 (%2)").arg(param->name()).arg(param->id()));
  }

  m_parameterListBox->setEnabled(true);
  m_updateInspectorButton->setEnabled(true);
  m_updateWidgetButton->setEnabled(true);

  m_typeButtonGroup->setEnabled(false);
  m_valueGroupBox->setEnabled(false);
  m_infoGroupBox->setEnabled(false);
}

void ParameterInspector::slotParameterTypeChanged(int index)
{
  kdDebug() << k_funcinfo << " index: " << index << endl;
  switch (index) {
    case 0:
      // Unset
    case 2:
      // Piped
      m_valueGroupBox->setEnabled(false);
      break;
    case 1:
      // Static
      m_valueGroupBox->setEnabled(true);
      enableButtons();
      break;
  }

  emit changed();
}

void ParameterInspector::slotValueItemSelected()
{
  enableButtons();
}

void ParameterInspector::slotParameterSelected()
{
  if (m_parameter)
    commitChanges();

  updateInspector();
}

void ParameterInspector::showDatatypeInfo()
{}

void ParameterInspector::moveDown()
{
  QListBoxItem* item = m_valueListBox->selectedItem();
  int index = m_valueListBox->index(item);
  QString text = item->text();

  m_valueListBox->removeItem(index);
  m_valueListBox->insertItem(text, index + 1);

  enableButtons();
  emit changed();
}

void ParameterInspector::moveUp()
{
  QListBoxItem* item = m_valueListBox->selectedItem();
  int index = m_valueListBox->index(item);
  QString text = item->text();

  m_valueListBox->removeItem(index);
  m_valueListBox->insertItem(text, index - 1);

  enableButtons();
  emit changed();
}

void ParameterInspector::deleteValue()
{
  int index = m_valueListBox->index(m_valueListBox->selectedItem());
  m_valueListBox->removeItem(index);
  enableButtons();
  emit changed();
}

void ParameterInspector::editValue()
{
  QListBoxItem* item = m_valueListBox->selectedItem();
  int index = m_valueListBox->index(item);
  QString text = getValue(item->text());
  if (!text.isNull()) {
    m_valueListBox->removeItem(index);
    m_valueListBox->insertItem(text, index);
  }
  enableButtons();
  emit changed();
}

void ParameterInspector::addValue()
{
  QString text = getValue();
  if (!text.isNull()) {
    m_valueListBox->insertItem(text);
  }
  enableButtons();
  emit changed();
}

void ParameterInspector::enableButtons()
{
  int count = m_valueListBox->count();
  int index = m_valueListBox->index(m_valueListBox->selectedItem());

  m_addButton->setEnabled(true);
  m_editButton->setEnabled(index != -1);
  m_deleteButton->setEnabled(index != -1);
  m_moveUpButton->setEnabled(count > 1 && index != -1 && index != 0);
  m_moveDownButton->setEnabled(count > 1 && index != -1 && index != count - 1);
}

QString ParameterInspector::getValue(const QString& oldvalue)
{
  QString text = KInputDialog::getText(i18n("Input value"), i18n("Value"), oldvalue /*, &ok, this*/);
  return text;
}

void ParameterInspector::commitChanges()
{
  if (m_unsetRadio->isChecked()) {
    m_parameter->clear();
  } else if (m_staticRadio->isChecked()) {
    Value value(m_parameter->typeId());
    for (unsigned int i = 0; i != m_valueListBox->count(); i++) {
      QString item = m_valueListBox->text(i);
      value.addString(item);
    }
    m_parameter->setStaticValue(value);
  } else if (m_pipedRadio->isChecked()) {
    // TODO: does that work???
  }
}

void ParameterInspector::slotClosePressed()
{
  commitChanges();
  close();
}

void ParameterInspector::updateInspector()
{
  int index = m_parameterListBox->index(m_parameterListBox->selectedItem());

  Parameter* param = 0;

  int count = 0;
  ParList parameters = m_command->parameterList();
  for (ParList::ConstIterator i = parameters.begin(); i != parameters.end(); ++i, ++count) {
    if (count == index) {
      param = *i;
      break;
    }
  }

  if (!param)  {
    kdWarning() << k_funcinfo << " param == 0" << endl;
    return;
  }

  m_parameter = param;

  m_typeButtonGroup->setEnabled(true);
  m_infoGroupBox->setEnabled(true);


  m_idEdit->setText(param->id());
  m_nameEdit->setText(param->name());
  m_datatypeEdit->setText(QString("%1 (%2)").arg(param->type()->name()).arg(param->typeId()));

  m_valueListBox->clear();
  switch (param->kind()) {
    case Parameter::Unset: {
      m_unsetRadio->setChecked(true);
      m_valueGroupBox->setEnabled(false);
      break;
    }
    case Parameter::Static: {
      m_staticRadio->setChecked(true);
      m_valueGroupBox->setEnabled(true);

      Value value = param->value();
      QStringList valueItems = value.asStringList();

      m_valueListBox->insertStringList(valueItems);

      break;
    }
    case Parameter::Piped: {
      m_pipedRadio->setChecked(true);
      m_valueGroupBox->setEnabled(false);
      break;
    }
  }
}

void ParameterInspector::updateWidget()
{
  commitChanges();
  m_command->updateWidget();
}

void ParameterInspector::slotAutoUpdateChecked(bool check)
{
  if (check) {
    connect(this, SIGNAL(changed()), this, SLOT(updateWidget()));
    connect(m_command, SIGNAL(modified(Command*)), this, SLOT(updateInspector()));
  } else {
    disconnect(this, SIGNAL(changed()), this, SLOT(updateWidget()));
    disconnect(m_command, SIGNAL(modified(Command*)), this, SLOT(updateInspector()));
  }
}

void ParameterInspector::showEvent(QShowEvent*)
{
  emit visibilityChanged();
}

void ParameterInspector::hideEvent(QHideEvent*)
{
  emit visibilityChanged();
}

#include "parameterinspector.moc"
