/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage budgets.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgbudgetpluginwidget.h"

#include <QDomDocument>
#include <QKeyEvent>

#include "skgbudgetdelegate.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgbudgetobject.h"
#include "skgbudgetruleobject.h"
#include "skgcategoryobject.h"
#include "skgtransactionmng.h"
#include "skgtraces.h"
#include "skgdocument.h"

SKGBudgetPluginWidget::SKGBudgetPluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEINFUNC(1);
    if (!iDocument) {
        return;
    }

    ui.setupUi(this);

    ui.kUseScheduledOperation->hide();

    ui.kPeriodLbl->setText(i18n("%1:", iDocument->getDisplay("t_period")));
    ui.kYearLbl->setText(i18n("%1:", iDocument->getDisplay("i_year")));
    ui.kMonthLbl->setText(i18n("%1:", iDocument->getDisplay("i_month")));
    ui.kAmountLabel->setText(i18n("%1:", iDocument->getDisplay("f_value")));
    ui.kCategoryLabel->setText(i18n("%1:", iDocument->getDisplay("t_category")));
    ui.kYearCheck->setText(i18n("%1:", iDocument->getDisplay("i_year")));
    ui.kMonthCheck->setText(i18n("%1:", iDocument->getDisplay("i_month")));
    ui.kCategoryCheck->setText(i18n("%1:", iDocument->getDisplay("t_category")));
    ui.kPeriodLbl2->setText(i18n("%1:", iDocument->getDisplay("t_period")));
    ui.kAmountLabel2->setText(i18n("%1:", iDocument->getDisplay("f_value")));
    ui.kCategoryTransferCheck->setText(i18n("%1:", iDocument->getDisplay("t_category")));

    ui.kYear->setValue(QDate::currentDate().year());
    ui.kYearAuto->setValue(QDate::currentDate().year());
    ui.kYearAutoBase->setValue(QDate::currentDate().year());
    ui.kMonth->setValue(QDate::currentDate().month());
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Monthly"));
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Yearly"));
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Individual"));


    ui.kView->getShowWidget()->addItem("all", i18nc("Noun, budget items to display", "All"), "",
                                       "",
                                       "current;currentYear;currentMonth;previousYear;previousMonth",
                                       "none", "", "",
                                       Qt::META + Qt::Key_A);
    ui.kView->getShowWidget()->addItem("none", i18nc("Noun, budget items to display", "None"), "",
                                       "1=0",
                                       "",  // Check when checked
                                       "all;current;currentYear;currentMonth;previousYear;previousMonth",       // Uncheck when checked
                                       "",       // Check when unchecked
                                       "all",    // Uncheck when unchecked
                                       Qt::META + Qt::Key_0);
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("current", i18nc("Noun, budget items to display", "Current"), "",
                                       "t_PERIOD>=STRFTIME('%Y-%m', date('now')) OR t_PERIOD=STRFTIME('%Y', date('now'))",
                                       "currentMonth",  // Check when checked
                                       "none",       // Uncheck when checked
                                       "",       // Check when unchecked
                                       "all",    // Uncheck when unchecked
                                       Qt::META + Qt::Key_1);
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("currentYear", i18nc("Noun, budget items to display", "Current year"), "view-calendar-month",
                                       "t_PERIOD LIKE STRFTIME('%Y', date('now'))||'%'",
                                       "",
                                       "none",
                                       "",
                                       "all",
                                       Qt::META + Qt::Key_3);;
    ui.kView->getShowWidget()->addItem("currentMonth", i18nc("Noun, budget items to display", "Current month"), "view-calendar-week",
                                       "t_PERIOD=STRFTIME('%Y-%m', date('now'))",
                                       "",
                                       "none",
                                       "",
                                       "all;current",
                                       Qt::META + Qt::Key_2);
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("previousYear", i18nc("Noun, budget items to display", "Previous year"), "view-calendar-month",
                                       "t_PERIOD LIKE STRFTIME('%Y', date('now','-1 Year'))||'%'",
                                       "",
                                       "none",
                                       "",
                                       "all",
                                       Qt::META + Qt::Key_5);
    ui.kView->getShowWidget()->addItem("previousMonth", i18nc("Noun, budget items to display", "Previous month"), "view-calendar-week",
                                       "t_PERIOD=STRFTIME('%Y-%m', date('now','-1 Month'))",
                                       "",
                                       "none",
                                       "",
                                       "all",
                                       Qt::META + Qt::Key_4);
    ui.kView->getShowWidget()->setDefaultState("current");

    m_timer.setSingleShot(true);
    connect(&m_timer, &QTimer::timeout, this, &SKGBudgetPluginWidget::refreshInfoZone, Qt::QueuedConnection);

    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "All"), SKGBudgetRuleObject::ALL);
    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "Negative"), SKGBudgetRuleObject::NEGATIVE);
    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "Positive"), SKGBudgetRuleObject::POSITIVE);

    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Next"), SKGBudgetRuleObject::NEXT);
    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Current"), SKGBudgetRuleObject::CURRENT);
    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Current year"), SKGBudgetRuleObject::YEAR);

    ui.kView->getView()->setItemDelegate(new SKGBudgetDelegate(ui.kView->getView(), getDocument()));

    SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
    if (doc) {
        ui.kUnitCmb->addItem("%");
        ui.kUnitCmb->addItem(doc->getPrimaryUnit().Symbol);
        ui.kUnit->setText(doc->getPrimaryUnit().Symbol);

        // Bind operation view
        m_objectModel = new SKGObjectModel(doc, "v_budget_display", "1=0", this, "", false);
        ui.kView->setModel(m_objectModel);
    }

    // Add registered global action in contextual menu
    if (SKGMainPanel::getMainPanel()) {
        ui.kView->getView()->insertGlobalAction();
        ui.kView->getView()->insertGlobalAction("open");
        ui.kView->getView()->insertGlobalAction();
        ui.kView->getView()->insertGlobalAction("edit_delete");
    }
    connect(ui.kView->getView(), &SKGTreeView::doubleClicked, SKGMainPanel::getMainPanel()->getGlobalAction("open"), &QAction::trigger);
    connect(ui.kView->getView(), &SKGTreeView::selectionChangedDelayed, [ = ] {this->onSelectionChanged();});

    // Add Standard KDE Icons to buttons to Operations
    ui.kModifyBtn->setIcon(QIcon::fromTheme("dialog-ok-apply"));
    ui.kAddBtn->setIcon(QIcon::fromTheme("list-add"));
    QAction* processAction = SKGMainPanel::getMainPanel()->getGlobalAction("tool_process_budget_rules");
    if (processAction) {
        ui.kProcessBtn->setIcon(processAction->icon());
        connect(ui.kProcessBtn, &QPushButton::clicked, processAction, &QAction::trigger);
    }

    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGManualSection);
        list.push_back(ui.SKGEditionButtonsSection);
        ui.kWidgetSelector->addButton(QIcon::fromTheme("user-properties"), i18n("Manual"), i18n("Display the edit panel for standard budget"), list);
    }
    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGAutoSection);
        list.push_back(ui.SKGEditionButtonsSection);
        ui.kWidgetSelector->addButton(QIcon::fromTheme("games-solve"), i18n("Auto"), i18n("Display the edit panel for automatic budgets"), list);
    }
    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGRuleSection);
        list.push_back(ui.SKGEditionButtonsSection);
        ui.kWidgetSelector->addButton(QIcon::fromTheme("view-financial-transfer"), i18n("Rules"), i18n("Display the edit panel for rules"), list);
    }
    connect(ui.kWidgetSelector, &SKGWidgetSelector::selectedModeChanged, this, &SKGBudgetPluginWidget::onBtnModeClicked, Qt::QueuedConnection);

    ui.kWidgetSelector->setSelectedMode(0);

    // Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);

    // Refresh
    connect(getDocument(), &SKGDocument::tableModified, this, &SKGBudgetPluginWidget::dataModified, Qt::QueuedConnection);
    dataModified("", 0);
}

SKGBudgetPluginWidget::~SKGBudgetPluginWidget()
{
    SKGTRACEINFUNC(1);
    m_objectModel = NULL;
}

bool SKGBudgetPluginWidget::eventFilter(QObject* iObject, QEvent* iEvent)
{
    if (iEvent && iEvent->type() == QEvent::KeyPress) {
        QKeyEvent* keyEvent = static_cast<QKeyEvent*>(iEvent);
        if ((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && iObject == this) {
            if (QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddBtn->isEnabled()) {
                ui.kAddBtn->click();
            } else if (QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyBtn->isEnabled()) {
                ui.kModifyBtn->click();
            }
        }
    }

    return false;
}

QString SKGBudgetPluginWidget::getState()
{
    SKGTRACEINFUNC(10);
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    root.setAttribute("currentPage", SKGServices::intToString(ui.kWidgetSelector->getSelectedMode()));
    if (m_objectModel && m_objectModel->getRealTable() == "budget") {
        root.setAttribute("view", ui.kView->getState());
        root.setAttribute("viewRule", m_viewRule);
    } else {
        root.setAttribute("view", m_viewBudget);
        root.setAttribute("viewRule", ui.kView->getState());
    }

    return doc.toString();
}

void SKGBudgetPluginWidget::setState(const QString& iState)
{
    SKGTRACEINFUNC(10);
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString currentPage = root.attribute("currentPage");
    if (currentPage.isEmpty()) {
        currentPage = '0';
    }

    ui.kWidgetSelector->setSelectedMode(SKGServices::stringToInt(currentPage));

    m_viewBudget = root.attribute("view");
    m_viewRule = root.attribute("viewRule");
    if (m_objectModel && m_objectModel->getRealTable() == "budget") {
        ui.kView->setState(m_viewBudget);
    } else {
        ui.kView->setState(m_viewRule);
    }
}

QString SKGBudgetPluginWidget::getDefaultStateAttribute()
{
    return "SKGBUDGET_DEFAULT_PARAMETERS";
}

QWidget* SKGBudgetPluginWidget::mainWidget()
{
    return ui.kView->getView();
}

void SKGBudgetPluginWidget::refresh()
{
    SKGTRACEINFUNC(1);

    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if (db != NULL) {
        // Refresh yours widgets here
    }
}

void SKGBudgetPluginWidget::dataModified(const QString& iTableName, int iIdTransaction, bool iLightTransaction)
{
    SKGTRACEINFUNC(10);
    Q_UNUSED(iIdTransaction);

    // Refresh widgets
    if (iTableName == "budget" || iTableName.isEmpty()) {
        // Refresh info area
        m_timer.start(300);
    }

    if (!iLightTransaction) {
        if (iTableName == "category" || iTableName.isEmpty()) {
            // Set type category
            SKGMainPanel::fillWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");
            SKGMainPanel::fillWithDistinctValue(ui.kCategoryRule, getDocument(), "category", "t_fullname", "");
            SKGMainPanel::fillWithDistinctValue(ui.kCategoryTransfer, getDocument(), "category", "t_fullname", "");
        }
    }
}

void SKGBudgetPluginWidget::onBtnModeClicked(int mode)
{
    SKGTRACEINFUNC(10);
    if (!m_objectModel) {
        return;
    }

    if (mode == 2 && m_objectModel->getTable() != "v_budgetrule_display") {
        ui.kView->getShowWidget()->setEnabled(false);
        m_viewBudget = ui.kView->getState();
        m_objectModel->setFilter("");
        m_objectModel->setTable("v_budgetrule_display");
        ui.kView->setState(m_viewRule);
    } else if (mode != 2 && m_objectModel->getTable() != "v_budget_display") {
        ui.kView->getShowWidget()->setEnabled(true);
        m_viewRule = ui.kView->getState();
        m_objectModel->setTable("v_budget_display");
        ui.kView->setState(m_viewBudget);
    }

    onCreatorModified();
}

void SKGBudgetPluginWidget::onAddClicked()
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);

    if (ui.kWidgetSelector->getSelectedMode() == 2) {
        // Creation of a rule
        QStringList uniqueIDs;
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget rule creation")  , err);
            SKGBudgetRuleObject budgetRule(getDocument());
            IFOKDO(err, updateBudgetRule(budgetRule))
            uniqueIDs.push_back(budgetRule.getUniqueID());

            // Send message
            IFOKDO(err, budgetRule.getDocument()->sendMessage(i18nc("An information to the user", "The budget rule '%1' has been added", budgetRule.getDisplayName()), SKGDocument::Hidden));
        }
        // status bar
        IFOK(err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget rule created"));
            ui.kView->getView()->selectObjects(uniqueIDs);
        } else {
            err.addError(ERR_FAIL, i18nc("Error message", "Budget rule creation failed"));
        }
    } else {
        // Creation of a budget
        QStringList uniqueIDs;
        {
            SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget creation")  , err, 2);
            if (ui.kWidgetSelector->getSelectedMode() == 0) {
                // Manual creation
                int mode = ui.kPeriod->currentIndex();
                if (mode == 0) {  // Monthly
                    for (int m = 1; !err && m <= 12; ++m) {
                        SKGBudgetObject budget(getDocument());
                        IFOKDO(err, updateBudget(budget, m))

                        // Send message
                        IFOKDO(err, getDocument()->sendMessage(i18nc("An information to the user", "The budget '%1' has been added", budget.getDisplayName()), SKGDocument::Hidden));

                        uniqueIDs.push_back(budget.getUniqueID());
                    }
                } else if (mode == 1) {  // Yearly
                    SKGBudgetObject budget(getDocument());

                    IFOKDO(err, updateBudget(budget, 0))

                    // Send message
                    IFOKDO(err, getDocument()->sendMessage(i18nc("An information to the user", "The budget '%1' has been added", budget.getDisplayName()), SKGDocument::Hidden));

                    uniqueIDs.push_back(budget.getUniqueID());
                } else {  // Individual
                    SKGBudgetObject budget(getDocument());
                    IFOKDO(err, updateBudget(budget))

                    // Send message
                    IFOKDO(err, getDocument()->sendMessage(i18nc("An information to the user", "The budget '%1' has been added", budget.getDisplayName()), SKGDocument::Hidden));

                    uniqueIDs.push_back(budget.getUniqueID());
                }
            } else {
                // Automatic creation
                if (ui.kAutoBudgetCheck->isChecked()) err = SKGBudgetObject::createAutomaticBudget(static_cast<SKGDocumentBank*>(getDocument()),
                            ui.kYearAuto->value(),
                            ui.kYearAutoBase->value(),
                            ui.kUseScheduledOperation->isChecked(),
                            ui.kRemovePrevious->isChecked());
                IFOKDO(err, getDocument()->stepForward(1))

                IFOKDO(err, SKGBudgetObject::balanceBudget(static_cast<SKGDocumentBank*>(getDocument()),
                        ui.kYearAuto->value(), (ui.kBalancingMonthly->isChecked() ? 0 : -1),
                        ui.kBalancingAnnual->isChecked()));
                IFOKDO(err, getDocument()->stepForward(2))
            }
        }

        // status bar
        IFOK(err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget created"));
            ui.kView->getView()->selectObjects(uniqueIDs);
        } else {
            err.addError(ERR_FAIL, i18nc("Error message", "Budget creation failed"));
        }
    }

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGBudgetPluginWidget::onUpdateClicked()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    // Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if (ui.kWidgetSelector->getSelectedMode() == 2) {
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget rule update"), err);
            SKGBudgetRuleObject rule(selection.at(0));
            IFOKDO(err, updateBudgetRule(rule))

            // Send message
            IFOKDO(err, rule.getDocument()->sendMessage(i18nc("An information to the user", "The budget rule '%1' has been updated", rule.getDisplayName()), SKGDocument::Hidden));
        }

        // status bar
        IFOK(err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget rule updated"));
        } else {
            err.addError(ERR_FAIL, i18nc("Error message",  "Budget rule update failed"));
        }
    } else {
        {
            int nb = selection.count();
            SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget update"), err, nb);
            for (int i = 0; !err && i < nb; ++i) {
                SKGBudgetObject budget(selection.at(i));
                int mode = ui.kPeriod->currentIndex();
                if (mode == 1) {  // Yearly
                    err = updateBudget(budget, 0);
                } else {  // Individual
                    err = updateBudget(budget);
                }

                IFOKDO(err, getDocument()->stepForward(i + 1))
            }
        }

        // status bar
        IFOK(err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget updated"));
        } else {
            err.addError(ERR_FAIL, i18nc("Error message",  "Budget update failed"));
        }
    }


    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

SKGError SKGBudgetPluginWidget::updateBudget(SKGBudgetObject& iBudget, int iMonth)
{
    SKGError err;
    if (!err && ui.kYear->isEnabled()) {
        err = iBudget.setYear(ui.kYear->value());
    }
    if (!err && ui.kMonth->isEnabled()) {
        err = iBudget.setMonth(iMonth != -1 ? iMonth : ui.kMonth->value());
    }

    SKGCategoryObject cat;
    QString catName = ui.kCategoryEdit->text().trimmed();
    IFOKDO(err, SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true))
    IFOKDO(err, iBudget.setCategory(cat))
    IFOKDO(err, iBudget.enableSubCategoriesInclusion(ui.kIncludingSubCategories->isChecked()))

    double val = ui.kAmountEdit->value();
    // Is the sign forced ?
    if (ui.kAmountEdit->sign() == 0) {
        // No
        SKGObjectBase cat2(cat.getDocument(), "v_category_display", cat.getID());

        // Are we able to find to sign with the category ?
        if (cat2.getAttribute("t_TYPEEXPENSE") == "-") {
            val = -val;
        }
    }
    IFOKDO(err, iBudget.setBudgetedAmount(val))

    IFOKDO(err, iBudget.save())
    return err;
}

SKGError SKGBudgetPluginWidget::updateBudgetRule(SKGBudgetRuleObject& iRule)
{
    SKGError err;
    SKGCategoryObject cat;
    QString catName = ui.kCategoryRule->text().trimmed();
    IFOKDO(err, SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true))
    SKGCategoryObject catchange;
    QString catchangeName = ui.kCategoryTransfer->text().trimmed();
    IFOKDO(err, SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catchangeName, catchange, true))
    IFOKDO(err, iRule.enableCategoryCondition(ui.kCategoryCheck->isChecked()))
    IFOKDO(err, iRule.setBudgetCategory(cat))
    IFOKDO(err, iRule.enableYearCondition(ui.kYearCheck->isChecked()))
    IFOKDO(err, iRule.setBudgetYear(ui.kYearRule->value()))
    IFOKDO(err, iRule.enableMonthCondition(ui.kMonthCheck->isChecked()))
    IFOKDO(err, iRule.setBudgetMonth(ui.kMonthRule->value()))
    IFOK(err) {
        bool absolute = (ui.kUnitCmb->currentIndex() == 1);
        double val = ui.kAmountEdit2->value();
        if (!absolute) {
            val = qMin(qMax(static_cast<double>(0), val), static_cast<double>(100));
        }
        err = iRule.setQuantity(val, absolute);
    }
    IFOKDO(err, iRule.setCondition(static_cast<SKGBudgetRuleObject::Condition>(ui.kConditionCmb->itemData(ui.kConditionCmb->currentIndex()).toInt())))
    IFOKDO(err, iRule.enableCategoryChange(ui.kCategoryTransferCheck->isChecked()))
    IFOKDO(err, iRule.setTransfer(static_cast<SKGBudgetRuleObject::Mode>(ui.kModeCmb->itemData(ui.kModeCmb->currentIndex()).toInt()), catchange))
    IFOKDO(err, iRule.save())
    return err;
}

void SKGBudgetPluginWidget::onCreatorModified()
{
    bool test = !ui.kAmountEdit->text().isEmpty() && !ui.kYear->text().isEmpty();
    ui.kAddBtn->setEnabled(test || ui.kWidgetSelector->getSelectedMode() != 0);
    ui.kModifyBtn->setEnabled((test && ui.kPeriod->currentIndex() != 0 && ui.kWidgetSelector->getSelectedMode() == 0 && getNbSelectedObjects())
                              || (ui.kWidgetSelector->getSelectedMode() == 2 && getNbSelectedObjects() == 1));

    bool monthCondition = (ui.kPeriod->currentIndex() == 2 || ui.kWidgetSelector->getSelectedMode() == 2);
    ui.kMonthLbl->setVisible(monthCondition);
    ui.kMonth->setVisible(monthCondition);
}

void SKGBudgetPluginWidget::onSelectionChanged()
{
    SKGTRACEINFUNC(10);
    if (!m_objectModel) {
        return;
    }

    SKGObjectBase::SKGListSKGObjectBase objs = getSelectedObjects();
    int nb = objs.count();
    int mode = ui.kWidgetSelector->getSelectedMode();
    if (nb) {
        if (m_objectModel->getRealTable() == "budget") {
            SKGBudgetObject budget(objs.at(0));
            ui.kYear->setValue(budget.getYear());
            ui.kMonth->setValue(budget.getMonth());
            ui.kAmountEdit->setValue(budget.getBudgetedAmount());
            ui.kCategoryEdit->setText(budget.getAttribute("t_CATEGORY"));
            ui.kPeriod->setCurrentIndex(budget.getMonth() == 0 ? 1 : 2);  // Set yearly or individual
            ui.kIncludingSubCategories->setChecked(budget.isSubCategoriesInclusionEnabled());

            if (mode > 0) {
                ui.kWidgetSelector->setSelectedMode(0);
            }
        } else {
            SKGBudgetRuleObject rule(objs.at(0));
            ui.kYearCheck->setChecked(rule.isYearConditionEnabled());
            ui.kYearRule->setValue(rule.getBudgetYear());
            ui.kMonthCheck->setChecked(rule.isMonthConditionEnabled());
            ui.kMonthRule->setValue(rule.getBudgetMonth());
            ui.kCategoryCheck->setChecked(rule.isCategoryConditionEnabled());
            ui.kCategoryRule->setText(rule.getAttribute("t_CATEGORYCONDITION"));
            ui.kCategoryTransferCheck->setChecked(rule.isCategoryChangeEnabled());
            ui.kCategoryTransfer->setText(rule.getAttribute("t_CATEGORY"));
            ui.kUnitCmb->setCurrentIndex(rule.isAbolute() ? 1 : 0);
            ui.kAmountEdit2->setValue(rule.getQuantity());
            ui.kModeCmb->setCurrentIndex(ui.kModeCmb->findData(rule.getTransferMode()));
            ui.kConditionCmb->setCurrentIndex(ui.kConditionCmb->findData(rule.getCondition()));
        }
    }

    ui.kPeriod->setEnabled(nb <= 1);
    ui.kYear->setEnabled(nb <= 1);
    ui.kMonth->setEnabled(nb <= 1);

    onCreatorModified();
    refreshInfoZone();
    Q_EMIT selectionChanged();
}

void SKGBudgetPluginWidget::activateEditor()
{
    if (ui.kWidgetSelector->getSelectedMode() == -1) {
        ui.kWidgetSelector->setSelectedMode(0);
    }
    ui.kAmountEdit->setFocus();
}

bool SKGBudgetPluginWidget::isEditor()
{
    return true;
}

void SKGBudgetPluginWidget::refreshInfoZone()
{
    SKGTRACEINFUNC(10);
    SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
    if (doc && ui.kWidgetSelector->getSelectedMode() != 2) {
        SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();
        SKGServices::SKGUnitInfo secondary = doc->getSecondaryUnit();
        // Refresh info area with selection
        SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();

        double budgeted = 0;
        double modified = 0;
        int nb = selection.count();
        for (int i = 0; i < nb; ++i) {
            SKGBudgetObject budget(selection.at(i));
            budgeted += budget.getBudgetedAmount();
            modified += budget.getBudgetedModifiedAmount();
        }

        QString budgetedS = doc->formatMoney(budgeted, primary);
        QString modifiedS = doc->formatMoney(modified, primary);
        QString v = (budgetedS == modifiedS ? budgetedS : modifiedS % " <s><small>" % budgetedS % "</small></s>");
        if (nb) {
            ui.kInfo->setText(i18np("Selection: %1 budget for %2", "Selection: %1 budgets for %2", nb, v));
            if (!secondary.Symbol.isEmpty() && secondary.Value) {
                budgetedS = doc->formatMoney(budgeted, secondary);
                modifiedS = doc->formatMoney(modified, secondary);
                v = (budgetedS == modifiedS ? budgetedS : modifiedS % " <s><small>" % budgetedS % "</small></s>");
            }
            ui.kInfo->setToolTip(i18np("Selection: %1 budget for %2", "Selection: %1 budgets for %2", nb, v));
        } else {
            ui.kInfo->setText(i18nc("Noun", "Selection: none"));
            ui.kInfo->setToolTip(i18nc("Noun", "Selection: none"));
        }
    }
}



