/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A plugin to highlight objects
 *
 * @author Stephane MANKOWSKI
 */
#include "skghighlightplugin.h"

#include <KActionCollection>
#include <KStandardAction>
#include <kaboutdata.h>
#include <kpluginfactory.h>
#include <klocalizedstring.h>

#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGHighlightPluginFactory, registerPlugin<SKGHighlightPlugin>();)

SKGHighlightPlugin::SKGHighlightPlugin(QWidget* iWidget, QObject* iParent, const QVariantList& /*iArg*/) :
    SKGInterfacePlugin(iParent), m_currentDocument(NULL)
{
    Q_UNUSED(iWidget);
    SKGTRACEINFUNC(10);
}

SKGHighlightPlugin::~SKGHighlightPlugin()
{
    SKGTRACEINFUNC(10);
    m_currentDocument = NULL;
}

bool SKGHighlightPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
    SKGTRACEINFUNC(10);
    Q_UNUSED(iArgument);

    m_currentDocument = iDocument;

    setComponentName("skg_highlight", title());
    setXMLFile("skg_highlight.rc");

    // ------------
    QStringList tmp;
    m_currentDocument->getDistinctValues("sqlite_master", "name", "type='table' AND sql LIKE '%t_bookmarked%'", tmp);
    QAction* actSwitchHighLight = new QAction(SKGServices::fromTheme("bookmarks"), i18nc("Verb", "Switch highlight"), this);
    connect(actSwitchHighLight, &QAction::triggered, this, &SKGHighlightPlugin::onSwitchHighLight);
    actionCollection()->setDefaultShortcut(actSwitchHighLight, Qt::CTRL + Qt::Key_H);
    registerGlobalAction("edit_switch_highlight", actSwitchHighLight, tmp, 1, -1, 301);

    // ------------
    m_currentDocument->getDistinctValues("sqlite_master", "name", "type='table' AND sql LIKE '%t_close%'", tmp);
    QAction* actSwitchClose = new QAction(SKGServices::fromTheme("dialog-close"), i18nc("Verb", "Switch close"), this);
    connect(actSwitchClose, &QAction::triggered, this, &SKGHighlightPlugin::onSwitchClose);
    registerGlobalAction("edit_switch_close", actSwitchClose, tmp, 1, -1, 301);

    // Create yours actions here
    return true;
}

QString SKGHighlightPlugin::title() const
{
    return toolTip();
}

QString SKGHighlightPlugin::icon() const
{
    return "bookmarks";
}

QString SKGHighlightPlugin::toolTip() const
{
    return i18nc("The tool tip", "Highlight");
}

int SKGHighlightPlugin::getOrder() const
{
    return 6;
}

void SKGHighlightPlugin::onSwitchClose()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    // Get Selection
    if (SKGMainPanel::getMainPanel() && m_currentDocument) {
        SKGObjectBase::SKGListSKGObjectBase selection = SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb = selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*m_currentDocument, i18nc("Noun, name of the user action", "Close"), err, nb);
            for (int i = 0; !err && i < nb; ++i) {
                SKGObjectBase obj(selection.at(i));
                IFOKDO(err, obj.setAttribute("t_close", obj.getAttribute("t_close") == "Y" ? "N" : "Y"))
                IFOKDO(err, obj.save())

                // Send message
                IFOKDO(err, m_currentDocument->sendMessage(i18nc("An information to the user", "The close status of '%1' has been changed", obj.getDisplayName()), SKGDocument::Hidden));

                IFOKDO(err, m_currentDocument->stepForward(i + 1))
            }
        }

        // status bar
        IFOKDO(err, SKGError(0, i18nc("Successful message after an user action", "Closed.")))
        else {
            err.addError(ERR_FAIL, i18nc("Error message",  "Closure failed"));
        }

        // Display error
        SKGMainPanel::displayErrorMessage(err);
    }
}

void SKGHighlightPlugin::onSwitchHighLight()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    // Get Selection
    if (SKGMainPanel::getMainPanel() && m_currentDocument) {
        SKGObjectBase::SKGListSKGObjectBase selection = SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb = selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*m_currentDocument, i18nc("Noun, name of the user action", "Highlight"), err, nb);
            for (int i = 0; !err && i < nb; ++i) {
                SKGObjectBase obj(selection.at(i));
                IFOKDO(err, obj.setAttribute("t_bookmarked", obj.getAttribute("t_bookmarked") == "Y" ? "N" : "Y"))
                IFOKDO(err, obj.save())

                // Send message
                IFOKDO(err, m_currentDocument->sendMessage(i18nc("An information to the user", "The highlight status of '%1' has been changed", obj.getDisplayName()), SKGDocument::Hidden));

                IFOKDO(err, m_currentDocument->stepForward(i + 1))
            }
        }

        // status bar
        IFOKDO(err, SKGError(0, i18nc("Successful message after an user action", "Highlighted.")))
        else {
            err.addError(ERR_FAIL, i18nc("Error message",  "Highlight failed"));
        }

        // Display error
        SKGMainPanel::displayErrorMessage(err);
    }
}

#include <skghighlightplugin.moc>
